package org.tip.puck.spacetime.workers;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

import org.tip.puck.PuckException;
import org.tip.puck.geo.GeoLevel;
import org.tip.puck.geo.Geography;
import org.tip.puck.geo.Place;
import org.tip.puck.graphs.Graph;
import org.tip.puck.graphs.GraphComparatorByArcCount;
import org.tip.puck.graphs.GraphProfile;
import org.tip.puck.graphs.Link;
import org.tip.puck.graphs.Node;
import org.tip.puck.graphs.workers.GraphUtils;
import org.tip.puck.net.Individual;
import org.tip.puck.net.Individuals;
import org.tip.puck.net.KinType;
import org.tip.puck.net.relations.Actor;
import org.tip.puck.net.relations.Relation;
import org.tip.puck.net.relations.RelationModel;
import org.tip.puck.net.relations.Relations;
import org.tip.puck.net.relations.Role;
import org.tip.puck.net.relations.workers.RelationWorker;
import org.tip.puck.net.workers.IndividualValuator;
import org.tip.puck.net.workers.NetUtils;
import org.tip.puck.partitions.Cluster;
import org.tip.puck.partitions.Partition;
import org.tip.puck.spacetime.Sequence;
import org.tip.puck.spacetime.SequenceProfile;
import org.tip.puck.spacetime.workers.SpaceTimeCriteria.RelationClassificationType;
import org.tip.puck.util.MathUtils;
import org.tip.puck.util.PuckUtils;
import org.tip.puck.util.ToolBox;
import org.tip.puck.util.Trafo;
import org.tip.puck.util.Value;

import com.mysql.jdbc.StringUtils;

import fr.devinsy.util.StringList;

/**
 * 
 * @author Klaus Hamberger
 *
 */
public class SequenceCensus {
	
	private Individual ego;
	private List<Relation> events;
	
	private String egoRoleName;
	private int[] maxDepths;
	private GeoLevel level;
	private List<String> placeNames;
	private List<String> relationModelNames; 
	private List<String> roleNames; 
	private List<RelationClassificationType> types; 
//	private String alterRoleName; // for ego networks (temp)
	private List<RelationClassificationType>  mainEventTypes; // for parcours morphology and parcours intersection norks
	private String relationModelName;
	private Geography geography;
	private List<Place> placeList;
	private String chainClassification;
	private String alterAttributeLabel; // for parcours intersection norks
	private String alterAttributeValue; // for parcours intersection norks
	
	private int length;
	private int threshold; // child age <= threshold
	
	private Map<String,Set<Individual>> altersByRoles;
	private Map<Individual,List<String>> rolesByAlter;
	private Map<Individual,List<String>> relationsByAlter;
	private Map<String,List<String>> relationsByRoles;
	private Map<Individual,List<Relation>> eventsByAlter;
	private Map<Integer,List<Individual>> coverage;
	private Map<RelationClassificationType,Map<Relation,String>> eventEvaluations;
	private Map<String,Double> specificDensities;
	private List<String> impersonalRelations;
	
	private Map<String, GraphProfile<Individual>> networkProfiles;
	private Map<String, GraphProfile<Cluster<Relation>>> eventTypeNetworkProfiles;
	private Map<String, GraphProfile<Relation>> eventNetworkProfiles;
	
	private Map<String,List<String>> partitionLabels;

	private Map<RelationClassificationType,Graph<Set<Individual>>> parcoursSimilarityTrees;

	private Map<RelationClassificationType,SequenceProfile> sequenceProfiles;
	private String pattern;
	
	private String impersonalLabel = "NOHOST";
	private String dateLabel;
	
	
	public SequenceCensus (Sequence source, SpaceTimeCriteria criteria){
		
		this.ego = source.getEgo();
		this.events = new ArrayList<Relation>();
		this.length = 0;
		for (Relation event : source.getEvents().values()){
			events.add(event);
			if (!isBirth(event) && !isDeath(event)){
				length++;
			}
		}		
		this.egoRoleName = criteria.getEgoRoleName();
		this.pattern = criteria.getPattern();
		this.maxDepths = ToolBox.stringsToInts(criteria.getPattern());
		this.level = criteria.getLevel();
		this.partitionLabels = criteria.getPartitionLabels();
		this.relationModelName = criteria.getRelationModelName();
		this.relationModelNames = criteria.getRelationModelNames(); 
//		this.alterRoleName = criteria.getAlterFilterRoleName();
		this.geography = criteria.getGeography();
		this.roleNames = criteria.getRoleNames();
		this.types = criteria.getTypes();
		types.add(RelationClassificationType.DATE);
		types.add(RelationClassificationType.AGE);
		
		this.chainClassification = criteria.getChainClassification();
		this.alterAttributeLabel = criteria.getAlterFilterAttributeLabel();
		this.alterAttributeValue = criteria.getAlterFilterAttributeValue();
		this.mainEventTypes = criteria.getMainRelationClassificationTypes();
		this.placeNames = criteria.getMinimalPlaceNames();
		this.threshold = 20; // INTEGRER DANS MASQUE!
		this.dateLabel = criteria.getDateLabel();
		
		if (mainEventTypes.contains(RelationClassificationType.TREES)){
			for (Relation event : events){
				event.updateReferents(criteria.getDefaultReferentRoleName());
			}
		}
		
		if ((types.contains(RelationClassificationType.MIG) || types.contains(RelationClassificationType.HOSTMIG) || types.contains(RelationClassificationType.MIGRATIONTYPE)|| types.contains(RelationClassificationType.CHILDMIGRATIONTYPE)) && (!roleNames.contains("MIG"))){
			roleNames.add("MIG");
		}
		if ((types.contains(RelationClassificationType.HOST) || types.contains(RelationClassificationType.HOSTMIG) || types.contains(RelationClassificationType.MIGRATIONTYPE) || types.contains(RelationClassificationType.CHILDMIGRATIONTYPE)) && (!roleNames.contains("HOST"))){
			roleNames.add("HOST");
		}
				
		this.sequenceProfiles = new HashMap<RelationClassificationType,SequenceProfile>();
		this.networkProfiles = new HashMap<String,GraphProfile<Individual>>();
		this.eventNetworkProfiles = new HashMap<String,GraphProfile<Relation>>();
		this.eventTypeNetworkProfiles = new HashMap<String,GraphProfile<Cluster<Relation>>>();
		
		setAlters();
		setAlterRelations("ALL");
		evaluateEvents();
		
		for (String networkTitle : criteria.getNetworkTitles()){
			if (networkTitle.contains("Ego Network")){
				setEgoNetworks();
			} else if (networkTitle.contains("Parcours Network")){
				setParcoursNetworks();
			} else if (networkTitle.contains("Parcours Intersection Network")){
				setParcoursIntersectionNetworks();
			} else if (networkTitle.contains("Parcours Similarity Network")){
				setParcoursSimilarityNetworks();
			}
		}
		
		
	}
	
	private void setEgoNetworks(){

		networkProfiles.put("Ego Network", new GraphProfile<Individual>(getEgoNetwork(),ego));
		networkProfiles.put("Nonmediated Ego Network", new GraphProfile<Individual>(getNonMediatedEgoNetwork(),ego));
		
		setSpecificDensities("Ego Network");
		setComponentTypes("Nonmediated Ego Network");

	}
	
	private List<String> getNetworkRelations(Graph<Individual> graph){
		List<String> result;
		
		result = new ArrayList<String>();
		for (Node<Individual> node : graph.getNodes().toListSortedById()){
			Individual referent = node.getReferent();
			for (String relation : relationsByAlter.get(referent)){
				if (!result.contains(relation)){
					result.add(relation);
				}
			}
		}
		Collections.sort(result);
		
		//
		return result;

	}
	
	private List<String[]> getConnectedRelations(Partition<Node<Individual>> partition){
		List<String[]> result;
		
		result = new ArrayList<String[]>();
		
		for (Cluster<Node<Individual>> cluster : partition.getClusters()){
			List<String> relations = new ArrayList<String>();
			for (Node<Individual> node : cluster.getItems()){
				Individual referent = node.getReferent();
				for (String relation : relationsByAlter.get(referent)){
					if (!relations.contains(relation)){
						relations.add(relation);
					}
				}
			}
			Collections.sort(relations);
			for (String firstRelation : relations){
				for (String secondRelation : relations){
					if (!firstRelation.equals(secondRelation)){
						String[] pair = new String[]{firstRelation,secondRelation};
						if (!result.contains(pair)){
							result.add(pair);
						}
					}
				}
			}
		}
		
		//
		return result;
	}
	
	private String getComponentType(Cluster<Node<Individual>> component){
		String result;
		
		result = null;
		
		List<String> relations = new ArrayList<String>();
		for (Node<Individual> node : component.getItems()){
			Individual referent = node.getReferent();
			relations.addAll(relationsByAlter.get(referent));
		}
		if (PuckUtils.containsStrings(relations, "FATHER;PATERNAL_HOME") && PuckUtils.containsStrings(relations,"MOTHER")){
			result = "PARENT";
		} else if (PuckUtils.containsStrings(relations,"FATHER;PATERNAL_HOME")){
			result = "FATHER";
		} else if (PuckUtils.containsStrings(relations,"MOTHER;MATERNAL_HOME")){
			result = "MOTHER";
		} else if (PuckUtils.containsStrings(relations,"SPOUSE;MARITAL_HOME")){
			result = "SPOUSE";
		} else if (PuckUtils.containsStrings(relations,"RELATIVE;RELATIVE_AGNATIC;RELATIVE_UTERINE;RELATIVE_COGNATIC")){
			result = "RELATIVE";
		} else if (PuckUtils.containsStrings(relations, "AFFINE")){
			result = "AFFINE";
		} else if (PuckUtils.containsStrings(relations, "EMPLOYER")){
			result = "EMPLOYER";
		}
		if (result == null) {
			List<String> list = new ArrayList<String>();
			for (String relationModelName : relationModelNames){
				if (ego.relations().getByModelName(relationModelName).size()>0){
					RelationModel model = ego.relations().getByModelName(relationModelName).getFirst().getModel();
					for (Role role : model.roles()){
						if (relations.contains(role.getName()) && !list.contains(role.getName())){
							list.add(role.getName());
						}
					}
				}
			}
			Collections.sort(list);
			if (list.size()>0){
				result = "";
				for (String item : list){
					if (result.length()>0){
						result+="-";
					}
					result+=item;
				}
			}
		}
		if (result == null) {
			if (relations.size()>0){
				result = "OTHER";
			} else {
				result = "UNKNOWN";
			}
		}
		//
		return result;
	}
	
	private void setParcoursNetworks(){
		
		for (RelationClassificationType relationClassificationType : mainEventTypes){

			eventTypeNetworkProfiles.put("Parcours Network_"+relationClassificationType, new GraphProfile<Cluster<Relation>>(getParcoursNetwork(relationClassificationType)));
			
			if (partitionLabels.get("Parcours Network_"+relationClassificationType).contains("COMPONENT")){
				setComponentTypes("Parcours Network_"+relationClassificationType);
			}
		}
		
		eventNetworkProfiles.put("Parcours", new GraphProfile<Relation>(getParcoursGraph()));
		eventNetworkProfiles.put("Extended Parcours", new GraphProfile<Relation>(getExtendedParcoursGraph()));
		eventNetworkProfiles.put("Multiple Parcours", new GraphProfile<Relation>(getMultipleParcoursGraph()));
		


		
	}
	
	private void setParcoursIntersectionNetworks(){
		
		networkProfiles.put("Parcours Intersection Network", new GraphProfile<Individual>(getParcoursIntersectionNetwork(),ego));
			
		if (partitionLabels.get("Parcours Intersection Network").contains("COMPONENT")){
			setComponentTypes("Parcours Intersection Network");
		}
	}
	
	private void setParcoursSimilarityNetworks(){
		
		parcoursSimilarityTrees = new HashMap<RelationClassificationType,Graph<Set<Individual>>>();
		
		for (RelationClassificationType relationClassificationType : mainEventTypes){

			networkProfiles.put("Parcours Similarity Network_"+relationClassificationType, new GraphProfile<Individual>(getParcoursSimilarityNetwork(relationClassificationType),ego));
			
			if (partitionLabels.get("Parcours Similarity Network_"+relationClassificationType).contains("COMPONENT")){
				setComponentTypes("Parcours Similarity Network_"+relationClassificationType);
			}
		}
		
	}
	
/*	private void makeAlterRelationPartition (Graph<Individual> network){
		
		for (Node<Individual> node : network.getNodes()){
			List<String> relations = relationsByAlter.get(node.getReferent());
			if (node.getReferent()==ego){
				node.setAttribute("EGO-RELATION", "EGO");
			} else if (relations != null){
				node.setAttribute("EGO-RELATION", relations.toString());
			}
		}
	}*/

	private <E> void makeAlterRelationPartition (Graph<E> network){
		
		for (Node<E> node : network.getNodes()){
			E referent = node.getReferent();
			Individual alter = null;
			if (referent instanceof Individual){
				alter = (Individual)referent;
			} else if (referent instanceof Set<?>){
				if (((Set) referent).size()==1){
					for (Individual element : (Set<Individual>)referent){
						alter = element;
					}
				}
			} 
			
			if (alter!=null){
				List<String> relations = relationsByAlter.get(alter);
				if (alter==ego){
					node.setAttribute("EGO-RELATION", "EGO");
				} else if (relations != null){
					node.setAttribute("EGO-RELATION", relations.toString());
				}
			}
		}
	}
	
	void writePajekNetwork (StringList pajekBuffer, String title) throws PuckException{

		if (title.contains("Parcours Similarity Tree") || title.contains("Parcours Network Fused")){
/*			RelationClassificationType relationClassificationType = RelationClassificationType.valueOf(title.substring(title.lastIndexOf("_")+1));
			Graph<Set<Individual>> network = parcoursSimilarityTrees.get(relationClassificationType);
			makeAlterRelationPartition(network);
			GraphUtils.getDepthPartition(network);
			GraphUtils.addNodeLabelsFromPartition(network, "EGO-RELATION");
			pajekBuffer.addAll(PuckUtils.writePajekNetwork(network,partitionLabels.get(title)));*/
		} else if (title.contains("Parcours Network") && !title.contains("Fused")){
			Graph<Cluster<Relation>> network = eventTypeNetworkProfiles.get(title).getGraph();
			if (network.nodeCount()>0){
				pajekBuffer.addAll(PuckUtils.writePajekNetwork(network,partitionLabels.get(title)));
			}
		} else if (title.contains("Network")){
			Graph<Individual> network = networkProfiles.get(title).getGraph();
			makeAlterRelationPartition(network);
			GraphUtils.addNodeLabelsFromPartition(network, "EGO-RELATION");
			pajekBuffer.addAll(PuckUtils.writePajekNetwork(network,partitionLabels.get(title)));
		} else if (title.contains("Parcours")){
			Graph<Relation> network = eventNetworkProfiles.get(title).getGraph();
			if (network.nodeCount()>0){
				pajekBuffer.addAll(PuckUtils.writePajekNetwork(network,partitionLabels.get(title)));
			}
		}
	}
	
	public Set<Individual> getAllAlters(){
		return altersByRoles.get("ALL");
	}
	
	public Map<Individual, List<String>> getRelationsByAlter() {
		return relationsByAlter;
	}

	private void setAlters() {
		
		altersByRoles = new HashMap<String,Set<Individual>>();
		altersByRoles.put("ALL", new HashSet<Individual>());
		altersByRoles.put("SELECTED", new HashSet<Individual>());
		for (String alterRoleName : roleNames){
			altersByRoles.put(alterRoleName, new HashSet<Individual>());
		}
		rolesByAlter = new HashMap<Individual, List<String>>();
		eventsByAlter = new HashMap<Individual, List<Relation>>();
		coverage = new TreeMap<Integer,List<Individual>>();
		
		//
		for (Relation event : events){
			if (event.hasRole(ego, egoRoleName)){
				// Add parents as birth actors
				if (isBirth(event)){
					for (Individual parent : ego.getParents()){
						if (!event.getIndividuals().contains(parent)){
							event.actors().add(new Actor(parent,new Role("OTHER")));
						}
					}
				}
				for (Actor actor : event.actors()){
					Individual alter = actor.getIndividual();
					String roleName = actor.getRole().getName();
					if (!roleName.equals(egoRoleName) || alter!=ego){
						altersByRoles.get("ALL").add(alter);
						if (!altersByRoles.containsKey(roleName)){
							altersByRoles.put(roleName, new HashSet<Individual>());
						}
						altersByRoles.get(actor.getRole().getName()).add(alter);
						if (roleNames.contains(roleName)){
							altersByRoles.get("SELECTED").add(alter);
						}
						List<String> roleNameList = rolesByAlter.get(alter);
						if (roleNameList == null){
							roleNameList = new ArrayList<String>();
							rolesByAlter.put(alter, roleNameList);
						}
						roleNameList.add(roleName);
						List<Relation> eventList = eventsByAlter.get(alter);
						if (eventList == null){
							eventList = new ArrayList<Relation>();
							eventsByAlter.put(alter, eventList);
						}
						if (!eventList.contains(event)){
							eventList.add(event);
						}
					}
				}
			}
		}
		for (Individual alter : eventsByAlter.keySet()){
			Integer cov = eventsByAlter.get(alter).size();
			List<Individual> list = coverage.get(cov);
			if (list == null){
				list = new ArrayList<Individual>();
				coverage.put(cov, list);
			}
			if (!list.contains(alter)){
				list.add(alter);
			}
		}
		
	}
	
	private void setAlterRelations (String alterRoleName){
		
		//
		relationsByAlter = NetUtils.getAlterRelations1(ego,altersByRoles.get(alterRoleName),maxDepths,relationModelNames, chainClassification,null,null);
		
		relationsByRoles = new HashMap<String,List<String>>();
		for (String roleName : altersByRoles.keySet()){
			List<String> relations = new ArrayList<String>();
			relationsByRoles.put(roleName, relations);
			for (Individual alter : altersByRoles.get(roleName)){
				if (relationsByAlter.get(alter).size()==0){
					relations.add("UNKNOWN");
//					System.err.println("Unknown relation of "+ego+" to "+alterRoleName+" "+alter);
				} else {
					for (String relation : relationsByAlter.get(alter)){
						if (!relations.contains(relation)){
							relations.add(relation);
						}
					}
				}
			}
		}

		// Add impersonal relations
		impersonalRelations = new ArrayList<String>();
		for (Relation event : events){
			String impersonalRelation = impersonalRelation(event);
			if (impersonalRelation!=null){
				impersonalRelations.add(impersonalRelation);
			}
		}

		// Add other kin types (for orks)
		for (KinType kinType : KinType.values()){
			for (Individual alter : ego.getKin(kinType)){
				if (relationsByAlter.get(alter)==null){
					List<String> list = new ArrayList<String>();
					list.add(kinType.toString());
					relationsByAlter.put(alter, list);
				}
			}
		}
	}
	
	private String impersonalRelation(Relation event){
		String result;
		
		result = null;
		Actor egoActor = event.actors().get(ego.getId(), egoRoleName);
		if (egoActor!=null && egoActor.attributes()!=null){
			result = egoActor.getAttributeValue(impersonalLabel);
		}
		//
		return result;

	}
	
	private void setPlaceList(){
		
		//
		placeList = new ArrayList<Place>();
		
		for (Relation relation : events){
			Place place = geography.getPlace(relation.getAttributeValue("END_PLACE"),level);
			if (place==null){
				place = new Place(level,"UNKNOWN");
			}
			placeList.add(place);
		}
	}
	
	private Relation getPreviousEvent(Relation event){
		Relation result;
		
		int index = events.indexOf(event)-1;
		
		if (index>-1){
			result = events.get(index);
		} else {
			result = null;
		}
		//
		return result;
	}
	
	private Double getTurnover (Relation event){
		Double result;
		
		result = 0.;
		
		Relation previousEvent = getPreviousEvent(event);
		
		if (previousEvent!=null){
			
			Individuals previousAlters = previousEvent.getIndividuals(roleNames);
			Individuals currentAlters = event.getIndividuals(roleNames);

			Double n = new Double(previousAlters.size());
			
			for (Individual individual: previousAlters){
				if (currentAlters.contains(individual)){
					result++;
				}
			}
			
			result = result/n;
			
		}
		
		//
		return result;
		
	}
	
	private GeoLevel getMaxDistance(){
		GeoLevel result;
		
		result = null;
		
		for (Relation event : events){
			if (!isBirth(event) && !isDeath(event)){
				String distance = eventEvaluations.get(RelationClassificationType.DISTANCE).get(event);
				if (!StringUtils.isNullOrEmpty(distance) && !distance.equals("UNDEFINED")){
					if (result == null || GeoLevel.valueOf(distance).compareTo(result)<0){
						result = GeoLevel.valueOf(distance);
					}
				}
			}
		}
				
		//
		return result;
	}
	
	private List<String> getRelationTypes (Individuals alters, String roleName){
		List<String> result;
		
		result = new ArrayList<String>();
		for (Individual alter : alters){
			List<String> relations = relationsByAlter.get(alter);
			if (relations!=null){
				for (String relation : relations){
					if (!result.contains(relation)){
						result.add(relation);
					}
				}
			}
		}
		//
		Collections.sort(result);
		
		//
		return result;
	}
	
	private List<String> getRelationTypes (Relation event, String roleName){
		List<String> result;
		
		Individuals alters = event.getIndividuals(roleName);
		if (roleName.equals("MIG")){
			alters.removeById(ego.getId());
		}

		result = getRelationTypes(alters, roleName);

		if (alters.size()==0){
			Actor egoActor = event.actors().get(ego.getId(), egoRoleName);
			
			if (egoActor.attributes()!=null && egoActor.getAttributeValue(impersonalLabel)!=null){
				result.add(egoActor.getAttributeValue(impersonalLabel));
			} else {
				for (Actor alterActor : event.actors().getByRole(egoRoleName)){
					if (alterActor!=null && !alterActor.equals(egoActor) && alterActor.attributes()!=null){
						String relation = alterActor.getAttributeValue(impersonalLabel);
						if (relation!=null){
							List<String> relations = relationsByAlter.get(alterActor.getIndividual());
							if (relations!=null){
								for (String egoRelation : relations){
									egoRelation+="S_"+relation;
									if (!result.contains(egoRelation)){
										result.add(egoRelation);
									}
								}
							}
						}
					}
				}
			}
		}  

		//
		return result;
	}
	
	private String getChildMigrationType (Relation event) {
		String result;
		
		List<String> hosts = getRelationTypes(event,"HOST");
		List<String> migs = getRelationTypes(event,"MIG");
		
		result = null;
		
		if (isBirth(event)){
			result = "BIRTH";
		} else if (getAge(event)<=threshold){
			if (!hosts.contains("FATHER") && !hosts.contains("MOTHER") && !migs.contains("FATHER") && !migs.contains("MOTHER")){
				result = "NOPARENTS";
			}
		}
		//
		return result;
	}
	
	private String getHostType (Relation event){
		String result;
		
		List<String> hosts = new ArrayList<String>();
		List<String> allhosts = reduceRelations(getRelationTypes(event,"HOST"),"HOST");

		for (String host : allhosts){
			boolean consider = true;
			if (host.contains("FRIEND")){
				for (String otherhost: allhosts){
					if (!otherhost.contains("FRIEND")){
						consider = false;
						break;
					}
				}
				if (!consider){
					continue;
				}
			}
			hosts.add(host);
		}

		if (isBirth(event)){
			result = "PARENTS";
		} else if (isDeath(event)){
			result = "DEATH";
		} else if (hosts.size()==0){
			result = "UNKNOWN";
			System.err.println(result+" "+ego+" "+event);
		} else if (hosts.contains("TRANSITION")){
			result = "TRANSITION";
		} else if (PuckUtils.containsStrings(hosts, "FATHER OR PATERNAL HOME;MOTHER OR MATERNAL HOME")){
			result = "PARENTS";
		} else if (PuckUtils.containsStrings(hosts, "SIBLING OR FRATERNAL HOME")){
			result = "RELATIVE";
		} else if (PuckUtils.containsStrings(hosts, "SPOUSE OR MARITAL HOME")){
			result = "SPOUSE";
		} else if (PuckUtils.containsStrings(hosts, "CHILD OR FILIAL HOME")){
			result = "CHILD";
		} else if (PuckUtils.containsStrings(hosts, "RELATIVE;RELATIVE_AGNATIC;RELATIVE_UTERINE;RELATIVE_COGNATIC;RELATIVE_OR_AFFINE")){
			result = "RELATIVE";
		} else if (hosts.contains("AFFINE")){
			result = "AFFINE";
		} else if (hosts.contains("LANDLORD")){
			result = "RENT";
		} else if (hosts.contains("UNRELATED")){
			result = "UNRELATED";
		} else if (hosts.contains("HOTEL-HOSTEL") || (hosts.contains("WITHOUT FIXED DOMICILE"))){
			result = "PUBLIC";
		} else {
			result = "";
			for (String host : hosts){
				host = host.replaceAll("RELATIVE_OR_AFFINE","RELATIVE").replaceAll("SIBLING", "RELATIVE").replaceAll("FATHER", "PARENTS").replaceAll("MOTHER", "PARENTS");
				if (host.equals("RELATIVES_MARITAL HOME")){
					result = "AFFINE";
				} else if (host.equals("SPOUSES_PATERNAL HOME")){
					result = "SPOUSE";
				} else if (host.equals("SPOUSES_MARITAL HOME")){
					result = "PARENTS";
				} else {
					String[] hostParts = host.split("S_");
					if (hostParts.length>1){
						host = "via "+hostParts[0];
					}
					result += host+" ";
				}
				// simplify
				if (result.contains("via PARENTS")){
					result = "via PARENTS";
				} else if (result.contains("via SPOUSE")){
					result = "via SPOUSE";
				} else if (result.contains("via RELATIVE")){
					result = "via RELATIVE";
				} else if (result.contains("via MASTER")){
					result = "via MASTER";
				} else if (result.lastIndexOf("via")>0){
					System.err.println("double reference "+result);
				}
			}
			if (result.contains("null")){
				result = "UNKNOWN";
				System.err.println(result+" "+ego+" "+event);
			}
		}
		if (result.contains("LODGER")){
			System.err.println("Inverted landlord-lodger relation for "+ego+" "+event);
		}
		//
		result = result.trim();
		
		//
		return result;
		
	}
	
	private String getMigrationType (Relation event){
		String result;
		
		List<String> hosts = getRelationTypes(event,"HOST");
		List<String> migs = getRelationTypes(event,"MIG");
		
		if (isBirth(event)){
			result = "BIRTH";
		} else if (isDeath(event)){
			result = "DEATH";
		} else {
			
			
			
			
			
			String hostType = "OTHER";
			if (hosts.size()==0){
				hostType = "NOBODY";
			} else {
				String[] types = new String[]{"EGO","FATHER","MOTHER","SPOUSE","CHILD","SIBLING"};
				boolean end = false;
				for (String type : types){
					if (hosts.contains(type)){
						hostType = type;
						end = true;
						break;
					}
				}
				if (!end && (hosts.contains("RELATIVE_AGNATIC") || hosts.contains("RELATIVE_UTERINE") || hosts.contains("RELATIVE_COGNATIC")|| hosts.contains("RELATIVE"))){
					hostType = "KIN";
				} else if (!end && hosts.contains("AFFINE")){
					hostType = "AFFINE";
				}
			}
			
			String migType = "OTHER";
			if (migs.size()==0){
				migType = "NOBODY";
			} else {
				String[] types = new String[]{"EGO","FATHER","MOTHER","SPOUSE","CHILD","SIBLING"};
				boolean end = false;
				for (String type : types){
					if (migs.contains(type)){
						migType = type;
						end = true;
						break;
					}
				}
				if (!end && (migs.contains("RELATIVE_AGNATIC") || migs.contains("RELATIVE_UTERINE") || migs.contains("RELATIVE_COGNATIC") || migs.contains("RELATIVE"))){
					migType = "KIN";
				} else if (!end && migs.contains("AFFINE")){
					migType = "AFFINE";
				}

			}
			result = "WITH_"+migType+"_TO_"+hostType;
			result = result.replaceAll("WITH_NOBODY", "ALONE");
			
		}
		
		//
		return result;
	}
	
/*	public void putEvents (Partition<String> eventPartition, EventType eventType){
		
		for (Relation event : events){
			String type = getEventType(eventType,event);
			eventPartition.put(ego.getId()+" "+event.getTypedId(), Value.valueOf(type));
		}
	}
	
	public void putEventPairs (Partition<String> eventPairPartition, EventType eventType){
		
		String previousType = null;
		for (Relation event : events){
			if (previousType == null){
				previousType = getEventType(eventType,event);
			} else {
				String nextType = getEventType(eventType,event);
				eventPairPartition.put(ego.getId()+" "+event.getTypedId(), new Value(new String[]{previousType,nextType}));
				previousType = nextType;
			}
		}
	}

	
	public void putSubSequences (Partition<String> sequencePartition, EventType eventType){
		
		String type = null;
		for (Relation event :events){
			if (type == null){
				type = getEventType(eventType,event);
			} else {
				type += "-"+getEventType(eventType,event);
			}
			if (type!=null){
				sequencePartition.put(ego.getId()+" "+event.getTypedId(), new Value(type));
			}
		}
	}*/
	
/*	public Relation getEvent(int i){
		Relation result;
		
		if (events!=null && events.size()>i){
			result = events.get(i);
		} else {
			result = null;
		}
		//
		return result;
	}*/
	
	public static boolean isBirth (Relation event) {
		return event.getAttributeValue("END_PLACE")!=null && event.getAttributeValue("START_PLACE")==null;
	}
	
	public static boolean isDeath (Relation event) {
		return event.getAttributeValue("END_PLACE")==null && event.getAttributeValue("START_PLACE")!=null;
	}
	
	private Relation getFirstMigration (){
		Relation result;

		result = null;
		for (Relation event : events){
			if (!isBirth(event)){
				if (!isDeath(event)){
					result = event;
				}
				break;
			}
		}
		
		//
		return result;
		
	}
	
	private Relation getFirstEvent(String label, String value){
		Relation result;
		
		Map<Relation, String> eventType = eventEvaluations.get(RelationClassificationType.valueOf(label));
		
		result = null;
		if (eventType != null){
			for (Relation event : events){
				if (value.equals(eventType.get(event))){
					result = event;
					break;
				}
			}
		}
		//
		return result;
	}
	
/*	public String getEventType (EventType type, Relation event){
		String result;
		
		if (type==null || event == null){
			result = null;
		} else {
			result = eventTypes.get(type).get(event);
		}
		return result;
	}
	
	public Set<String> getEventTypes (EventType type){
		Set<String> result;
		
		result = new HashSet<String>();
		for (Relation event : events){
			result.add(getEventType(type,event));
		}
		
		//
		return result;
	}*/
	
	private String getRelationTypesAsShortCutString (Relation event, String roleName){
		String result;
		
		result = Trafo.asShortCutString(getRelationTypes(event, roleName), 2);
		
		if (result == null && roleName.equals("MIG")){
			result = "SG";
		}
		
		//
		return result;
	}

	/**
	 * Generalize within RelationValuator
	 */
	private void evaluateEvents (){
		
		eventEvaluations = new HashMap<RelationClassificationType,Map<Relation,String>>();
		
		for (RelationClassificationType type : types){
			
			eventEvaluations.put(type, new HashMap<Relation,String>());
			
			if (altersByRoles==null && (type==RelationClassificationType.HOST || type==RelationClassificationType.MIG || type==RelationClassificationType.HOSTMIG || type==RelationClassificationType.MIGRATIONTYPE)){
//				setAlters();
//				setAlterRelations("ALL");
			} else	if (type==RelationClassificationType.MOVEMENT){
				setPlaceList();
			} 
			
			int i = 0;
			for (Relation event : events){
				String eventType = null;
					switch (type){
					case TYPEDID:
						eventType = event.getTypedId()+"";
						break;
					case HOST: 
						eventType = getHostType(event);
						break;
					case MIG:					
						eventType = getRelationTypesAsShortCutString(event,"MIG");
						break;
					case HOSTMIG:
						eventType = getRelationTypesAsShortCutString(event,"HOST")+":"+getRelationTypesAsShortCutString(event,"MIG");
						break;
					case MIGRATIONTYPE:
						eventType = getMigrationType(event);
						break;
					case CHILDMIGRATIONTYPE:
						eventType = getChildMigrationType(event);
						break;
					case TREES:
						eventType = RelationWorker.getLinkTrees(event, true, "GENDER", pattern);
						break;
					case DISTANCE: 
						
						if (isBirth(event)){
							eventType = "BIRTH";
						} else if (isDeath(event)){
							eventType = "DEATH";
						} else {
							GeoLevel distance = getDistance(event);
							if (distance != null){
								eventType = distance.dynamic().toString();
							}
						}
						break;
						
					case PLACE: 
						
						if (isDeath(event)){
							eventType = "DEATH";
						} else {
							Place place = geography.get(event.getAttributeValue("END_PLACE"),level);
							if (place!=null){
								eventType = place.getToponym();
							} 
						}

						break;
						
					case REGION: 
						Place region = geography.getSup(event.getAttributeValue("END_PLACE"),placeNames);
						if (region!=null){
							eventType = region.getName();
						}
						break;
						
					case MOVEMENT:
						eventType = SequenceWorker.getMovement(placeList, i);	
						break;
						
					case TURNOVER:
						eventType = MathUtils.round(getTurnover(event), 2)+"";
						break;
						
					case DATE:
						eventType = event.getAttributeValue(dateLabel);
						break;
						
					case AGE:
						eventType = getAge(event)+"";
						break;
						
					default:
						eventType = null;
					}
					
				if (eventType==null){
					System.err.println("Event type "+type+" is null for event "+event);
					eventType = "UNDEFINED";
				}
					
				eventEvaluations.get(type).put(event, eventType);
				i++;
			}
		}
	}
	
	private GeoLevel getDistance (Relation event){
		GeoLevel result;
		
		result = null;
		
		if (!isBirth(event) && !isDeath(event)){
			Place start = geography.getByHomonym(event.getAttributeValue("START_PLACE"));
			Place end = geography.getByHomonym(event.getAttributeValue("END_PLACE"));
			if (start != null && end != null){
				Place commonAncestor = geography.getCommonAncestor(start, end);
				if (commonAncestor!=null){
					result = commonAncestor.getLevel();
				}
			}
		}
		
		//
		return result;
	}
	
	public Map<GeoLevel, Integer> getDistanceProfile (){
		Map<GeoLevel, Integer> result;
		
		result = new HashMap<GeoLevel, Integer>();
		
		for (Relation event : events){
			GeoLevel distance = getDistance(event);
			if (distance!=null){
				distance = distance.dynamic();
				if (result.containsKey(distance)){
					result.put(distance, result.get(distance)+1);
				} else {
					result.put(distance, 1);
				}
			}
		}
		//
		return result;
	}

	
	public Graph getExtendedParcoursGraph () {
		Graph result;
		
		result = new Graph<Relation>();
		result.setLabel("Extended Parcours "+ego);
		
		Relation lastEvent = null;
		for (Relation event : events){
			if (lastEvent != null){
				Link<Relation> eventLink = result.addArc(lastEvent, event,2);
				eventLink.setTag(ego.getId()+"");
				for (String label : roleNames){
					for (Individual alter : event.getIndividuals(label)){
						if (label!=egoRoleName || alter!=ego){
							Link alterLink = result.addArc(event, alter, 1);
							alterLink.setTag(label);
						}
					}
				}
			}
			lastEvent = event;
		}
		
		if (events.size()==1){
			result.addNode(lastEvent);
		}
		
		for (String label : roleNames){
			Set<Individual> cluster = altersByRoles.get(label);
			if (cluster!=null){
				for (Individual alter : cluster){
					for (Relation event : alter.relations().getByModel(lastEvent.getModel())){
						if (!events.contains(event)){
							if(!isBirth(event) && !isDeath(event) && Sequence.getYear(event)!=null && Sequence.getYear(lastEvent)!=null && Sequence.getYear(event)<=Sequence.getYear(lastEvent)){
								for (Individual tertius : event.getIndividuals()){
									if (tertius.getId()> alter.getId() && result.getNode(tertius)!=null){
										Link link1 = result.addArc(event, tertius, 1);
										link1.setTag(event.getRoleNamesAsString(tertius));
										Link link2 = result.addArc(event, alter, 1);
										link2.setTag(event.getRoleNamesAsString(alter));
									}
								}
							}
						}
					}
				}
			}
		}
		
		int order = 1;
		for (Object obj : result.getNodes()){
			Node node = (Node)obj;
			
			if (node.getReferent() instanceof Individual){
				
				node.setAttribute("TYPE", "INDIVIDUAL");

				for (RelationClassificationType type : types){
					
					List<String> list = null;
					if (type==RelationClassificationType.HOST || type==RelationClassificationType.MIG || type==RelationClassificationType.HOSTMIG){
						list = relationsByAlter.get((Individual)node.getReferent());
					} else {
						list = rolesByAlter.get((Individual)node.getReferent());
					}
					if (list!=null){
						node.setAttribute(type.toString(), list.toString());
					}
				}
				
				node.setAttribute("ORDER", "0");
				
			} else if (node.getReferent() instanceof Relation){
				
				node.setAttribute("TYPE", "EVENT");
				
				for (RelationClassificationType type : types){
					
					String value = eventEvaluations.get(type).get((Relation)node.getReferent());
					if (value!=null){
						node.setAttribute(type.toString(), value);
					}
					
				}
				
				node.setAttribute("ORDER", order+"");
				order++;
			}
		}
		
		//
		return result;
	}
	
	private List<String> reduceRelations (List<String> relations, String alterRole){
		List<String> result;
		
		result = new ArrayList<String>();
		
		for (String relation : relations){
			String reducedRelation = null;
			if (relation.equals("EGO")){
				reducedRelation = "OWN HOME";
			} else if ((alterRole.equals("HOST") && relation.equals("FATHER")) || relation.equals("PATERNAL HOME")){
				reducedRelation = "FATHER OR PATERNAL HOME";
			} else if ((alterRole.equals("HOST") && relation.equals("CHILD")) ||  relation.equals("FILIAL HOME")){
				reducedRelation = "CHILD OR FILIAL HOME";
			} else if ((alterRole.equals("HOST") && relation.equals("MOTHER")) || relation.equals("MATERNAL HOME")){
				reducedRelation = "MOTHER OR MATERNAL HOME";
			} else if ((alterRole.equals("HOST") && relation.equals("SIBLING")) || relation.equals("FRATERNAL HOME")){
				reducedRelation = "SIBLING OR FRATERNAL HOME";
			} else if ((alterRole.equals("HOST") && relation.equals("SPOUSE")) || relation.equals("MARITAL HOME")){
				reducedRelation = "SPOUSE OR MARITAL HOME";
			} else if (relation.equals("STATE") || relation.equals("NGO")){
				reducedRelation = "STATE OR NGO";
			} else if (relation.equals("WORKPLACE") || relation.equals("HOMELESS")){
				reducedRelation = "WITHOUT FIXED DOMICILE";
			} else if (relation.equals("FRIEND") || relation.equals("EMPLOYERS_EMPLOYEE") || relation.equals("LANDLORDS_LODGER") || relation.equals("MASTERS_APPRENTICE") || relation.equals("FRIENDS_FRIEND")){
				reducedRelation = "FRIEND OR COLLEGUE";
			} else if (relation.contains("UNRELATEDS_")){
				reducedRelation = "UNRELATED";
			} else if (relation.contains("S_")){
				reducedRelation = relation.replaceAll("_AGNATIC", "").replaceAll("_UTERINE", "").replaceAll("_COGNATIC", "").replaceAll("RELATIVE", "RELATIVE_OR_AFFINE");
			} else {
				reducedRelation = relation;
			}
			reducedRelation = reduceRelation(reducedRelation);
			if (!result.contains(reducedRelation)){
				result.add(reducedRelation);
			}
		}
		//
		return result;
	}
	
	private String reduceRelation (String relation){
		String result;
		
		result = null;
		
		if (relation.equals("FATHER OR PATERNAL HOME") || relation.equals("MOTHER OR MATERNAL HOME")){
			result = "PARENTS";
		} else if (relation.equals("SPOUSE OR MARITAL HOME")){
			result = "SPOUSE";
		} else if (relation.equals("CHILD OR FILIAL HOME")){
			result = "CHILD";
		} else if (relation.equals("LANDLORD")){
			result = "RENT";
		} else if (relation.equals("HOTEL-HOSTEL") || (relation.equals("WITHOUT FIXED DOMICILE"))){
			result = "PUBLIC";
		} else {
			result = relation.replaceAll("RELATIVE_OR_AFFINE","RELATIVE").replaceAll("FATHER", "PARENTS").replaceAll("MOTHER", "PARENTS");

			if (result.equals("RELATIVES_MARITAL HOME")){
				result = "AFFINE";
			} else if (result.equals("SIBLINGS_MARITAL HOME")){
				result = "AFFINE";
			} else if (result.equals("SPOUSES_PATERNAL HOME")){
				result = "SPOUSE";
			} else if (result.equals("SPOUSES_MARITAL HOME")){
				result = "PARENTS";
			} else {
				String[] relationParts = result.split("S_");
				if (relationParts.length>1){
					result = "via "+relationParts[0];
					// simplify
					if (result.contains("via PARENTS")){
						result = "via PARENTS";
					} else if (result.contains("via SPOUSE")){
						result = "via SPOUSE";
					} else if (result.contains("via SIBLING")){
						result = "via SIBLING";
					} else if (result.contains("via RELATIVE")){
						result = "via RELATIVE";
					} else if (result.contains("via MASTER")){
						result = "via MASTER";
					} else if (result.lastIndexOf("via")>0){
						System.err.println("double reference "+result);
					}
				}
			}  
		}
		//
		return result;
	}
	
	private static String parameter (String label){
		String result;
		
		result = null;
		
		if (label.contains("_")){
			String[] parts = label.split("_");
			if (parts.length>1){
				result = parts[parts.length-1];
			}
		}
		
		//
		return result;
	}
	
	private boolean isRelationClassificationType(String parameter){
		boolean result;
		
		result = false;
		
		if (parameter!=null){
			try {
				if (SpaceTimeCriteria.RelationClassificationType.valueOf(parameter)!=null){
					result = true;
				}
			} catch (IllegalArgumentException iae) {
			}
		}
		
		//
		return result;
	}

	
	public Value getValue (String label){
		Value result;
				
		result = null;
		
		GraphProfile networkProfile = null;
		String parameter = parameter(label);
		
		if (isRelationClassificationType(parameter)){
			if (eventTypeNetworkProfiles!=null){
				networkProfile = eventTypeNetworkProfiles.get("Parcours Network_"+parameter);
			}
			if (networkProfile == null && networkProfiles!=null){
				networkProfile = networkProfiles.get("Parcours Intersection Network_"+parameter);
			}
		} else if (networkProfiles!=null){
			networkProfile = networkProfiles.get("Nonmediated Ego Network");
		}
		
		if (length == 0){
			
			result = null;
			
		} else if (label.contains("PROFILE")){
			
			SequenceProfile profile = getProfile(RelationClassificationType.valueOf(parameter));
			if (profile == null){
				result = null;
			} else {
				result = Value.valueOf(profile.getProfile());
			}

		} else if (label.contains("SUPPORT")){
			
			SequenceProfile profile = getProfile(RelationClassificationType.valueOf(parameter));
			if (profile == null){
				result = null;
			} else {
				result = Value.valueOf(profile.getSupportAsList());
			}

		} else if (label.contains("NRSTATIONS")){
			
			SequenceProfile profile = getProfile(RelationClassificationType.valueOf(parameter));
			if (profile == null){
				result = null;
			} else {
				result = Value.valueOf(profile.getNrStations());
			}
			
		} else if (label.contains("NRCENTERSNOSTART")){
			
			SequenceProfile profile = getProfile(RelationClassificationType.valueOf(parameter));
			if (profile == null){
				result = null;
			} else {
				result = Value.valueOf(profile.getNrCentersWithoutStart());
			}

		} else if (label.contains("NRCENTERS")){
			
			SequenceProfile profile = getProfile(RelationClassificationType.valueOf(parameter));
			if (profile == null){
				result = null;
			} else {
				result = Value.valueOf(profile.getNrCenters());
			}

		} else if (label.contains("CENTERSNOSTART")){
			
			SequenceProfile profile = getProfile(RelationClassificationType.valueOf(parameter));
			if (profile == null){
				result = null;
			} else {
				result = Value.valueOf(profile.getCentersWithoutStartAsList());
			}

		} else if (label.contains("CENTERS")){
			
			SequenceProfile profile = getProfile(RelationClassificationType.valueOf(parameter));
			if (profile == null){
				result = null;
			} else {
				result = Value.valueOf(profile.getCentersAsList());
			}

		} else if (label.contains("NRINTERNALMOVES")){
			
			SequenceProfile profile = getProfile(RelationClassificationType.valueOf(parameter));
			if (profile == null){
				result = null;
			} else {
				result = Value.valueOf(profile.getNrLoops());
			}

		} else if (label.contains("NREXTERNALMOVES")){
			
			result = new Value(length - getValue("NRINTERNALMOVES_PLACE").intValue());

		} else if (label.contains("NRDIRECTRETURNS_NORM")){
			
			result = Value.valueOf(MathUtils.percent(getValue("NRDIRECTRETURNS_"+parameter).intValue(), length-1));

		} else if (label.contains("NRDIRECTRETURNS")){
			
			SequenceProfile profile = getProfile(RelationClassificationType.valueOf(parameter));
			if (profile == null){
				result = null;
			} else {
				result = Value.valueOf(profile.getNrDirectReturns());
			}

		} else if (label.contains("NRCYCLES_NORM")){
			
			result = Value.valueOf(MathUtils.percent(getValue("NRCYCLES_"+parameter).intValue(), length-2));

		} else if (label.contains("NRCYCLES")){
			
			SequenceProfile profile = getProfile(RelationClassificationType.valueOf(parameter));
			if (profile == null){
				result = null;
			} else {
				result = Value.valueOf(profile.getNrCycles());
			}

		} else if (label.contains("NRALTERSPEREVENT")){
			
			result = new Value(getMeanNr(parameter));
			
		} else if (label.contains("AGEFIRST")){
			
			String[] subLabels = label.split("_");
			Integer age = null;
			if (subLabels.length==1){
				age = getAge(getFirstMigration());
			} else if (subLabels.length==3){
				age = getAge(getFirstEvent(subLabels[1],subLabels[2]));
			}
			if (age!=null){
				result = new Value(age);
			}
			
		} else if (label.equals("MAX_DISTANCE")){
			
			GeoLevel maxDistance = getMaxDistance();
			if (maxDistance != null){
				result = new Value(maxDistance);
			}
			
		} else if (label.equals("MEAN_NR_MOVES")){
			
			Map<GeoLevel, Integer> distanceProfile = getDistanceProfile();
			if (distanceProfile != null){
				result = new Value(distanceProfile);
			}
			
		} else if (label.equals("MEAN_COVERAGE")){
			
			if (length==0){
				result = null;
			} else {
				result = new Value(getMeanCoverage());
			}
			
		} else if (label.equals("MAX_COVERAGE")){
			
			if (length==0){
				result = null;
			} else {
				result = new Value(getMaxCoverage());
			}
			
		} else if (label.contains("MEAN_BETWEENNESS")){
			
			result = new Value(networkProfile.getMeanBetweenness());
			
		} else if (label.contains("MAX_BETWEENNESS")){

			result = new Value(networkProfile.getMaxNonEgoBetweenness());
			
		} else if (label.equals("MAIN_ALTERS")){
			
			result = new Value(getMaxCoverageAlters());
			
		} else if (label.contains("CENTRAL_ALTERS")){
			
			if (networkProfile.nodeCount()<3){
				result = null;
			} else {
				result = new Value(networkProfile.getCentralAlters());
			}

		} else if (label.equals("MAIN_RELATIONS")){
			
			List<String> relations = new ArrayList<String>();
			for (Individual alter : getMaxCoverageAlters()){
				for (String relation : relationsByAlter.get(alter)){
					if (!relations.contains(relation)){
						relations.add(relation);
					}
				}
			}
			result = new Value(relations);
			
		} else if (label.contains("EVENTS_")){

			String typeName = label.substring(label.indexOf("_")+1);

			List<String> types = new ArrayList<String>();
			for (Relation event : events){
				String type = eventEvaluations.get(RelationClassificationType.valueOf(typeName)).get(event);
				if (!types.contains(type)){
					types.add(type);
				}
			}
			result = new Value(types);
			
		} else if (label.contains("CENTRAL_RELATIONS")){
			
			if (networkProfile.nodeCount()<3){
				result = null;
			} else {
				List<String> relations = new ArrayList<String>();
				for (Individual alter : ((GraphProfile<Individual>)networkProfile).getCentralAlters()){
					for (String relation : relationsByAlter.get(alter)){
						if (!relations.contains(relation)){
							relations.add(relation);
						}
					}
				}
				result = new Value(relations);
			}
			
		} else if (label.contains("CONNECTED_NETWORK_RELATIONS")){
			
			result = new Value(getConnectedRelations(networkProfile.getNonEgoComponents()));

		} else if (label.contains("NETWORK_RELATIONS")){
			
			result = new Value(getNetworkRelations(networkProfile.getGraphWithoutEgo()));

		} else if (label.contains("RELATIONS")){
			
			List<String> relations = relationsByRoles.get(parameter);
			if (parameter.equals("HOST")){
				relations.addAll(impersonalRelations);
			}
			
			relations = reduceRelations(relations, parameter);
					
			result = new Value(relations);
			
		} else if (label.equals("NREVENTS")){
			
			result = new Value(length);
			
		} else if (label.contains("NRALTERS")){
			
			result = new Value(altersByRoles.get(parameter).size());
			
		} else if (label.contains("SAMESEXALTERS")){

			int sameSexAlters = 0;
			for (Individual alter : altersByRoles.get(parameter)){
				if (alter.getGender()==ego.getGender()){
					sameSexAlters++;
				}
			}
			
			if (altersByRoles.get(parameter).size()==0){
				result = null;
			} else {
				result = new Value(MathUtils.percent(sameSexAlters, altersByRoles.get(parameter).size()));
			}
			
		} else if (label.contains("SAMEPLACEALTERS")){

			int samePlaceAlters = 0;
			Place egoBirthPlace = geography.getPlace(ego.getAttributeValue("BIRT_PLAC"),level);
			
			for (Individual alter : altersByRoles.get(parameter)){
				Place alterBirthPlace = geography.getPlace(alter.getAttributeValue("BIRT_PLAC"),level);
				if (egoBirthPlace != null && egoBirthPlace.equals(alterBirthPlace)){
					samePlaceAlters++;
				}
			}
			
			if (altersByRoles.get(parameter).size()==0){
				result = null;
			} else {
				result = new Value(MathUtils.percent(samePlaceAlters, altersByRoles.get(parameter).size()));
			}
			
		} else if (label.contains("EGO-BETWEENNESS")){
			
			result = new Value(networkProfile.getEgoBetweenness());
			
		} else if (label.contains("SIZE")){
			
			result = new Value(networkProfile.nodeCount());
			
		} else if (label.contains("TIES")){

			result = new Value(networkProfile.nonNullLineCount());
			
		} else if (label.contains("DENSITY")){

			result = new Value(networkProfile.density());
			
		} else if (label.contains("DENSITY_NOLOOPS")){

			result = new Value(networkProfile.densityWithoutLoops());
			
		} else if (label.contains("SDENSITY")){
			
			if (specificDensities.get(parameter)==0.){
				result = null;
			} else {
				result = new Value(specificDensities.get(parameter));
			}
			
		} else if (label.contains("MEANDEGREE")){

			result = new Value(networkProfile.meanDegree());
						
		} else if (label.contains("MEANDEGREE_NOLOOPS")){

			result = new Value(networkProfile.meanDegreeWithoutLoops());
			
		} else if (label.contains("MEANDEGREE_NORM")){

			result = new Value(networkProfile.meanDegreeNormalized());
			
		} else if (label.contains("MEANDEGREE_NOLOOPS_NORM")){

			result = new Value(networkProfile.meanDegreeWithoutLoopsNormalized());
						
		} else if (label.contains("NRCOMPONENTS")){

			result = new Value(networkProfile.getNonEgoComponents().size());
						
		} else if (label.contains("NRISOLATES")){

			result = new Value(networkProfile.getNonEgoComponents().nrSingletons());
			
		} else if (label.contains("MAXCOMPONENT")){

			result = new Value(networkProfile.getNonEgoComponents().maxClusterSize());
						
		} else if (label.contains("NRCOMPONENTS_NORM")){

			if (networkProfile.getNonEgoComponents().size()==0){
				result = null;
			} else {
				result = new Value(networkProfile.getNonEgoComponents().meanShare());
			}
			
		} else if (label.contains("MAXCOMPONENT_NORM")){

			if (networkProfile.getNonEgoComponents().size()==0){
				result = null;
			} else {
				result = new Value(networkProfile.getNonEgoComponents().maxShare());
			}
						
		} else if (label.contains("NRISOLATES_NORM")){

			if (networkProfile.getNonEgoComponents().size()==0){
				result = null;
			} else {
				result = new Value(networkProfile.getNonEgoComponents().singletonShare());
			}
			
		} else if (label.contains("BROKERAGE")){

			result = new Value(networkProfile.brokerage());
			
		} else if (label.contains("EFFICIENT_SIZE")){

			result = Value.valueOf(networkProfile.efficientSize());
			
		} else if (label.contains("EFFICIENCY")){

			result = Value.valueOf(networkProfile.efficiency());
			
		} else if (label.contains("SIMILARITY")){
			
			result = new Value(getParcoursSimilarities(RelationClassificationType.valueOf(parameter)));
			
		} else if (label.contains("ECCENTRICITY")){

			result = new Value(networkProfile.getEccentricity());

		} else if (label.contains("CONCENTRATION")) {

			result = new Value(MathUtils.herfindahl(networkProfile.getNonEgoComponents().clusterSizes()));
		}
	//
		return result;
	}
	
	private Integer getAge (Relation event){
		Integer result;
		
		result = null;

		if (event!=null){
			int year = Sequence.getYear(event);
			result = IndividualValuator.ageAtYear(ego, year);
		}
		//
		return result;
	}
	
	private double getMaxCoverage (){
		double result;
		
		if (coverage.size()>0){
			result = MathUtils.percent(((TreeMap<Integer,List<Individual>>)coverage).lastKey(),length);
		} else {
			result = 0;
		}
		//
		return result;
	}
	
	private double getMeanCoverage (){
		double result;
		
		result = 0;
		
		if (coverage.size()>0){
			double sum = 0.;
			
			for (int i : coverage.keySet()){
				result += i;
				sum++;
			}
			
			result = MathUtils.percent(result, sum*length);
		}
		//
		return result;
	}
	
	private void setSpecificDensities (String networkTitle){
		
		Graph<Individual> network = networkProfiles.get(networkTitle).getGraph();
		
		specificDensities = new HashMap<String,Double>();
		specificDensities.put("PARENT-CHILD", 0.);
		specificDensities.put("SPOUSE", 0.);
		specificDensities.put("SIBLING", 0.);
		specificDensities.put("RELATIVE", 0.);
		specificDensities.put("AFFINE", 0.);
		specificDensities.put("EMPLOYMENT", 0.);
		specificDensities.put("RENT", 0.);
		
		for (Link<Individual> link : network.getLinks()){
			String tag = link.getTag();
			tag = tag.substring(tag.indexOf("'")+1,tag.lastIndexOf("'"));
			String key = null;
			if (tag.equals("FATHER") || tag.equals("MOTHER") || tag.equals("CHILD")){
				key = "PARENT-CHILD";
			} else if (tag.equals("SPOUSE") || tag.equals("SIBLING") || tag.equals("AFFINE")){
				key = tag;
			} else if (tag.contains("RELATIVE")){
				key = "RELATIVE";
			} else if (tag.contains("EMPLOY")){
				key = "EMPLOYMENT";
			} else if (tag.equals("LANDLORD") || tag.equals("LODGER")){
				key = "RENT";
			}
			
			if (key!=null){
				specificDensities.put(key, specificDensities.get(key)+1.);
			}
		}
		
		for (String key : specificDensities.keySet()){
			specificDensities.put(key, MathUtils.percent(specificDensities.get(key),network.lineCount()));
		}
		
	}
	

	
	private List<Individual> getMaxCoverageAlters (){
		List<Individual> result;
		
		if (coverage.size()>1){
			result = coverage.get(((TreeMap<Integer,List<Individual>>)coverage).lastKey());
		} else {
			result = new ArrayList<Individual>();
		}
		//
		return result; 
	}
	
	private void setComponentTypes(String networkTitle){
		
		Partition<Node<Individual>> nonEgoComponents = networkProfiles.get(networkTitle).getNonEgoComponents();
		
		if (nonEgoComponents.size()>0){
			for (Cluster<Node<Individual>> component : nonEgoComponents.getClusters().toList()){
				String type = getComponentType(component);
				Value value = new Value(type);
				if (nonEgoComponents.getValues().contains(value)){
					int i = 1;
					while (nonEgoComponents.getValues().contains(value)){
						i++;
						value = new Value(type+" "+i);
					}
				}
				nonEgoComponents.changeClusterValue(component, value);
			}
		}
	}
	
	
	private double getMeanNr(String roleName){
		double result;
		
		result = MathUtils.percent(altersByRoles.get(roleName).size(), 100*length);
		
		//
		return result;
		
	}
	
	private Graph<Individual> getEgoNetwork (){
		Graph<Individual> result;

		//
		result = new Graph<Individual>("Ego network " + ego+" "+ relationModelName+" "+egoRoleName+"-"+getSelectedRolesAsString());

		Map<String,Integer> tagMap = new HashMap<String,Integer>();

		result.addNode(ego);
		//
		for (Individual alter : altersByRoles.get("SELECTED")) {
			result.addNode(alter);
		}
		
		List<Individual> referents = result.getReferents();
		
		//
		for (int i=0;i<referents.size();i++){
			Individual ego = referents.get(i);
			for (int j=0;j<i;j++){
				Individual alter = referents.get(j);
				
				for (String tag: NetUtils.getAlterRoles(ego, alter, maxDepths, relationModelNames, chainClassification)){
					Link<Individual> link = result.addArc(ego, alter,1);
					Integer tagNumber = tagMap.get(tag);
					if (tagNumber == null){
						tagNumber = tagMap.size()+1;
						tagMap.put(tag, tagNumber);
					}
					link.setTag(":"+tagNumber+" '"+tag+"'");
				}
			}
		}
		
		//
		NetUtils.setGenderShapes(result);
		
		//
		return result;
	}
	
	private Graph<Relation> getParcoursGraph (){
		Graph<Relation> result;
		
		result = new Graph<Relation>();
		result.setLabel("Parcours "+ego);
		
		Relation previousEvent = null;
		for (Relation event : events){
			if (previousEvent != null){
				Link<Relation> link = result.addArc(previousEvent, event,1);
				link.setTag(ego.getId()+"");
			}
			previousEvent = event;
		}
		
		if (events.size()==1){
			result.addNode(previousEvent);
		}
		
		//
		return result;
	}
	
	private Graph<Cluster<Relation>> getParcoursNetwork (RelationClassificationType relationClassificationType){
		Graph<Cluster<Relation>> result;
		
		result = new Graph<Cluster<Relation>>();
		result.setLabel("Parcours Network_"+relationClassificationType+" "+ego);
		
		Partition<Relation> partition = new Partition<Relation>();
		Map<Relation,String> valueMap = eventEvaluations.get(relationClassificationType);

		Map<String,Individuals> hostsMap = new HashMap<String,Individuals>();
		hostsMap.put("SPOUSE", new Individuals());
		
		for (Relation event : events){
			String eventType = valueMap.get(event);
			Individuals hosts = hostsMap.get(eventType);
			if (hosts!=null){
				for (Individual host : event.getIndividuals("HOST")){
					if (eventType.equals("SPOUSE") && !host.spouses().contains(ego)){
						System.err.println("Non-spouse as spouse for "+ego+": "+host+" "+event);
						continue;
					}
					hosts.put(host);
				}
				if (hosts.size()>1){
					eventType += "_"+hosts.size();
				}
			} else {
				if (eventType.contains("via ")){
					Individuals viaHosts = hostsMap.get(eventType.replace("via ", ""));
					if (viaHosts!=null && viaHosts.size()>1){
						eventType += viaHosts.size();
					}
				}
			}
			partition.put(event, new Value(eventType));
		}
		
		Relation previousEvent = null;
		
		int t=1;
		for (Relation event : events){
//			if (ego.getId()==4440) System.out.println(event+" "+partition.getCluster(event).getValue()+" "+event.getIndividuals("HOST"));
			if (previousEvent != null){
				if (impersonalRelation(previousEvent)!=null && impersonalRelation(previousEvent).equals("TRANSITION")){
					if (partition.getCluster(previousEvent).getValue().toString().equals("TRANSITION")){
						partition.getCluster(event).put(previousEvent);
						result.addNode(partition.getCluster(event));
//						System.out.println("replaced "+ego+" "+previousEvent+" "+result.getNode(partition.getCluster(previousEvent))+" > "+event+" "+result.getNode(partition.getCluster(event)));
						result.transferLinks(result.getNode(partition.getCluster(previousEvent)), result.getNode(partition.getCluster(event)));
					} else {
						System.err.println("Ambiguous transition "+ego+" "+t+" "+previousEvent+" "+previousEvent.getIndividuals("HOST")+" *"+partition.getCluster(previousEvent).getValue()+"*");
					}
				} else {
					Link<Cluster<Relation>> link = result.getArc(partition.getCluster(previousEvent), partition.getCluster(event));
					if (link==null){
						link = result.addArc(partition.getCluster(previousEvent), partition.getCluster(event),t);
						link.setTag(ego.getId()+" "+t);
					} else {
						String tag = link.getTag();
						double weight = link.getWeight();
						link.setTag(tag+" "+t);
						link.setWeight(weight*0.1+t);
					}
				}
			}
			previousEvent = event;
			t++;
		}
		
		if (events.size()==1){
			result.addNode(partition.getCluster(previousEvent));
		}
				
		for (Node<Cluster<Relation>> node : result.getNodes().toList()){
			if (node.getReferent().getValue().toString().equals("TRANSITION")){
				result.removeNode(node);
				break;
			}
		}
		
		//
		result.renumberNodes();
		
		if (result.getNodesByLabel("PARENTS").size()==0){
			System.err.println("No parents: "+result);
		}
		
		//
		return result;
	}

	
	private Graph<Relation> getMultipleParcoursGraph(){
		Graph<Relation> result;
		
		List<Graph<Relation>> graphs = new ArrayList<Graph<Relation>>();
		
		SpaceTimeCriteria criteria = new SpaceTimeCriteria();
		criteria.setMaxAge(1000);
		criteria.setRelationModelNames(relationModelNames);
		criteria.setRelationModelName(relationModelName);
		
		for (Individual alter : altersByRoles.get("SELECTED")) {
			Sequence fullBiography = SequenceMaker.createPersonalSequence(alter, criteria);
			Sequence sortedBiography = SequenceWorker.split(fullBiography).getById(1);
			
			SequenceCensus census = new SequenceCensus(sortedBiography,criteria);
			if (sortedBiography.getNrEvents()>0){
				Graph<Relation> alterGraph = census.getParcoursGraph();
				graphs.add(alterGraph);
			}
		}
		result = GraphUtils.fuse(graphs);
		result.setLabel("Multiple parcours "+ego);
		//
		return result;
	}
	
	public static Graph<String> getFlatParcoursNetworkNoLoops(Graph<Cluster<Relation>> parcoursNetwork){
		Graph<String>  result;
		result = new Graph<String>(parcoursNetwork.getLabel());
		for (Link<Cluster<Relation>> link : parcoursNetwork.getLinks()){
			if (!link.isLoop()){
				result.addArc(link.getSourceNode().getReferent().getLabel(),link.getTargetNode().getReferent().getLabel());
			}
		}
		//
		return result;
	}
	
	private Relations getEvents (Individual indi){
		Relations result;
		
		result = new Relations();
		
		for (Relation event : indi.relations().getByModelName(relationModelName)){
			if (!isBirth(event)){
				result.add(event);
			}
		}
		//
		return result;
	}

	private Graph<Individual> getParcoursSimilarityNetwork (RelationClassificationType relationClassificationType){
		Graph<Individual> result;
		
		//
		result = new Graph<Individual>("Parcours Similarity Network_"+relationClassificationType +" "+ ego+" "+ relationModelName+" "+egoRoleName+"-"+getSelectedRolesAsString());
		
		result.addNode(ego);
		//
		for (Individual alter : altersByRoles.get("SELECTED")) {
			if (alter.hasAttributeValue(alterAttributeLabel,alterAttributeValue)){
				result.addNode(alter);
			}
		}
		List<Graph<String>> flatParcoursNetworksNoLoops = new ArrayList<Graph<String>>();
		
		Map<Graph<String>,Individual> index = new HashMap<Graph<String>,Individual>();
		Set<Value> clusterValues = new HashSet<Value>();
		
		for (Node<Individual> node : result.getNodes().toListSortedById()){

			Individual referent = node.getReferent();
			SpaceTimeCriteria criteria = new SpaceTimeCriteria();
			criteria.getRelationClassificationTypes().add(relationClassificationType);
			criteria.setEgoRoleName(egoRoleName);
			criteria.setRoleNames(roleNames);
			criteria.setRelationModelName(relationModelName);
			criteria.setRelationModelNames(relationModelNames);
			criteria.setPattern(pattern);
			
			Sequence sequence = SequenceMaker.createPersonalSequence(referent, criteria);
			SequenceCensus census = new SequenceCensus(sequence,criteria);

			Graph<Cluster<Relation>> parcoursNetwork = census.getParcoursNetwork(relationClassificationType);
			node.setAttribute("NRTRANSITIONS", parcoursNetwork.arcCount()+"");
			Graph<String> flatParcoursNetworkNoLoops = SequenceCensus.getFlatParcoursNetworkNoLoops(parcoursNetwork);
			flatParcoursNetworksNoLoops.add(flatParcoursNetworkNoLoops);
			
			index.put(flatParcoursNetworkNoLoops, referent);
			for (Node<Cluster<Relation>> clusterNode : parcoursNetwork.getNodes()){
				clusterValues.add(clusterNode.getReferent().getValue());
			}
		}

		Collections.sort(flatParcoursNetworksNoLoops, new GraphComparatorByArcCount<String>());
		Graph<Graph<String>> distanceGraph = GraphUtils.createDistanceGraph(flatParcoursNetworksNoLoops);
		
		for (Link<Graph<String>> link : distanceGraph.getEdges()){
			Link<Individual> edge = result.addEdge(index.get(link.getSourceNode().getReferent()), index.get(link.getTargetNode().getReferent()));
			List<Graph<String>> pair = new ArrayList<Graph<String>>();
			pair.add(link.getSourceNode().getReferent());
			pair.add(link.getTargetNode().getReferent());
			double maxArcs = new Double(GraphUtils.fuse(pair).arcCount());
			double weight = (maxArcs-link.getWeight())/maxArcs;
//			System.out.println(link+" "+maxArcs +" "+weight);
			// As Pajek does not allow decimal values:
			edge.setWeight(100*weight);
			edge.setTag(weight+"");
		}
		
		//
		NetUtils.setGenderShapes(result);
		
		//
		return result;
	}
	
	
	private Graph<Individual> getParcoursIntersectionNetwork (){
		Graph<Individual> result;
		
		//
		result = new Graph<Individual>("Parcours Intersection Network "+ ego+" "+ relationModelName+" "+egoRoleName+"-"+getSelectedRolesAsString());

		Map<Individual,Set<Relation>> eventMap = new HashMap<Individual,Set<Relation>>();

		result.addNode(ego);
		result.getNode(ego).setAttribute("NREVENTS", getEvents(ego).size()+"");
		Set<Relation> egoEvents = new HashSet<Relation>();
		eventMap.put(ego, egoEvents);
		for (Relation event : getEvents(ego)){
			egoEvents.add(event);
		}
		
		//
		for (Individual alter : altersByRoles.get("SELECTED")) {
			if (alter.hasAttributeValue(alterAttributeLabel,alterAttributeValue)){
				result.addNode(alter);
				result.getNode(alter).setAttribute("NREVENTS", getEvents(alter).size()+"");
				Set<Relation> alterEvents = new HashSet<Relation>();
				eventMap.put(alter, alterEvents);
				for (Relation event : getEvents(alter)){
					alterEvents.add(event);
				}
			}
		}
		
		List<Individual> referents = result.getReferents();
		
		//
		for (int i=0;i<referents.size();i++){
			Individual first = referents.get(i);
			for (int j=0;j<i;j++){
				Individual second = referents.get(j);
				
				double[] shares = MathUtils.intersectionRates(eventMap.get(first), eventMap.get(second));
				
				if (shares[0]>0){
					Link<Individual> outArc = result.addArc(first, second,shares[0]);
				}
				if (shares[1]>0){
					Link<Individual> inArc = result.addArc(second, first,shares[1]);
				}
			}
		}
		
		//
		NetUtils.setGenderShapes(result);
		
		//
		return result;
	}
	
	public Partition<Link<Individual>> getParcoursLinkPartition(RelationClassificationType relationClassificationType){
		Partition<Link<Individual>> result;
		
		GraphProfile<Individual> parcoursNetworkProfile = networkProfiles.get("Parcours Intersection Network_"+relationClassificationType);
		result = parcoursNetworkProfile.getLinkPartition();
		if (result==null){
			result = GraphUtils.getLinkPartitionByKinship(parcoursNetworkProfile.getGraph());
			parcoursNetworkProfile.setLinkPartition(result);
		}
		//
		return result;
	}
	
	public Map<Value, Double[]> getParcoursSimilarities(RelationClassificationType relationClassificationType) {
		Map<Value, Double[]> result;
		
		GraphProfile<Individual> parcoursNetworkProfile = networkProfiles.get("Parcours Intersection Network_"+relationClassificationType);
		if (parcoursNetworkProfile.getLinkPartition()==null){
			parcoursNetworkProfile.setLinkPartition(GraphUtils.getLinkPartitionByKinship(parcoursNetworkProfile.getGraph()));
		}
		result = parcoursNetworkProfile.aggregateWeights();
		
		//
		return result;
	}
	
	private String getSelectedRolesAsString (){
		String result;
		
		result = "";
		for (String roleName : roleNames){
			if (result.length()>0){
				result+=" ";
			}
			result+=roleName;
		}
		//
		return result;
	}
	

	private Graph<Individual> getNonMediatedEgoNetwork (){
		Graph<Individual> result;

		//
		result = new Graph<Individual>("Nonmediated Ego Network " + ego+" "+ relationModelName+" "+egoRoleName+"-"+getSelectedRolesAsString());

		Map<String,Integer> tagMap = new HashMap<String,Integer>();

		result.addNode(ego);
		//
		for (Individual alter : altersByRoles.get("SELECTED")) {
			result.addNode(alter);
		}
		Set<Individual> others = new HashSet<Individual>();
		for (Individual medius : altersByRoles.get("ALL")){
			if (!altersByRoles.get("SELECTED").contains(medius)){
				others.add(medius);
			}
		}
		
		List<Individual> referents = result.getReferents();
		Graph<Individual> barredEdges = new Graph<Individual>();
		
		// get direct links
		for (int i=0;i<referents.size();i++){
			Individual ego = referents.get(i);
			for (int j=0;j<i;j++){
				Individual alter = referents.get(j);
				
				List<String> alterRoles = NetUtils.getAlterRoles(ego, alter, maxDepths, relationModelNames, chainClassification, referents, barredEdges);
				
				for (String tag: alterRoles){
					Link<Individual> link = result.addEdge(ego, alter,1);
					Integer tagNumber = tagMap.get(tag);
					if (tagNumber == null){
						tagNumber = tagMap.size()+1;
						tagMap.put(tag, tagNumber);
					}
					link.setTag(":"+tagNumber+" '"+tag+"'");
				}
			}
		}
		
		// get indirect links mediated by nodes outside the network
		for (int i=0;i<referents.size();i++){
			Individual ego = referents.get(i);
			for (int j=0;j<i;j++){
				Individual alter = referents.get(j);
				if (result.getEdge(ego, alter)==null && barredEdges.getEdge(ego, alter)==null){
					List<String> alterRoles = new ArrayList<String>();
					for (Individual medius : others){
						if (medius!=ego){
							List<String> firstRoles = NetUtils.getAlterRoles(ego, medius, maxDepths, relationModelNames, chainClassification, referents, barredEdges);
							if (firstRoles.size()>0){
								List<String> secondRoles = NetUtils.getAlterRoles(medius, alter, maxDepths, relationModelNames, chainClassification, referents, barredEdges);
								for (String firstRole : firstRoles){
									for (String secondRole : secondRoles){
										if (!(firstRole.equals("CHILD") && (secondRole.contains("RELATIVE")))){
											String role = firstRole+"S_"+secondRole;
											String reducedRole = NetUtils.reduced(role);
											while (!reducedRole.equals(role)){
												role = reducedRole;
												reducedRole = NetUtils.reduced(role);
											}
											alterRoles.add(reducedRole);
										}
									}
								}
							}
						}
					}
					for (String tag: alterRoles){
						Link<Individual> link = result.addEdge(ego, alter,1);
						Integer tagNumber = tagMap.get(tag);
						if (tagNumber == null){
							tagNumber = tagMap.size()+1;
							tagMap.put(tag, tagNumber);
						}
						link.setTag(":"+tagNumber+" '"+tag+"'");
					}
				}
			}
		}
		
		//
		NetUtils.setGenderShapes(result);
		
		//
		return result;
	}

	Partition<Node<Individual>> getComponents(String networkTitle) {
		return networkProfiles.get(networkTitle).getNonEgoComponents();
	}

	private SequenceProfile getProfile(RelationClassificationType relationClassificationType){
		SequenceProfile result;
		
		result = sequenceProfiles.get(relationClassificationType);
		if (result==null){
			result = new SequenceProfile(events, eventEvaluations, relationClassificationType);
			sequenceProfiles.put(relationClassificationType, result);
		}
		
		//
		return result;
	}
	
	public static String getEgoRolePartners(Relation event, SpaceTimeCriteria criteria, Individual ego){
		String result;
		
		result = "";
		if (event.getModel().getName().equals(criteria.getRelationModelName())){
			Individuals partners = new Individuals();
			for (Individual partner : event.getIndividuals(criteria.getEgoRoleName())){
				if (partner!=ego){
					partners.put(partner);
				}
			}
			
			result = partners.toStringAsNameList();
		}
		
		//
		return result;
	}
	
	public Graph<Cluster<Relation>> getEventTypeNetwork (String title){
		Graph<Cluster<Relation>> result;
		
		if (eventTypeNetworkProfiles.get(title)==null){
			result = null;
		} else {
			
			result = eventTypeNetworkProfiles.get(title).getGraph();

		}
		//
		return result;
	}
	
	
	

}
