package org.tip.puck.spacetime;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.TreeMap;

import org.tip.puck.census.workers.CensusCriteria;
import org.tip.puck.net.Individual;
import org.tip.puck.net.relations.Actor;
import org.tip.puck.net.relations.Actors;
import org.tip.puck.net.relations.Relation;
import org.tip.puck.net.relations.workers.RelationWorker;
import org.tip.puck.net.workers.IndividualValuator;
import org.tip.puck.net.workers.NetUtils;
import org.tip.puck.partitions.Partition;
import org.tip.puck.segmentation.Segmentation;
import org.tip.puck.spacetime.workers.SpaceTimeCriteria;
import org.tip.puck.util.MathUtils;
import org.tip.puck.util.ToolBox;
import org.tip.puck.util.Value;

public class HouseStatistics {
	
	Houses houses;
	Integer[] dates;
	Map<House,Map<Ordinal,Map<String,Object>>> valuesByHouses;
	Map<Individual,Map<Ordinal,Map<String,Object>>> valuesByIndividuals;
	Map<Individual,Map<Ordinal,Relation>> relationsByIndividuals;
	String pattern;
	String affiliationLabel;
	String localUnitLabel;
	List<String> indicators;
	String dateLabel;
	String placeLabel;
	String relationModelName;
	String startDateLabel;
	String endDateLabel;
	String egoRoleName;
	
	public HouseStatistics (Houses houses, SpaceTimeCriteria criteria, List<String> indicators){
		
		this.houses = houses;
		this.dates = criteria.getDates();
		this.pattern = criteria.getPattern();
		this.affiliationLabel = criteria.getGroupAffiliationLabel();
		this.localUnitLabel = criteria.getLocalUnitLabel();
		this.valuesByHouses = new TreeMap<House,Map<Ordinal,Map<String,Object>>>();
		this.valuesByIndividuals = new TreeMap<Individual,Map<Ordinal,Map<String,Object>>>();
		this.relationsByIndividuals = new TreeMap<Individual,Map<Ordinal,Relation>>();
		this.indicators = indicators;
		this.dateLabel = criteria.getDateLabel();
		this.placeLabel = criteria.getPlaceLabel();
		this.relationModelName = criteria.getRelationModelName();
		this.startDateLabel = criteria.getStartDateLabel();
		this.endDateLabel = criteria.getEndDateLabel();
		this.egoRoleName = criteria.getEgoRoleName();
		
		for (House house : houses.toSortedList()){
			
			Map<Ordinal,Map<String,Object>> houseValues = new TreeMap<Ordinal,Map<String,Object>>();
			valuesByHouses.put(house, houseValues);

			for (Integer year : dates){
				
				Ordinal time = new Ordinal(year);
				houseValues.put(time, new HashMap<String,Object>());
			}
		}
		
	}
	
	public void getStatistics(){
		
		for (House house : houses.toSortedList()){
			
			Map<Ordinal,Map<String,Object>> houseValues = valuesByHouses.get(house);
			
			for (Integer year : dates){
				
				Ordinal time = new Ordinal(year);
				Relation relation = house.getByOrdinal(time);
				houseValues.get(time).putAll(RelationWorker.getStatistics(relation, indicators, pattern));
					
			}
		}

	}
	
	public void getMemberValues(){
		
		for (Individual member : houses.getMembers().toSortedList()){
			
			Map<Ordinal,Map<String,Object>> memberValues = new TreeMap<Ordinal,Map<String,Object>>();
			Map<Ordinal,Relation> memberRelations = new TreeMap<Ordinal,Relation>();
			
			relationsByIndividuals.put(member, memberRelations);
			valuesByIndividuals.put(member, memberValues);

			Ordinal previousTime = null;
			
			for (Integer year : dates){
				
				Ordinal time = new Ordinal(year);
				memberValues.put(time, new HashMap<String,Object>());
				// Warning! supposes that there is only one relation by year...
				Relation relation = member.relations().getByModelName(relationModelName).getByTime(dateLabel, year).getFirst();
				memberRelations.put(time, relation);
				
				// Extrapolation of previous stations 
				if (relation!=null && previousTime!=null && memberRelations.get(previousTime)==null){
					// Warning! supposes that there is only one relation by year...
					Relation previousRelation = member.relations().getByModelName(relationModelName).getPredecessors(relation, member, egoRoleName, dateLabel, startDateLabel, endDateLabel, previousTime.getYear()).getFirst();
					if (previousRelation!=null){
						memberRelations.put(previousTime, previousRelation);
					} 
				}
				
				previousTime = time;
			}
		}

		
		for (Individual member : houses.getMembers().toSortedList()){
			
			Map<Ordinal,Map<String,Object>> memberValues = valuesByIndividuals.get(member);
			Map<Ordinal,Relation> memberRelations = relationsByIndividuals.get(member);
			
			
			for (Integer year : dates){
				
				Map<String,Object> valueMap = new HashMap<String,Object>();

				String lifeStatus = IndividualValuator.lifeStatusAtYear(member, year);
				
				Ordinal time = new Ordinal(year);
				Relation relation = memberRelations.get(time);
				if (relation==null){
					if (year.toString().equals(member.getAttributeValue("BIRT_DATE"))){
						lifeStatus = "UNBORN";
					} else if (year.toString().equals(member.getAttributeValue("DEAT_DATE"))){
						lifeStatus = "DEAD";
					}
				}
				
				for (String indicator : indicators){
					if (lifeStatus.equals("DEAD") || lifeStatus.equals("UNBORN")){
						valueMap.put(indicator, lifeStatus);
					} else if (relation!=null){
						if (indicator.equals("PLACE")){
							String placeValue = relation.getAttributeValue(localUnitLabel);
							if (placeValue==null){
								placeValue = relation.getAttributeValue(placeLabel);
							}
							valueMap.put(indicator, placeValue);
						} else {
							Actor actor = relation.getActor(member, egoRoleName);
							if (indicator.equals("REFERENT")){
								valueMap.put(indicator, actor.getReferent());
							} else if (indicator.equals("REFERENT_KIN")){
								valueMap.put(indicator, RelationWorker.getReferentRole(actor, pattern, affiliationLabel, relation));
							} else if (indicator.equals("REFERENT_CHAIN")){
								valueMap.put(indicator, RelationWorker.getReferentChain(actor, affiliationLabel, relation));
							}
						}
					}
				}
				memberValues.put(time, valueMap);
			}
		}
	}
	
	public void getReferentKinCensus(){
		
		for (House house : houses.toSortedList()){
			
			Map<Ordinal,Map<String,Object>> houseValues = valuesByHouses.get(house);
			
			for (Integer year : dates){
				
				Ordinal time = new Ordinal(year);
				Relation relation = house.getByOrdinal(time);
				houseValues.get(time).putAll(RelationWorker.getReferentKinCensus(relation, pattern, affiliationLabel));
				
				for (String indicator : indicators){
					if (!houseValues.get(time).containsKey(indicator)){
						houseValues.get(time).put(indicator, 0.);
					}
				}
					
			}
		}
	}
	
	public void getReferentChainCensus(){
		
		for (House house : houses.toSortedList()){
			
			Map<Ordinal,Map<String,Object>> houseValues = valuesByHouses.get(house);
			
			for (Integer year : dates){
				
				Ordinal time = new Ordinal(year);
				Relation relation = house.getByOrdinal(time);
				houseValues.get(time).putAll(RelationWorker.getReferentChainCensus(relation, affiliationLabel));
				
				for (String indicator : houseValues.get(time).keySet()){
					if (!indicators.contains(indicator)){
						indicators.add(indicator);
					}
				}
					
			}
		}
	}
	

	public void getAllKinCensus(Segmentation segmentation, CensusCriteria censusCriteria){
		
		
		for (House house : houses.toSortedList()){
			
			Map<Ordinal,Map<String,Object>> houseValues = valuesByHouses.get(house);
			
			for (Integer year : dates){
				
				Ordinal time = new Ordinal(year);
				Relation relation = house.getByOrdinal(time);
				houseValues.get(time).putAll(RelationWorker.getAllKinCensus(segmentation, relation, pattern, censusCriteria));
				
				for (String indicator : houseValues.get(time).keySet()){
					if (!indicators.contains(indicator)){
						indicators.add(indicator);
					}
				}
					
			}
		}
	}
	
	public void put(House house, Ordinal time, String indicator, Object value){
		
		Map<Ordinal,Map<String,Object>> houseValues = valuesByHouses.get(house);
		
		if (houseValues != null){
			
			Map<String,Object> relationValues = houseValues.get(time);
			
			if (relationValues != null){
				
				relationValues.put(indicator,value);
			}
		}
	}
	
	/***
	 * gets all actors that participate in relation alpha but not in relation beta
	 * @param alpha
	 * @param beta
	 * @return
	 */
	private Actors getNewActors (Relation alpha, Relation beta){
		Actors result;
		
		result = new Actors();
		
		if (alpha!=null){
			
			if (beta==null){
				
				result.addAll(alpha.actors());

			} else {
			
				for (Actor actor: alpha.actors()){
					if (!beta.actors().contains(actor)){
						result.add(actor);
					}
				}
			}
		}
		
		//
		return result;
	}
	
	public Partition<String> getFlows (String direction){
		Partition<String> result;
		
		result = new Partition<String>();
		
		int[] maxDegrees = ToolBox.stringsToInts(pattern);
		
		Ordinal former = null;
		for (Integer year : dates){
			Ordinal later = new Ordinal(year);
			if (former!=null){
				
				Ordinal current = null;
				if (direction.equals("OUT")){
					current = former;
				} else if (direction.equals("IN")){
					current = later;
				}
				
				for (House house : houses){
					
					Relation formerRelation = house.getByOrdinal(former);
					Relation laterRelation = house.getByOrdinal(later);
					Relation currentRelation = house.getByOrdinal(current);
					Actors migrants = null;
					
					if (direction.equals("OUT")){
						migrants = getNewActors(currentRelation,laterRelation);
					} else if (direction.equals("IN")){
						migrants = getNewActors(currentRelation,formerRelation);
					}
					
					for (Actor actor : migrants){
						Individual referent = actor.getReferent();
						String link = "UNKNOWN";
						if (referent!=null){
							link = 	NetUtils.getAlterRole(actor.getIndividual(), referent, maxDegrees, null);
						}
						Individual otherReferent = null;
						Actor otherActor = RelationWorker.getClosestHomologue(currentRelation, actor, current.getYear(), dateLabel,direction);
						if (otherActor!=null){
							otherReferent = otherActor.getReferent();
						} else {
							System.err.println("Missing homologue "+actor+" "+direction+" "+year);
						}
						
						String otherLink = "UNKNOWN";
						if (otherReferent!=null){
							otherLink = NetUtils.getAlterRole(actor.getIndividual(), otherReferent, maxDegrees, null);
						} 
						String change = null;
						if (referent!=null && referent.equals(otherReferent)){
							change = "IDENTICAL";
						} else if (direction.equals("OUT")){
							change = link+">"+otherLink;
						} else if (direction.equals("IN")){
							change = otherLink+">"+link;
						}
						result.put(house+"\t"+former+"-"+later+"\t"+direction+"\t"+actor.getIndividual()+"\t"+referent+"\t"+otherReferent, new Value(change));
					}
				}
			}
			former = later;
		}

		//
		return result;
	}
	
	public Object get(House house, Ordinal time, String indicator){
		Object result;
		
		result = null;
		
		Map<Ordinal,Map<String,Object>> houseValues = valuesByHouses.get(house);
		
		if (houseValues != null){
			
			Map<String,Object> relationValues = houseValues.get(time);
			
			if (relationValues != null){
				
				result = relationValues.get(indicator);
			}
		}
		
		//
		return result;
		
	}
	
	public Object getByMember(Individual member, Ordinal time, String indicator){
		Object result;
		
		result = null;
		
		Map<Ordinal,Map<String,Object>> memberValues = valuesByIndividuals.get(member);
		
		if (memberValues != null){
			
			Map<String,Object> relationValues = memberValues.get(time);
			
			if (relationValues != null){
				
				result = relationValues.get(indicator);
			}
		}
		
		//
		return result;
		
	}
	
	public String getTrend(House house, String indicator){
		String result;
		
		result = null;
		Object lastValue = null;
		
		for (Integer year : dates){
			
			Object value = get(house,new Ordinal(year),indicator);

			if (lastValue != null){

				if (!(lastValue instanceof Number)){
					
					break;
					
				} else {
				
					int comp = ((Comparable)value).compareTo(lastValue);
					String trend = null;
					
					if (comp < 0){
						trend = "DECLINING";
					} else if (comp > 0) {
						trend = "AUGMENTING";
					} else if (comp == 0){
						trend = "CONSTANT";
					}
					
					if (result == null || result.equals("CONSTANT")){
						result = trend;
					} else if (!trend.equals("CONSTANT") && !trend.equals(result)){
						result = "VARIABLE";
						break;
					} 
				}
			}

			lastValue = value;
				
		}
		
		//
		return result;
	}
	
	public Double getMean (Ordinal time, String indicator){
		Double result;
		
		result = null;
		Double sum = 0.;
		
		for (House house : houses){
			
			Object value = get(house,time,indicator);
			
			if (value instanceof Number){
				
				sum += ((Number)value).doubleValue();
				
			} else {
				
				result = null;
				break;
			}
		}
		
		result = sum/new Double(houses.size());
		
		//
		return result;
	}
	
	public String getMeanTrend (String indicator){
		String result;
		
		result = "";
		
		Map<String,Double> trendCounts = new HashMap<String,Double>();
		
		for (House house : houses){
			
			String trend = getTrend(house,indicator);
			Double count = trendCounts.get(trend);
			if (count == null){
				trendCounts.put(trend,1.);
			} else {
				trendCounts.put(trend,count+1.);
			}
		}
		
		for (String trend : trendCounts.keySet()){
			trendCounts.put(trend,MathUtils.percent(trendCounts.get(trend), new Double(houses.size())));
		}
		
	    List<Entry<String,Double>> sortedEntries = new ArrayList<Entry<String,Double>>(trendCounts.entrySet());

	    Collections.sort(sortedEntries, 
	            new Comparator<Entry<String,Double>>() {
	                @Override
	                public int compare(Entry<String,Double> e1, Entry<String,Double> e2) {
	                    return e2.getValue().compareTo(e1.getValue());
	                }
	            }
	    );

		for (Entry<String,Double> entry: sortedEntries){
			result += entry.getKey()+" "+entry.getValue()+" ";
		}
		
		//
		return result;
	}

	public List<String> indicators() {
		return indicators;
	}
	
	

}
