package org.tip.puck.statistics;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tip.puck.PuckException;
import org.tip.puck.PuckManager;
import org.tip.puck.census.chains.Chain;
import org.tip.puck.census.workers.CircuitFinder;
import org.tip.puck.io.paj.PAJFile;
import org.tip.puck.net.Family;
import org.tip.puck.net.FamilyComparator;
import org.tip.puck.net.Individual;
import org.tip.puck.net.IndividualComparator;
import org.tip.puck.net.IndividualComparator.Sorting;
import org.tip.puck.net.Individuals;
import org.tip.puck.net.Net;
import org.tip.puck.partitions.Cluster;
import org.tip.puck.partitions.Clusters;
import org.tip.puck.partitions.MultiPartition;
import org.tip.puck.partitions.Partition;
import org.tip.puck.partitions.PartitionCriteria;
import org.tip.puck.partitions.PartitionCriteria.CumulationType;
import org.tip.puck.partitions.PartitionCriteria.SizeFilter;
import org.tip.puck.partitions.PartitionCriteria.ValueFilter;
import org.tip.puck.partitions.PartitionMaker;
import org.tip.puck.report.Report;
import org.tip.puck.report.ReportAttributes;
import org.tip.puck.report.ReportChart;
import org.tip.puck.report.ReportChart.GraphType;
import org.tip.puck.report.ReportChart.LogarithmType;
import org.tip.puck.report.ReportRawData;
import org.tip.puck.report.ReportTable;
import org.tip.puck.segmentation.Segmentation;
import org.tip.puck.statistics.StatisticsWorker.Indicator;
import org.tip.puck.util.Chronometer;
import org.tip.puck.util.MathUtils;
import org.tip.puck.util.PuckUtils;
import org.tip.puck.util.ToolBox;
import org.tip.puck.util.Value;

import fr.devinsy.util.StringList;

/**
 * 
 * 
 * @author TIP
 */
public class StatisticsReporter {

	static private final Logger logger = LoggerFactory.getLogger(StatisticsReporter.class);

	/**
	 * 
	 * @param partitions
	 * @return
	 * @throws PuckException
	 */
	public static <E> ReportChart createArrayChart(final String title, final double[][] array, final String[] lineTitles) {
		ReportChart result;

		// == Build Report Chart.
		result = new ReportChart(title, GraphType.LINES);

		//
		{
			for (int j = 0; j < array[0].length; j++) {
				result.setLineTitle(lineTitles[j], j);

				for (int i = 0; i < array.length; i++) {
					result.setHeader(i + "", i);
					result.setValue(array[i][j], j, i);
				}
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @param net
	 * @return
	 */
	public static ReportChart createCompletenessChart(final FiliationCounts counts) {
		ReportChart result;

		//
		result = new ReportChart("Genealogical Completeness", GraphType.LINES);
		result.setHeadersLegend("Generational depth");
		result.setLinesLegend("% ascendants");

		//
		result.setLineTitle("Overall", 0);
		result.setLineTitle("Agnatic", 1);
		result.setLineTitle("Uterine", 2);

		//
		for (int ascendingIndex = 1; ascendingIndex < counts.size(); ascendingIndex++) {
			FiliationCount count = counts.get(ascendingIndex);
			if (count.isPositive()) {
				result.addValue(ascendingIndex, count.getCognatic(), 0);
				result.addValue(ascendingIndex, count.getAgnatic(), 1);
				result.addValue(ascendingIndex, count.getUterine(), 2);
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @param individuals
	 * @return
	 * @throws PuckException
	 */
	public static ReportChart createComponentsChart(final Individuals individuals, final int resolution) throws PuckException {
		ReportChart result;

		//
		FiliationCounts counts = StatisticsWorker.components(individuals, resolution);

		//
		result = new ReportChart("Components", GraphType.LINES);
		result.setLogarithmType(LogarithmType.HORIZONTAL);
		result.setHeadersLegend("% Individuals");
		result.setLinesLegend("% Components");

		//
		result.setLineTitle("Agnatic", 0);
		result.setLineTitle("Uterine", 1);

		//
		for (int ascendingIndex = 0; ascendingIndex < counts.size(); ascendingIndex++) {
			result.addValue(MathUtils.percent(ascendingIndex, resolution), counts.get(ascendingIndex).getAgnatic(), 0);
			result.addValue(MathUtils.percent(ascendingIndex, resolution), counts.get(ascendingIndex).getUterine(), 1);
		}

		//
		return result;
	}

	/**
	 * 
	 * @param net
	 * @return
	 * @throws PuckException
	 */
	public static ReportChart createFratryDistributionChart(final FiliationCounts counts) throws PuckException {
		ReportChart result;

		//
		result = new ReportChart("Fratry Distribution", GraphType.LINES);
		result.setHeadersLegend("Size");
		result.setLinesLegend("Number");

		//
		result.setLineTitle("Uterine", 0);
		result.setLineTitle("Agnatic", 1);

		//
		for (int countIndex = 2; countIndex < counts.size(); countIndex++) {
			result.setHeader(MathUtils.toString(countIndex), countIndex);
			result.addValue(countIndex, counts.get(countIndex).getUterine(), 0);
			result.addValue(countIndex, counts.get(countIndex).getAgnatic(), 1);
		}

		//
		return result;
	}

	/**
	 * 
	 * @param net
	 * @return
	 * @throws PuckException
	 */
	public static ReportTable createFratryDistributionTable(final FiliationCounts counts) throws PuckException {
		ReportTable result;

		//
		result = new ReportTable(counts.size() + 1, 5);
		result.setTitle("Fratry Distribution");

		//
		result.set(0, 0, " ");
		result.set(0, 1, "Uterine");
		result.set(0, 2, "Agnatic");
		result.set(0, 3, "% Agnatic");
		result.set(0, 4, "% Uterine");

		// TODO 2 ?

		//
		int agnaticSum = counts.agnaticSum();
		int uterineSum = counts.uterineSum();
		for (int countIndex = 0; countIndex < counts.size(); countIndex++) {
			result.set(countIndex + 1, 0, countIndex);
			result.set(countIndex + 1, 1, MathUtils.toString(counts.get(countIndex).getUterine()));
			result.set(countIndex + 1, 2, MathUtils.toString(counts.get(countIndex).getAgnatic()));
			result.set(countIndex + 1, 3, MathUtils.percent(counts.get(countIndex).getUterine(), uterineSum));
			result.set(countIndex + 1, 4, MathUtils.percent(counts.get(countIndex).getAgnatic(), agnaticSum));
		}

		//
		return result;
	}

	/**
	 * 
	 * @param counts
	 * @return
	 */
	public static ReportChart createGenderBIASNetWeightChart(final BIASCounts counts) {
		ReportChart result;

		//
		result = new ReportChart("Gender BIAS (net weight)", GraphType.LINES);
		result.setHeadersLegend("Generational Distance");
		result.setLinesLegend("% Individuals");

		//
		result.setLineTitle("Only Uterine Ancestor Known", 0);
		result.setLineTitle("Only Agnatic Ancestor Known", 1);

		//
		for (int countIndex = 1; countIndex < counts.size(); countIndex++) {
			BIASCount count = counts.get(countIndex);
			if (count.isPositive()) {
				result.setHeader(MathUtils.toString(countIndex), countIndex);
				result.addValue(countIndex, count.getUterine(), 0);
				result.addValue(countIndex, count.getAgnatic(), 1);
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @param net
	 * @return
	 * @throws PuckException
	 */
	public static ReportTable createGenderBIASNetWeightTable(final BIASCounts counts) throws PuckException {
		ReportTable result;

		//
		result = new ReportTable(counts.size(), 5);
		result.setTitle("Gender BIAS (net weight)");

		//
		result.set(0, 0, " ");
		result.set(0, 1, "A or U");
		result.set(0, 2, "U not A");
		result.set(0, 3, "A not U");
		result.set(0, 4, "Diff");

		//
		for (int countIndex = 1; countIndex < counts.size(); countIndex++) {
			//
			BIASCount count = counts.get(countIndex);

			// Extract from Net.java Puck 1.0:
			// i
			// c
			// bias[0][i]
			// bias[1][i]
			// Math.round(100.*(bias[0][i]-bias[1][i]))/100.
			result.set(countIndex, 0, countIndex);
			result.set(countIndex, 1, count.getCoo());
			result.set(countIndex, 2, count.getUterine());
			result.set(countIndex, 3, count.getAgnatic());
			result.set(countIndex, 4, MathUtils.roundHundredth(count.getUterine() - count.getAgnatic()));
		}

		//
		return result;
	}

	/**
	 * 
	 * @param counts
	 * @return
	 */
	public static ReportChart createGenderBIASWeightChart(final BIASCounts counts) {
		ReportChart result;

		//
		result = new ReportChart("Gender BIAS (weight)", GraphType.LINES);
		result.setHeadersLegend("Generational Distance");
		result.setLinesLegend("% Individuals");

		//
		result.setLineTitle("Uterine Ancestor Known", 0);
		result.setLineTitle("Agnatic Ancestor Known", 1);
		result.setLineTitle("Agn. and Ut. Ancestor Known", 2);

		//
		for (int countIndex = 1; countIndex < counts.size(); countIndex++) {
			BIASCount count = counts.get(countIndex);
			if (count.isPositive()) {
				result.setHeader(MathUtils.toString(countIndex), countIndex);
				result.addValue(countIndex, count.getUterine(), 0);
				result.addValue(countIndex, count.getAgnatic(), 1);
				result.addValue(countIndex, count.getCognatic(), 2);
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @param net
	 * @return
	 * @throws PuckException
	 */
	public static ReportTable createGenderBIASWeightTable(final BIASCounts counts) throws PuckException {
		ReportTable result;

		//
		result = new ReportTable(counts.size(), 6);
		result.setTitle("Gender BIAS (weight)");

		//
		result.set(0, 0, " ");
		result.set(0, 1, "A or U");
		result.set(0, 2, "U");
		result.set(0, 3, "A");
		result.set(0, 4, "A and U");
		result.set(0, 5, "Diff");

		//
		for (int countIndex = 1; countIndex < counts.size(); countIndex++) {
			//
			BIASCount count = counts.get(countIndex);

			// Extract from Net.java Puck 1.0:
			// i
			// c
			// bias[0][i]
			// bias[1][i]
			// bias[2][i]
			// Math.round(100.*(bias[0][i]-bias[1][i]))/100.)
			result.set(countIndex, 0, countIndex);
			result.set(countIndex, 1, count.getCoo());
			result.set(countIndex, 2, count.getUterine());
			result.set(countIndex, 3, count.getAgnatic());
			result.set(countIndex, 4, count.getCognatic());
			result.set(countIndex, 5, MathUtils.roundHundredth(count.getUterine() - count.getAgnatic()));
		}

		//
		return result;
	}

	public static <E> ReportChart createMapChart(final Map<Value, Double[]> map, final int idx, final String label) throws PuckException {
		ReportChart result;

		if (map == null) {
			result = null;
		} else {
			//
			result = new ReportChart(label, GraphType.STACKED_BARS);

			//
			int index = 0;
			for (Value key : map.keySet()) {
				//
				result.setHeader(key.toString(), index);

				//
				result.addValue(map.get(key)[idx], 0);

				//
				index += 1;
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @param partitionCriteria
	 * @param splitCriteria
	 * @return
	 * @throws PuckException
	 */
	public static ReportChart createMeanClusterValuesChart(final String criteriaString, final Map<Cluster<Individual>, Double> source) throws PuckException {
		ReportChart result;

		if ((criteriaString == null) || (source == null)) {
			result = null;
		} else {
			result = new ReportChart("Mean Cluster Values (" + criteriaString + ")", GraphType.SCATTER);

			result.setHeadersLegend("Size");
			result.setLinesLegend("Mean Values");

			result.setLineTitle("Clusters", 0);
			int columnIndex = 0;
			for (Cluster<Individual> cluster : source.keySet()) {

				result.setHeader(cluster.getLabel(), columnIndex);
				result.addValue(cluster.size(), source.get(cluster), 0);

				columnIndex += 1;
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @param net
	 * @return
	 * @throws PuckException
	 */
	public static ReportTable createMeanClusterValuesTable(final String criteriaString, final Clusters<Individual> source,
			final Map<Cluster<Individual>, Double> means) throws PuckException {
		ReportTable result;

		if ((criteriaString == null) || (source == null) || (means == null)) {
			result = null;
		} else {
			result = new ReportTable(source.size() + 1, 4);
			result.setTitle("Mean Cluster Values (" + criteriaString + ")");

			result.set(0, 0, "Cluster");
			result.set(0, 1, "Name");
			result.set(0, 2, "Mean Value");
			result.set(0, 3, "Size");

			int rowIndex = 1;
			for (Cluster<Individual> cluster : source.toListSortedByDescendingSize()) {

				result.set(rowIndex, 0, rowIndex);
				result.set(rowIndex, 1, cluster.getLabel());
				result.set(rowIndex, 2, MathUtils.toString(means.get(cluster)));
				result.set(rowIndex, 3, MathUtils.toString(cluster.size()));

				rowIndex += 1;
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @param partitions
	 * @return
	 * @throws PuckException
	 */
	public static <E> ReportChart createMultiPartitionChart(final MultiPartition<E> partitions) throws PuckException {
		ReportChart result;

		// == Build Report Chart.
		result = new ReportChart(partitions.getLabel(), GraphType.STACKED_BARS);

		//
		{
			int columnIndex = 0;
			for (Value columnValue : partitions.colValues()) {
				result.setLineTitle(columnValue.toString(), columnIndex);

				int rowIndex = 0;
				for (Value rowValue : partitions.rowValues()) {
					result.setHeader(rowValue.toString(), rowIndex);
					result.setValue(partitions.frequency(rowValue, columnValue), columnIndex, rowIndex);
					rowIndex += 1;
				}

				columnIndex += 1;
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @param partitionCriteria
	 * @param splitCriteria
	 * @return
	 * @throws PuckException
	 */
	public static <E> ReportChart createPartitionChart(final Partition<E> partition) throws PuckException {
		ReportChart result;

		if (partition == null) {
			result = null;
		} else {
			//
			result = new ReportChart(partition.getLabel(), GraphType.STACKED_BARS);

			//
			int clusterIndex = 0;
			for (Cluster<E> cluster : partition.getClusters().toListSortedByValue()) {
				//
				result.setHeader(cluster.getLabel(), clusterIndex);

				//
				result.addValue(cluster.size(), 0);

				//
				clusterIndex += 1;
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @param partitionCriteria
	 * @param splitCriteria
	 * @return
	 * @throws PuckException
	 */
	public static ReportChart createPartitionChart(final Partition<Individual> source, final PartitionCriteria partitionCriteria,
			final PartitionCriteria splitCriteria) throws PuckException {
		ReportChart result;

		if ((source == null) || (partitionCriteria == null) || (!partitionCriteria.isValid())) {
			result = null;
		} else {
			//
			Partition<Individual> partition = source;

			//
			if ((partitionCriteria.getSizeFilter() != null) && (partitionCriteria.getSizeFilter() != SizeFilter.NONE)) {
				partition = PartitionMaker.createCleaned(partition, partitionCriteria.getSizeFilter());
			}
			if ((partitionCriteria.getValueFilter() != null) && (partitionCriteria.getValueFilter() != ValueFilter.NONE)) {
				partition = PartitionMaker.createCleaned(partition, partitionCriteria.getValueFilter());
			}

			//
			result = new ReportChart(partition.getLabel() + " " + partitionCriteria.toShortString(), GraphType.STACKED_BARS);

			//
			if (partitionCriteria.getCumulationType() == CumulationType.ASCENDANT) {
				//
				int clusterIndex = 0;
				int cumulation = 0;
				for (Cluster<Individual> cluster : partition.getClusters().toListSortedByValue()) {
					//
					result.setHeader(cluster.getLabel(), clusterIndex);

					//
					cumulation += cluster.size();
					result.addValue(cumulation, 0);

					//
					clusterIndex += 1;
				}
			} else if (partitionCriteria.getCumulationType() == CumulationType.DESCENDANT) {
				//
				int clusterIndex = partition.size() - 1;
				int cumulation = 0;
				List<Cluster<Individual>> clusters = partition.getClusters().toListSortedByValue();
				Collections.reverse(clusters);
				for (Cluster<Individual> cluster : clusters) {
					//
					result.setHeader(cluster.getLabel(), clusterIndex);

					//
					cumulation += cluster.size();
					result.setValue(cumulation, 0, clusterIndex);

					//
					clusterIndex -= 1;
				}
			} else if ((splitCriteria != null) && (splitCriteria.isValid())) {
				//
				int clusterIndex = 0;
				HashMap<String, Integer> clusterValueToLineIndex = new HashMap<String, Integer>();
				for (Cluster<Individual> cluster : partition.getClusters().toListSortedByValue()) {

					//
					Partition<Individual> split = PartitionMaker.create("split", new Individuals(cluster.getItems()), splitCriteria);
					logger.debug("split cluster size=" + split.size());
					//
					result.setHeader(cluster.getLabel(), clusterIndex);
					for (Cluster<Individual> splitCluster : split.getClusters().toListSortedByValue()) {
						String splitClusterLabel;
						if (splitCluster.getLabel() == null) {
							splitClusterLabel = "null";
						} else {
							splitClusterLabel = splitCluster.getLabel();
						}

						//
						Integer lineIndex = clusterValueToLineIndex.get(splitClusterLabel);

						if (lineIndex == null) {
							lineIndex = clusterValueToLineIndex.size();

							clusterValueToLineIndex.put(splitClusterLabel, lineIndex);
							result.setLineTitle(splitClusterLabel, lineIndex);
						}

						//
						result.setValue(splitCluster.size(), lineIndex, clusterIndex);
					}

					//
					clusterIndex += 1;
				}
			} else {
				//
				int clusterIndex = 0;
				for (Cluster<Individual> cluster : partition.getClusters().toListSortedByValue()) {
					//
					result.setHeader(cluster.getLabel(), clusterIndex);

					//
					result.addValue(cluster.size(), 0);

					//
					clusterIndex += 1;
				}
			}
		}

		//
		return result;
	}

	/**
	 * Generates a report.
	 * 
	 * @param net
	 *            Source to report.
	 * @throws PuckException
	 */
	public static Report reportBasicInformation(final Net net, final Segmentation segmentation) throws PuckException {
		Report result;

		//
		Chronometer chrono = new Chronometer();
		result = new Report();

		//
		result.setTitle("Basic statistics about a corpus.");
		result.setOrigin("Statistics reporter");
		result.setTarget(net.getLabel());

		//
		if (segmentation.isOn()) {
			result.setInputComment("Segmentation:\n" + segmentation.getSummary());
		}

		//
		StatisticsWorker worker = new StatisticsWorker(net);
		ReportAttributes items = new ReportAttributes();

		for (StatisticsWorker.Indicator indicator : StatisticsWorker.Indicator.values()) {
			if (indicator.equals(Indicator.MEAN_DEPTH) || indicator.equals(Indicator.MEAN_SPOUSE_DISTANCE_GEN)) {
				continue;
			}
			worker.addItem(items, indicator);
		}

		result.outputs().append(items);
		result.outputs().appendln();

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;
	}

	/**
	 * Generates a report.
	 * 
	 * @param net
	 *            Source to report.
	 * @throws PuckException
	 */
	public static Report reportCompleteness(final Segmentation segmentation) throws PuckException {
		Report result;

		Chronometer chrono = new Chronometer();
		result = new Report();
		result.setTitle("Completeness statistics about a corpus.");
		result.setOrigin("Statistics reporter");
		result.setTarget(segmentation.getLabel());

		//
		if (segmentation.isOn()) {
			result.setInputComment(segmentation.getSummary());
		}

		//
		FiliationCounts counts = StatisticsWorker.completeness(segmentation.getCurrentIndividuals(), 10);
		ReportChart reportChart = createCompletenessChart(counts);

		result.outputs().append(reportChart);
		result.outputs().appendln();

		//
		result.outputs().append(ReportTable.transpose(reportChart.createReportTableWithSum()));
		result.outputs().appendln();

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;
	}

	/**
	 * Generates a report.
	 * 
	 * @param net
	 *            Source to report.
	 * @throws PuckException
	 */
	public static Report reportFamiliesByHusband(final Net net) throws PuckException {
		Report result;

		Chronometer chrono = new Chronometer();
		result = new Report();
		result.setTitle("Family Spouse Ids (by Husband).");
		result.setOrigin("Statistics reporter");
		result.setTarget(net.getLabel());

		//
		List<String> strings = new ArrayList<String>();
		for (Family family : net.families().toSortedList(FamilyComparator.Sorting.HUSBAND)) {
			//
			int husbandId;
			String husbandName;

			if (family.getHusband() == null) {
				husbandId = 0;
				husbandName = "";
			} else {
				husbandId = family.getHusband().getId();
				husbandName = family.getHusband().getName();
			}

			//
			int wifeId;
			String wifeName;

			if (family.getWife() == null) {
				wifeId = 0;
				wifeName = "";
			} else {
				wifeId = family.getWife().getId();
				wifeName = family.getWife().getName();
			}

			//
			String string = husbandId + "\t" + husbandName + "\t" + wifeId + "\t" + wifeName + "\t" + family.getId();

			//
			strings.add(string);
		}

		result.outputs().appendln("Husband\tWife\tFamily");
		for (String string : strings) {
			result.outputs().appendln(string);
		}

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;
	}

	/**
	 * Generates a report.
	 * 
	 * @param net
	 *            Source to report.
	 * @throws PuckException
	 */
	public static Report reportFamiliesByWife(final Net net) throws PuckException {
		Report result;

		Chronometer chrono = new Chronometer();
		result = new Report();
		result.setTitle("Family Spouse Ids (by Wife).");
		result.setOrigin("Statistics reporter");
		result.setTarget(net.getLabel());

		//
		List<String> strings = new ArrayList<String>();
		for (Family family : net.families().toSortedList(FamilyComparator.Sorting.WIFE)) {
			//
			int husbandId;
			String husbandName;

			if (family.getHusband() == null) {
				husbandId = 0;
				husbandName = "";
			} else {
				husbandId = family.getHusband().getId();
				husbandName = family.getHusband().getName();
			}

			//
			int wifeId;
			String wifeName;

			if (family.getWife() == null) {
				wifeId = 0;
				wifeName = "";
			} else {
				wifeId = family.getWife().getId();
				wifeName = family.getWife().getName();
			}

			//
			String string = wifeId + "\t" + wifeName + "\t" + husbandId + "\t" + husbandName + "\t" + family.getId();

			//
			strings.add(string);
		}

		result.outputs().appendln("Wife\tHusband\tFamily");
		for (String string : strings) {
			result.outputs().appendln(string);
		}

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;
	}

	/**
	 * Generates a report.
	 * 
	 * @param net
	 *            Source to report.
	 * @throws PuckException
	 */
	public static Report reportGraphicsStatistics(final Segmentation segmentation, final StatisticsCriteria criteria, final File sourceFile)
			throws PuckException {
		Report result;

		if ((segmentation == null) || (criteria == null)) {
			result = null;
		} else {
			//
			Chronometer chrono = new Chronometer();

			//
			result = new Report();
			result.setTitle("Basic graphics statistics about a corpus.");
			result.setOrigin("Statistics reporter");
			result.setTarget(segmentation.getLabel());

			//
			if (segmentation.isOn()) {
				result.setInputComment("Segmentation:\n" + segmentation.getSummary());
			}

			//
			List<ReportChart> charts = new ArrayList<ReportChart>(20);
			List<ReportTable> tables = new ArrayList<ReportTable>(20);

			//
			if (criteria.isGenderBIASWeight()) {
				//
				BIASCounts counts = StatisticsWorker.biasWeights(segmentation.getCurrentIndividuals());
				charts.add(createGenderBIASWeightChart(counts));
				tables.add(createGenderBIASWeightTable(counts));
			}

			//
			if (criteria.isGenderBIASNetWeight()) {
				//
				BIASCounts counts = StatisticsWorker.biasNetWeights(segmentation.getCurrentIndividuals());
				charts.add(createGenderBIASNetWeightChart(counts));
				tables.add(createGenderBIASNetWeightTable(counts));
			}

			//
			if (criteria.isComponents()) {
				//
				charts.add(createComponentsChart(segmentation.getCurrentIndividuals(), 1000));
			}

			//
			if (criteria.isGenealogicalCompleteness()) {
				//
				FiliationCounts counts = StatisticsWorker.completeness(segmentation.getCurrentIndividuals(), 10);
				ReportChart chart = createCompletenessChart(counts);
				charts.add(chart);
				tables.add(ReportTable.transpose(chart.createReportTable()));
			}

			//
			if (criteria.isFratryDistribution()) {
				//
				FiliationCounts counts = StatisticsWorker.fratryDistribution(segmentation.getCurrentIndividuals());
				charts.add(createFratryDistributionChart(counts));
				tables.add(createFratryDistributionTable(counts));
			}

			//
			if (criteria.isFourCousinMarriages()) {
				Partition<Chain> partition = CircuitFinder.createFirstCousinMarriages(segmentation);
				ReportChart chart = createPartitionChart(partition);
				if (chart != null) {
					//
					charts.add(chart);
					tables.add(ReportTable.transpose(chart.createReportTableWithSum()));
				}
			}

			//
			if (criteria.isAncestorTypes()) {
				MultiPartition<Chain> partition = CircuitFinder.createAncestorChains(segmentation, criteria.getAncestorTypesDegree());
				ReportChart chart = createMultiPartitionChart(partition);
				if (chart != null) {
					//
					charts.add(chart);
					tables.add(ReportTable.transpose(chart.createReportTableWithSum()));
				}
			}

			//
			if (criteria.isConsanguineChains()) {
				double[][] consCount = StatisticsWorker.countConsanguinePairs(segmentation, criteria.getConsanguineDegree());
				ReportChart chart = createArrayChart("Consanguines per person", consCount, new String[] { "HH", "FF", "HF", "FH" });
				if (chart != null) {
					chart.setHeadersLegend("Generational distance");
					chart.setLinesLegend("Consanguines");

					//
					charts.add(chart);
					tables.add(ReportTable.transpose(chart.createReportTable()));
				}
			}

			// Compute charts and tables.
			StringList pajekBuffer = new StringList();
			for (PartitionCriteria partitionCriteria : criteria.getPartitionCriterias()) {
				//
				if (partitionCriteria.isValid()) {
					Partition<Individual> partition = PartitionMaker.create(segmentation.getLabel(), segmentation.getCurrentIndividuals(), partitionCriteria);

					ReportChart chart = createPartitionChart(partition, partitionCriteria, criteria.getSplitCriteria());
					if (chart != null) {
						//
						charts.add(chart);
						tables.add(ReportTable.transpose(chart.createReportTableWithSum()));

						// Create Mean Cluster Values chart.
						if ((criteria.isMeanClusterValues()) && (criteria.getSplitCriteria() != null)) {
							try {
								Map<Cluster<Individual>, Double> means = StatisticsWorker.meanClusterValues(partition.getClusters(),
										criteria.getSplitCriteria());

								String criteriaString = partitionCriteria.toShortString() + "/" + criteria.getSplitCriteria().getLabel();

								ReportChart meanChart = createMeanClusterValuesChart(criteriaString, means);
								charts.add(meanChart);

								tables.add(createMeanClusterValuesTable(criteriaString, partition.getClusters(), means));
							} catch (ClassCastException exception) {
								logger.debug("ClassCastException => No mean cluster value.");
								// If mean can't be calculated, do not build
								// chart
								// and table.
							}
						}
					}

					//
					partition.setLabel(partition.getLabel() + "_" + partitionCriteria.getLabel());
					pajekBuffer.addAll(PuckUtils.writePajekPartition(partition, new IndividualComparator(Sorting.ID)));
				}
			}

			// Manage the number of chart by line.
			for (int chartIndex = 0; chartIndex < charts.size(); chartIndex++) {
				result.outputs().append(charts.get(chartIndex));
				if (chartIndex % 4 == 3) {
					result.outputs().appendln();
				}
			}

			// Add chart tables.
			for (ReportTable table : tables) {
				result.outputs().appendln(table.getTitle());
				result.outputs().appendln(table);
			}

			//
			if (pajekBuffer.length() != 0) {

				//
				File targetFile = ToolBox.setExtension(ToolBox.addToName(sourceFile, "-Partitions"), ".paj");
				ReportRawData rawData = new ReportRawData("Export Partitions to Pajek", "Pajek", "paj", targetFile);
				rawData.setData(PAJFile.convertToMicrosoftEndOfLine(pajekBuffer.toString()));

				//
				result.outputs().appendln("Partitions");
				result.outputs().append(rawData);
			}

			//
			result.setTimeSpent(chrono.stop().interval());
		}

		//
		return result;
	}

	/**
	 * Generates a report.
	 * 
	 * @param net
	 *            Source to report.
	 * @throws PuckException
	 */
	public static Report reportHomonyms(final Net net) throws PuckException {
		Report result;

		Chronometer chrono = new Chronometer();
		result = new Report();
		result.setTitle("Homonyms.");
		result.setOrigin("Statistics reporter");
		result.setTarget(net.getLabel());

		Map<String, String> ids = new TreeMap<String, String>();
		Map<String, Integer> count = new TreeMap<String, Integer>();

		for (Individual individual : net.individuals()) {
			String name = individual.getName();
			String id = ids.get(name);
			if (id == null) {
				ids.put(name, individual.getId() + "");
				count.put(name, 1);
			} else {
				ids.put(name, id + ";" + individual.getId());
				count.put(name, count.get(name) + 1);
			}
		}

		result.outputs().appendln("Name\tIds");
		for (String name : ids.keySet()) {
			if (count.get(name) > 1) {
				result.outputs().appendln(name + "\t" + ids.get(name));
			}
		}
		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;
	}

	/*
	 * 
	 */
	public static Report reportIndividuals(final Segmentation segmentation, final Sorting sorting) {
		Report result;

		Chronometer chrono = new Chronometer();
		result = new Report();
		result.setTitle("Individual List.");
		result.setOrigin("Statistics reporter");
		result.setTarget(segmentation.getLabel());

		result.inputs().add("Sorting", sorting.toString());

		for (Individual individual : segmentation.getCurrentIndividuals().toSortedList(sorting)) {
			String signature = null;

			String firstname = individual.getFirstName();
			if (firstname == null) {
				firstname = "-";
			}

			String lastname = individual.getLastName();
			if (lastname == null) {
				lastname = "-";
			}

			switch (sorting) {
				case ID:
					signature = individual.getId() + "\t" + firstname + "\t" + lastname;
				break;
				case FIRSTN:
					signature = firstname + "\t" + lastname + "\t" + individual.getId();
				break;
				case LASTN:
					signature = lastname + "\t" + firstname + "\t" + individual.getId();
			}

			if (signature != null) {
				result.outputs().append(signature);
				result.outputs().appendln();
			}
		}

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;
	}

	/**
	 * 
	 * @param segmentation
	 * @param partitionLabel
	 * @param sorting
	 * @return
	 * @throws PuckException
	 */
	public static Report reportIndividuals(final Segmentation segmentation, final String partitionLabel, final Sorting sorting) throws PuckException {
		Report result;

		Chronometer chrono = new Chronometer();
		result = new Report();
		result.setTitle("Individual List.");
		result.setOrigin("Statistics reporter");
		result.setTarget(segmentation.getLabel());

		result.inputs().add("Partition", partitionLabel);
		result.inputs().add("Sorting", sorting.toString());

		Partition<Individual> partition = PartitionMaker.createRaw(partitionLabel, segmentation.getCurrentIndividuals());

		for (Cluster<Individual> cluster : partition.getClusters().toListSortedByValue()) {
			if (cluster == null || cluster.getValue() == null) {
				continue;
			}
			result.outputs().append(cluster.getValue().toString() + "\t(" + cluster.count() + ")");
			result.outputs().appendln();

			for (Individual individual : new Individuals(cluster.getItems()).toSortedList(sorting)) {
				String signature = null;

				switch (sorting) {
					case ID:
						signature = individual.getId() + "\t" + individual.getFirstName() + "\t" + individual.getLastName();
					break;
					case FIRSTN:
						signature = individual.getFirstName() + "\t" + individual.getLastName() + "\t" + individual.getId();
					break;
					case LASTN:
						signature = individual.getLastName() + "\t" + individual.getFirstName() + "\t" + individual.getId();
				}

				if (signature != null) {
					result.outputs().append(signature);
					result.outputs().appendln();
				}
			}
			result.outputs().appendln();
		}

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;
	}

	public static Report reportSynopsis(final String directory) {
		Report result;

		//
		Chronometer chrono = new Chronometer();
		result = new Report();

		//
		result.setTitle("Basic statistics about a corpus.");
		result.setOrigin("Statistics reporter");

		StringList listStatistics = new StringList();
		String headline = "General statistics";
		for (StatisticsWorker.Indicator indicator : StatisticsWorker.Indicator.values()) {
			headline += "\t" + indicator.toString();
		}
		result.outputs().appendln(headline);

		StringList listBias = new StringList();
		listBias.appendln("Gender bias\tA1\tA2\tA3\tA4\tA5\tU1\tU2\tU3\tU4\tU5");

		StringList listCensus = new StringList();
		listCensus.appendln("First cousin marriages\tParPat\tCross\tParMat");

		String pattern = "XX(X)XX";
		String classificationType = "LINE";

		File folder = new File(directory);
		for (File file : folder.listFiles()) {
			try {
				Net net = PuckManager.loadCorpus(file);
				listStatistics.appendln(StatisticsWorker.getValueString(net));
				listBias.appendln(StatisticsWorker.listBiasWeights(net));
				listCensus.appendln(StatisticsWorker.listCircuitFrequencies(new Segmentation(net), classificationType, pattern));
			} catch (PuckException e) {
				System.err.println("Not a corpus file: " + file.getName());
			}
		}
		result.outputs().append(listStatistics);
		result.outputs().appendln();
		result.outputs().append(listBias);
		result.outputs().appendln();
		result.outputs().append(listCensus);
		result.outputs().appendln();

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;
	}

}
