package org.tip.puck.partitions;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tip.puck.PuckException;
import org.tip.puck.PuckExceptions;
import org.tip.puck.census.chains.Chain;
import org.tip.puck.census.workers.ChainValuator;
import org.tip.puck.graphs.Node;
import org.tip.puck.graphs.workers.NodeValuator;
import org.tip.puck.net.Families;
import org.tip.puck.net.Family;
import org.tip.puck.net.Individual;
import org.tip.puck.net.Individuals;
import org.tip.puck.net.Net;
import org.tip.puck.net.relations.Relation;
import org.tip.puck.net.relations.Relations;
import org.tip.puck.net.workers.FamilyValuator;
import org.tip.puck.net.workers.IndividualValuator;
import org.tip.puck.net.workers.RelationValuator;
import org.tip.puck.partitions.Interval.EndpointStatus;
import org.tip.puck.partitions.PartitionCriteria.SizeFilter;
import org.tip.puck.partitions.PartitionCriteria.ValueFilter;
import org.tip.puck.util.MathUtils;
import org.tip.puck.util.NumberedValues;
import org.tip.puck.util.Value;
import org.tip.puck.util.Values;

/**
 * Don't be afraid by the number of methods. In fact, each source type has only
 * one specific method: create(Type, PartitionCriteria). All other methods are
 * helpers, they contains only one line to call the specific method.
 * 
 * @author TIP
 */
public class PartitionMaker {
	static private final Logger logger = LoggerFactory.getLogger(PartitionMaker.class);

	static final double MAX_INTERVALS = 10000;
	static final double MIN_VALUE = Double.MIN_VALUE / 4;
	static final double MAX_VALUE = Double.MAX_VALUE / 4;

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Partition<Individual> create(final Net source, final PartitionCriteria criteria) throws PuckException {
		Partition<Individual> result;

		if ((source == null) || (criteria == null)) {
			result = null;
		} else {
			//
			result = create(source.getLabel() + " " + criteria.getLabel(), source.individuals(), criteria);
		}

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Partition<Family> create(final String title, final Families source, final PartitionCriteria criteria) throws PuckException {
		Partition<Family> result;

		if ((source == null) || (criteria == null)) {
			result = null;
		} else {
			//
			result = new Partition<Family>();
			result.setLabel(title);
			result.setCriteria(criteria);

			//
			NumberedValues values = FamilyValuator.get(source, criteria.getLabel(), criteria.getLabelParameter());

			//
			NumberedValues partitionValues = getPartitionValues(values, criteria);

			//
			for (Family family : source) {
				result.put(family, partitionValues.get(family.getId()));
			}

		}

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Partition<Individual> create(final String title, final Individuals source, final PartitionCriteria criteria) throws PuckException {
		Partition<Individual> result;
		
		if ((source == null) || (criteria == null)) {
			result = null;
		} else {
			//
			result = new Partition<Individual>();
			result.setLabel(title);
			result.setCriteria(criteria);

			//
			NumberedValues values = IndividualValuator.get(source, criteria.getLabel(), criteria.getLabelParameter());

			//
			NumberedValues partitionValues = getPartitionValues(values, criteria);

			//
			for (Individual individual : source) {
				result.put(individual, partitionValues.get(individual.getId()));
			}

		}

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static <E> Partition<Node<E>> create(final String title, final List<Node<E>> source, final PartitionCriteria criteria) throws PuckException {
		Partition<Node<E>> result;

		if ((source == null) || (criteria == null)) {
			result = null;
		} else {
			//
			result = new Partition<Node<E>>();
			result.setLabel(title);
			result.setCriteria(criteria);

			//
			Values values = NodeValuator.get(source, criteria.getLabel());

			//
			Values partitionValues = getPartitionValues(values, criteria);

			//
			for (int index = 0; index < source.size(); index++) {
				result.put(source.get(index), partitionValues.get(index));
			}
		}

		//
		return result;
	}

	/**
	 * 
	 */
	public static Partition<Value> create(final String title, final List<Value> values) {
		Partition<Value> result;

		result = new Partition<Value>();
		result.setLabel(title);

		for (Value value : values) {
			result.put(value, value);
		}

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Partition<Chain> create(final String title, final Partition<Chain> source, final PartitionCriteria criteria) throws PuckException {
		Partition<Chain> result;

		if ((source == null) || (criteria == null)) {
			result = null;
		} else {
			//
			result = new Partition<Chain>();
			result.setLabel(title);
			result.setCriteria(criteria);

			//
			for (Cluster<Chain> cluster : source.getClusters()) {
				result.putAll(cluster.getItems(), ChainValuator.get(cluster.getFirstItem(), criteria.getLabel(), criteria.getLabelParameter()));

			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Partition<Relation> create(final String title, final Relations source, final PartitionCriteria criteria) throws PuckException {
		Partition<Relation> result;

		if ((source == null) || (criteria == null)) {
			result = null;
		} else {
			//
			result = new Partition<Relation>();
			result.setLabel(title);
			result.setCriteria(criteria);

			//
			NumberedValues values = RelationValuator.get(source, criteria.getLabel(), criteria.getLabelParameter());

			//
			NumberedValues partitionValues = getPartitionValues(values, criteria);

			//
			for (Relation relation : source) {
				result.put(relation, partitionValues.get(relation.getId()));
			}

		}

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Partition<Individual> createBinarization(final Net source, final String label, final String pattern) throws PuckException {
		Partition<Individual> result;

		result = create(source, PartitionCriteria.createBinarization(label, null, pattern));

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Partition<Individual> createBinarization(final Net source, final String label, final String labelParameter, final String pattern)
			throws PuckException {
		Partition<Individual> result;

		result = create(source, PartitionCriteria.createBinarization(label, labelParameter, pattern));

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Partition<Individual> createBinarization(final String title, final Individuals source, final String label, final String pattern)
			throws PuckException {
		Partition<Individual> result;

		result = create(title, source, PartitionCriteria.createBinarization(label, null, pattern));

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Partition<Individual> createBinarization(final String title, final Individuals source, final String label, final String labelParameter,
			final String pattern) throws PuckException {
		Partition<Individual> result;

		result = create(title, source, PartitionCriteria.createBinarization(label, labelParameter, pattern));

		//
		return result;
	}

	/**
	 * Copy a new partition removing cluster with filtered value.
	 * 
	 * @param source
	 *            The partition to copy.
	 * @param filter
	 *            The filter of value to remove.
	 * 
	 * @return A copy of the partition source without filtered value.
	 */
	public static <E> Partition<E> createCleaned(final Partition<E> source, final SizeFilter filter) {
		Partition<E> result;

		if (source == null) {
			result = null;
		} else if (filter == null) {
			result = createCopy(source);
		} else {
			switch (filter) {
				case NONE:
					result = createCopy(source);
				break;

				case TRIM: {
					//
					List<Cluster<E>> clusters = source.getClusters().toListSortedByValue();

					// Find the first cluster not empty.
					int startIndex = 0;
					{
						boolean ended = false;
						int index = 0;
						while (!ended) {
							if (index < clusters.size()) {
								Cluster<E> cluster = clusters.get(index);
								if (cluster.isNotEmpty()) {
									ended = true;
									startIndex = index;
								} else {
									index += 1;
								}
							} else {
								ended = true;
								startIndex = 0;
							}
						}
					}

					// Find the last cluster non empty.
					int endIndex = 0;
					{
						boolean ended = false;
						int index = clusters.size() - 1;
						while (!ended) {
							if (index >= 0) {
								Cluster<E> cluster = clusters.get(index);
								if (cluster.isNotEmpty()) {
									ended = true;
									endIndex = index;
								} else {
									index += 1;
								}
							} else {
								ended = true;
								endIndex = 0;
							}
						}
					}

					result = new Partition<E>(source.getLabel());
					for (int clusterIndex = startIndex; clusterIndex <= endIndex; clusterIndex++) {
						Cluster<E> cluster = clusters.get(clusterIndex);

						result.putAll(cluster.getItems(), cluster.getValue());
					}
				}
				break;

				case EMPTY:
					result = new Partition<E>(source.getLabel());
					for (Cluster<E> cluster : source.getClusters()) {
						if (cluster.isNotEmpty()) {
							result.putAll(cluster.getItems(), cluster.getValue());
						}
					}
				break;

				case HOLES:
					if (source.getClusters().isNumericOrNull()) {
						//
						result = new Partition<E>(source.getLabel());

						//
						List<Cluster<E>> clusters = source.getClusters().toListSortedByValue();

						//
						Cluster<E> maxCluster = Collections.max(clusters);
						if (maxCluster.getValue() == null) {
							result = createCopy(source);
						} else {
							//
							int count = 0;
							for (int clusterIndex = 0; clusterIndex < clusters.size(); clusterIndex++) {
								//
								Cluster<E> cluster = clusters.get(clusterIndex);

								//
								if (cluster.getValue() == null) {
									result.putAll(cluster.getItems(), cluster.getValue());
								} else {
									//
									int currentCount = cluster.getValue().intValue();

									//
									while (count < currentCount) {
										result.putCluster(new Value(count));
										count += 1;
									}

									//
									result.putAll(cluster.getItems(), cluster.getValue());
									count += 1;
								}
							}
						}
					} else {
						result = createCopy(source);
					}
				break;

				default:
					result = createCopy(source);
			}
		}

		//
		return result;
	}

	/**
	 * Copy a new partition removing cluster with filtered value.
	 * 
	 * @param source
	 *            The partition to copy.
	 * @param filter
	 *            The filter of value to remove.
	 * 
	 * @return A copy of the partition source without filtered value.
	 */
	public static <E> Partition<E> createCleaned(final Partition<E> source, final ValueFilter filter) {
		Partition<E> result;

		if (source == null) {
			result = null;
		} else if (filter == null) {
			result = createCopy(source);
		} else {
			switch (filter) {
				case NONE:
					result = createCopy(source);
				break;

				case NULL:
					result = new Partition<E>(source.getLabel());
					for (Cluster<E> cluster : source.getClusters()) {
						if (cluster.getValue() != null) {
							result.putAll(cluster.getItems(), cluster.getValue());
						}
					}
				break;

				case ZERO:
					result = new Partition<E>(source.getLabel());
					for (Cluster<E> cluster : source.getClusters()) {
						Value value = cluster.getValue();
						if ((value != null) && ((value.isNotNumber()) || (value.doubleValue() != 0))) {
							result.putAll(cluster.getItems(), cluster.getValue());
						}
					}
				break;

				default:
					result = createCopy(source);
			}
		}

		//
		return result;
	}

	/**
	 * Copy a new partition.
	 * 
	 * @param source
	 *            The partition to copy.
	 * 
	 * @return A copy of the partition source.
	 */
	public static <E> Partition<E> createCopy(final Partition<E> source) {
		Partition<E> result;

		if (source == null) {
			result = null;
		} else {
			result = new Partition<E>(source.getLabel());
			for (Cluster<E> cluster : source.getClusters()) {
				result.putAll(cluster.getItems(), cluster.getValue());
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Partition<Individual> createCountedGrouping(final Net source, final String label, final Double start, final Double count, final Double end)
			throws PuckException {
		Partition<Individual> result;

		result = create(source, PartitionCriteria.createCountedGrouping(label, null, start, count, end));

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Partition<Individual> createCountedGrouping(final Net source, final String label, final String labelParameter, final Double start,
			final Double count, final Double end) throws PuckException {
		Partition<Individual> result;

		result = create(source, PartitionCriteria.createCountedGrouping(label, labelParameter, start, count, end));

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Partition<Individual> createCountedGrouping(final String title, final Individuals source, final String label, final Double start,
			final Double count, final Double end) throws PuckException {
		Partition<Individual> result;

		result = create(title, source, PartitionCriteria.createCountedGrouping(label, null, start, count, end));

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Partition<Individual> createCountedGrouping(final String title, final Individuals source, final String label, final String labelParameter,
			final Double start, final Double count, final Double end) throws PuckException {
		Partition<Individual> result;

		result = create(title, source, PartitionCriteria.createCountedGrouping(label, labelParameter, start, count, end));

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Partition<Individual> createGrouping(final Net source, final String label, final Intervals intervals) throws PuckException {
		Partition<Individual> result;

		result = create(source, PartitionCriteria.createGrouping(label, null, intervals));

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Partition<Individual> createGrouping(final Net source, final String label, final String steps) throws PuckException {
		Partition<Individual> result;

		result = create(source, PartitionCriteria.createGrouping(label, null, getIntervals(steps)));

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Partition<Individual> createGrouping(final String title, final Individuals source, final String label, final Intervals intervals)
			throws PuckException {
		Partition<Individual> result;

		result = create(title, source, PartitionCriteria.createGrouping(label, null, intervals));

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Partition<Individual> createGrouping(final String title, final Individuals source, final String label, final String steps)
			throws PuckException {
		Partition<Individual> result;

		result = create(title, source, PartitionCriteria.createGrouping(label, null, getIntervals(steps)));

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Partition<Individual> createGrouping(final String title, final Individuals source, final String label, final String labelParameter,
			final Intervals intervals) throws PuckException {
		Partition<Individual> result;

		result = create(title, source, PartitionCriteria.createGrouping(label, labelParameter, intervals));

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Partition<Individual> createGrouping(final String title, final Individuals source, final String label, final String labelParameter,
			final String steps) throws PuckException {
		Partition<Individual> result;

		result = create(title, source, PartitionCriteria.createGrouping(label, labelParameter, getIntervals(steps)));

		//
		return result;
	}

	/**
	 * 
	 * @return
	 */
	public static <E> Partition<E> createNumerized(final Partition<E> source) {
		Partition<E> result;

		if (source == null) {
			result = null;
		} else {
			if (source.isNumeric()) {
				result = source;
			} else {
				result = new Partition<E>();
				result.setLabel(source.getLabel());

				List<Cluster<E>> clusterList = source.getClusters().toListSortedByValue();
				for (int i = 1; i <= clusterList.size(); i++) {
					result.putAll(clusterList.get(i - 1).getItems(), new Value(i));
				}
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Partition<Individual> createRaw(final Net source, final String label) throws PuckException {
		Partition<Individual> result;

		result = create(source, PartitionCriteria.createRaw(label, null));

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Partition<Individual> createRaw(final Net source, final String label, final String labelParameter) throws PuckException {
		Partition<Individual> result;

		result = create(source, PartitionCriteria.createRaw(label, labelParameter));

		//
		return result;
	}

	/**
	 * 
	 * @param title
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Partition<Individual> createRaw(final String title, final Individuals source) throws PuckException {
		return create(title, source, PartitionCriteria.createRaw(title));
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Partition<Individual> createRaw(final String title, final Individuals source, final String label) throws PuckException {
		Partition<Individual> result;

		result = create(title, source, PartitionCriteria.createRaw(label, null));

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Partition<Individual> createRaw(final String title, final Individuals source, final String label, final String labelParameter)
			throws PuckException {
		Partition<Individual> result;

		result = create(title, source, PartitionCriteria.createRaw(label, labelParameter));

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Partition<Individual> createSizedGrouping(final Net source, final String label, final Double start, final Double size, final Double end)
			throws PuckException {
		Partition<Individual> result;

		result = create(source, PartitionCriteria.createSizedGrouping(label, null, start, size, end));

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Partition<Individual> createSizedGrouping(final Net source, final String label, final String labelParameter, final Double start,
			final Double size, final Double end) throws PuckException {
		Partition<Individual> result;

		result = create(source, PartitionCriteria.createSizedGrouping(label, labelParameter, start, size, end));

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Partition<Individual> createSizedGrouping(final String title, final Individuals source, final String label, final Double start,
			final Double size, final Double end) throws PuckException {
		Partition<Individual> result;

		result = create(title, source, PartitionCriteria.createSizedGrouping(label, null, start, size, end));

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Partition<Individual> createSizedGrouping(final String title, final Individuals source, final String label, final String labelParameter,
			final Double start, final Double size, final Double end) throws PuckException {
		Partition<Individual> result;

		result = create(title, source, PartitionCriteria.createSizedGrouping(label, labelParameter, start, size, end));

		//
		return result;
	}

	/**
	 * 
	 * @return
	 * @throws PuckException
	 */
	public static Value getBinarizedValue(final Value sourceValue, final String pattern) {
		Value result;

		//
		String targetPattern = pattern.replaceAll("\\*", ".*");

		//
		if (sourceValue == null) {
			result = null;
		} else if (StringUtils.isEmpty(pattern)) {
			result = new Value("nonnull");
		} else if (sourceValue.stringValue().matches(targetPattern)) {
			result = new Value(pattern);
		} else {
			result = null;
		}

		//
		return result;
	}

	/**
	 * 
	 * @return
	 * @throws PuckException
	 */
	public static NumberedValues getBinarizedValues(final NumberedValues source, final String pattern) {
		NumberedValues result;

		if (source == null) {
			result = null;
		} else if (pattern == null) {
			result = source;
		} else {
			//
			result = new NumberedValues(source.size());
			for (Integer id : source.keySet()) {
				Value sourceValue = source.get(id);
				Value targetValue = getBinarizedValue(sourceValue, pattern);
				result.put(id, targetValue);
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @return
	 * @throws PuckException
	 */
	public static Values getBinarizedValues(final Values source, final String pattern) {
		Values result;

		if (source == null) {
			result = null;
		} else if (pattern == null) {
			result = source;
		} else {
			//
			result = new Values(source.size());
			for (Value sourceValue : source) {
				result.add(getBinarizedValue(sourceValue, pattern));
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @return
	 * @throws PuckException
	 */
	public static Value getGroupingValue(final Value sourceValue, final Intervals intervals) {
		Value result;

		//
		if ((sourceValue == null) || (sourceValue.isNotNumber())) {
			result = null;
		} else {
			Interval interval = intervals.find(sourceValue.doubleValue());
			if (interval == null) {
				result = null;
			} else {
				result = new Value(interval);
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @return
	 * @throws PuckException
	 */
	public static NumberedValues getGroupingValues(final NumberedValues source, final Intervals intervals) {
		NumberedValues result;

		if (source == null) {
			result = null;
		} else if (intervals == null) {
			result = source;
		} else {
			//
			result = new NumberedValues(source.size());
			for (Integer id : source.keySet()) {
				Value sourceValue = source.get(id);
				Value targetValue = getGroupingValue(sourceValue, intervals);
				result.put(id, targetValue);
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @return
	 * @throws PuckException
	 */
	public static Values getGroupingValues(final Values source, final Intervals intervals) {
		Values result;

		if (source == null) {
			result = null;
		} else if (intervals == null) {
			result = source;
		} else {
			//
			result = new Values(source.size());
			for (Value sourceValue : source) {
				result.add(getGroupingValue(sourceValue, intervals));
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @param steps
	 * @return
	 */
	public static Intervals getIntervals(final String steps) {
		Intervals result;

		//
		result = new Intervals();

		//
		if (StringUtils.isNotBlank(steps)) {
			//
			List<Double> values = new ArrayList<Double>();
			String[] strings = steps.split("[\\s]+");
			for (String string : strings) {
				if (NumberUtils.isNumber(string)) {
					values.add(Double.parseDouble(string));
				}
			}

			//
			Collections.sort(values);

			//
			if (values.size() == 1) {
				Interval interval = new Interval(values.get(0), EndpointStatus.INCLUDED, values.get(0), EndpointStatus.INCLUDED);
				result.add(interval);
			} else if (values.size() > 1) {
				for (int index = 0; index < values.size() - 1; index++) {
					Interval interval;
					if (index == values.size() - 1) {
						interval = new Interval(values.get(index), EndpointStatus.INCLUDED, values.get(index + 1), EndpointStatus.INCLUDED);
					} else {
						interval = new Interval(values.get(index), EndpointStatus.INCLUDED, values.get(index + 1), EndpointStatus.EXCLUDED);
					}
					result.add(interval);
				}
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @param start
	 * @param size
	 * @param end
	 * @return
	 * @throws PuckException
	 */
	public static Intervals getIntervalsByCount(final Double start, final Double size, final Double end) throws PuckException {
		Intervals result;

		result = getIntervalsByCount(start, size, end, MIN_VALUE, MAX_VALUE);

		//
		return result;
	}

	/**
	 * 
	 * @param start
	 * @param size
	 * @param end
	 * @param min
	 * @param max
	 * 
	 * @return
	 * 
	 * @throws PuckException
	 */
	public static Intervals getIntervalsByCount(final Double start, final Double count, final Double end, final Double min, final Double max)
			throws PuckException {
		Intervals result;
		// logger.debug("start=" + start + ", count=" + count + ", end=" + end +
		// ", min=" + min + ", max=" + max);
		//
		Double targetCount;
		if ((count == null) || (count == 0)) {
			targetCount = 1.0;
		} else {
			targetCount = count;
		}

		//
		double targetStart;
		if (start == null) {
			if (min == null) {
				targetStart = MIN_VALUE;
			} else {
				targetStart = min;
			}
		} else {
			targetStart = start;
		}

		//
		double targetEnd;
		if (end == null) {
			if (max == null) {
				targetEnd = MAX_VALUE;
			} else {
				targetEnd = max;
			}
		} else {
			targetEnd = end;
		}

		//
		double targetSize;
		if ((MathUtils.isDecimalInteger(targetStart)) && (MathUtils.isDecimalInteger(targetEnd))) {
			targetSize = Math.ceil((targetEnd - targetStart) / targetCount);
		} else {
			targetSize = (targetEnd - targetStart) / targetCount;
		}

		//
		result = getIntervalsBySize(targetStart, targetSize, targetEnd, min, max);

		//
		return result;
	}

	/**
	 * 
	 * @param start
	 * @param size
	 * @param end
	 * @return
	 * @throws PuckException
	 */
	public static Intervals getIntervalsBySize(final Double start, final Double size, final Double end) throws PuckException {
		Intervals result;

		result = getIntervalsBySize(start, size, end, MIN_VALUE, MAX_VALUE);

		//
		return result;
	}

	/**
	 * 
	 * @param start
	 * @param size
	 * @param end
	 * @param min
	 * @param max
	 * 
	 * @return
	 * 
	 * @throws PuckException
	 */
	public static Intervals getIntervalsBySize(final Double start, final Double size, final Double end, final Double min, final Double max)
			throws PuckException {
		Intervals result;
		// logger.debug("start=" + start + ", size=" + size + ", end=" + end +
		// ", min=" + min + ", max=" + max);
		//
		double targetStart;
		if (start == null) {
			if (min == null) {
				targetStart = MIN_VALUE;
			} else {
				targetStart = min;
			}
		} else {
			targetStart = start;
		}

		//
		double targetEnd;
		if (end == null) {
			if (max == null) {
				targetEnd = MAX_VALUE;
			} else {
				targetEnd = max;
			}
		} else {
			targetEnd = end;
		}

		//
		double targetSize;
		if (size == null) {
			targetSize = 1;
		} else {
			targetSize = size;
		}

		//
		double intervalCount = (targetEnd - targetStart) / targetSize;
		if (intervalCount > MAX_INTERVALS) {
			throw PuckExceptions.OVERFLOW.create("Too numerous intervals: " + intervalCount + " > " + MAX_INTERVALS + ".");
		}

		//
		result = new Intervals();
		// logger.debug("targetStart=" + targetStart + ", targetSize=" +
		// targetSize + ", targetEnd=" + targetEnd);

		//
		if ((start == null) && (end != null)) {
			for (double value = targetEnd; value > targetStart; value -= targetSize) {
				Interval interval;
				if (value - targetSize <= targetStart) {
					interval = new Interval(targetStart, EndpointStatus.INCLUDED, value, EndpointStatus.INCLUDED);
				} else {
					interval = new Interval(value - targetSize, EndpointStatus.INCLUDED, value, EndpointStatus.EXCLUDED);
				}
				result.add(interval);
			}
			Collections.reverse(result);
		} else {
			for (double value = targetStart; value < targetEnd; value += targetSize) {
				Interval interval;
				if (value + targetSize >= targetEnd) {
					interval = new Interval(value, EndpointStatus.INCLUDED, targetEnd, EndpointStatus.INCLUDED);
				} else {
					interval = new Interval(value, EndpointStatus.INCLUDED, value + targetSize, EndpointStatus.EXCLUDED);
				}
				result.add(interval);
			}
		}

		//
		if ((start != null) && (min < start)) {
			result.add(new Interval(min, EndpointStatus.INCLUDED, start, EndpointStatus.EXCLUDED));
		}

		//
		if ((end != null) && (end < max)) {
			result.add(new Interval(end, EndpointStatus.EXCLUDED, max, EndpointStatus.INCLUDED));
		}

		// logger.debug("======" + result.toString());

		//
		return result;
	}

	/**
	 * 
	 * @param values
	 * @param criteria
	 * @return
	 * @throws PuckException
	 */
	public static NumberedValues getPartitionValues(final NumberedValues values, final PartitionCriteria criteria) throws PuckException {
		NumberedValues result;

		if (values == null) {
			result = null;
		} else if (criteria == null) {
			result = values;
		} else {
			//
			switch (criteria.getType()) {
				case RAW: {
					result = values;
				}
				break;

				case BINARIZATION: {
					result = getBinarizedValues(values, criteria.getPattern());
				}
				break;

				case FREE_GROUPING: {
					result = getGroupingValues(values, criteria.getIntervals());
				}
				break;

				case COUNTED_GROUPING: {
					Intervals intervals = getIntervalsByCount(criteria.getStart(), criteria.getCount(), criteria.getEnd(), values.min(), values.max());
					result = getGroupingValues(values, intervals);
				}
				break;

				case SIZED_GROUPING: {
					Intervals intervals = getIntervalsBySize(criteria.getStart(), criteria.getSize(), criteria.getEnd(), values.min(), values.max());
					result = getGroupingValues(values, intervals);
				}
				break;

				default:
					result = new NumberedValues();
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @param values
	 * @param criteria
	 * @return
	 * @throws PuckException
	 */
	public static Values getPartitionValues(final Values source, final PartitionCriteria criteria) throws PuckException {
		Values result;

		if (source == null) {
			result = null;
		} else if (criteria == null) {
			result = source;
		} else {
			//
			switch (criteria.getType()) {
				case RAW: {
					result = source;
				}
				break;

				case BINARIZATION: {
					result = getBinarizedValues(source, criteria.getPattern());
				}
				break;

				case FREE_GROUPING: {
					result = getGroupingValues(source, criteria.getIntervals());
				}
				break;

				case COUNTED_GROUPING: {
					Intervals intervals = getIntervalsByCount(criteria.getStart(), criteria.getCount(), criteria.getEnd(), source.min(), source.max());
					result = getGroupingValues(source, intervals);
				}
				break;

				case SIZED_GROUPING: {
					Intervals intervals = getIntervalsBySize(criteria.getStart(), criteria.getSize(), criteria.getEnd(), source.min(), source.max());
					result = getGroupingValues(source, intervals);
				}
				break;

				default:
					result = new Values();
			}
		}

		//
		return result;
	}

}
