package org.tip.puck.partitions;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tip.puck.util.Value;

/**
 * 
 * @author TIP
 */
public class Partition<E> {

	static private final Logger logger = LoggerFactory.getLogger(Partition.class);

	private String label;
	private PartitionCriteria criteria;
	private Clusters<E> clusters;
	private HashMap<E, Value> itemToValueShortcuts;

	/**
	 * 
	 */
	public Partition() {
		this.clusters = new Clusters<E>();
		this.itemToValueShortcuts = new HashMap<E, Value>();

	}

	/**
	 * 
	 */
	public Partition(final String label) {
		this.label = label;
		this.clusters = new Clusters<E>();
		this.itemToValueShortcuts = new HashMap<E, Value>();

	}

	/**
	 * 
	 * @param partition
	 */
	public void add(final Partition<E> partition) {

		for (Value value : partition.getValues()) {
			if (getCluster(value) == null) {
				putCluster(value);
			}
			for (E item : partition.getCluster(value).getItems()) {
				put(item, value);
			}
		}

		/*		for (E item : partition.getItems()){
					Value value = partition.itemToValueShortcuts.get(item);
					put(item,value);
				}*/
	}

	/**
	 * 
	 * @return
	 */
	public int clusteredItemsCount() {
		int result;

		result = 0;
		for (Cluster<E> cluster : getClusters()) {
			if (!cluster.isNull()) {
				result += cluster.count();
			}
		}
		//
		return result;
	}

	public Map<Integer, Integer> clusterSizeDistribution() {
		Map<Integer, Integer> result;

		result = new TreeMap<Integer, Integer>();

		for (Cluster<E> cluster : getClusters()) {
			if (cluster.getLabel() != null) {
				int size = cluster.size();
				Integer number = result.get(size);
				if (number == null) {
					result.put(size, 1);
				} else {
					result.put(size, number + 1);
				}
			}
		}
		//
		return result;
	}

	/**
	 * 
	 * @param value
	 * @return
	 */
	public boolean containsValue(final Value value) {
		boolean result;

		// TODO Improve algorithm.
		result = false;
		for (Cluster<E> cluster : getClusters()) {
			if (cluster.getValue().equals(value)) {
				result = true;
			}
		}

		return result;
	}

	/**
	 * Returns the cluster of an element.
	 * 
	 * @param item
	 * @return
	 */
	public Cluster<E> getCluster(final E item) {
		Cluster<E> result;

		if (this.itemToValueShortcuts.containsKey(item)) {
			Value value = getValue(item);
			result = getCluster(value);
		} else {
			result = null;
		}

		//
		return result;
	}

	/**
	 * Returns the cluster of a value.
	 * 
	 * @param value
	 * @return
	 */
	public Cluster<E> getCluster(final Value value) {
		Cluster<E> result;

		result = this.clusters.get(value);

		//
		return result;
	}

	public Clusters<E> getClusters() {
		return clusters;
	}

	public PartitionCriteria getCriteria() {
		return criteria;
	}

	/**
	 * 
	 * @return
	 */
	public String getCriteriaLabel() {
		String result;

		if (this.criteria == null) {
			result = null;
		} else {
			result = this.criteria.getLabel();
		}

		//
		return result;
	}

	/**
	 * 
	 * @return
	 */
	public Set<E> getItems() {
		Set<E> result;

		result = this.itemToValueShortcuts.keySet();

		//
		return result;
	}

	public List<E> getItemsAsSortedList(final Comparator<E> comparator) {
		List<E> result;

		result = new ArrayList<E>(this.itemToValueShortcuts.keySet());
		Collections.sort(result, comparator);

		//
		return result;
	}

	public String getLabel() {
		return label;
	}

	/**
	 * Returns the cluster value of a given item.
	 * 
	 * @param item
	 *            The vertex to be checked.
	 * 
	 * @return The cluster value of the vertex.
	 */
	public Value getValue(final E item) {
		Value result;

		result = this.itemToValueShortcuts.get(item);

		//
		return result;
	}

	/**
	 * 
	 * @param value
	 * @return
	 */
	public int getValueFrequency(final Value value) {
		int result;

		Cluster<E> cluster = this.clusters.get(value);
		if (cluster == null) {
			result = 0;
		} else {
			result = cluster.size();
		}

		//
		return result;
	}

	/**
	 * Returns the cluster value of a given item.
	 * 
	 * @param item
	 *            The vertex to be checked.
	 * 
	 * @return The cluster value of the vertex.
	 */
	public Value getValueNotNull(final E item) {
		Value result;

		result = getValue(item);

		if (result == null) {
			result = new Value(0);
		}

		//
		return result;
	}

	/**
	 * 
	 * @return
	 */
	public Collection<Value> getValues() {
		Collection<Value> result;

		result = this.getClusters().getValues();

		//
		return result;
	}

	/**
	 * 
	 * @return
	 */
	public boolean isNotNumeric() {
		boolean result;

		result = !isNumeric();

		//
		return result;
	}

	/**
	 * 
	 * @return
	 */
	public boolean isNumeric() {
		boolean result;

		if (label.equals("ID") || label.equals("ORDER")) {
			result = false;
		} else {
			result = this.clusters.isNumeric();
		}

		//
		return result;
	}

	/**
	 * 
	 * @return
	 */
	public int itemsCount() {
		int result;

		result = getItems().size();

		//
		return result;
	}

	/**
	 * Gets the maximal Cluster.
	 * 
	 * @return the maximal Cluster
	 */
	public Cluster<E> maxCluster() {
		Cluster<E> result;

		result = null;
		for (Cluster<E> cluster : this.clusters) {
			if ((result == null) || (cluster.size() > result.size())) {
				result = cluster;
			}
		}

		//
		return result;
	}

	public double maxShare() {
		return share(maxCluster());
	}

	public double meanShare() {
		return new Double(1) / new Double(size());
	}

	public double meanShare(final int minSize) {
		double result;

		double clustersCount = 0.;

		for (Cluster<E> cluster : clusters) {
			if (cluster.size() >= minSize) {
				clustersCount += 1;
			}
		}

		result = 1. / clustersCount;
		//
		return result;
	}

	/**
	 * 
	 * @param item
	 * @param partitionValue
	 */
	public void put(final E item, final Value partitionValue) {

		//
		Cluster<E> cluster = clusters.get(partitionValue);
		if (cluster == null) {
			cluster = new Cluster<E>(partitionValue);
			clusters.put(cluster);
		}

		//
		cluster.put(item);

		//
		this.itemToValueShortcuts.put(item, partitionValue);
	}

	/**
	 * 
	 * @param items
	 * @param partitionValue
	 */
	public void putAll(final List<E> items, final Value partitionValue) {

		if (items != null) {
			//
			if (items.isEmpty()) {
				putCluster(partitionValue);
			} else {
				//
				for (E item : items) {
					put(item, partitionValue);
				}
			}
		}
	}

	/**
	 * 
	 * @param value
	 */
	public void putCluster(final Value value) {
		Cluster<E> cluster = new Cluster<E>(value);
		clusters.put(cluster);
	}

	public void setCriteria(final PartitionCriteria criteria) {
		this.criteria = criteria;
	}

	public void setLabel(final String label) {
		this.label = label;
	}

	/**
	 * Gets the share of a cluster (the percentage of nodes contained in the
	 * cluster).
	 * 
	 * @param c
	 *            the cluster
	 */
	public double share(final Cluster<E> cluster) {
		double result;

		int itemsCount = itemsCount();

		if (itemsCount == 0) {
			result = 0;
		} else {
			result = Double.valueOf(cluster.count()) / Double.valueOf(itemsCount);
		}

		//
		return result;
	}

	/**
	 * 
	 * @return
	 */
	public int size() {
		int result;

		result = this.clusters.size();

		//
		return result;
	}

	public void swap(final E item, final Cluster<E> cluster1, final Cluster<E> cluster2) {

		//
		if (item != null && cluster1 != null && cluster1.getItems().contains(item)) {
			cluster1.remove(item);
			put(item, cluster2.getValue());
		}
	}

	public void swap(final E item, final Value value1, final Value value2) {

		//
		Cluster<E> cluster = clusters.get(value2);
		if (item != null && cluster != null && cluster.getItems().contains(item)) {
			cluster.remove(item);
			put(item, value2);
		}
	}
}
