package org.tip.puck.net.workers;

import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tip.puck.PuckException;
import org.tip.puck.PuckExceptions;
import org.tip.puck.census.chains.Chain;
import org.tip.puck.census.chains.Couple;
import org.tip.puck.census.chains.Notation;
import org.tip.puck.census.workers.CircuitFinder;
import org.tip.puck.graphs.Graph;
import org.tip.puck.graphs.Link;
import org.tip.puck.graphs.Node;
import org.tip.puck.net.Attribute;
import org.tip.puck.net.Families;
import org.tip.puck.net.Family;
import org.tip.puck.net.Individual;
import org.tip.puck.net.Individuals;
import org.tip.puck.net.KinType;
import org.tip.puck.net.Net;
import org.tip.puck.net.relations.Actor;
import org.tip.puck.net.relations.Relation;
import org.tip.puck.net.relations.RelationModel;
import org.tip.puck.net.relations.Role;
import org.tip.puck.net.workers.UpdateWorker.UpdateMode;
import org.tip.puck.partitions.Cluster;
import org.tip.puck.partitions.Partition;
import org.tip.puck.partitions.PartitionMaker;
import org.tip.puck.segmentation.Segment;
import org.tip.puck.segmentation.Segmentation;

/**
 * 
 * Methods added in this classes has to be static and autonomous.
 * 
 * @author TIP
 */
public class NetUtils {

	static private final Logger logger = LoggerFactory.getLogger(NetUtils.class);

	/**
	 * Anonymizes a net leaving only the first name.
	 * 
	 * @param net
	 *            Source to anonymize.
	 */
	public static void anonymizeByFirstName(final Individuals source) {
		for (Individual individual : source) {
			individual.setName(individual.getFirstName());
		}
	}

	/**
	 * Anonymizes a net leaving only the first name.
	 * 
	 * @param net
	 *            Source to anonymize.
	 */
	public static void anonymizeByFirstName(final Net source) {
		anonymizeByFirstName(source.individuals());
	}

	/**
	 * Anonymizes a net replacing name by gender and id.
	 * 
	 * @param net
	 *            Source to anonymize.
	 */
	public static void anonymizeByGenderAndId(final Individuals source) {
		for (Individual individual : source) {
			individual.setName(individual.getGender().toChar() + " " + individual.getId());
		}
	}

	/**
	 * Anonymizes a net replacing name by gender and id.
	 * 
	 * @param net
	 *            Source to anonymize.
	 */
	public static void anonymizeByGenderAndId(final Net source) {
		anonymizeByGenderAndId(source.individuals());
	}

	/**
	 * Anonymizes a net leaving only the first name.
	 * 
	 * @param net
	 *            Source to anonymize.
	 */
	public static void anonymizeByLastName(final Individuals source) {
		for (Individual individual : source) {
			individual.setName(individual.getLastName() + " " + individual.getId());
		}
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Net buildCleanedNet(final Net source) throws PuckException {
		Net result;

		//
		result = new Net();

		//
		result.setLabel(source.getLabel());

		//
		for (Attribute attribute : source.attributes().toSortedList()) {
			result.attributes().put(attribute.getLabel(), attribute.getValue());
		}

		// Create individuals (with bad data).
		for (Individual sourceIndividual : source.individuals()) {

			// Set individual data. Warning:
			// <ul>
			// <li> origin family is setted with source data and it will be
			// fixed later
			// <li> personal families is setted with source data and it will be
			// fixed later
			// <li> realtions is not setted and it will be done later.
			// </ul>
			Individual targetIndividual = new Individual(sourceIndividual.getId());
			targetIndividual.setGender(sourceIndividual.getGender());
			targetIndividual.setName(sourceIndividual.getName());
			targetIndividual.setOriginFamily(sourceIndividual.getOriginFamily());
			targetIndividual.getPersonalFamilies().add(sourceIndividual.getPersonalFamilies());
			targetIndividual.attributes().addAll(sourceIndividual.attributes());

			//
			result.individuals().add(targetIndividual);
		}

		// Create family and fix data.
		for (Family sourceFamily : source.families()) {
			//
			int fatherId;
			if (sourceFamily.getHusband() == null) {
				fatherId = 0;
			} else {
				fatherId = sourceFamily.getHusband().getId();
			}

			//
			int motherId;
			if (sourceFamily.getWife() == null) {
				motherId = 0;
			} else {
				motherId = sourceFamily.getWife().getId();
			}

			// Get parent family or create it.
			// Note: If one parent is unknown, we can't associate
			// the individual to an existent family, we have to
			// avoid german relation. So, if one parent is
			// unknown, we have to create a new family.
			Family targetFamily;
			if ((fatherId == 0) || (motherId == 0)) {
				targetFamily = null;
			} else {
				targetFamily = result.families().getBySpouses(fatherId, motherId);
			}

			if (targetFamily == null) {
				targetFamily = new Family(result.families().size() + 1);
				result.families().add(targetFamily);

				//
				if (sourceFamily.getHusband() != null) {
					targetFamily.setHusband(result.individuals().getById(sourceFamily.getHusband().getId()));
				}
				if (sourceFamily.getWife() != null) {
					targetFamily.setWife(result.individuals().getById(sourceFamily.getWife().getId()));
				}
				targetFamily.setUnionStatus(sourceFamily.getUnionStatus());

				for (Individual sourceChild : sourceFamily.getChildren()) {
					Individual targetChild = result.individuals().getById(sourceChild.getId());
					targetFamily.getChildren().add(targetChild);
					targetChild.setOriginFamily(targetFamily);
				}

				targetFamily.attributes().addAll(sourceFamily.attributes());

			} else {
				for (Individual sourceChild : sourceFamily.getChildren()) {
					if (targetFamily.getChildren().getById(sourceChild.getId()) == null) {
						Individual targetChild = result.individuals().getById(sourceChild.getId());
						targetFamily.getChildren().add(targetChild);
						targetChild.setOriginFamily(targetFamily);
					}
				}

				if (sourceFamily.isMarried()) {
					targetFamily.setMarried(true);
				}

				targetFamily.attributes().addAll(sourceFamily.attributes());
			}

			//
			if (sourceFamily.getHusband() != null) {
				Individual targetParent = result.individuals().getById(sourceFamily.getHusband().getId());
				targetParent.getPersonalFamilies().removeById(sourceFamily.getId());
				targetParent.getPersonalFamilies().add(targetFamily);
			}

			//
			if (sourceFamily.getWife() != null) {
				Individual targetParent = result.individuals().getById(sourceFamily.getWife().getId());
				targetParent.getPersonalFamilies().removeById(sourceFamily.getId());
				targetParent.getPersonalFamilies().add(targetFamily);
			}
		}

		//
		for (RelationModel model : source.relationModels()) {
			result.relationModels().add(new RelationModel(model));
		}

		//
		for (Relation sourceRelation : source.relations()) {
			//
			Relation targetRelation = new Relation(sourceRelation.getId(), sourceRelation.getTypedId(), result.relationModels().getByName(
					sourceRelation.getModel().getName()), sourceRelation.getName());
			result.relations().add(targetRelation);

			//
			for (Actor sourceActor : sourceRelation.actors()) {
				result.createRelationActor(targetRelation, sourceActor.getId(), sourceActor.getRole().getName());
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 */
	public static Net copy(final Net source) {
		Net result;

		result = new Net(source);

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 */
	public static Net copyWithMarriedCoparents(final Net source) {
		Net result;

		result = new Net(source);
		marryCoparents(result);

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Net copyWithoutMarkedDoubles(final Net source) throws PuckException {
		Net result;

		result = new Net(source);
		eliminateMarkedDoubles(result);

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 */
	public static Net copyWithoutSingles(final Net source) {
		Net result;

		result = new Net(source);
		eliminateSingles(result);

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 */
	public static Net copyWithoutStructuralChildren(final Net source) {
		Net result;

		result = new Net(source);
		eliminateStructuralChildren(result);

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 */
	public static Net copyWithoutVirtuals(final Net source) {
		Net result;

		result = new Net(source);
		eliminateVirtuals(result);

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Graph<Individual> createOreGraph(final Segmentation source) throws PuckException {
		Graph<Individual> result;

		//
		result = new Graph<Individual>("Ore graph " + source.getLabel());

		//
		for (Individual individual : source.getCurrentIndividuals().toSortedList()) {
			result.addNode(individual);
		}

		//
		for (Family family : source.getCurrentFamilies()) {
			Individual father = family.getHusband();
			Individual mother = family.getWife();
			if (father != null && mother != null && family.isMarried()) {
				result.addEdge(father, mother, 1);
			}
			if (!family.isSterile()) {
				for (Individual child : family.getChildren()) {
					if (father != null) {
						result.addArc(father, child, 1);
					}
					if (mother != null) {
						result.addArc(mother, child, 1);
					}
				}
			}
		}

		//
		NetUtils.setGenderShapes(result);

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Graph<Family> createPGraph(final Segmentation source) throws PuckException {
		Graph<Family> result;
		//
		result = new Graph<Family>("Pgraph " + source.getLabel());

		// getAllFamilies would be better
		int familyCount = source.getCurrentFamilies().getLastId();

		for (Individual individual : source.getCurrentIndividuals()) {
			Family originFamily = individual.getOriginFamily();
			if (originFamily == null) {
				familyCount++;
				originFamily = new Family(familyCount);
				originFamily.getChildren().add(individual);
			}
			Families personalFamilies = individual.getPersonalFamilies();
			if (personalFamilies.size() == 0) {
				familyCount++;
				Family personalFamily = new Family(familyCount);
				if (individual.isMale()) {
					personalFamily.setHusband(individual);
				} else if (individual.isFemale()) {
					personalFamily.setWife(individual);
				}
			}
			for (Family personalFamily : personalFamilies) {
				int weight = 1 - 2 * (individual.getGender().toInt() % 2);

				result.addArc(originFamily, personalFamily, weight);
			}
		}

		//
		return result;

	}

	/**
	 * 
	 * 
	 * @param net
	 * @throws PuckException
	 */
	public static RelationModel createRelationsFromCircuits(final Net net, final CircuitFinder finder, final RelationModel model) throws PuckException {
		RelationModel result;

		//
		model.roles().add(new Role("PIVOT"));
		model.roles().add(new Role("INTERMEDIARY"));

		//
		int i = 0;
		for (Cluster<Chain> cluster : finder.getCircuits().getClusters().toListSortedByValue()) {
			for (Chain r : cluster.getItems()) {
				//
				Relation relation = net.createRelation(i, r.signature(Notation.NUMBERS), model);
				for (int j = 0; j < 2 * r.dim(); j++) {
					Individual pivot = r.getPivot(j);
					if (!relation.hasActor(pivot)) {
						net.createRelationActor(relation, pivot.getId(), "PIVOT");
					}
				}
				relation.attributes().add(new Attribute("TYPE", cluster.getValue().toString()));
				relation.attributes().add(new Attribute("CLASSIC", r.signature(Notation.CLASSIC_GENDERED)));

				for (Individual indi : r) {
					if (!relation.hasActor(indi)) {
						if (indi instanceof Couple) {
							if (((Couple) indi).getFirstId() > 0) {
								net.createRelationActor(relation, ((Couple) indi).getFirstId(), "INTERMEDIARY");
							}
							if (((Couple) indi).getSecondId() > 0) {
								net.createRelationActor(relation, ((Couple) indi).getSecondId(), "INTERMEDIARY");
							}
						} else {
							net.createRelationActor(relation, indi.getId(), "INTERMEDIARY");
						}
					}
				}
				i++;
			}
		}

		//
		result = model;

		//
		return result;
	}

	/**
	 * 
	 * 
	 * @param net
	 * @throws PuckException
	 */
	public static RelationModel createRelationsFromFamilies(final Net net) throws PuckException {
		RelationModel result;

		//
		RelationModel model = net.createRelationModel("FamiliesPlus");

		//
		model.roles().add(new Role("HUSBAND"));
		model.roles().add(new Role("WIFE"));
		model.roles().add(new Role("CHILD"));

		//
		for (Family family : net.families()) {

			//
			Relation relation = net.createRelation(family.getId(), family.hashKey(), model);

			//
			if (family.getHusband() != null) {
				net.createRelationActor(relation, family.getHusband().getId(), "HUSBAND");
			}

			if (family.getWife() != null) {
				net.createRelationActor(relation, family.getWife().getId(), "WIFE");
			}

			for (Individual child : family.getChildren()) {
				net.createRelationActor(relation, child.getId(), "CHILD");
			}

			//
			relation.attributes().addAll(family.attributes());
		}

		//
		result = model;

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Graph<Individual> createTipGraph(final Segmentation source) throws PuckException {

		Graph<Individual> result;
		//
		result = new Graph<Individual>("Tip graph " + source.getLabel());

		for (Individual individual : source.getCurrentIndividuals().toSortedList()) {
			result.addNode(individual);
		}

		//
		for (Family family : source.getCurrentFamilies()) {
			//
			Individual father = family.getHusband();
			Individual mother = family.getWife();

			//
			if (father != null && mother != null && family.isMarried()) {
				Link<Individual> link = result.addArc(mother, father, 1);
				link.setTag(":1 'H.F'");
			}

			//
			if (!family.isSterile()) {
				for (Individual child : family.getChildren()) {
					if (father != null) {
						//
						int relationCode;
						String relationPattern;
						switch (child.getGender()) {
							case FEMALE:
								relationCode = 4;
								relationPattern = "'F(H)'";
							break;
							case MALE:
								relationCode = 5;
								relationPattern = "'H(H)'";
							break;
							default:
								relationCode = 7;
								relationPattern = "'X(H)'";
						}

						//
						Link<Individual> link = result.addArc(father, child, relationCode);
						link.setTag(":" + relationCode + " " + relationPattern);
					}

					if (mother != null) {
						//
						int relationCode;
						String relationPattern;
						switch (child.getGender()) {
							case FEMALE:
								relationCode = 2;
								relationPattern = "'F(F)'";
							break;
							case MALE:
								relationCode = 3;
								relationPattern = "'M(F)'";
							break;
							default:
								relationCode = 6;
								relationPattern = "'X(F)'";
						}

						//
						Link<Individual> link = result.addArc(mother, child, relationCode);
						link.setTag(":" + relationCode + " " + relationPattern);
					}
				}
			}
		}

		//
		NetUtils.setGenderShapes(result);

		//
		return result;
	}

	/**
	 * Eliminates the individuals marked as double (i.e., individuals having a
	 * number instead of a name).
	 * <p>
	 * The "name" of a double is the ID number of the original.
	 * 
	 * @param net
	 *            Source.
	 * 
	 * @return The number of single individuals eliminated.
	 * @throws PuckException
	 */
	public static int eliminateMarkedDoubles(final Net source) throws PuckException {
		int result;

		result = eliminateMarkedDoubles(source, source.individuals());

		//
		return result;
	}

	/**
	 * Eliminates the individuals marked as double (i.e., individuals having a
	 * number instead of a name).
	 * <p>
	 * The "name" of a double is the ID number of the original.
	 * 
	 * @param net
	 *            Source.
	 * 
	 * @return The number of single individuals eliminated.
	 * @throws PuckException
	 */
	public static int eliminateMarkedDoubles(final Net source, final Individuals sample) throws PuckException {
		int result;

		List<Individual> individuals = sample.toList();

		result = 0;
		for (Individual sourceIndividual : individuals) {
			if (NumberUtils.isNumber(sourceIndividual.getFirstName())) {
				//
				int originalId = Integer.parseInt(sourceIndividual.getFirstName());
				Individual targetIndividual = source.individuals().getById(originalId);
				targetIndividual.setAttribute("DOUBLE", String.valueOf(sourceIndividual.getId()));

				// Append attributes.
				UpdateWorker.update(targetIndividual.attributes(), sourceIndividual.attributes(), UpdateMode.APPEND);

				// Append parents.
				if ((targetIndividual.getFather() == null) && (sourceIndividual.getFather() != null)) {
					NetUtils.setFatherRelation(source, sourceIndividual.getFather().getId(), targetIndividual.getId());
				}
				if ((targetIndividual.getMother() == null) && (sourceIndividual.getMother() != null)) {
					NetUtils.setMotherRelation(source, sourceIndividual.getMother().getId(), targetIndividual.getId());
				}

				// Append or fuse personalFamilies
				for (Family sourceFamily : sourceIndividual.getPersonalFamilies()) {
					Family targetFamily;
					if (sourceIndividual == sourceFamily.getHusband()) {
						targetFamily = source.families().getBySpouses(targetIndividual, sourceFamily.getWife());
						if (targetFamily == null) {
							sourceFamily.setHusband(targetIndividual);
							targetIndividual.addPersonalFamily(sourceFamily);
						} else {
							targetFamily.getChildren().add(sourceFamily.getChildren());
							for (Individual child : sourceFamily.getChildren()) {
								child.setOriginFamily(targetFamily);
							}
							if (sourceFamily.getWife() != null) {
								sourceFamily.getWife().getPersonalFamilies().removeById(sourceFamily.getId());
							}
							source.families().removeById(sourceFamily.getId());
						}
					} else if (sourceIndividual == sourceFamily.getWife()) {
						targetFamily = source.families().getBySpouses(sourceFamily.getHusband(), targetIndividual);
						if (targetFamily == null) {
							sourceFamily.setWife(targetIndividual);
							targetIndividual.addPersonalFamily(sourceFamily);
						} else {
							targetFamily.getChildren().add(sourceFamily.getChildren());
							for (Individual child : sourceFamily.getChildren()) {
								child.setOriginFamily(targetFamily);
							}
							if (sourceFamily.getHusband() != null) {
								sourceFamily.getHusband().getPersonalFamilies().removeById(sourceFamily.getId());
							}
							source.families().removeById(sourceFamily.getId());
						}
					}
				}

				/*
				 * for (Individual partner : sourceIndividual.getPartners()) {
				 * NetUtils.setSpouseRelationAndFixRoles(source,
				 * targetIndividual.getId(), partner.getId()); }
				 * 
				 * // Append children. for (Individual child :
				 * sourceIndividual.getChildren()) { if
				 * (sourceIndividual.isFemale()){
				 * NetUtils.setMotherRelation(source, targetIndividual.getId(),
				 * child.getId()); } else { NetUtils.setFatherRelation(source,
				 * targetIndividual.getId(), child.getId()); } }
				 */
				result += 1;
				source.remove(sourceIndividual);
			}
		}

		//
		return result;
	}

	/**
	 * Eliminates the individuals marked as double (i.e., individuals having a
	 * number instead of a name).
	 * <p>
	 * The "name" of a double is the ID number of the original.
	 * 
	 * @param net
	 *            Source.
	 * 
	 * @return The number of single individuals eliminated.
	 */
	public static int eliminateMarkedDoubles1(final Net source, final Individuals sample) {
		int result;

		List<Individual> individuals = sample.toList();
		result = 0;
		for (Individual individual : individuals) {
			if (NumberUtils.isNumber(individual.getFirstName())) {
				//
				int originalId = Integer.parseInt(individual.getFirstName());
				Individual original = source.individuals().getById(originalId);
				original.setAttribute("DOUBLE", String.valueOf(individual.getId()));
				source.remove(individual);
				result += 1;

				// Update Attributes.
				// updateAttributes(firstIndi, secondIndi);
				for (Attribute sourceAttribute : individual.attributes()) {
					Attribute targetAttribute = original.attributes().get(sourceAttribute.getLabel());
					if (targetAttribute == null) {
						original.attributes().put(sourceAttribute.getLabel(), sourceAttribute.getValue());
					} else {
						targetAttribute.setValue(sourceAttribute.getValue());
					}
				}

				// TODO
				// updateKin(firstIndi, secondIndi, erase);
				// {
				// KinType kinType = KinType.CHILD;
				// int position = -1;
				// for (Individual child : individual.getChildren()) {
				// position++;
				// addRelation(original, child, kinType, position, true);
				// removeRelative(child, individual, kinType, false);
				// }
				// }
				//
				// {
				// int position = -1;
				// for (Individual spouse : individual.getSpouses()) {
				// position++;
				// addRelation(original, spouse, kinType, position, true);
				// removeRelative(spouse, individual, kinType, false);
				// }
				// }
				//
				// {
				// if (individual.getMother() != null) {
				// Individual mother = individual.getMother();
				// addRelation(original, mother, kinType, position, true);
				// removeRelative(mother, individual, kinType, false);
				// }
				// if (individual.getFather() != null) {
				// Individual father = individual.getFather();
				// addRelation(original, father, kinType, position, true);
				// removeRelative(father, individual, kinType, false);
				// }
				// }
			}
		}

		//
		return result;
	}

	/**
	 * Eliminates single individual.
	 * 
	 * @param net
	 *            Source.
	 * 
	 * @return The number of single individuals eliminated.
	 */
	public static int eliminateSingles(final Net source) {
		int result;

		result = eliminateSingles(source, source.individuals());

		//
		return result;
	}

	/**
	 * Eliminates single individual.
	 * 
	 * @param net
	 *            Source.
	 * 
	 * @return The number of single individuals eliminated.
	 */
	public static int eliminateSingles(final Net source, final Individuals sample) {
		int result;

		List<Individual> individuals = sample.toList();
		result = 0;
		for (Individual individual : individuals) {
			if (individual.isSingle()) {
				//
				source.remove(individual);

				//
				result += 1;
			}
		}

		//
		return result;
	}

	/**
	 * @param net
	 *            Source.
	 * 
	 * @return The number of single individuals eliminated.
	 */
	public static int eliminateStructuralChildren(final Net source) {
		int result;

		result = eliminateStructuralChildren(source, source.individuals());

		//
		return result;
	}

	/**
	 * @param net
	 *            Source.
	 * 
	 * @return The number of single individuals eliminated.
	 */
	public static int eliminateStructuralChildren(final Net source, final Individuals sample) {
		int result;

		List<Individual> individuals = sample.toList();
		result = 0;
		for (Individual individual : individuals) {
			if ((individual.isSingle()) && (individual.isSterile())) {
				source.remove(individual);
				result += 1;
			}
		}

		//
		return result;
	}

	/**
	 * @param net
	 *            Source.
	 * 
	 * @return The number of single individuals eliminated.
	 */
	public static int eliminateVirtuals(final Net source) {
		int result;

		result = eliminateVirtuals(source, source.individuals());

		//
		return result;
	}

	/**
	 * @param net
	 *            Source.
	 * 
	 * @return The number of single individuals eliminated.
	 */
	public static int eliminateVirtuals(final Net source, final Individuals sample) {
		int result;

		List<Individual> individuals = sample.toList();
		result = 0;
		for (Individual individual : individuals) {
			if ((StringUtils.isBlank(individual.getName())) || individual.getName().equals("?") || (individual.getName().charAt(0) == '#')) {
				source.remove(individual);
				result += 1;
			}
		}

		//
		return result;
	}

	private static void expand(final Individuals individuals, final Individual ego, final KinType kinType) {
		if (!individuals.contains(ego)) {
			individuals.add(ego);

			Individuals kin;
			if (kinType == null) {
				kin = ego.getKin();
			} else {
				kin = ego.getKin(kinType);
			}

			for (Individual alter : kin) {
				expand(individuals, alter, kinType);
			}
		}
	}

	public static Net expand(final Net source, final Segment segment, final KinType direction) {
		Net result;

		Individuals individuals = new Individuals();
		for (Individual individual : segment.getCurrentIndividuals()) {
			expand(individuals, individual, direction);
		}

		result = extract(source, individuals);

		String label = result.getLabel() + "_" + segment.getLabel() + "_expanded";
		if (direction != null) {
			label = label + "_" + direction;
		}
		result.setLabel(label);

		//
		return result;

	}

	/**
	 * TODO manage relations
	 * 
	 * @param target
	 *            Net to update.
	 * @param source
	 *            Net containing data to put in target.
	 */
	public static Net extract(final Net source, final Individuals sourceIndividuals) {
		Net result;

		//
		result = new Net();
		result.setLabel(source.getLabel());

		// Extract individuals.
		for (Individual sourceIndividual : sourceIndividuals) {
			//
			Individual targetIndividual = new Individual(sourceIndividual.getId());
			targetIndividual.setGender(sourceIndividual.getGender());
			targetIndividual.setName(sourceIndividual.getName());
			targetIndividual.setOriginFamily(sourceIndividual.getOriginFamily());
			targetIndividual.getPersonalFamilies().add(sourceIndividual.getPersonalFamilies());
			targetIndividual.attributes().addAll(sourceIndividual.attributes());

			//
			result.individuals().add(targetIndividual);
		}

		// Extract families.
		for (Family sourceFamily : source.families()) {
			//
			Individual sourceFather = sourceFamily.getFather();
			Individual targetFather;
			if (sourceFather == null) {
				targetFather = null;
			} else {
				targetFather = result.individuals().getById(sourceFather.getId());
			}

			//
			Individual sourceMother = sourceFamily.getMother();
			Individual targetMother;
			if (sourceMother == null) {
				targetMother = null;
			} else {
				targetMother = result.individuals().getById(sourceMother.getId());
			}

			//
			if ((targetFather != null) || (targetMother != null)) {
				//
				Family targetFamily = new Family(sourceFamily.getId());
				targetFamily.setMarried(sourceFamily.isMarried());
				targetFamily.setFather(targetFather);
				targetFamily.setMother(targetMother);

				//
				for (Individual sourceChild : sourceFamily.getChildren()) {
					Individual targetChild = result.individuals().getById(sourceChild.getId());
					if (targetChild != null) {
						targetFamily.getChildren().add(targetChild);
					}
				}

				//
				if (((targetFather != null) && (targetMother != null)) || (!targetFamily.getChildren().isEmpty())) {
					result.families().add(targetFamily);
				}
			}
		}

		//
		for (Individual targetIndividual : result.individuals()) {
			//
			Family sourceOriginFamily = targetIndividual.getOriginFamily();
			if (sourceOriginFamily != null) {
				Family targetOriginFamily = result.families().getById(sourceOriginFamily.getId());

				targetIndividual.setOriginFamily(targetOriginFamily);
			}

			//
			for (Family sourcePersonalFamily : targetIndividual.getPersonalFamilies().toList()) {
				//
				Family targetPersonalFamily = result.families().getById(sourcePersonalFamily.getId());

				//
				targetIndividual.getPersonalFamilies().removeById(sourcePersonalFamily.getId());

				//
				if (targetPersonalFamily != null) {
					targetIndividual.getPersonalFamilies().add(targetPersonalFamily);
				}
			}
		}

		//
		return result;
	}

	public static Net extractByClusterSize(final Net source, final String label, final int minimalNumberOfMembers) throws PuckException {
		Net result;

		Partition<Individual> partition = PartitionMaker.createRaw(source, label);
		Individuals individuals = new Individuals();

		for (Cluster<Individual> cluster : partition.getClusters()) {
			if (!cluster.isNull() && cluster.count() >= minimalNumberOfMembers) {
				individuals.add(cluster.getItems());
			}
		}

		result = extract(source, individuals);
		result.setLabel(result.getLabel() + "_" + label + "_min_" + minimalNumberOfMembers);

		//
		return result;
	}

	public static Net extractByClusterValue(final Net source, final String label, final int minimalValue) throws PuckException {
		Net result;

		Partition<Individual> partition;

		String[] labels = label.split("\\s");
		if (labels.length == 2) {
			partition = PartitionMaker.createRaw(source, labels[0], labels[1]);
		} else {
			partition = PartitionMaker.createRaw(source, label);
		}

		Individuals individuals = new Individuals();

		for (Cluster<Individual> cluster : partition.getClusters()) {
			if (!cluster.isNull() && cluster.getValue().doubleValue() >= minimalValue) {
				individuals.add(cluster.getItems());
			}
		}

		result = extract(source, individuals);
		result.setLabel(result.getLabel() + "_" + label + "_min_" + minimalValue);

		//
		return result;
	}

	/**
	 * 
	 * @param parent1
	 * @param parent2
	 * @return
	 */
	public static Individual fixFatherByGender(final Individual ego, final Individual alter) {
		Individual result;

		if ((ego == null) && (alter == null)) {
			result = null;
		} else if (ego == null) {
			if (alter.isFemale()) {
				result = ego;
			} else {
				result = alter;
			}
		} else if (alter == null) {
			if (ego.isMale()) {
				result = ego;
			} else {
				result = alter;
			}
		} else {
			if (ego.isMale() || (alter.isFemale() && ego.isUnknown())) {
				result = ego;
			} else {
				result = alter;
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @param parent1
	 * @param parent2
	 * @return
	 */
	public static Individual fixMotherByGender(final Individual ego, final Individual alter) {
		Individual result;

		Individual newHusband = fixFatherByGender(ego, alter);

		if (newHusband == ego) {
			result = alter;
		} else {
			result = ego;
		}

		//
		return result;
	}

	/**
	 * 
	 * @param family
	 */
	public static void fixSpouseRolesByGender(final Family family) {

		if (family != null) {
			//
			Individual newHusband = fixFatherByGender(family.getHusband(), family.getWife());

			if (newHusband != family.getHusband()) {
				swapParents(family);
			}
		}
	}

	/**
	 * 
	 * @param family
	 */
	public static void fixSpouseRolesByGender2(final Family family) {

		if (family != null) {
			Individual ego = family.getHusband();
			Individual alter = family.getWife();

			Individual husband;
			Individual wife;
			if (ego.isMale() || (alter.isFemale() && ego.isUnknown())) {
				husband = ego;
				wife = alter;
			} else {
				husband = alter;
				wife = ego;
			}

			family.setHusband(husband);
			family.setWife(wife);
		}
	}

	/**
	 * 
	 * @param father
	 * @param mother
	 * @return
	 */
	public static boolean isFemaleFatherOrMaleMother(final Individual father, final Individual mother) {
		boolean result;

		if ((father == null) || (mother == null)) {
			result = false;
		} else if ((father.isFemale()) || (mother.isMale())) {
			result = true;
		} else {
			result = false;
		}

		//
		return result;
	}

	/**
	 * 
	 * @param father
	 * @param mother
	 * @return
	 */
	public static boolean isParentChildMarriage(final Individual parent, final Individual children) {
		boolean result;

		if ((parent == null) || (children == null)) {
			result = false;
		} else if (children.getSpouses().contains(parent)) {
			result = true;
		} else {
			result = false;
		}

		//
		return result;
	}

	/**
	 * 
	 * @param father
	 * @param mother
	 * @return
	 */
	public static boolean isParentChildMarriage(final Individual parent, final Individual... children) {
		boolean result;

		if (parent == null) {
			result = false;
		} else {

			boolean ended = false;
			result = false;
			int childIndex = 0;
			while (!ended) {
				if (childIndex < children.length) {
					if (isParentChildMarriage(parent, children[childIndex])) {
						ended = true;
						result = true;
					} else {
						childIndex += 1;
					}
				} else {
					ended = true;
					result = false;
				}
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @param father
	 * @param mother
	 * @return
	 */
	public static boolean isParentChildMarriage(final Individual father, final Individual mother, final Individual... children) {
		boolean result;

		if ((isParentChildMarriage(father, children)) || (isParentChildMarriage(mother, children))) {
			result = true;
		} else {
			result = false;
		}

		//
		return result;
	}

	/**
	 * 
	 * @param parent1
	 * @param parent2
	 * @return
	 */
	public static boolean isRolesFixedByGender(final Individual ego, final Individual alter) {
		boolean result;

		Individual newHusband = fixFatherByGender(ego, alter);

		if (newHusband == ego) {
			result = true;
		} else {
			result = false;
		}

		//
		return result;
	}

	/**
	 * 
	 * @return
	 */
	public static boolean isSame(final Individual ego, final Individual alter) {
		boolean result;

		if ((ego == null) || (alter == null)) {
			result = false;
		} else if (ego == alter) {
			result = true;
		} else {
			result = false;
		}

		//
		return result;
	}

	/**
	 * 
	 * @return
	 */
	public static boolean isSameSex(final Individual ego, final Individual alter) {
		boolean result;

		if ((ego == null) || (alter == null) || (ego.isUnknown()) || (alter.isUnknown())) {
			result = false;
		} else if (ego.getGender() == alter.getGender()) {
			result = true;
		} else {
			result = false;
		}

		//
		return result;
	}

	public static int marryCoparents(final Net source) {
		int result;

		//
		result = 0;

		for (Family family : source.families()) {
			if (!family.isSterile() && !family.isSingleParent() && !family.isMarried()) {
				family.setMarried();
				result++;
			}
		}

		//
		return result;

	}

	/**
	 * Adds the identity number between parentheses to the original name (e.g.
	 * "Pierre Dupont (5)").
	 * 
	 * @param net
	 *            Source to modify.
	 */
	public static void numberNames(final Individuals source) {
		if (source != null) {
			for (Individual individual : source) {
				individual.setName(individual.getTrimmedName() + " (" + individual.getId() + ")");
			}
		}
	}

	/**
	 * Adds the identity number between parentheses to the original name (e.g.
	 * "Pierre Dupont (5)").
	 * 
	 * @param net
	 *            Source to modify.
	 */
	public static void numberNames(final Net source) {
		if (source != null) {
			numberNames(source.individuals());
		}
	}

	/**
	 * 
	 * @param family
	 */
	public static void removeFather(final Family family) {
		if (family != null) {
			family.getFather().getPersonalFamilies().removeById(family.getId());
			family.setFather(null);
		}
	}

	/**
	 * 
	 * @param family
	 */
	public static void removeMother(final Family family) {
		if ((family != null) && (family.getMother() != null)) {
			family.getMother().getPersonalFamilies().removeById(family.getId());
			family.setMother(null);
		}
	}

	/**
	 * 
	 * @param family
	 */
	public static void removeSpouse(final Family family, final Individual spouse) {
		if ((family != null) && (spouse != null)) {
			if (family.getFather() == spouse) {
				removeFather(family);
			} else if (family.getMother() == spouse) {
				removeMother(family);
			}
		}
	}

	/**
	 * Re-numberate a net.
	 * 
	 * @param net
	 *            Source to renumberate.
	 */
	public static void renumberate(final Net source) {
		// TODO.
	}

	/**
	 * 
	 * @param parentId
	 * @param childId
	 * @throws PuckException
	 */
	public static void setAttribute(final Net net, final int individualId, final String label, final String value) throws PuckException {
		if ((net != null) && (individualId != 0) && (StringUtils.isNotBlank(label)) && (StringUtils.isNotBlank(value))) {
			//
			Individual individual = net.individuals().getById(individualId);
			if (individual == null) {
				individual = new Individual(individualId);
				net.individuals().add(individual);
			}

			//
			individual.attributes().put(label, value);
		}
	}

	/**
	 * WARNING: the use of this method can created duplicated families. WARNING:
	 * use only if source does not contain family ids.
	 * 
	 * @param parentId
	 * @param childId
	 * @throws PuckException
	 */
	public static void setFatherRelation(final Net net, final int fatherId, final int childId) throws PuckException {
		if ((net != null) && (fatherId != 0) && (childId != 0)) {
			//
			Individual parent = net.individuals().getById(fatherId);
			if (parent == null) {
				parent = new Individual(fatherId);
				net.individuals().add(parent);
			}

			//
			Individual child = net.individuals().getById(childId);
			if (child == null) {
				child = new Individual(childId);
				net.individuals().add(child);
			}

			//
			Family childSourceFamily;
			if (child.getOriginFamily() == null) {
				childSourceFamily = new Family(net.families().size() + 1);
				net.families().add(childSourceFamily);
				child.setOriginFamily(childSourceFamily);
				childSourceFamily.getChildren().add(child);
			} else {
				childSourceFamily = child.getOriginFamily();
				if (childSourceFamily.getFather() != null) {
					childSourceFamily.getFather().getPersonalFamilies().removeById(childSourceFamily.getId());
				}
			}
			childSourceFamily.setFather(parent);

			//
			Family parentPersonalFamily = parent.getPersonalFamilies().getById(childSourceFamily.getId());
			if (parentPersonalFamily == null) {
				parent.getPersonalFamilies().add(childSourceFamily);
			}
		}
	}

	/**
	 * 
	 * @param graph
	 */
	public static void setGenderShapes(final Graph<Individual> graph) {
		for (Node<Individual> node : graph.getNodes()) {
			node.setTag(node.getReferent().getGender().toShapeString());
		}
	}

	/**
	 * 
	 * This method needs not null parameters.
	 * 
	 * The case "ego equals alter" is managed normally to be detectable in
	 * controls.
	 * 
	 * In case of PARENT kintype, the gender will fix the role (father, mother).
	 * 
	 * @param ego
	 *            spouse or parent
	 * @param alter
	 *            spouse or child
	 * @param type
	 * 
	 * @throws PuckException
	 */
	public static void setKin(final Net net, final Individual ego, final Individual alter, final KinType type) throws PuckException {

		if ((ego == null) || (alter == null) || (type == null)) {
			throw PuckExceptions.INVALID_PARAMETER.create("Null parameter.");
		} else {
			switch (type) {
				case PARENT:
					setKinParent(net, ego, alter);
				break;

				case CHILD:
					setKinParent(net, alter, ego);
				break;

				case SPOUSE:
					setKinSpouse(net, ego, alter);
				break;
			}
		}
	}

	/**
	 * 
	 * @param net
	 * @param family
	 * @param newFather
	 */
	public static void setKinFather(final Family family, final Individual newFather) {
		if (family != null) {
			//
			if (family.getFather() != null) {
				removeFather(family);
			}

			//
			if (newFather != null) {
				if ((family.getMother() == null) || (newFather.getId() != family.getMother().getId())) {
					family.setFather(newFather);
					newFather.getPersonalFamilies().add(family);
				}
			}
		}
	}

	/**
	 * 
	 * @param net
	 * @param father
	 * @param child
	 */
	public static void setKinFather(final Net net, final Individual father, final Individual child) {

		setKinParent(net, father, KinType.FATHER, child);
	}

	/**
	 * 
	 * @param net
	 * @param family
	 * @param newFather
	 */
	public static void setKinMother(final Family family, final Individual newMother) {
		if (family != null) {
			//
			if (family.getMother() != null) {
				removeMother(family);
			}

			//
			if (newMother != null) {
				if ((family.getFather() == null) || (newMother.getId() != family.getFather().getId())) {
					family.setMother(newMother);
					newMother.getPersonalFamilies().add(family);
				}
			}
		}
	}

	/**
	 * 
	 * @param net
	 * @param mother
	 * @param child
	 */
	public static void setKinMother(final Net net, final Individual mother, final Individual child) {

		setKinParent(net, mother, KinType.MOTHER, child);
	}

	/**
	 * This method sets a parent kinship determining parent role by gender.
	 * 
	 * In case of another parent gender value than FATHER or MOTHER, this method
	 * do nothing.
	 * 
	 * 
	 * @param net
	 * @param ego
	 * @param alter
	 */
	public static void setKinParent(final Net net, final Individual parent, final Individual child) {

		//
		switch (parent.getGender()) {
			case FEMALE:
				setKinParent(net, parent, KinType.MOTHER, child);
			break;

			case MALE:
				setKinParent(net, parent, KinType.FATHER, child);
			break;
		}
	}

	/**
	 * In case of another parentRole value than FATHER or MOTHER, this method do
	 * nothing.
	 * 
	 * Action is done with the point of child view.
	 * 
	 * @param net
	 * @param newParent
	 * @param parentRole
	 * @param child
	 */
	public static void setKinParent(final Net net, final Individual newParent, final KinType parentRole, final Individual child) {
		// Notation:
		// F = Father
		// M = Mother
		// f = family
		// C = Child
		// C1 = Child number 1
		// (A) = kinship objects
		// + A = new kin parameter
		// ' = new object
		if ((child != null) && (parentRole != null) && ((parentRole == KinType.FATHER) || (parentRole == KinType.MOTHER))) {

			if (parentRole == KinType.FATHER) {
				// Remove father.
				if ((child.getFather() != null) && (child.getFather() != newParent)) {
					//
					Family previousFamily = child.getOriginFamily();
					previousFamily.getChildren().removeById(child.getId());
					child.setOriginFamily(null);

					//
					if (previousFamily.getMother() != null) {
						net.createFamily((Individual) null, previousFamily.getMother(), child);
					}

					// Clean
					if ((previousFamily.isSterile()) && (previousFamily.isSingleParent()) && (previousFamily.attributes().size() == 0)) {
						net.remove(previousFamily);
					}
				}

				// Add father.
				if (newParent != null) {
					if (child.getOriginFamily() == null) {
						net.createFamily(newParent, null, child);
					} else if ((child.getFather() == null) && (child.getMother() == null)) {
						child.getOriginFamily().setFather(newParent);
					} else if ((child.getFather() == null) && (child.getMother() != null)) {
						Family newOriginFamily = net.families().getBySpouses(newParent, child.getOriginFamily().getMother());
						if (newOriginFamily == null) {
							child.getOriginFamily().setFather(newParent);
							child.getFather().getPersonalFamilies().add(child.getOriginFamily());
						} else {
							//
							Family previousFamily = child.getOriginFamily();
							previousFamily.getChildren().removeById(child.getId());
							if (previousFamily.isEmpty()) {
								net.remove(previousFamily);
							}

							//
							child.setOriginFamily(newOriginFamily);
							newOriginFamily.getChildren().add(child);

							// Clean
							if ((previousFamily.isSterile()) && (previousFamily.isSingleParent()) && (previousFamily.attributes().size() == 0)) {
								net.remove(previousFamily);
							}
						}
					}
				}
			} else if (parentRole == KinType.MOTHER) {
				// Remove Mother.
				if ((child.getMother() != null) && (child.getMother() != newParent)) {
					//
					Family previousFamily = child.getOriginFamily();
					previousFamily.getChildren().removeById(child.getId());
					child.setOriginFamily(null);

					//
					if (previousFamily.getFather() != null) {
						net.createFamily(previousFamily.getFather(), (Individual) null, child);
					}

					// Clean
					if ((previousFamily.isSterile()) && (previousFamily.isSingleParent()) && (previousFamily.attributes().size() == 0)) {
						net.remove(previousFamily);
					}
				}

				// Add mother.
				if (newParent != null) {
					if (child.getOriginFamily() == null) {
						net.createFamily((Individual) null, newParent, child);
					} else if ((child.getFather() == null) && (child.getMother() == null)) {
						child.getOriginFamily().setMother(newParent);
					} else if ((child.getFather() != null) && (child.getMother() == null)) {
						Family newOriginFamily = net.families().getBySpouses(child.getOriginFamily().getFather(), newParent);
						if (newOriginFamily == null) {
							child.getOriginFamily().setMother(newParent);
							child.getMother().getPersonalFamilies().add(child.getOriginFamily());
						} else {
							//
							Family previousFamily = child.getOriginFamily();
							previousFamily.getChildren().removeById(child.getId());
							if (previousFamily.isEmpty()) {
								net.remove(previousFamily);
							}

							//
							child.setOriginFamily(newOriginFamily);
							newOriginFamily.getChildren().add(child);

							// Clean.
							if ((previousFamily.isSterile()) && (previousFamily.isSingleParent()) && (previousFamily.attributes().size() == 0)) {
								net.remove(previousFamily);
							}
						}
					}
				}
			}
		}
	}

	/**
	 * In case of another parentRole value than FATHER or MOTHER, this method do
	 * nothing.
	 * 
	 * Bad way because there is no matter about attributes.
	 * 
	 * @param net
	 * @param newParent
	 * @param parentRole
	 * @param child
	 */
	public static void setKinParent2(final Net net, final Individual newParent, final KinType parentRole, final Individual child) {
		// Notation:
		// F = Father
		// M = Mother
		// f = family
		// C = Child
		// C1 = Child number 1
		// (A) = kinship objects
		// + A = new kin parameter
		// ' = new object

		if ((parentRole != null) && ((parentRole == KinType.FATHER) || (parentRole == KinType.MOTHER))) {
			//
			if (child.isOrphan()) {
				// Rule: (null-C) + F-C -> F-f-C
				// Rule: (null-C) + M-C -> M-f-C

				if (child.getOriginFamily() == null) {
					if (parentRole == KinType.MOTHER) {
						child.getOriginFamily().setMother(newParent);
					} else {
						child.getOriginFamily().setFather(newParent);
					}
				} else {
					if (parentRole == KinType.MOTHER) {
						net.createFamily((Individual) null, newParent, child);
					} else {
						net.createFamily(newParent, null, child);
					}
				}

				//
				newParent.getPersonalFamilies().add(child.getOriginFamily());

			} else if ((child.isOrphanOfMother()) && (parentRole == KinType.FATHER)) {
				// Check if the newParent is already parent.
				if (child.getOriginFamily().getFather() != newParent) {

					//
					if (child.getOriginFamily().getChildren().size() == 1) {
						// Rule: (F-f-C) + F'-C -> (F,F'-f-C)

						//
						Individual previousFather = child.getOriginFamily().getFather();
						previousFather.getPersonalFamilies().removeById(child.getOriginFamily().getId());

						//
						child.getOriginFamily().setFather(newParent);
					} else {
						// Rule: (F-f-C1C2) + F'-C1 -> (F-f-C2,F'-f'-C1)

						//
						child.getOriginFamily().getChildren().removeById(child.getId());

						//
						net.createFamily(newParent, null, child);
					}
				}

			} else if ((child.isOrphanOfFather()) && (parentRole == KinType.MOTHER)) {
				// Check if the newParent is already parent.
				if (child.getOriginFamily().getMother() != newParent) {

					//
					if (child.getOriginFamily().getChildren().size() == 1) {
						// Rule: (M-f-C) + M'-C -> (M,M'-f-C)

						//
						child.getOriginFamily().getMother().getPersonalFamilies().removeById(child.getOriginFamily().getId());

						//
						child.getOriginFamily().setMother(newParent);
					} else {
						// Rule: (M-f-C1C2) + M'-C1 -> (M-f-C2,M'-f'-C1)

						//
						child.getOriginFamily().getChildren().removeById(child.getId());

						//
						net.createFamily(child.getFather(), newParent, child);
					}
				}

			} else if ((child.isOrphanOfMother()) && (parentRole == KinType.MOTHER)) {

				//
				Family newChildOriginFamily = net.families().getBySpouses(child.getOriginFamily().getFather(), newParent);

				if (newChildOriginFamily == null) {
					if (child.getOriginFamily().getChildren().size() == 1) {
						// Rule: (F-f-C) + M-C -> (FM-f-C)
						child.getOriginFamily().setMother(newParent);
						newParent.getPersonalFamilies().add(child.getOriginFamily());
					} else {
						// Rule: (F-f-C1C2) + M-C1 -> (F-f-C2, FM-f'-C)

						//
						child.getOriginFamily().getChildren().removeById(child.getId());

						//
						net.createFamily(child.getFather(), newParent, child);
					}
				} else {
					if (child.getOriginFamily().getChildren().size() == 1) {
						// Rule: (F-f1-C, FM-f2) + M-C -> (f1 removed, FM-f2-C)
						child.getFather().getPersonalFamilies().removeById(child.getOriginFamily().getId());
						child.getOriginFamily().getChildren().removeById(child.getId());
						net.families().removeById(child.getOriginFamily().getId());

						//
						child.setOriginFamily(newChildOriginFamily);
					} else {
						// Rule: (F-f1-C1C2, FM-f2) + M-C1 -> (F-f1-C2,
						// FM-f2-C1)
						child.getOriginFamily().getChildren().removeById(child.getId());
						child.setOriginFamily(newChildOriginFamily);
						child.getOriginFamily().getChildren().add(child);
					}
				}

			} else if ((child.isOrphanOfFather()) && (parentRole == KinType.FATHER)) {
				//
				Family newChildOriginFamily = net.families().getBySpouses(newParent, child.getOriginFamily().getMother());

				if (newChildOriginFamily == null) {
					if (child.getOriginFamily().getChildren().size() == 1) {
						// Rule: (M-f-C) + F-C -> (FM-f-C)
						child.getOriginFamily().setFather(newParent);
						newParent.getPersonalFamilies().add(child.getOriginFamily());
					} else {
						// Rule: (M-f-C1C2) + F-C1 -> (F-f-C2, FM-f'-C)

						//
						child.getOriginFamily().getChildren().removeById(child.getId());

						//
						net.createFamily(child.getFather(), newParent, child);
					}
				} else {
					if (child.getOriginFamily().getChildren().size() == 1) {
						// Rule: (M-f1-C, FM-f2) + F-C -> (f1 removed, FM-f2-C)
						child.getMother().getPersonalFamilies().removeById(child.getOriginFamily().getId());
						child.getOriginFamily().getChildren().removeById(child.getId());
						net.families().removeById(child.getOriginFamily().getId());

						//
						child.setOriginFamily(newChildOriginFamily);
					} else {
						// Rule: (M-f1-C1C2, FM-f2) + F-C1 -> (F-f1-C2,
						// FM-f2-C1)
						child.getOriginFamily().getChildren().removeById(child.getId());
						child.setOriginFamily(newChildOriginFamily);
						child.getOriginFamily().getChildren().add(child);
					}
				}

			} else {
				// Note: child is not orphan.

				if ((parentRole == KinType.FATHER) && (child.getFather() != newParent)) {
					//
					Family newChildOriginFamily = net.families().getBySpouses(newParent, child.getOriginFamily().getMother());

					if (newChildOriginFamily == null) {
						// Rule: (FM-f-C) + F'-C -> (FM-f, F'M-f'-C)

						//
						child.getOriginFamily().getChildren().removeById(child.getId());

						//
						net.createFamily(newParent, child.getMother(), child);
					} else {
						// Rule: (FM-f1-C, F'M-f2) + F'-C -> (FM-f1, F'M-f2-C)

						//
						child.getOriginFamily().getChildren().removeById(child.getId());
						child.setOriginFamily(newChildOriginFamily);
						child.getOriginFamily().getChildren().put(child);
					}
				} else if ((parentRole == KinType.MOTHER) && (child.getMother() != newParent)) {
					//
					Family newChildOriginFamily = net.families().getBySpouses(child.getOriginFamily().getFather(), newParent);

					if (newChildOriginFamily == null) {
						// Rule: (FM-f-C) + M'-C -> (FM-f, FM'-f'-C)

						//
						child.getOriginFamily().getChildren().removeById(child.getId());

						//
						net.createFamily(child.getFather(), newParent, child);
					} else {
						// Rule: (FM-f1-C, FM'-f2) + M'-C -> (FM-f1, FM'-f2-C)

						//
						child.getOriginFamily().getChildren().removeById(child.getId());
						child.setOriginFamily(newChildOriginFamily);
						child.getOriginFamily().getChildren().put(child);
					}
				}
			}
		}
	}

	/**
	 * WARNING: the use of this method can created duplicated families. WARNING:
	 * use only if source does not contain family ids.
	 * 
	 * @param net
	 * @param egoId
	 * @param alterId
	 * @param kinType
	 * @throws PuckException
	 */
	public static void setKinRelation(final Net net, final Individual ego, final Individual alter, final KinType kinType) throws PuckException {
		int egoId = ego.getId();
		int alterId = alter.getId();

		switch (kinType) {
			case PARENT:
				switch (alter.getGender()) {
					case MALE:
						setFatherRelation(net, alterId, egoId);
					break;
					case FEMALE:
						setMotherRelation(net, alterId, egoId);
					break;
				}
			break;
			case CHILD:
				switch (ego.getGender()) {
					case MALE:
						setFatherRelation(net, egoId, alterId);
					break;
					case FEMALE:
						setMotherRelation(net, egoId, alterId);
					break;
				}
			break;
			case SPOUSE:
				switch (ego.getGender()) {
					case MALE:
						setSpouseRelation(net, egoId, alterId);
					break;
					case FEMALE:
						setSpouseRelation(net, alterId, egoId);
					break;
				}
			break;
		}
	}

	/**
	 * This method sets a spouse kinship determining role by gender.
	 * 
	 * @param net
	 * @param ego
	 * @param alter
	 */
	public static void setKinSpouse(final Net net, final Individual ego, final Individual alter) {
		//
		Individual husband;
		Individual wife;
		if (ego.isMale() || (alter.isFemale() && ego.isUnknown())) {
			husband = ego;
			wife = alter;
		} else {
			wife = ego;
			husband = alter;
		}

		//
		setKinSpouseByRole(net, husband, wife);
	}

	/**
	 * This method sets a spouse kinship.
	 * 
	 * @param net
	 * @param ego
	 * @param alter
	 */
	public static void setKinSpouseByRole(final Net net, final Individual husband, final Individual wife) {
		//
		Family family = net.families().getBySpouses(husband, wife);

		//
		if (family == null) {
			family = new Family(net.families().getFirstFreeId());
			net.families().add(family);

			//
			husband.getPersonalFamilies().add(family);
			wife.getPersonalFamilies().add(family);
		}

		//
		family.setHusband(husband);
		family.setWife(wife);

		//
		family.setMarried(true);
	}

	/**
	 * WARNING: the use of this method can created duplicated families. WARNING:
	 * use only if source does not contain family ids.
	 * 
	 * @param parentId
	 * @param childId
	 * @throws PuckException
	 */
	public static void setMotherRelation(final Net net, final int motherId, final int childId) throws PuckException {
		if ((net != null) && (motherId != 0) && (childId != 0)) {

			//
			Individual parent = net.individuals().getById(motherId);
			if (parent == null) {
				parent = new Individual(motherId);
				net.individuals().add(parent);
			}

			//
			Individual child = net.individuals().getById(childId);
			if (child == null) {
				child = new Individual(childId);
				net.individuals().add(child);
			}

			//
			Family childSourceFamily;
			if (child.getOriginFamily() == null) {
				childSourceFamily = new Family(net.families().size() + 1);
				net.families().add(childSourceFamily);
				child.setOriginFamily(childSourceFamily);
				childSourceFamily.getChildren().add(child);
			} else {
				childSourceFamily = child.getOriginFamily();
				if (childSourceFamily.getMother() != null) {
					childSourceFamily.getMother().getPersonalFamilies().removeById(childSourceFamily.getId());
				}
			}
			childSourceFamily.setMother(parent);

			//
			Family parentPersonalFamily = parent.getPersonalFamilies().getById(childSourceFamily.getId());
			if (parentPersonalFamily == null) {
				parent.getPersonalFamilies().add(childSourceFamily);
			}
		}
	}

	/**
	 * WARNING: the parent role is determined by gender.
	 * 
	 * WARNING: the use of this method can created duplicated families.
	 * 
	 * WARNING: use only if source does not contain family ids.
	 * 
	 * @param parentId
	 * @param childId
	 * @throws PuckException
	 */
	public static void setParentRelation(final Net net, final int parentId, final int childId) throws PuckException {
		if ((net != null) && (parentId != 0) && (childId != 0)) {
			//
			Individual parent = net.individuals().getById(parentId);
			if (parent == null) {
				parent = new Individual(parentId);
				net.individuals().add(parent);
			}

			//
			Individual child = net.individuals().getById(childId);
			if (child == null) {
				child = new Individual(childId);
				net.individuals().add(child);
			}

			//
			Family childSourceFamily;
			if (child.getOriginFamily() == null) {
				childSourceFamily = new Family(net.families().size() + 1);
				net.families().add(childSourceFamily);
				child.setOriginFamily(childSourceFamily);
				childSourceFamily.getChildren().add(child);
			} else {
				childSourceFamily = child.getOriginFamily();
			}

			//
			switch (parent.getGender()) {
				case FEMALE:
					childSourceFamily.setMother(parent);
				break;

				case MALE:
				case UNKNOWN:
					childSourceFamily.setFather(parent);
			}

			//
			Family parentPersonalFamily = parent.getPersonalFamilies().getById(childSourceFamily.getId());
			if (parentPersonalFamily == null) {
				parent.getPersonalFamilies().add(childSourceFamily);
			}
		}
	}

	/**
	 * WARNING: the use of this method can created duplicated families.
	 * 
	 * @param parentId
	 * @param childId
	 * @throws PuckException
	 */
	public static Family setSpouseRelation(final Net net, final int husbandId, final int wifeId) throws PuckException {
		Family result;

		if ((net == null) || (husbandId == 0) || (wifeId == 0)) {
			result = null;
		} else {
			//
			boolean isNewFamily = false;

			//
			Individual husband = net.individuals().getById(husbandId);
			if (husband == null) {
				husband = new Individual(husbandId);
				net.individuals().add(husband);
				isNewFamily = true;
			}

			//
			Individual wife = net.individuals().getById(wifeId);
			if (wife == null) {
				wife = new Individual(wifeId);
				net.individuals().add(wife);
				isNewFamily = true;
			}

			//
			Family family;
			if (isNewFamily) {
				family = null;
			} else {
				family = net.families().getBySpouses(husband, wife);
			}

			//
			if (family == null) {
				family = new Family(net.families().size() + 1);
				net.families().add(family);
				family.setHusband(husband);
				family.setWife(wife);
				husband.getPersonalFamilies().add(family);
				wife.getPersonalFamilies().add(family);
			}

			//
			family.setMarried(true);

			//
			result = family;
		}

		//
		return result;
	}

	/**
	 * WARNING: the use of this method can created duplicated families.
	 * 
	 * @param parentId
	 * @param childId
	 * @throws PuckException
	 */
	public static void setSpouseRelationAndFixRoles(final Net net, final int husbandId, final int wifeId) throws PuckException {
		//
		Family family = setSpouseRelation(net, husbandId, wifeId);
		if (family != null) {
			fixSpouseRolesByGender(family);
		}
	}

	/**
	 * 
	 * @param source
	 */
	public static void swapParents(final Family source) {
		if (source != null) {
			Individual pivot = source.getFather();
			source.setFather(source.getMother());
			source.setMother(pivot);
		}
	}
}
