package org.tip.puck.net.workers;

import java.io.File;

import org.tip.puck.PuckException;
import org.tip.puck.PuckExceptions;
import org.tip.puck.graphs.Graph;
import org.tip.puck.mas.MAS;
import org.tip.puck.net.Family;
import org.tip.puck.net.Individual;
import org.tip.puck.net.Net;
import org.tip.puck.report.Report;
import org.tip.puck.report.ReportRawData;
import org.tip.puck.segmentation.Segmentation;
import org.tip.puck.util.Chronometer;
import org.tip.puck.util.PuckUtils;
import org.tip.puck.util.ToolBox;
import org.tip.puckgui.views.RandomCorpusCriteria;
import org.tip.puckgui.views.mas.CousinsWeightFactor;
import org.tip.puckgui.views.mas.Divorce2WeightFactor;
import org.tip.puckgui.views.mas.DivorceWeightFactor;
import org.tip.puckgui.views.mas.NormalAgeDifferenceWeightFactor;
import org.tip.puckgui.views.mas.NormalAgeWeightFactor;
import org.tip.puckgui.views.mas.PregnancyWeightFactor;
import org.tip.puckgui.views.mas.WeightFactor;

import fr.devinsy.util.StringList;

/**
 * 
 * @author TIP
 */
public class NetReporter {

	/**
	 * Generates a report.
	 * 
	 * @param net
	 *            Source to report.
	 * @throws PuckException
	 */
	public static Report reportOreGraph(final Segmentation source, final File currentCorpus) throws PuckException {
		Report result;

		if ((source == null) || (currentCorpus == null)) {
			throw PuckExceptions.INVALID_PARAMETER.create("Null parameter detected.");
		} else {
			result = new Report();

			Chronometer chrono = new Chronometer();

			// Compute.

			Graph<Individual> graph = NetUtils.createOreGraph(source);

			StringList partitionLabels = new StringList();

			partitionLabels.append("SEX");
			partitionLabels.append("BIRT_PLAC");

			// Build Pajek data.
			ReportRawData rawData = new ReportRawData("Export to Pajek", "Pajek", "paj", new File(currentCorpus.getParent() + File.separator
					+ ToolBox.clean(graph.getLabel()) + ".paj"));

			rawData.setData(PuckUtils.writePajekNetwork(graph, partitionLabels).toString());

			// Build report.

			//
			result.setTimeSpent(chrono.stop().interval());
		}

		//
		return result;
	}

	/**
	 * Generates a report.
	 * 
	 * @param net
	 *            Source to report.
	 * @throws PuckException
	 */
	public static Report reportPGraph(final Segmentation source, final File currentCorpus) throws PuckException {
		Report result;

		if ((source == null) || (currentCorpus == null)) {
			throw PuckExceptions.INVALID_PARAMETER.create("Null parameter detected.");
		} else {
			result = new Report();

			Chronometer chrono = new Chronometer();

			// Compute.

			Graph<Family> graph = NetUtils.createPGraph(source);

			StringList partitionLabels = new StringList();

			// Build Pajek data.
			ReportRawData rawData = new ReportRawData("Export to Pajek", "Pajek", "paj", new File(currentCorpus.getParent() + File.separator
					+ ToolBox.clean(graph.getLabel()) + ".paj"));

			rawData.setData(PuckUtils.writePajekNetwork(graph, partitionLabels).toString());

			// Build report.

			//
			result.setTimeSpent(chrono.stop().interval());
		}

		//
		return result;
	}

	/**
	 * Generates a report.
	 * 
	 * @param net
	 *            Source to report.
	 * @throws PuckException
	 */
	public static Report reportRandomCorpus(final RandomCorpusCriteria criteria, final Net target) throws PuckException {
		Report result;

		if ((criteria == null) || (target == null)) {
			throw PuckExceptions.INVALID_PARAMETER.create("Null parameter detected.");
		} else {
			//
			Chronometer chrono = new Chronometer();

			//
			result = new Report("Random Corpus");
			result.setOrigin("NetReporter.reportRandomCorpus()");
			result.setTarget(target.getLabel());

			//
			StringList buffer = new StringList();
			buffer.appendln();

			//
			result.inputs().add("Initial Population", criteria.getInitialPopulation());
			result.inputs().add("Years", criteria.getYear());
			result.inputs().add("Fertility rate", criteria.getFertilityRate());
			result.inputs().add("Marriage rate", criteria.getMarriageRate());
			result.inputs().add("Divorce rate", criteria.getDivorceRate());
			result.inputs().add("Max age", criteria.getMaxAge());
			result.inputs().add("Min age", criteria.getMinAge());
			result.inputs().add("Normal age difference (mean)", criteria.getMeanAgeDifference());
			result.inputs().add("Normal age difference (stdev)", criteria.getStdevAgeDifference());
			result.inputs().add("Cousin preferences (type)", criteria.getCousinPreferenceType().toString());
			result.inputs().add("Cousin preferences (weight)", criteria.getCousinPreferenceWeight());

			//
			result.setTimeSpent(chrono.stop().interval());
		}

		//
		return result;
	}

	/**
	 * Generates a report.
	 * 
	 * @param net
	 *            Source to report.
	 * @throws PuckException
	 */
	public static Report reportRandomCorpusMAS(final RandomCorpusCriteria criteria, final MAS mas, final Net target) throws PuckException {
		Report result;

		if ((criteria == null) || (target == null)) {
			throw PuckExceptions.INVALID_PARAMETER.create("Null parameter detected.");
		} else {
			//
			Chronometer chrono = new Chronometer();

			//
			result = new Report("Random Corpus MAS");
			result.setOrigin("NetReporter.reportRandomCorpusMAS()");
			result.setTarget(target.getLabel());

			//
			StringList buffer = new StringList();

			//
			buffer.appendln("Weight forces input:");
			for (WeightFactor factor : criteria.weightFactors()) {
				switch (factor.getType()) {
					case DIVORCE:
						DivorceWeightFactor divorceFactor = (DivorceWeightFactor) factor;
						buffer.append("DIVORCE\tprobability=").appendln(divorceFactor.getProbability());
					break;
					case DIVORCE2:
						Divorce2WeightFactor divorce2Factor = (Divorce2WeightFactor) factor;
						buffer.append("DIVORCE2\tfemale=").appendln(divorce2Factor.getFemaleProbability()).append("\tmale=")
								.appendln(divorce2Factor.getMaleProbability()).append("\tboth=").appendln(divorce2Factor.getBothProbability());
					break;
					case NORMAL_AGE:
						NormalAgeWeightFactor normalAgeFactor = (NormalAgeWeightFactor) factor;
						buffer.append("NORMAL_AGE\tgender=").append(normalAgeFactor.getGender().toString()).append("\tmean=").append(normalAgeFactor.getMean())
								.append("\tstdev=").appendln(normalAgeFactor.getStdev());
					break;
					case NORMAL_AGE_DIFFERENCE:
						NormalAgeDifferenceWeightFactor differenceFactor = (NormalAgeDifferenceWeightFactor) factor;
						buffer.append("NORMAL_AGE_DIFFERENCE\tmean=").append(differenceFactor.getMean()).append("\tstdev=")
								.appendln(differenceFactor.getStdev());
					break;
					case COUSINS:
						CousinsWeightFactor cousinsFactor = (CousinsWeightFactor) factor;
						buffer.append("COUSINS\talpha=").appendln(cousinsFactor.getFirst()).append("\tbravo=").appendln(cousinsFactor.getSecond())
								.append("\tcharlie=").appendln(cousinsFactor.getThird());
					break;
					case PREGNANCY:
						PregnancyWeightFactor pregnancyFactor = (PregnancyWeightFactor) factor;
						buffer.append("PREGNANCY\talpha=").appendln(pregnancyFactor.getFirst());
					break;
					default:
				}
			}
			buffer.appendln();

			//
			buffer.appendln("Illegal marriages input:");
			buffer.appendln(criteria.getIllegalMarriages());
			buffer.appendln();
			buffer.appendln("MAS configuration:");
			buffer.appendln(criteria.toMASConfig());
			result.setInputComment(buffer.toString());

			//
			result.inputs().add("Year", criteria.getYear());
			result.inputs().add("Initial Population", criteria.getInitialPopulation());
			result.inputs().add("Same marriage probability", criteria.getSameMarriageProbability());
			result.inputs().add("Fertility rate", criteria.getFertilityRate());
			result.inputs().add("Max age", criteria.getMaxAge());

			//
			result.setTimeSpent(chrono.stop().interval());
		}

		//
		return result;
	}

	/**
	 * Generates a report.
	 * 
	 * @param net
	 *            Source to report.
	 * @throws PuckException
	 */
	public static Report reportTipGraph(final Segmentation source, final File currentCorpus) throws PuckException {
		Report result;

		if ((source == null) || (currentCorpus == null)) {
			throw PuckExceptions.INVALID_PARAMETER.create("Null parameter detected.");
		} else {
			result = new Report();

			Chronometer chrono = new Chronometer();

			// Compute.

			Graph<Individual> graph = NetUtils.createTipGraph(source);

			StringList partitionLabels = new StringList();

			partitionLabels.append("SEX");
			partitionLabels.append("BIRT_PLAC");

			// Build Pajek data.
			ReportRawData rawData = new ReportRawData("Export to Pajek", "Pajek", "paj", new File(currentCorpus.getParent() + File.separator
					+ ToolBox.clean(graph.getLabel()) + ".paj"));

			rawData.setData(PuckUtils.writePajekNetwork(graph, partitionLabels).toString());

			// Build report.

			//
			result.setTimeSpent(chrono.stop().interval());
		}

		//
		return result;
	}

}
