package org.tip.puck.graphs.workers;

import java.util.Collections;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tip.puck.PuckException;
import org.tip.puck.graphs.Graph;
import org.tip.puck.graphs.Link;
import org.tip.puck.graphs.Links;
import org.tip.puck.graphs.Node;
import org.tip.puck.graphs.Nodes;
import org.tip.puck.matrix.Matrix;
import org.tip.puck.partitions.Cluster;
import org.tip.puck.partitions.Partition;
import org.tip.puck.partitions.PartitionCriteria;
import org.tip.puck.partitions.PartitionMaker;
import org.tip.puck.util.MathUtils;
import org.tip.puck.util.Value;

import fr.devinsy.util.StringList;

/**
 * 
 * @author TIP
 */
public class GraphUtils {

	static private final Logger logger = LoggerFactory.getLogger(GraphUtils.class);

	public static <E> Graph<E> createGraphFromMatrix(final Nodes<E> nodes, final Matrix matrix) {
		Graph<E> result;

		result = new Graph<E>(0, nodes.size());

		result.addNodes(nodes);
		result.addArcs(matrix);

		//
		return result;
	}

	private static <E> Node<E> maxStrengthNode(final List<Node<E>> source, final Cluster<Node<E>> cluster1, final Cluster<Node<E>> cluster2) {
		Node<E> result = null;

		double maxStrength = 1.;

		for (Node<E> node : source) {
			double strength = strengthRate(node, cluster1, cluster2);
			if (strength >= maxStrength) {
				maxStrength = strength;
				result = node;
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @return
	 */
	public static <E> Graph<E> reduce(final Graph<E> source, final int minimalNumberOfLinks, final double minimalNodeStrength, final double minimalLinkWeight) {
		Graph<E> result;

		logger.debug("reduce starting... [minimalNumberOfLinks={}][minimalNodeStrength={}][minimalLinkWeight={}]", minimalNumberOfLinks, minimalNodeStrength,
				minimalLinkWeight);

		if ((minimalNumberOfLinks == 0) && (minimalLinkWeight == 0)) {
			result = source;
		} else {
			//
			result = new Graph<E>(source.getLabel());

			//
			for (Node<E> node : source.getNodes()) {
				if (((minimalNumberOfLinks == 0) || (node.getDegree() >= minimalNumberOfLinks))
						&& (((minimalNodeStrength == 0) || (node.getForce() >= minimalNodeStrength)))
						&& (((minimalLinkWeight == 0) || (MathUtils.compare(node.getMaxLinkWeight(), minimalLinkWeight) >= 0)))) {
					result.addNode(node.getReferent());
				}
			}

			//
			for (Node<E> newSourceNode : result.getNodes()) {
				//
				Node<E> oldSourceNode = source.getNode(newSourceNode.getReferent());

				//
				for (Link<E> oldLink : oldSourceNode.getOutArcs()) {
					//
					Node<E> oldTargetNode = oldLink.getTargetNode();
					Node<E> newTargetNode = result.getNode(oldTargetNode.getReferent());

					//
					if (newTargetNode != null) {
						if (oldLink.getWeight() >= minimalLinkWeight) {
							result.addArc(newSourceNode, newTargetNode, oldLink.getWeight());
						}
					}
				}

				//
				for (Link<E> oldLink : oldSourceNode.getEdges()) {
					//
					Node<E> oldTargetNode = oldLink.getTargetNode();
					Node<E> newTargetNode = result.getNode(oldTargetNode.getReferent());

					//
					if (newTargetNode != null) {
						if (oldLink.getWeight() >= minimalLinkWeight) {
							result.addEdge(newSourceNode, newTargetNode, oldLink.getWeight());
						}
					}
				}
			}
		}

		logger.debug("reduce done.");

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 */
	public static <E> Graph<E> reduceArcToEdge(final Graph<E> source) {
		Graph<E> result;

		if (source == null) {
			result = null;
		} else {
			//
			result = new Graph<E>(source.getLabel());

			//
			for (Node<E> node : source.getNodes()) {
				result.addNode(node.getReferent());
			}

			//
			for (Node<E> newSourceNode : result.getNodes()) {
				//
				Node<E> oldSourceNode = source.getNode(newSourceNode.getReferent());

				//
				for (Link<E> oldLink : oldSourceNode.getOutArcs()) {
					//
					Node<E> oldTargetNode = oldLink.getTargetNode();
					Node<E> newTargetNode = result.getNode(oldTargetNode.getReferent());

					//
					if (result.getEdge(newSourceNode, newTargetNode) == null) {
						//
						double newWeight = oldLink.getWeight();
						if (newSourceNode != newTargetNode) {
							Link<E> oldReverseLink = oldTargetNode.getArcWith(oldSourceNode);
							if (oldReverseLink != null) {
								newWeight += oldReverseLink.getWeight();
							}
						}

						//
						result.addEdge(newSourceNode, newTargetNode, newWeight);
					}
				}
			}

		}

		//
		return result;
	}

	public static <E> double sidedness(final Graph<E> source, final Partition<Node<E>> sides) {
		double result;

		double exoWeight = 0.;
		double totalWeight = 0;

		for (Link<E> link : source.getLinks()) {
			if (sides.getCluster(link.getSourceNode()) != sides.getCluster(link.getTargetNode())) {
				exoWeight += link.getWeight();
			}
			totalWeight += link.getWeight();

		}

		result = exoWeight / totalWeight;
		//
		return result;

	}

	/**
	 * sidedness algorithm of Douglas R. White
	 * 
	 * @param source
	 * @return
	 */
	public static <E> Partition<Node<E>> sides(final Graph<E> source) {
		Partition<Node<E>> result;

		result = new Partition<Node<E>>();

		Cluster<Node<E>> side1 = new Cluster<Node<E>>(new Value(1));
		Cluster<Node<E>> side2 = new Cluster<Node<E>>(new Value(2));
		Cluster<Node<E>> side3 = new Cluster<Node<E>>(new Value(3));

		result.getClusters().put(side1);
		result.getClusters().put(side2);

		double maxStrength = 0;
		Node<E> maxStrengthNode = null;

		for (Node<E> node : source.getNodes()) {
			double strength = node.getForce();
			if (strength > 0.) {
				side3.put(node);
				if (strength > maxStrength) {
					maxStrength = strength;
					maxStrengthNode = node;
				}
			}
		}
		result.swap(maxStrengthNode, side3, side1);
		result.swap(maxStrengthNode(side3.getItems(), side1, side3), side3, side2);

		int count = side3.count() + 1;

		while (side3.count() < count) {
			count = side3.count();
			result.swap(maxStrengthNode(side3.getItems(), side1, side2), side3, side2);
			result.swap(maxStrengthNode(side3.getItems(), side2, side1), side3, side1);
		}

		double sidedness = sidedness(source, result) - 1;

		while (sidedness < sidedness(source, result)) {

			sidedness = sidedness(source, result);

			result.swap(maxStrengthNode(side1.getItems(), side1, side2), side1, side2);
			result.swap(maxStrengthNode(side2.getItems(), side2, side1), side2, side1);

		}

		//
		return result;

	}

	private static <E> double strengthRate(final Node<E> node, final Cluster<Node<E>> cluster1, final Cluster<Node<E>> cluster2) {
		double result;

		double strength1 = 0.;
		double strength2 = 0;
		for (Link<E> link : node.getLinks()) {
			Node<E> other = link.getOtherNode(node);
			if (cluster1.getItems().contains(other)) {
				strength1 = +link.getWeight();
			} else if (cluster2.getItems().contains(other)) {
				strength2 = +link.getWeight();
			}
		}

		result = strength1 / strength2;
		//
		return result;
	}

	/**
	 * Generates a Pajek file content from a graph.
	 * 
	 * @param source
	 *            A graph to write in Pajek format.
	 * 
	 * @param partitionLabels
	 *            Labels of endogenous attributes of a graph node.
	 * 
	 * @return The Pajet content for this graph.
	 * @throws PuckException
	 */
	public static <E> StringList writePajekNetwork(final Graph<E> source, final List<String> partitionLabels) throws PuckException {
		StringList result;

		//
		result = new StringList(100);

		//
		result.appendln("*Network " + source.getLabel());
		result.appendln();

		// Node list
		result.appendln("*vertices " + source.nodeCount());
		for (int nodeIndex = 1; nodeIndex <= source.nodeCount(); nodeIndex++) {
			Node<E> node = source.getNodes().get(nodeIndex);
			if (StringUtils.isBlank(node.getTag())) {
				result.appendln((nodeIndex) + " '" + node.getLabel() + "'");
			} else {
				result.appendln((nodeIndex) + " '" + node.getLabel() + "' " + node.getTag());
			}
		}

		// Arc List
		Links<E> arcs = source.getArcs();
		if (arcs.isNotEmpty()) {
			//
			List<String> tags = arcs.getTags();
			if (tags.isEmpty()) {
				result.appendln("*arcs");
				for (Link<E> arc : arcs) {
					result.appendln(arc.getSourceNode().getId() + " " + arc.getTargetNode().getId() + " " + arc.getWeightAsInt());
				}
			} else {
				Collections.sort(tags);
				for (String tag : tags) {
					result.appendln("*arcs " + tag);
					for (Link<E> arc : arcs.getByTag(tag)) {
						result.appendln(arc.getSourceNode().getId() + " " + arc.getTargetNode().getId() + " " + arc.getWeightAsInt());
					}
				}
			}
		}

		// Edge List
		Links<E> edges = source.getEdges();
		if (edges.isNotEmpty()) {
			result.appendln("*edges");
			for (Link<E> edge : edges) {
				result.appendln(edge.getSourceNode().getId() + " " + edge.getTargetNode().getId() + " " + edge.getWeightAsInt());
			}
		}

		// Partitions
		for (String label : partitionLabels) {
			result.appendln();
			logger.debug("label=[" + label + "]");
			Partition<Node<E>> partition = PartitionMaker.create(label, source.getNodes().toList(), PartitionCriteria.createRaw(label));

			String type;
			if (partition.isNumeric()) {
				type = "Vector";
			} else {
				type = "Partition";
				partition = PartitionMaker.createNumerized(partition);
			}

			result.appendln("*" + type + " " + partition.getLabel());
			result.appendln("*vertices " + source.nodeCount());
			for (int nodeIndex = 1; nodeIndex <= source.nodeCount(); nodeIndex++) {
				Node<E> node = source.getNodes().get(nodeIndex);
				result.appendln(partition.getValue(node).intValue());
			}
		}

		//
		return result;
	}

}
