package org.tip.puck.net.workers;

import java.util.ResourceBundle;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tip.puck.net.Family;
import org.tip.puck.net.Individual;
import org.tip.puck.net.Net;
import org.tip.puck.report.Report;
import org.tip.puck.util.Chronometer;

import fr.devinsy.util.StringList;

/**
 * 
 * @author TIP
 */
public class ControlReporter {

	public enum ControlType {
		UNKNOWN_RELATIVES,
		SAME_SEX_SPOUSES,
		FEMALE_FATHERS_OR_MALE_MOTHERS,
		MULTIPLE_FATHERS_OR_MOTHERS,
		CYCLIC_DESCENT_CASES,
		UNKNOWN_SEX_PERSONS,
		NAMELESS_PERSONS,
		PARENT_CHILD_MARRIAGES,
		AUTO_MARRIAGE
	}

	static private final Logger logger = LoggerFactory.getLogger(ControlReporter.class);

	/**
	 * checks for filiation cycles in the network
	 * 
	 * @param control
	 *            the protocol to note found cycles
	 * @param k
	 *            the maximal cycle length to be checked
	 * @param firstEgo
	 *            the last vertex of the chain
	 * @param str
	 *            a string containing the preceding kinship chain in positional
	 *            notation
	 */
	public static void checkCycles(final StringList control, final Individual ego, final Individual firstEgo, final int k, String str) {

		if (k == 0 || ego.getId() < firstEgo.getId()) {
			return;
		}
		String[] parentType = { "F", "M", "X" };

		str = str + " " + ego.getName() + " (" + ego.getId() + ")";

		if (ego.equals(firstEgo)) {
			control.appendln(str);
		}
		for (Individual alter : ego.getParents()) {
			checkCycles(control, alter, firstEgo, k - 1, str + " - " + parentType[alter.getGender().toInt()]);
		}
	}

	/**
	 * checks for filiation cycles in the network
	 * 
	 * @param control
	 *            the protocol to note found cycles
	 * @param k
	 *            the maximal cycle length to be checked
	 */
	public static void checkCycles(final StringList control, final Individual ego, final int k) {

		String[] parentType = { "F", "M", "X" };

		for (Individual alter : ego.getParents()) {
			checkCycles(control, alter, ego, k, ego.getName() + " (" + ego.getId() + ") - " + parentType[alter.getGender().toInt()]);
		}
	}

	/*
	 * Checks for possible errors (special features) of a given type in the
	 * dataset (as specified in the options) and counts them.
	 * 
	 * @param k the index of the type of possible error
	 * 
	 * @return the number of special features in the dataset
	 * 
	 * @see maps.Net#control()
	 */
	public static Report reportAutoMarriages(final Net net, final ResourceBundle bundle) {
		Report result;

		//
		Chronometer chrono = new Chronometer();

		//
		int errorCount = 0;
		StringList errors = new StringList();
		for (Individual individual : net.individuals()) {

			if (individual.getSpouses().contains(individual)) {
				errors.appendln(individual.getTrimmedName() + " (" + individual.getId() + ")");
				errorCount += 1;
			}
		}

		//
		result = new Report();
		result.setTitle("Checks for Auto-Marriages.");
		result.setOrigin("Control reporter");
		result.setTarget(net.getLabel());

		//
		errors.add(0, errorCount + " " + Report.translate(bundle, "Auto-Marriages") + "\n");
		result.outputs().append(errors.toString());

		//
		result.setStatus(errorCount);

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;
	}

	/*
	 * Checks for possible errors (special features) of a given type in the
	 * dataset (as specified in the options) and counts them.
	 * 
	 * @param k the index of the type of possible error
	 * 
	 * @return the number of special features in the dataset
	 * 
	 * @see maps.Net#control()
	 */
	public static Report reportControl(final Net net, final ResourceBundle bundle, final ControlType controlType) {
		Report result;

		switch (controlType) {
			case CYCLIC_DESCENT_CASES:
				result = reportCyclicDescentCases(net, bundle);
			break;
			case FEMALE_FATHERS_OR_MALE_MOTHERS:
				result = reportFemaleFathersOrMaleMothers(net, bundle);
			break;
			case MULTIPLE_FATHERS_OR_MOTHERS:
				result = reportMultipleFathersOrMothers(net, bundle);
			break;
			case NAMELESS_PERSONS:
				result = reportNamelessPersons(net, bundle);
			break;
			case AUTO_MARRIAGE:
				result = reportAutoMarriages(net, bundle);
			break;
			case PARENT_CHILD_MARRIAGES:
				result = reportParentChildMarriages(net, bundle);
			break;
			case SAME_SEX_SPOUSES:
				result = reportSameSexSpouses(net, bundle);
			break;
			case UNKNOWN_SEX_PERSONS:
				result = reportUnknownSexPersons(net, bundle);
			break;
			default:
				result = null;
		}

		//
		return result;
	}

	/**
	 * 
	 * @param net
	 * @return
	 */
	public static Report reportControls(final Net net, final ResourceBundle bundle) {
		Report result;

		result = reportControls(net, bundle, ControlType.values());

		//
		return result;
	}

	/**
	 * 
	 * @param net
	 * @return
	 */
	public static Report reportControls(final Net net, final ResourceBundle bundle, final ControlType... controlTypes) {
		Report result;

		//
		Chronometer chrono = new Chronometer();

		//
		result = new Report();
		result.setTitle("Checks for possible errors (special features).");
		result.setOrigin("Control reporter");
		result.setTarget(net.getLabel());

		//
		for (ControlType type : controlTypes) {
			result.inputs().add(type.toString(), "true");
		}

		//
		int errorCount = 0;
		for (ControlType controlType : controlTypes) {
			Report report = reportControl(net, bundle, controlType);
			if ((report != null) && (report.status() != 0)) {
				errorCount += report.status();
				result.outputs().append(report.outputs());
				result.outputs().appendln();
			}
		}

		//
		result.setStatus(errorCount);

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;
	}

	/*
	 * Checks for possible errors (special features) of a given type in the
	 * dataset (as specified in the options) and counts them.
	 * 
	 * @param k the index of the type of possible error
	 * 
	 * @return the number of special features in the dataset
	 * 
	 * @see maps.Net#control()
	 */
	public static Report reportCyclicDescentCases(final Net net, final ResourceBundle bundle) {
		Report result;

		//
		Chronometer chrono = new Chronometer();

		//
		StringList errors = new StringList();
		for (Individual individual : net.individuals()) {
			checkCycles(errors, individual, 5);
		}

		// Temporary solution for Stringlist counts a carriage return as a
		// separate line!
		int errorCount = errors.size() / 2;

		//
		result = new Report();
		result.setTitle("Checks for Cases of Cyclic Descent.");
		result.setOrigin("Control reporter");
		result.setTarget(net.getLabel());

		//
		errors.add(0, errorCount + " " + Report.translate(bundle, "Cases of Cyclic Descent") + "\n");
		result.outputs().append(errors.toString());

		//
		result.setStatus(errorCount);

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;
	}

	/*
	 * Checks for possible errors (special features) of a given type in the
	 * dataset (as specified in the options) and counts them.
	 * 
	 * @param k the index of the type of possible error
	 * 
	 * @return the number of special features in the dataset
	 * 
	 * @see maps.Net#control()
	 */
	public static Report reportFemaleFathersOrMaleMothers(final Net net, final ResourceBundle bundle) {
		Report result;

		//
		Chronometer chrono = new Chronometer();

		//
		int errorCount = 0;
		StringList errors = new StringList();
		for (Individual individual : net.individuals()) {
			Individual father = individual.getFather();
			if ((father != null) && (father.getGender().isFemale())) {
				errors.appendln(String.format("%c %s (%d)\t%s %s (%d)", father.getGender().toChar(), father.getTrimmedName(), father.getId(),
						Report.translate(bundle, "for"), individual.getName().trim(), individual.getId()));

				errorCount += 1;
			}
			Individual mother = individual.getMother();
			if ((mother != null) && (mother.getGender().isMale())) {
				errors.appendln(String.format("%c %s (%d)\t%s %s (%d)", mother.getGender().toChar(), mother.getName().trim(), mother.getId(),
						Report.translate(bundle, "for"), individual.getName().trim(), individual.getId()));

				errorCount += 1;
			}
		}

		//
		result = new Report();
		result.setTitle("Checks for Female Fathers or Male Mothers.");
		result.setOrigin("Control reporter");
		result.setTarget(net.getLabel());

		//
		errors.add(0, errorCount + " " + Report.translate(bundle, "Female Fathers or Male Mothers") + "\n");
		result.outputs().append(errors.toString());

		//
		result.setStatus(errorCount);

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;
	}

	public static Report reportMultipleFathersOrMothers(final Net net, final ResourceBundle bundle) {
		Report result;

		//
		Chronometer chrono = new Chronometer();

		//
		int errorCount = 0;
		StringList errors = new StringList();
		for (Individual individual : net.individuals()) {
			for (Family family : individual.getPersonalFamilies()) {
				Individual parent = family.getParent(individual.getGender());
				if (parent != null && parent != individual) {
					errors.appendln(String.format("%c %s (%d)\t%s %s (%d)", individual.getGender().toChar(), individual.getTrimmedName(), individual.getId(),
							Report.translate(bundle, "and"), parent.getName().trim(), parent.getId()));
					errorCount += 1;
				}
			}
		}

		//
		result = new Report();
		result.setTitle("Checks for Multiple Fathers or Mothers.");
		result.setOrigin("Control reporter");
		result.setTarget(net.getLabel());

		//
		errors.add(0, errorCount + " " + Report.translate(bundle, "Multiple Fathers/Mothers") + "\n");
		result.outputs().append(errors.toString());

		//
		result.setStatus(errorCount);

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;
	}

	/*
	 * Checks for possible errors (special features) of a given type in the
	 * dataset (as specified in the options) and counts them.
	 * 
	 * @param k the index of the type of possible error
	 * 
	 * @return the number of special features in the dataset
	 * 
	 * @see maps.Net#control()
	 */
	public static Report reportNamelessPersons(final Net net, final ResourceBundle bundle) {
		Report result;

		//
		Chronometer chrono = new Chronometer();

		//
		int errorCount = 0;
		StringList errors = new StringList();
		for (Individual individual : net.individuals()) {

			if (StringUtils.isBlank(individual.getName())) {
				errors.appendln(individual.getGender().toChar() + " " + individual.getId());
				errorCount += 1;
			}
		}

		//
		result = new Report();
		result.setTitle("Checks for Nameless Persons.");
		result.setOrigin("Control reporter");
		result.setTarget(net.getLabel());

		//
		errors.add(0, errorCount + " " + Report.translate(bundle, "Nameless Persons") + "\n");
		result.outputs().append(errors.toString());

		//
		result.setStatus(errorCount);

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;
	}

	/*
	 * Checks for possible errors (special features) of a given type in the
	 * dataset (as specified in the options) and counts them.
	 * 
	 * @param k the index of the type of possible error
	 * 
	 * @return the number of special features in the dataset
	 * 
	 * @see maps.Net#control()
	 */
	public static Report reportParentChildMarriages(final Net net, final ResourceBundle bundle) {
		Report result;

		//
		Chronometer chrono = new Chronometer();

		//
		int errorCount = 0;
		StringList errors = new StringList();
		for (Individual individual : net.individuals()) {

			for (Individual parent : individual.getParents()) {
				if (individual.getSpouses().contains(parent)) {
					errors.appendln(String.format("%s (%d) %s %s (%d)", individual.getTrimmedName(), individual.getId(), Report.translate(bundle, "and"),
							parent.getName().trim(), parent.getId()));
					errorCount += 1;
				}
			}
		}

		//
		result = new Report();
		result.setTitle("Checks for Parent-Child Marriages.");
		result.setOrigin("Control reporter");
		result.setTarget(net.getLabel());

		//
		errors.add(0, errorCount + " " + Report.translate(bundle, "Parent-Child Marriages") + "\n");
		result.outputs().append(errors.toString());

		//
		result.setStatus(errorCount);

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;
	}

	/*
	 * Checks for possible errors (special features) of a given type in the
	 * dataset (as specified in the options) and counts them.
	 * 
	 * @param k the index of the type of possible error
	 * 
	 * @return the number of special features in the dataset
	 * 
	 * @see maps.Net#control()
	 */
	public static Report reportSameSexSpouses(final Net net, final ResourceBundle bundle) {
		Report result;

		//
		Chronometer chrono = new Chronometer();

		//
		int errorCount = 0;
		StringList errors = new StringList();
		for (Individual individual : net.individuals()) {
			for (Individual spouse : individual.getSpouses()) {
				if (individual.getId() <= spouse.getId() && individual.getGender() == spouse.getGender()) {
					errors.appendln(String.format("%c %s (%d)\t%s %s (%d)", spouse.getGender().toChar(), spouse.getTrimmedName(), spouse.getId(),
							Report.translate(bundle, "for"), individual.getName().trim(), individual.getId()));
					errorCount += 1;
				}
			}
		}

		//
		result = new Report();
		result.setTitle("Checks for Same-Sex Couples.");
		result.setOrigin("Control reporter");
		result.setTarget(net.getLabel());

		//
		errors.add(0, errorCount + " " + Report.translate(bundle, "Same-Sex Couples") + "\n");
		result.outputs().append(errors.toString());

		//
		result.setStatus(errorCount);

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;
	}

	/*
	 * Checks for possible errors (special features) of a given type in the
	 * dataset (as specified in the options) and counts them.
	 * 
	 * @param k the index of the type of possible error
	 * 
	 * @return the number of special features in the dataset
	 * 
	 * @see maps.Net#control()
	 */
	public static Report reportUnknownSexPersons(final Net net, final ResourceBundle bundle) {
		Report result;

		//
		Chronometer chrono = new Chronometer();

		//
		int errorCount = 0;
		StringList errors = new StringList();
		for (Individual individual : net.individuals()) {
			if (individual.getGender().isUnknown()) {
				errors.appendln(individual.getTrimmedName() + " (" + individual.getId() + ")");
				errorCount += 1;
			}
		}

		//
		result = new Report();
		result.setTitle("Checks for Persons of unknown Sex.");
		result.setOrigin("Control reporter");
		result.setTarget(net.getLabel());

		//
		errors.add(0, errorCount + " " + Report.translate(bundle, "Persons of unknown Sex") + "\n");
		result.outputs().append(errors.toString());

		//
		result.setStatus(errorCount);

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;
	}
}
