package org.tip.puck.net.workers;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tip.puck.net.Attribute;
import org.tip.puck.net.Attributes;
import org.tip.puck.net.Family;
import org.tip.puck.net.Individual;
import org.tip.puck.net.Individuals;
import org.tip.puck.net.Net;
import org.tip.puck.net.relations.Relation;
import org.tip.puck.net.relations.Relations;
import org.tip.puck.util.NumberedValues;
import org.tip.puck.util.Value;

/**
 * 
 * Methods added in this classes has to be static and autonomous.
 * 
 * @author TIP
 */
public class AttributeWorker {

	public enum EmptyType {
		VOID,
		BLANK,
		FORCED_BLANK
	}

	public enum Scope {
		ALL,
		CORPUS,
		INDIVIDUALS,
		FAMILIES,
		RELATION,
		RELATIONS
	}

	static private final Logger logger = LoggerFactory.getLogger(AttributeWorker.class);

	/**
	 * 
	 * @param attributeLabel
	 * @return
	 */
	public static int cleanAttribute(final Individuals source, final String attributeLabel) {
		int result;

		result = 0;
		for (Individual individual : source) {
			//
			Attribute attribute = individual.attributes().get(attributeLabel);

			//
			if ((attribute != null) && ((StringUtils.isBlank(attribute.getValue()) || (attribute.getValue().trim().equals("0"))))) {
				individual.attributes().remove(attributeLabel);
				logger.debug("remove ===>" + individual.getId() + " " + individual.attributes().get(attributeLabel));
				result += 1;
			}
		}

		//
		return result;
	}

	/**
	 * @param individuals
	 * @return
	 */
	public static List<String> getExogenousAttributeLabels(final Net source, final Integer limit) {
		List<String> result;

		//
		result = new ArrayList<String>(20);

		//
		if (source != null) {

			//
			HashSet<String> buffer = new HashSet<String>();

			//
			{
				for (Attribute attribute : source.attributes()) {
					buffer.add(attribute.getLabel());
				}
			}

			//
			{
				int index = 0;
				Iterator<Individual> iterator = source.individuals().iterator();
				while ((iterator.hasNext()) && ((limit == null) || (index < limit))) {
					Individual individual = iterator.next();
					for (Attribute attribute : individual.attributes()) {
						buffer.add(attribute.getLabel());
					}
					index += 1;
				}
			}

			//
			{
				int index = 0;
				Iterator<Family> iterator = source.families().iterator();
				while ((iterator.hasNext()) && ((limit == null) || (index < limit))) {
					Family family = iterator.next();
					for (Attribute attribute : family.attributes()) {
						buffer.add(attribute.getLabel());
					}
					index += 1;
				}
			}

			//
			{
				int index = 0;
				Iterator<Relation> iterator = source.relations().iterator();
				while ((iterator.hasNext()) && ((limit == null) || (index < limit))) {
					Relation relation = iterator.next();
					for (Attribute attribute : relation.attributes()) {
						buffer.add(attribute.getLabel());
					}
					index += 1;
				}
			}

			//
			for (String string : buffer) {
				result.add(string);
			}

			//
			Collections.sort(result);
		}

		//
		return result;
	}

	/**
	 * 
	 * @param target
	 *            ALL, CORPUS, INDIVIDUAL, FAMILY or relation model name.
	 * @param label
	 * @param type
	 * @return
	 */
	public static int removeAllAttribute(final Net net, final Scope scope, final String optionalRelationModelName, final EmptyType type) {
		int result;

		if (net == null) {
			throw new IllegalArgumentException("net parameter is null");
		} else if (scope == null) {
			throw new IllegalArgumentException("target parameter is null");
		} else if (type == null) {
			throw new IllegalArgumentException("type parameter is null");
		} else {
			//
			result = 0;

			//
			if ((scope == Scope.ALL) || (scope == Scope.CORPUS)) {
				result += net.attributes().size();
				net.attributes().clear();
			}

			//
			if ((scope == Scope.ALL) || (scope == Scope.INDIVIDUALS)) {
				for (Individual individual : net.individuals()) {
					result += individual.attributes().size();
					individual.attributes().clear();
				}
			}

			//
			if ((scope == Scope.ALL) || (scope == Scope.FAMILIES)) {
				for (Family family : net.families()) {
					result += family.attributes().size();
					family.attributes().clear();
				}
			}

			//
			if ((scope == Scope.ALL) || (scope == Scope.RELATIONS)) {
				for (Relation relation : net.relations()) {
					result += relation.attributes().size();
					relation.attributes().clear();
				}
			}

			//
			if (scope == Scope.RELATION) {
				Relations relations = net.relations().getByModelName(optionalRelationModelName);
				for (Relation relation : relations) {
					result += relation.attributes().size();
					relation.attributes().clear();
				}
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @param net
	 * @param target
	 * @param label
	 * @param type
	 * @return
	 */
	public static int removeAttribute(final Attributes source, final String label, final EmptyType type) {
		int result;

		if (label == null) {
			throw new IllegalArgumentException("label parameter is null");
		} else if (type == null) {
			throw new IllegalArgumentException("type parameter is null");
		} else if (source == null) {
			result = 0;
		} else {
			Attribute target = source.get(label);
			switch (type) {
				case VOID:
					if (target == null) {
						result = 0;
					} else {
						source.remove(label);
						result = 1;
					}
				break;

				case BLANK:
					if (target == null) {
						result = 0;
					} else {
						target.setValue("");
						result = 1;
					}
				break;

				case FORCED_BLANK:
					source.put(label, "");
					result = 1;
				break;

				default:
					result = 0;
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @param target
	 *            ALL, CORPUS, INDIVIDUAL, FAMILY or relation model name.
	 * @param label
	 * @param type
	 * @return
	 */
	public static int removeAttribute(final Net net, final Scope scope, final String optionalRelationModelName, final String label, final EmptyType type) {
		int result;

		logger.debug("removeAttribute [net={}][scope={}][optionalName={}][label={}][type={}]", net, scope, optionalRelationModelName, label, type);

		if (net == null) {
			throw new IllegalArgumentException("net parameter is null");
		} else if (scope == null) {
			throw new IllegalArgumentException("target parameter is null");
		} else if (label == null) {
			throw new IllegalArgumentException("label parameter is null");
		} else if (type == null) {
			throw new IllegalArgumentException("type parameter is null");
		} else {
			//
			result = 0;

			//
			if ((scope == Scope.ALL) || (scope == Scope.CORPUS)) {
				result += removeAttribute(net.attributes(), label, type);
			}

			//
			if ((scope == Scope.ALL) || (scope == Scope.INDIVIDUALS)) {
				for (Individual individual : net.individuals()) {
					result += removeAttribute(individual.attributes(), label, type);
				}
			}

			//
			if ((scope == Scope.ALL) || (scope == Scope.FAMILIES)) {
				for (Family family : net.families()) {
					result += removeAttribute(family.attributes(), label, type);
				}
			}

			//
			if ((scope == Scope.ALL) || (scope == Scope.RELATIONS)) {
				for (Relation relation : net.relations()) {
					result += removeAttribute(relation.attributes(), label, type);
				}
			}

			//
			if (scope == Scope.RELATION) {
				Relations relations = net.relations().getByModelName(optionalRelationModelName);
				for (Relation relation : relations) {
					result += removeAttribute(relation.attributes(), label, type);
				}
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @param sourceLabel
	 * @param targetLabel
	 * @param valuePrefix
	 * @return
	 */
	public static int valuateExogenousAttribute(final Individuals source, final String sourceLabel, final String targetLabel, final String valuePrefix) {
		int result;

		logger.debug("valuateExogenousAttribute [source={}][sourceLabel={}][targetLabel={}][valuePrefix={}]", source, sourceLabel, targetLabel, valuePrefix);

		if (source == null) {
			throw new IllegalArgumentException("source parameter is null");
		} else if (sourceLabel == null) {
			throw new IllegalArgumentException("source label parameter is null");
		} else if (StringUtils.isBlank(targetLabel)) {
			throw new IllegalArgumentException("target label parameter is null");
		} else {
			//
			result = 0;
			NumberedValues values = IndividualValuator.get(source, sourceLabel);
			for (Individual individual : source) {
				//
				Value value = values.get(individual.getId());

				//
				String attributeValue;
				if (value == null) {
					attributeValue = "";
				} else {
					attributeValue = value.toString();
				}

				//
				String prefix;
				if (valuePrefix == null) {
					prefix = "";
				} else {
					prefix = valuePrefix;
				}

				//
				individual.attributes().put(targetLabel, prefix + attributeValue);
				result += 1;
			}
		}

		//
		return result;
	}
}
