/**
 * Copyright 2013-2015 Christian Pierre MOMON, DEVINSY, UMR 7186 LESC.
 * 
 * christian.momon@devinsy.fr
 * 
 * This file is part of Kiwa. This software (Kiwa) is a computer program whose
 * purpose is to be the Kinsources Web Application, an open interactive platform
 * for archiving, sharing, analyzing and comparing kinship data used in
 * scientific inquiry.
 * 
 * This software is governed by the CeCILL license under French law and abiding
 * by the rules of distribution of free software. You can use, modify and/ or
 * redistribute the software under the terms of the CeCILL license as circulated
 * by CEA, CNRS and INRIA at the following URL "http://www.cecill.info".
 * 
 * As a counterpart to the access to the source code and rights to copy, modify
 * and redistribute granted by the license, users are provided only with a
 * limited warranty and the software's author, the holder of the economic
 * rights, and the successive licensors have only limited liability.
 * 
 * In this respect, the user's attention is drawn to the risks associated with
 * loading, using, modifying and/or developing or reproducing the software by
 * the user in light of its specific status of free software, that may mean that
 * it is complicated to manipulate, and that also therefore means that it is
 * reserved for developers and experienced professionals having in-depth
 * computer knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions enabling
 * the security of their systems and/or data to be ensured and, more generally,
 * to use and operate it in the same conditions as regards security.
 * 
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 */
package org.kinsources.kiwa.website.kidarep;

import java.io.IOException;
import java.util.Locale;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.kinsources.kiwa.accounts.Account;
import org.kinsources.kiwa.kernel.Kiwa;
import org.kinsources.kiwa.kidarep.Dataset;
import org.kinsources.kiwa.kidarep.DatasetFile;
import org.kinsources.kiwa.website.charter.ErrorView;

import fr.devinsy.kiss4web.Redirector;
import fr.devinsy.util.StacktraceWriter;
import fr.devinsy.util.xml.XMLTools;
import fr.devinsy.xidyn.data.TagDataManager;
import fr.devinsy.xidyn.presenters.URLPresenter;

/**
 *
 */
public class Dataset_raw_statistics_xhtml extends HttpServlet
{
	private static final long serialVersionUID = -8903536311059605390L;
	private static org.slf4j.Logger logger = org.slf4j.LoggerFactory.getLogger(Dataset_raw_statistics_xhtml.class);
	private static Kiwa kiwa = Kiwa.instance();
	private static URLPresenter xidyn = new URLPresenter("/org/kinsources/kiwa/website/kidarep/dataset_raw_statistics.html");

	/** 
	 *
	 */
	@Override
	public void doGet(final HttpServletRequest request, final HttpServletResponse response) throws IOException, ServletException
	{
		try
		{
			//
			logger.debug("doGet starting...");
			kiwa.logPageHit("pages.kidarep.dataset_statistics", request);

			//
			Locale locale = kiwa.getUserLocale(request);
			Account account = kiwa.getAuthentifiedAccount(request, response);
			Long accountId = Account.getId(account);

			// Get parameters.
			// ===============
			String datasetId = request.getParameter("dataset_id");
			logger.info("datasetId=[{}]", datasetId);

			// Use parameters.
			// ===============
			if (datasetId == null)
			{
				throw new IllegalArgumentException("Invalid dataset id");
			}
			else
			{
				Dataset dataset = kiwa.kidarep().getDatasetById(datasetId);

				if (dataset == null)
				{
					throw new IllegalArgumentException("Invalid dataset.");
				}
				else
				{
					//
					TagDataManager data = new TagDataManager();

					if (isNotGrantedAccess(dataset, account))
					{
						data.setContent("dataset", "Permission denied.");
					}
					else
					{
						//
						data.setContent("dataset_name", XMLTools.escapeXmlBlank(dataset.getName()));

						data.setContent("dataset_id", dataset.getId());

						// Statistics.
						DatasetFile sourceFile = kiwa.getDatasetFile(dataset, account);

						//
						if ((sourceFile == null) || (sourceFile.statistics().isEmpty()))
						{
							//
							data.setContent("statistic_count", 0);
							data.setContent("statistics_section", "None.");

						}
						else
						{
							//
							data.setContent("statistic_count", sourceFile.statistics().size());

							//
							int line = 0;
							for (String key : sourceFile.statistics().keySortedList())
							{
								//
								data.setContent("statistic", line, "statistic_name", key);
								Object value = sourceFile.statistics().getValue(key);
								if ((value == null) || (value instanceof byte[]))
								{
									data.setContent("statistic", line, "statistic_value", "n/a");
								}
								else
								{
									data.setContent("statistic", line, "statistic_value", value.toString());
								}

								//
								line += 1;
							}
						}

						data.setContent("button_back", "Back");
						data.setAttribute("button_back", "href", kiwa.permanentURI(dataset) + "#statistics_section");
					}

					// Send response.
					// ==============
					StringBuffer content = xidyn.dynamize(data);

					//
					StringBuffer page = kiwa.getCharterView().getHtml(accountId, locale, content);

					//
					response.setContentType("application/xhtml+xml; charset=UTF-8");
					response.getWriter().println(page);
				}
			}

		}
		catch (final IllegalArgumentException exception)
		{
			logger.error(StacktraceWriter.toString(exception));

			Redirector.redirect(response, "/");

		}
		catch (final Exception exception)
		{
			ErrorView.show(request, response, exception);
		}
	}

	/**
	 *
	 */
	@Override
	public void doPost(final HttpServletRequest request, final HttpServletResponse response) throws ServletException, IOException
	{
		doGet(request, response);
	}

	/**
	 *
	 */
	@Override
	public void init() throws ServletException
	{
	}

	/**
	 * 
	 * @param dataset
	 * @param account
	 * @return
	 */
	public static boolean isGrantedAccess(final Dataset dataset, final Account account)
	{
		boolean result;

		if ((dataset.getStatus() == Dataset.Status.VALIDATED) || (dataset.isContributor(account) || (Kiwa.instance().isWebmaster(account))))
		{
			result = true;
		}
		else if ((Kiwa.instance().isSciboarder(account)) && (dataset.getStatus() == Dataset.Status.SUBMITTED))
		{
			result = true;
		}
		else
		{
			result = false;
		}

		//
		return result;
	}

	/**
	 * 
	 * @param dataset
	 * @param account
	 * @return
	 */
	public static boolean isNotGrantedAccess(final Dataset dataset, final Account account)
	{
		boolean result;

		result = !isGrantedAccess(dataset, account);

		//
		return result;
	}
}

// ////////////////////////////////////////////////////////////////////////
