/**
 * Copyright 2013-2015 Christian Pierre MOMON, DEVINSY, UMR 7186 LESC.
 * 
 * christian.momon@devinsy.fr
 * 
 * This file is part of Kiwa. This software (Kiwa) is a computer program whose
 * purpose is to be the Kinsources Web Application, an open interactive platform
 * for archiving, sharing, analyzing and comparing kinship data used in
 * scientific inquiry.
 * 
 * This software is governed by the CeCILL license under French law and abiding
 * by the rules of distribution of free software. You can use, modify and/ or
 * redistribute the software under the terms of the CeCILL license as circulated
 * by CEA, CNRS and INRIA at the following URL "http://www.cecill.info".
 * 
 * As a counterpart to the access to the source code and rights to copy, modify
 * and redistribute granted by the license, users are provided only with a
 * limited warranty and the software's author, the holder of the economic
 * rights, and the successive licensors have only limited liability.
 * 
 * In this respect, the user's attention is drawn to the risks associated with
 * loading, using, modifying and/or developing or reproducing the software by
 * the user in light of its specific status of free software, that may mean that
 * it is complicated to manipulate, and that also therefore means that it is
 * reserved for developers and experienced professionals having in-depth
 * computer knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions enabling
 * the security of their systems and/or data to be ensured and, more generally,
 * to use and operate it in the same conditions as regards security.
 * 
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 */
package org.kinsources.kiwa.actalog;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.ResourceBundle;

import org.joda.time.DateTime;
import org.kinsources.kiwa.actalog.EventLog.Category;
import org.slf4j.helpers.MessageFormatter;

import fr.devinsy.util.StringListWriter;
import fr.devinsy.util.rss.RSSElement;
import fr.devinsy.util.rss.RSSWriter;

/**
 * 
 * @author Christian P. Momon
 */
public class Actalog
{
	private static org.slf4j.Logger logger = org.slf4j.LoggerFactory.getLogger(Actalog.class);
	public static final String RSS_NAME = "kinsources_website";
	private long lastId;
	private EventLogs eventLogs;

	/**
	 * 
	 */
	public Actalog()
	{
		this.lastId = 0;
		this.eventLogs = new EventLogs();
	}

	/**
	 * 
	 */
	public Actalog(final long autoIncrementValue)
	{
		this.lastId = autoIncrementValue;
		this.eventLogs = new EventLogs();
	}

	/**
	 * 
	 */
	public void clear()
	{
		this.lastId = 0;
		this.eventLogs.clear();
	}

	/**
	 * 
	 * @return
	 */
	public EventLogs eventLogs()
	{
		return this.eventLogs;
	}

	/**
	 * 
	 * @return
	 */
	public long lastId()
	{
		long result;

		result = this.lastId;

		//
		return result;
	}

	/**
	 * 
	 */
	synchronized public EventLog logEvent(final Category category, final String... arguments)
	{
		EventLog result;

		result = new EventLog(nextId(), DateTime.now(), category, arguments);
		this.eventLogs.add(result);

		//
		return result;
	}

	/**
	 * 
	 * @return
	 */
	synchronized public long nextId()
	{
		long result;

		this.lastId += 1;

		result = this.lastId;

		//
		return result;
	}

	/**
	 * 
	 */
	public void purge(final int days)
	{
		//
		DateTime limit = DateTime.now().minusDays(days);

		//
		Iterator<EventLog> iterator = this.eventLogs.iterator();
		while (iterator.hasNext())
		{
			//
			EventLog eventLog = iterator.next();

			//
			if (eventLog.getTime().isBefore(limit))
			{
				iterator.remove();
			}
		}
	}

	/**
	 * 
	 */
	public void resetLastId()
	{
		this.lastId = this.eventLogs.lastId();
	}

	/**
	 * 
	 * @param language
	 * @return
	 * @throws IOException
	 */
	public String toRSS(final int feedSize, final Locale locale, final String websiteUrl) throws IOException
	{
		String result;

		//
		ResourceBundle bundle = ResourceBundle.getBundle("org.kinsources.kiwa.actalog.rss", locale);

		StringListWriter target = new StringListWriter();
		RSSWriter out = new RSSWriter(target);

		//
		List<RSSElement> elements = new ArrayList<RSSElement>();
		elements.add(new RSSElement("pubDate", DateTime.now()));
		elements.add(new RSSElement("lastBuildDate", DateTime.now()));
		elements.add(new RSSElement("generator", "Generated by Kiwa"));
		elements.add(new RSSElement("language", locale.getLanguage()));

		//
		out.writeChannel(bundle.getString("FEED_TITLE"), websiteUrl + "rss/kinsources_website-en.rss", bundle.getString("FEED_DESCRIPTION"), elements.toArray(new RSSElement[0]));

		//
		if (this.eventLogs.isEmpty())
		{
			out.writeItem("No event.", null, null, new RSSElement("category", "system"), new RSSElement("author", "webmaster@kinsources.net"));
		}
		else
		{
			for (EventLog eventLog : this.eventLogs.lastest(feedSize))
			{
				//
				elements.clear();
				elements.add(new RSSElement("guid", "kinsources_event_log_" + eventLog.getId(), "isPermaLink", "false"));
				elements.add(new RSSElement("pubDate", eventLog.getTime()));
				elements.add(new RSSElement("author", "webmaster@kinsources.net (kiwa bot)"));
				elements.add(new RSSElement("category", eventLog.getCategory().toString()));

				//
				String title = toString(eventLog, locale);

				//
				out.writeItem(title, null, null, elements.toArray(new RSSElement[0]));
			}
		}

		//
		out.close();

		//
		result = target.toString();

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @param locale
	 * @return
	 */
	public String toString(final EventLog source, final Locale locale)
	{
		String result;

		//
		ResourceBundle bundle = ResourceBundle.getBundle("org.kinsources.kiwa.actalog.rss", locale);

		//
		result = MessageFormatter.arrayFormat(bundle.getString(source.getCategory().toString()), source.getArguments()).getMessage();

		//
		return result;
	}
}
