package org.tip.puck.spacetime.workers;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

import org.tip.puck.PuckException;
import org.tip.puck.geo.GeoLevel;
import org.tip.puck.graphs.Graph;
import org.tip.puck.graphs.GraphComparatorByArcCount;
import org.tip.puck.graphs.Link;
import org.tip.puck.graphs.Node;
import org.tip.puck.graphs.workers.GraphUtils;
import org.tip.puck.graphs.workers.NodeValuator;
import org.tip.puck.net.Gender;
import org.tip.puck.net.Individual;
import org.tip.puck.net.relations.Relation;
import org.tip.puck.partitions.Cluster;
import org.tip.puck.partitions.Partition;
import org.tip.puck.partitions.PartitionCriteria;
import org.tip.puck.partitions.PartitionMaker;
import org.tip.puck.spacetime.CorrelationMatrix;
import org.tip.puck.spacetime.EgoRelationSequence;
import org.tip.puck.spacetime.EgoRelationSequences;
import org.tip.puck.spacetime.workers.SpaceTimeCriteria.RelationClassificationType;
import org.tip.puck.util.MathUtils;
import org.tip.puck.util.NumberedValues;
import org.tip.puck.util.PuckUtils;
import org.tip.puck.util.Trafo;
import org.tip.puck.util.Value;
import org.tip.puck.util.Values;

import fr.devinsy.util.StringList;

/**
 * 
 * @author Klaus Hamberger
 *
 */
public class SequencesCensus {
	
	private EgoRelationSequences sequences;
	private Map<String, NumberedValues> valuesMap;
	private List<String> labels;
	private Map<String, StringList> pajekBuffers;
	
	private Map<String,CorrelationMatrix> eventSequenceMatrices;
	private Map<String,CorrelationMatrix> subSequenceMatrices;
		
	private Map<String,Map<Integer,Partition<Node<Individual>>>> componentsMap;
	private CorrelationMatrix relationConnectionMatrix;

	private Map<RelationClassificationType,Partition<Link<Individual>>> linkPartitions;
	private Map<RelationClassificationType,Map<Value,Double[]>> similaritiesMaps;
	
	Map<RelationClassificationType,Map<String,Integer>> nrValues;
	private Map<String,Map<String,Map<String,Value>>> parcoursNetworkStatistics;
	
	public SequencesCensus (EgoRelationSequences sequences, SpaceTimeCriteria criteria) throws PuckException{
		
		this.sequences = sequences;
		this.labels = new ArrayList<String>();
		this.labels.addAll(criteria.getCensusOperationLabels());
		this.labels.add("PROFILE_AGE");
		this.labels.add("PROFILE_"+criteria.getDateLabel());
		valuesMap = new HashMap<String, NumberedValues>();

		for (String label : labels){
			if (label.contains("SIMILARITY")){
				if (similaritiesMaps == null){
					similaritiesMaps = new HashMap<RelationClassificationType,Map<Value,Double[]>>();
					linkPartitions = new HashMap<RelationClassificationType,Partition<Link<Individual>>>();
				}
				RelationClassificationType relationClassificationType = RelationClassificationType.valueOf(label.substring(label.lastIndexOf("_")+1));
				similaritiesMaps.put(relationClassificationType, new HashMap<Value,Double[]>());
				linkPartitions.put(relationClassificationType, new Partition<Link<Individual>>());
			} 
			valuesMap.put(label, new NumberedValues());
		}
		
		pajekBuffers = new HashMap<String, StringList>();
		for (String title : criteria.getNetworkTitles()){
			pajekBuffers.put(title, new StringList());
		}
		
		for (String networkTitle : criteria.getNetworkTitles()){
			if (networkTitle.contains("Ego Network") || networkTitle.contains("Parcours Similarity Network")){
				if (componentsMap==null){
					componentsMap = new HashMap<String,Map<Integer,Partition<Node<Individual>>>>();
				}
				componentsMap.put(networkTitle, new HashMap<Integer,Partition<Node<Individual>>>());
			}
		}

		eventSequenceMatrices = new HashMap<String,CorrelationMatrix>();
		subSequenceMatrices = new HashMap<String,CorrelationMatrix>();
		nrValues = new HashMap<RelationClassificationType,Map<String,Integer>>();

		for (RelationClassificationType relationClassificationType : criteria.getTrajectoriesRelationClassificationTypes()){
			nrValues.put(relationClassificationType, new HashMap<String,Integer>());
		}
		
		// Optionalize!
		Map<String,Map<Integer,Graph<Cluster<Relation>>>> parcoursNetworksMap = new HashMap<String,Map<Integer,Graph<Cluster<Relation>>>>();
		for (String networkTitle : criteria.getNetworkTitles()){
			if (networkTitle.contains("Parcours Network") && !networkTitle.contains("Fused")){
				parcoursNetworksMap.put(networkTitle.substring(networkTitle.lastIndexOf("_")+1),new HashMap<Integer,Graph<Cluster<Relation>>>());
			}
		}


		for (EgoRelationSequence sequence : sequences){
			
			SequenceCensus census = new SequenceCensus(sequence, criteria);
			
			for (String label : labels){
				if (label.contains("SIMILARITY")){
					
					RelationClassificationType relationClassificationType = RelationClassificationType.valueOf(label.substring(label.lastIndexOf("_")+1));
					linkPartitions.get(relationClassificationType).add(census.getParcoursLinkPartition(relationClassificationType));
					
				} 
				valuesMap.get(label).put(sequence.getEgo().getId(), census.getValue(label));				
			}
			
			if (componentsMap!=null){
				
				for (String networkTitle : criteria.getNetworkTitles()){
					Map<Integer,Partition<Node<Individual>>> components = componentsMap.get(networkTitle);
					if (components!=null){
						if (networkTitle.contains("Ego Network")){
							components.put(sequence.getEgo().getId(), census.getComponents("Nonmediated Ego Network"));
						} else if (networkTitle.contains("Parcours Similarity Network")){
							RelationClassificationType relationClassificationType = RelationClassificationType.valueOf(networkTitle.substring(networkTitle.lastIndexOf("_")+1));
							components.put(sequence.getEgo().getId(), census.getComponents("Parcours Similarity Network_"+relationClassificationType));
						}
					}
				}
			}

			for (String networkTitle : criteria.getNetworkTitles()){
				if (networkTitle.contains("Parcours Network") && !networkTitle.contains("Fused")){
					parcoursNetworksMap.get(networkTitle.substring(networkTitle.lastIndexOf("_")+1)).put(sequence.getEgo().getId(), census.getEventTypeNetwork(networkTitle));
				}
			}


			
/*			for (EventType eventType : criteria.getMainEventTypes()){
				
				String eventTypeName = eventType.toString();

				if (criteria.getNetworkTitles().contains("Event Type Network")){
					
					Partition<String> eventPartition = eventPartitions.get(eventTypeName);
					Partition<String> eventPairPartition = eventPairPartitions.get(eventTypeName);
					
					census.putEvents(eventPartition,EventType.valueOf(eventTypeName));
					census.putEventPairs(eventPairPartition, EventType.valueOf(eventTypeName));

					
				} 
				
				if (criteria.getNetworkTitles().contains("Sequence Type Network")){

					Partition<String> subSequencePartition = subSequencePartitions.get(eventTypeName);
					
					census.putSubSequences(subSequencePartition,EventType.valueOf(eventTypeName));
				} 
			}*/
			
			
			for (String title : criteria.getNetworkTitles()){
				
				if (!title.equals("Event Type Network") && !title.equals("Sequence Type Network")){
					
					census.writePajekNetwork(pajekBuffers.get(title), title);

				}
				
			}
			
		}
		
		// Create union graphs, similarity networks and phylogenetic trees 
		
		for (String networkTitle : criteria.getNetworkTitles()){ // Add condition for the two operations
			if (networkTitle.contains("Parcours Network") && !networkTitle.contains("Fused")){
									
				Map<Integer,Graph<Cluster<Relation>>> parcoursNetworks = parcoursNetworksMap.get(networkTitle.substring(networkTitle.lastIndexOf("_")+1));
				List<Graph<String>> flatParcoursNetworksNoLoops = new ArrayList<Graph<String>>();
				
				for (Graph<Cluster<Relation>> parcoursNetwork : parcoursNetworks.values()){
					Graph<String> flatParcoursNetworkNoLoops = SequenceCensus.getFlatParcoursNetworkNoLoops(parcoursNetwork);
/*					Graph<String> flatParcoursNetworkNoLoops = new Graph<String>(parcoursNetwork.getLabel());
					for (Link<Cluster<Relation>> link : parcoursNetwork.getLinks()){
						if (!link.isLoop()){
							flatParcoursNetworkNoLoops.addArc(link.getSourceNode().getReferent().getLabel(),link.getTargetNode().getReferent().getLabel());
						}
					}*/
					flatParcoursNetworksNoLoops.add(flatParcoursNetworkNoLoops);
				}
				
				Collections.sort(flatParcoursNetworksNoLoops, new GraphComparatorByArcCount<String>());
				
				// Make phylogenetic tree
				Graph<Set<Graph<String>>> tree = GraphUtils.createPhylogeneticTree(flatParcoursNetworksNoLoops);
				tree.setLabel(networkTitle+"_Tree");
				for (Node<Set<Graph<String>>> node : tree.getNodes()){
					node.setLabel(node.getLabel().replaceAll(networkTitle+" ",""));
					if (node.getReferent().size()==1){
						node.setAttribute("TYPE", "1");
						String[] splitLabel = Trafo.noParentheses(node.getReferent().toString()).split(" ");
						Integer egoId = Integer.parseInt(splitLabel[splitLabel.length-1]);
						node.setAttribute("GENDER", sequences.getByEgoId(egoId).getEgo().getGender().toString());
					} else {
						node.setAttribute("TYPE", "0");
					}
				}
				
				List<String> treePartitionLabels = new ArrayList<String>();
				treePartitionLabels.add("TYPE");
				treePartitionLabels.add("GENDER");
				
				pajekBuffers.get(networkTitle.replaceAll("Network", "Similarity Tree")).addAll(PuckUtils.writePajekNetwork(tree,treePartitionLabels)); 		

				// Make union graphs
				List<Graph<Cluster<Relation>>> unions = new ArrayList<Graph<Cluster<Relation>>>();
				PartitionCriteria partitionCriteria = new PartitionCriteria(criteria.getPartitionLabel());
				Partition<Individual> individualPartition = PartitionMaker.create("", sequences.egos(), partitionCriteria);
				Partition<Graph<Cluster<Relation>>> graphPartition = new Partition<Graph<Cluster<Relation>>>();
				
				for (Individual ego : individualPartition.getItems()){
					graphPartition.put(parcoursNetworks.get(ego.getId()), individualPartition.getValue(ego));
				}
				
				parcoursNetworkStatistics = new TreeMap<String,Map<String,Map<String,Value>>>();
				
				for (Cluster<Graph<Cluster<Relation>>> graphCluster : graphPartition.getClusters()){
					Graph<Cluster<Relation>> union = GraphUtils.fuseGraphs(graphCluster.getItems());
					union.setLabel(networkTitle+"_"+graphCluster.getValue());
					unions.add(union);
					parcoursNetworkStatistics.put(graphCluster.getValue()+"",GraphUtils.getNodeStatisticsByLabel(union, criteria.getNodeStatisticsLabels()));
				}
				Graph<Cluster<Relation>> totalUnion = GraphUtils.fuseGraphs(new ArrayList<Graph<Cluster<Relation>>>(parcoursNetworks.values()));
				totalUnion.setLabel(networkTitle+"_Total");
				parcoursNetworkStatistics.put("Total",GraphUtils.getNodeStatisticsByLabel(totalUnion, criteria.getNodeStatisticsLabels()));
				unions.add(totalUnion);
				
				List<String> unionPartitionLabels = new ArrayList<String>();
				unionPartitionLabels.add("NUMBER");
				unionPartitionLabels.add("SIZE");
				unionPartitionLabels.add("BETWEENNESS");
				unionPartitionLabels.add("DEGREE");

				for (Graph<Cluster<Relation>> union : unions){
					pajekBuffers.get(networkTitle.replaceAll("Network", "Network Fused")).addAll(PuckUtils.writePajekNetwork(union,unionPartitionLabels)); 		
				}
			}
		}
		

			
		
		
		// Make Event Type Networks
		
		if (criteria.getNetworkTitles().contains("Event Type Network")){

			for (RelationClassificationType relationClassificationType : criteria.getTrajectoriesRelationClassificationTypes()){
				
				String eventTypeName = relationClassificationType.toString();
				Map<Individual,List<String>> singles = new TreeMap<Individual,List<String>>();
				Map<Individual,List<String[]>> pairs  = new TreeMap<Individual,List<String[]>>();
				
				for (EgoRelationSequence sequence : sequences){
					
					Individual ego = sequence.getEgo();
					
					Value singlesValue = valuesMap.get("PROFILE_"+eventTypeName).get(ego.getId());
					
					if (singlesValue != null){

						List<String> singlesList = (List<String>)singlesValue.listValue();
						List<String[]> pairsList = new ArrayList<String[]>();
						for (int i=1;i<singlesList.size();i++){
							pairsList.add(new String[]{singlesList.get(i-1),singlesList.get(i)});
						}
						
						//
						singles.put(ego, singlesList);
						pairs.put(ego, pairsList);
					}
				}
				
				eventSequenceMatrices.put(eventTypeName, new CorrelationMatrix("Event Type Network",eventTypeName,singles,pairs));
			}
		}
		
		if (criteria.getNetworkTitles().contains("Sequence Type Network")){

			for (RelationClassificationType relationClassificationType : criteria.getTrajectoriesRelationClassificationTypes()){
				
				String eventTypeName = relationClassificationType.toString();
				Map<Individual,List<String>> singles = new TreeMap<Individual,List<String>>();
				Map<Individual,List<String[]>> pairs  = new TreeMap<Individual,List<String[]>>();
				
				for (EgoRelationSequence sequence : sequences){
					
					Individual ego = sequence.getEgo();
					
					Value singlesValue = valuesMap.get("PROFILE_"+eventTypeName).get(ego.getId());
					
					if (singlesValue != null){

						List<String> singlesList = PuckUtils.cumulateList((List<String>)singlesValue.listValue());
						List<String[]> pairsList = new ArrayList<String[]>();
						for (int i=1;i<singlesList.size();i++){
							pairsList.add(new String[]{singlesList.get(i-1),singlesList.get(i)});
						}
						//
						singles.put(ego, singlesList);
						pairs.put(ego, pairsList);
					}
				}
				
				subSequenceMatrices.put(eventTypeName, new CorrelationMatrix("Sequence Type Network",eventTypeName,singles,pairs));
			}
		}
		
		
		if (labels.contains("CONNECTED_NETWORK_RELATIONS")){

			Map<Individual,List<String>> singles = new TreeMap<Individual,List<String>>();
			Map<Individual,List<String[]>> pairs  = new TreeMap<Individual,List<String[]>>();
			
			for (EgoRelationSequence sequence : sequences){
				
				Individual ego = sequence.getEgo();
				Value singlesValue = valuesMap.get("NETWORK_RELATIONS").get(ego.getId());
				
				if (singlesValue != null){
					singles.put(ego, (List<String>)singlesValue.listValue());
				}
								
				Value pairsValue = valuesMap.get("CONNECTED_NETWORK_RELATIONS").get(ego.getId());
				
				if (pairsValue != null){
					pairs.put(ego, (List<String[]>)pairsValue.listValue());
				}
			}
			
			relationConnectionMatrix = new CorrelationMatrix("Component connections",null,singles,pairs);
			
		}

/*		for (String networkTitle : criteria.getNetworkTitles()){
			if (networkTitle.contains("Parcours Similarity Network")){
				
				for (RelationClassificationType relationClassificationType : linkPartitions.keySet()){
					Partition<Link<Individual>> linkPartition = linkPartitions.get(relationClassificationType);
					Map<Value,Double[]> similaritiesMap = similaritiesMaps.get(relationClassificationType);
					
					for (Value linkValue : linkPartition.getValues()){
						
						Double[] values = new Double[]{0.,0.,0.,0.,0.};
						Double[] sums = new Double[]{0.,0.,0.,0.,0.};

						for (Link<Individual> link : linkPartition.getCluster(linkValue).getItems()){
							Individual ego = (Individual)link.getSourceNode().getReferent();
							Individual alter = (Individual)link.getTargetNode().getReferent();
							
							int egoGender = ego.getGender().toInt();
							int alterGender = alter.getGender().toInt();
							int idx = egoGender + 2 * alterGender;
							
							values[idx] += link.getWeight();
							values[4] += link.getWeight();
							sums[idx]++;
							sums[4]++;
							
						}
						
						for (int idx=0;idx<sums.length;idx++){
							values[idx] = MathUtils.percent(values[idx], 100*sums[idx]);
						}
						
						similaritiesMap.put(linkValue, values);
					}
				}
			}
		}*/
		
		for (RelationClassificationType relationClassificationType : criteria.getTrajectoriesRelationClassificationTypes()){

			if (criteria.getNetworkTitles().contains("Event Type Network")){

				CorrelationMatrix matrix = eventSequenceMatrices.get(relationClassificationType.toString());

				Graph<Cluster<String>>[] eventTypeNetworks = matrix.getSequenceNetworks();
				
				List<String> partitionLabels = new ArrayList<String>();
				partitionLabels.add(relationClassificationType.toString());
				Map<String,Map<Value,Integer>> partitionNumbersMaps = getPartitionNumbersMaps(partitionLabels, eventTypeNetworks[2]);
				for (String label : partitionLabels){
					if (partitionNumbersMaps.get(label)!=null){
						nrValues.get(relationClassificationType).put(label, partitionNumbersMaps.get(label).size());
					}
				}
				partitionLabels.add("SIZE");

				for (Gender gender : Gender.values()){
					
					pajekBuffers.get("Event Type Network").addAll(PuckUtils.writePajekNetwork(eventTypeNetworks[gender.toInt()],partitionLabels,partitionNumbersMaps));

				}
			}
				
			if (criteria.getNetworkTitles().contains("Sequence Type Network")){
				
				CorrelationMatrix matrix = subSequenceMatrices.get(relationClassificationType.toString());

				Graph<Cluster<String>>[] sequenceTypeNetworks = matrix.getSequenceNetworks();
				matrix.getDepthPartitions();
						
				List<String> partitionLabels = new ArrayList<String>();
				partitionLabels.add(relationClassificationType.toString());
				Map<String,Map<Value,Integer>> partitionNumbersMaps = getPartitionNumbersMaps(partitionLabels, sequenceTypeNetworks[2]);
				for (String label : partitionLabels){
					nrValues.get(relationClassificationType).put(label, partitionNumbersMaps.get(label).size());
				}
				partitionLabels.add("SIZE");
				partitionLabels.add("STEP");

				for (Gender gender : Gender.values()){
					pajekBuffers.get("Sequence Type Network").addAll(PuckUtils.writePajekNetwork(sequenceTypeNetworks[gender.toInt()],partitionLabels,partitionNumbersMaps));
				}
			}
		}
	}
	
	public static <E> Map<String,Map<Value,Integer>> getPartitionNumbersMaps(List<String> labels, Graph<E> model){
		Map<String,Map<Value,Integer>> result;
		
		result = new HashMap<String,Map<Value,Integer>>();
		
		for (String label : labels){
			
			Values values = NodeValuator.get(model, label);
			if (!values.isNumeric()) {
				Partition<Value> partition = PartitionMaker.create(label, values);
				result.put(label, PartitionMaker.getPartitionNumbersMap(partition));
			}

		}
		
		//
		return result;
		
	}
	

	public Map<String, StringList> getPajekBuffers() {
		return pajekBuffers;
	}



	public Map<Integer, Partition<Node<Individual>>> getComponents(String networkTitle) {
		return componentsMap.get(networkTitle);
	}
	
	public Map<Value,Double[]> getMeanNrMoves (){
		Map<Value, Double[]> result;
		
		result = new TreeMap<Value, Double[]>();
		
		Map<GeoLevel, Integer[]> sequenceMap = new HashMap<GeoLevel, Integer[]>();
		Map<GeoLevel, Integer[]> eventMap = new HashMap<GeoLevel, Integer[]>();
		
		for (GeoLevel level : new GeoLevel[]{GeoLevel.TRANSNATIONAL,GeoLevel.TRANSREGIONAL,GeoLevel.REGIONAL,GeoLevel.LOCAL}){
			sequenceMap.put(level, new Integer[]{0,0,0});
			eventMap.put(level, new Integer[]{0,0,0});
			result.put(new Value(level), new Double[]{0.,0.,0.});
		}
		
		
		for (EgoRelationSequence sequence : sequences){
			int gender = sequence.getEgo().getGender().toInt();
			Value value = valuesMap.get("MEAN_NR_MOVES").get(sequence.getEgo().getId());
			if (value!=null){
				Map<GeoLevel,Integer> distanceProfile = value.mapValue();
				for (GeoLevel level : distanceProfile.keySet()){
					if (sequenceMap.get(level)!=null){
						sequenceMap.get(level)[gender] += 1;
						sequenceMap.get(level)[2] += 1;
						eventMap.get(level)[gender] += distanceProfile.get(level);
						sequenceMap.get(level)[2] += distanceProfile.get(level);
					}
				}
			}
		}
		
		for (Value value : result.keySet()){
			for (int i=0;i<3;i++){
				result.get(value)[i] = new Double(eventMap.get(value.enumValue())[i])/new Double(sequenceMap.get(value.enumValue())[i]);
			}
		}
		//
		return result;
	}


	public NumberedValues getValues (String label){
		
		return valuesMap.get(label);
	}
	
	public Graph<Cluster<String>> getSequenceNetwork (String title, RelationClassificationType relationClassificationType, Partition<String> partition){
		Graph<Cluster<String>> result;
		
		if (sequences == null) {
			throw new IllegalArgumentException("Null parameter detected.");
		} else {
			
			result = new Graph<Cluster<String>>(title+"_"+relationClassificationType);
			
			//
			for (Cluster<String> cluster : partition.getClusters().toListSortedByDescendingSize()) {
				if (!cluster.isNull()) {
					result.addNode(cluster);
				}
			}
			
			//
			for (EgoRelationSequence biography : sequences){
				Cluster<String> previous = null;
				for (Relation event : biography.getStations().values()){
					Cluster<String> next = partition.getCluster(biography.getEgo().getId()+" "+event.getTypedId());
					if (previous!=null){
						result.incArcWeight(previous, next);
					}
					previous = next;
				}
			}
			
			for (Node<Cluster<String>> node : result.getNodes()){
				Cluster<String> referent = node.getReferent();
				if (referent !=null){
					Value clusterValue = referent.getValue();
					if (clusterValue!=null){
						String value = clusterValue.toString();
						if (value.lastIndexOf("-")>-1){
							value = value.substring(value.lastIndexOf("-")+1);
						}
						node.setAttribute(relationClassificationType.toString(), value);
					}
				}
			}
		}
		
		//
		return result;
	}




/*	public Graph<Cluster<String>> getEventTypeNetwork(String eventTypeName) {
		return eventTypeNetworks.get(eventTypeName);
	}




	public Graph<Cluster<String>> getSequenceTypeNetwork(String eventTypeName) {
		return sequenceTypeNetworks.get(eventTypeName);
	}




	public Partition<String> getEventPartition(String eventTypeName) {
		return eventPartitions.get(eventTypeName);
	}*/
	
	public Integer getNrValues (RelationClassificationType relationClassificationType, String label){
		Integer result;
		
		if (relationClassificationType==null || label==null || nrValues.get(relationClassificationType)==null){
			result = null;
		} else {
			result = nrValues.get(relationClassificationType).get(label);
		}
		
		return result;
	}




/*	public Partition<String> getSubSequencePartition(String eventTypeName) {
		return subSequencePartitions.get(eventTypeName);
	}*/
	
	public Set<Relation> events(){
		Set<Relation> result;
		
		result = new HashSet<Relation>();
		
		for (EgoRelationSequence sequence : sequences){
			for (Relation event : sequence.getStations().values()){
				result.add(event);
			}
		}
		//
		return result;
		
	}
	
	public int nrEvents(){
		return events().size();
	}
	
	public int nrSequences(){
		return sequences.size();
	}


/*	public Partition<String> getEventPairPartition(String eventTypeName) {
		return eventPairPartitions.get(eventTypeName);
	}*/


	public Map<Value, Double[]> getSimilaritiesMap(RelationClassificationType relationClassificationType) {
		return similaritiesMaps.get(relationClassificationType);
	}

/*	public Partition<Node<Cluster<String>>>[] getDepthPartition(String eventTypeName) {
		return depthPartitions.get(eventTypeName);
	}*/

	public CorrelationMatrix getRelationConnectionMatrix() {
		return relationConnectionMatrix;
	}
	
	public CorrelationMatrix getEventSequenceMatrix(String eventTypeName) {
		CorrelationMatrix result;
		
		if (eventSequenceMatrices == null || eventTypeName == null){
			result = null;
		} else {
			result = eventSequenceMatrices.get(eventTypeName);
		}
		
		return result;
	}
	
	public CorrelationMatrix getSubSequenceMatrix(String eventTypeName) {
		CorrelationMatrix result;
		
		if (subSequenceMatrices == null || eventTypeName == null){
			result = null;
		} else {
			result = subSequenceMatrices.get(eventTypeName);
		}
		
		return result;
	}

	public Map<String, Map<String, Map<String, Value>>> getParcoursNetworkStatistics() {
		return parcoursNetworkStatistics;
	}

	

}
