package org.tip.puck.geo2;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tip.puck.net.Attributes;

import fr.devinsy.util.StringList;
import fr.devinsy.util.StringSet;

/**
 * 
 */
public class Geography2 {

	public enum Status {
		PERFECT,
		COMPLETE,
		WORKABLE,
		UNWORKABLE
	}

	private static final Logger logger = LoggerFactory.getLogger(Geography2.class);

	private String label;
	private Places2 places;
	private IndexOfToponyms indexOfToponyms;
	private Attributes attributes;

	/**
	 * 
	 */
	public Geography2() {
		//
		this.places = new Places2();
		this.indexOfToponyms = new IndexOfToponyms();
		this.attributes = new Attributes();
	}

	/**
	 * 
	 * @param place
	 */
	public Place2 addPlace(final Place2 place) {
		if (place != null) {
			//
			this.places.add(place);

			//
			this.indexOfToponyms.index(place);
		}

		//
		return place;
	}

	public Attributes attributes() {
		return this.attributes;
	}

	/**
	 * 
	 */
	public void clear() {
		this.label = "";
		this.attributes.clear();
		this.places.clear();
		this.indexOfToponyms.clear();
	}

	/**
	 * 
	 * @param toponym
	 * @return
	 */
	public boolean contains(final String toponym) {
		boolean result;

		if (toponym == null) {
			result = false;
		} else {
			if (get(toponym) == null) {
				result = false;
			} else {
				result = true;
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @return
	 */
	public int countOfConflictedPlaces() {
		int result;

		result = getConflictedPlaces().size();

		//
		return result;
	}

	/**
	 * 
	 * @return
	 */
	public int countOfHomonyms() {
		int result;

		result = 0;

		for (Place2 place : this.places) {
			//
			result += place.getHomonyms().size();
		}

		//
		return result;
	}

	/**
	 * 
	 * @return
	 */
	public int countOfPlaces() {
		int result;

		result = this.places.size();

		//
		return result;
	}

	/**
	 * 
	 * @return
	 */
	public int countOfSingleHomonyms() {
		int result;

		result = getHomonyms().size();

		//
		return result;
	}

	/**
	 * 
	 * @return
	 */
	public int countOfToponyms() {
		int result;

		result = this.indexOfToponyms.size();

		//
		return result;
	}

	/**
	 * 
	 * @return
	 */
	public int countOfUngeocodedPlaces() {
		int result;

		result = getUngeocodedPlaces().size();

		//
		return result;
	}

	/**
	 * 
	 * @param toponym
	 * @return
	 */
	public Place2 get(final String toponym) {
		Place2 result;

		if (toponym == null) {
			//
			result = null;

		} else {
			//
			result = this.indexOfToponyms.get(toponym);
		}

		//
		return result;
	}

	/**
	 * This method returns the list of lacking places.
	 * 
	 * A lacking place is a place with missing coordinates.
	 * 
	 * 
	 * @return
	 */
	public Places2 getBlankPlaces() {
		Places2 result;

		result = new Places2();

		for (Place2 place : this.places) {
			//
			if ((StringUtils.isBlank(place.getComment())) && (place.getHomonyms().isEmpty()) && (place.getCoordinate() == null)) {
				//
				result.add(place);
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @return
	 */
	public Places2 getConflictedPlaces() {
		Places2 result;

		Set<Place2> conflictedPlaces = new HashSet<Place2>();

		for (Place2 place : this.places) {
			//
			StringList names = new StringList(1 + place.getHomonyms().size());
			names.add(place.getToponym());
			names.addAll(place.getHomonyms());

			//
			for (String name : names) {
				//
				Place2 reference = this.indexOfToponyms.get(name);
				if ((reference != place) && (reference != null)) {
					//
					conflictedPlaces.add(reference);
					conflictedPlaces.add(place);
				}
			}
		}
		// logger.debug("conflicted1=" + conflictedPlaces.size());

		//
		result = new Places2();
		for (Place2 place : conflictedPlaces) {
			//
			result.add(place);
		}
		// logger.debug("conflicted2=" + result.size());

		//
		return result;
	}

	/**
	 * 
	 * @return
	 */
	public StringList getHomonyms() {
		StringList result;

		StringSet names = new StringSet();

		for (Place2 place : this.places) {
			//
			names.addAll(place.getHomonyms());
		}

		result = names.toStringList();

		//
		return result;
	}

	public String getLabel() {
		return this.label;
	}

	/**
	 * 
	 * @return
	 */
	public StringList getMainToponyms() {
		StringList result;

		StringSet names = new StringSet();

		for (Place2 place : this.places) {
			//
			names.add(place.getToponym());
		}

		result = names.toStringList();

		//
		return result;
	}

	/**
	 * 
	 * @return
	 */
	public Places2 getPlaces() {
		Places2 result;

		//
		result = new Places2(this.places);

		//
		return result;
	}

	/**
	 * 
	 * @return
	 */
	public Status getStatus() {
		Status result;

		if (isLacking()) {
			//
			if (isConflicted()) {
				//
				result = Status.UNWORKABLE;

			} else if (countOfUngeocodedPlaces() == countOfPlaces()) {
				//
				result = Status.UNWORKABLE;

			} else {
				//
				result = Status.WORKABLE;
			}
		} else {
			//
			if (isConflicted()) {
				//
				result = Status.UNWORKABLE;

			} else if (isEmpty()) {
				//
				result = Status.UNWORKABLE;

			} else {
				//
				result = Status.COMPLETE;
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @return
	 */
	public StringList getToponyms() {
		StringList result;

		result = new StringList();

		for (String toponym : this.indexOfToponyms.getToponyms()) {
			//
			result.add(toponym);
		}

		//
		return result;
	}

	/**
	 * This method returns the list of lacking places.
	 * 
	 * A lacking place is a place with missing coordinates.
	 * 
	 * 
	 * @return
	 */
	public Places2 getUngeocodedPlaces() {
		Places2 result;

		result = new Places2();

		for (Place2 place : this.places) {
			//
			if (place.getCoordinate() == null) {
				//
				result.add(place);
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @return
	 */
	public boolean isConflicted() {
		boolean result;

		boolean ended = false;
		StringSet toponyms = new StringSet();
		Iterator<Place2> iterator = this.places.iterator();
		result = true;
		while (!ended) {
			//
			if (iterator.hasNext()) {
				//
				Place2 place = iterator.next();

				//
				if (toponyms.contains(place.getToponym())) {
					//
					ended = true;
					result = true;

				} else {
					//
					boolean ended2 = false;
					Iterator<String> iterator2 = place.getHomonyms().iterator();
					while (!ended2) {
						//
						if (iterator2.hasNext()) {
							//
							String homonym = iterator2.next();

							if (toponyms.contains(homonym)) {
								//
								ended2 = true;
								ended = true;
								result = true;
							}
						} else {
							//
							ended2 = true;

							//
							toponyms.put(place.getToponym());
							toponyms.put(place.getHomonyms());
						}
					}
				}
			} else {
				//
				ended = true;
				result = false;
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @return
	 */
	public boolean isEmpty() {
		boolean result;

		result = this.places.isEmpty();

		//
		return result;
	}

	/**
	 * A geography is lacking when it contains ungeocoded place.
	 * 
	 * @return
	 */
	public boolean isLacking() {
		boolean result;

		result = false;

		boolean ended = false;
		Iterator<Place2> iterator = this.places.iterator();
		while (!ended) {
			//
			if (iterator.hasNext()) {
				//
				Place2 place = iterator.next();

				if (place.getCoordinate() == null) {
					//
					ended = true;
					result = true;
				}

			} else {
				//
				ended = true;
				result = false;
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @return
	 */
	public boolean isNotConflicted() {
		boolean result;

		result = !isConflicted();

		//
		return result;
	}

	/**
	 * 
	 * @return
	 */
	public boolean isNotLacking() {
		boolean result;

		result = !isLacking();

		//
		return result;
	}

	/**
	 * 
	 * @return
	 */
	public boolean isVoid() {
		boolean result;

		result = (this.places.isEmpty() && this.attributes.isEmpty());

		//
		return result;
	}

	/**
	 * 
	 */
	public void rebuildIndexes() {
		//
		this.indexOfToponyms.clear();
		this.indexOfToponyms.index(this.places);
	}

	/**
	 * 
	 * @param place
	 */
	public void removePlace(final Place2 place) {
		this.places.remove(place);
		rebuildIndexes();
	}

	/**
	 * 
	 * @param toponym
	 * @return
	 */
	public Place2 searchByToponym(final String toponym) {
		Place2 result;

		if (toponym == null) {
			//
			result = null;

		} else {
			//
			result = this.indexOfToponyms.get(toponym);
		}

		//
		return result;
	}

	public void setLabel(final String label) {
		this.label = label;
	}
}
