package org.tip.puck.spacetime.workers;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.tip.puck.geo.GeoLevel;
import org.tip.puck.geo.Geography;
import org.tip.puck.net.FiliationType;
import org.tip.puck.net.workers.ExpansionMode;

import fr.devinsy.util.StringList;

/**
 * 
 * @author Klaus Hamberger
 * 
 */
public class SpaceTimeCriteria {

	public enum CensusType {
		GENERAL,
		EGONETWORKS,
		PARCOURSNETWORKS,
		PARCOURSSIMILARITYNETWORKS,
		PARCOURSINTERSECTIONNETWORKS,
		PARCOURS;

		@Override
		public String toString() {
			String result;

			switch (this) {
				case GENERAL:
					result = "General";
				break;
				case EGONETWORKS:
					result = "Ego Networks";
				break;
				case PARCOURSNETWORKS:
					result = "Parcours Networks";
				break;
				case PARCOURSINTERSECTIONNETWORKS:
					result = "Parcours Intersection Networks";
				break;
				case PARCOURSSIMILARITYNETWORKS:
					result = "Parcours Similarity Networks";
				break;
				case PARCOURS:
					result = "Parcours";
				break;
				default:
					result = null;
			}

			//
			return result;
		}
	}

	public enum EgoNetworksOperation {
		GENERAL,
		CENTRALITY,
		COHESION,
		RELATIONS,
		EXPORT_EGO_NETWORKS
	}

	public enum ParcoursNetworksOperation {
		GENERAL,
		CENTRALITY,
		COHESION, // change
		RELATIONS, // change
		EXPORT_PARCOURS,
		EXPORT_EXTENDED_PARCOURS,
		EXPORT_MULTIPLE_PARCOURS,
		EXPORT_PARCOURS_NETWORKS,
		EXPORT_SIMILARY_TREES
	}

	public enum ParcoursIntersectionNetworksOperation {
		GENERAL,
		CENTRALITY,
		COHESION,
		RELATIONS,
		EXPORT_PARCOURS_INTERSECTION_NETWORKS,
		EXPORT_SIMILARY_TREES
	}

	public enum ParcoursSimilarityNetworksOperation {
		GENERAL,
		CENTRALITY,
		COHESION,
		RELATIONS,
		EXPORT_PARCOURS_SIMILARITY_NETWORKS,
		EXPORT_SIMILARY_TREES
	}

	public enum RelationClassificationType {
		TYPEDID,
		HOST,
		MIG,
		HOSTMIG,
		MIGRATIONTYPE,
		CHILDMIGRATIONTYPE,
		TREES,
		DISTANCE,
		PLACE,
		REGION,
		MOVEMENT,
		TURNOVER,
		COMPONENTS,
		DATE,
		AGE
	}

	public enum SequenceGeneralStatistics {
		EVENTS,
		AGE,
		RELATIONS
	}

	public enum SequenceReportType {
		ITINERARIES_SURVEY,
		ITINERARIES_DETAILS,
		BIOGRAPHIES,
		EXTENDED_BIOGRAPHIES,
		ACTOR_EVENT_TABLES,
		INTERACTION_TABLES,
		EXPORT_RELATION_GRAPH
	}

	public enum SliceGeneralStatistics {
		STATICS,
		DYNAMICS
	}

	public enum SliceReportType {
		MEMBERSHIP
	}

	public enum TrajectoriesOperation {
		GENERAL,
		LIST_TREES,
		EXPORT_EVENT_TYPE_NETWORK,
		EXPORT_SEQUENCE_TYPE_NETWORK,
		DRAW
	}

	// General criteria for space time analysis

	private String relationModelName;

	// Attribute labels

	private String dateLabel;
	private String startDateLabel;
	private String endDateLabel;
	private String placeLabel;
	private String startPlaceLabel;
	private String endPlaceLabel;
	private String localUnitLabel;
	private String groupAffiliationLabel;

	// Role names

	private String egoRoleName;
	private String defaultReferentRoleName;
	private StringList roleNames;

	// Time criteria

	private int maxAge;
	private int minAge;
	private Integer[] dates;

	// Geography criteria [temporary]

	private Geography geography;
	private GeoLevel level;
	private StringList minimalPlaceNames;

	// Parameters for sequence expansion

	private ExpansionMode expansionMode;
	private FiliationType filiationType;

	// General census criteria

	private CensusType censusType;

	// Circuit census criteria

	private String pattern;
	private String chainClassification;
	private List<String> relationModelNames;

	// Partition criteria

	private String individualClassificationType;
	private List<RelationClassificationType> relationClassificationTypes;
	private List<RelationClassificationType> mainRelationClassificationTypes; // for
																				// parcours
																				// networks

	private List<RelationClassificationType> trajectoriesRelationClassificationTypes;

	// Census criteria

	private List<String> censusOperationLabels;
	private List<String> networkTitles;
	private Map<String, List<String>> partitionLabels;
	
	// Node statistics criteria
	
	private List<String> nodeStatisticsLabels;

	// private double threshold;

	// Filter for alters in ego and parcours intersection networks

	private String alterFilterRoleName;
	private String alterFilterAttributeLabel;
	private String alterFilterAttributeValue;

	//
	private List<SequenceReportType> sequenceReportTypes;
	private List<SliceReportType> sliceReportTypes;
	private List<SequenceGeneralStatistics> sequenceGeneralStatistics;
	private List<SliceGeneralStatistics> sliceGeneralStatistics;
	private List<EgoNetworksOperation> egoNetworksOperations;
	private List<TrajectoriesOperation> trajectoriesOperations;
	private List<ParcoursNetworksOperation> parcoursNetworkdsOperations;
	private List<ParcoursIntersectionNetworksOperation> parcoursIntersectionNetworksOperations;
	private List<ParcoursSimilarityNetworksOperation> parcoursSimilarityNetworksOperations;

	/**
	 * 
	 */
	public SpaceTimeCriteria() {
		//
		this.sequenceReportTypes = new ArrayList<SequenceReportType>();
		this.sliceReportTypes = new ArrayList<SliceReportType>();
		this.sequenceGeneralStatistics = new ArrayList<SequenceGeneralStatistics>();
		this.sliceGeneralStatistics = new ArrayList<SliceGeneralStatistics>();
		this.egoNetworksOperations = new ArrayList<EgoNetworksOperation>();
		this.trajectoriesRelationClassificationTypes = new ArrayList<RelationClassificationType>();
		this.trajectoriesOperations = new ArrayList<TrajectoriesOperation>();
		this.parcoursNetworkdsOperations = new ArrayList<ParcoursNetworksOperation>();
		this.parcoursIntersectionNetworksOperations = new ArrayList<ParcoursIntersectionNetworksOperation>();
		this.parcoursSimilarityNetworksOperations = new ArrayList<ParcoursSimilarityNetworksOperation>();

		//
		setDefaultCriteria();
	}
	
	public void setCensusParameters(){
		
		// Set Census Operation Labels
		
		this.censusOperationLabels = new ArrayList<String>();

		this.censusOperationLabels.add("NREVENTS");


		if (!getSequenceGeneralStatistics().isEmpty()){
			if (!this.relationClassificationTypes.contains(RelationClassificationType.PLACE)) {
				this.relationClassificationTypes.add(RelationClassificationType.PLACE);
			}
			if (!this.relationClassificationTypes.contains(RelationClassificationType.DISTANCE)) {
				this.relationClassificationTypes.add(RelationClassificationType.DISTANCE);
			}
		}

		if (getSequenceGeneralStatistics().contains(SequenceGeneralStatistics.EVENTS)){

			this.censusOperationLabels.add("NREXTERNALMOVES");
			this.censusOperationLabels.add("MAX_DISTANCE");
			this.censusOperationLabels.add("MEAN_NR_MOVES");
			
			for (RelationClassificationType type : this.relationClassificationTypes) {
				this.censusOperationLabels.add("EVENTS_" + type);
			}

		}
		
		if (getSequenceGeneralStatistics().contains(SequenceGeneralStatistics.AGE)){
			this.censusOperationLabels.add("AGEFIRST");
			this.censusOperationLabels.add("AGEFIRST_CHILDMIGRATIONTYPE_NOPARENTS");
			this.censusOperationLabels.add("AGEFIRST_DISTANCE_TRANSNATIONAL");
		}
		
		if (getSequenceGeneralStatistics().contains(SequenceGeneralStatistics.RELATIONS)){
			this.censusOperationLabels.add("SAMESEXALTERS_ALL");
			this.censusOperationLabels.add("SAMEPLACEALTERS_ALL");

			for (String roleName : this.roleNames) {
				this.censusOperationLabels.add("NRALTERS_" + roleName);
				this.censusOperationLabels.add("NRALTERSPEREVENT_" + roleName);
			}
			
			this.censusOperationLabels.add("MEAN_COVERAGE");
			this.censusOperationLabels.add("MAX_COVERAGE");

			for (String roleName : this.roleNames) {
				this.censusOperationLabels.add("RELATIONS_" + roleName);
			}

			this.censusOperationLabels.add("MAIN_ALTERS");
			this.censusOperationLabels.add("MAIN_RELATIONS");

		}
		
		if (getEgoNetworksOperations().contains(EgoNetworksOperation.GENERAL)){
			this.censusOperationLabels.add("SIZE");
			this.censusOperationLabels.add("TIES");
			this.censusOperationLabels.add("DENSITY");
			this.censusOperationLabels.add("DENSITY_NOLOOPS");
			this.censusOperationLabels.add("MEANDEGREE");
			this.censusOperationLabels.add("MEANDEGREE_NOLOOPS");
			this.censusOperationLabels.add("MEANDEGREE_NOLOOPS_NORM");
		}
		
		if (getEgoNetworksOperations().contains(EgoNetworksOperation.CENTRALITY)){
			this.censusOperationLabels.add("EGO-BETWEENNESS");
			this.censusOperationLabels.add("MEAN_BETWEENNESS");
			this.censusOperationLabels.add("MAX_BETWEENNESS");
			this.censusOperationLabels.add("ECCENTRICITY");
		}
		
		if (getEgoNetworksOperations().contains(EgoNetworksOperation.COHESION)){
			this.censusOperationLabels.add("NRCOMPONENTS");
			this.censusOperationLabels.add("NRISOLATES");
			this.censusOperationLabels.add("MAXCOMPONENT");
			this.censusOperationLabels.add("NRCOMPONENTS_NORM");
			this.censusOperationLabels.add("NRISOLATES_NORM");
			this.censusOperationLabels.add("MAXCOMPONENT_NORM");
			this.censusOperationLabels.add("CONCENTRATION");
			this.censusOperationLabels.add("BROKERAGE");
			this.censusOperationLabels.add("EFFICIENT_SIZE");
			this.censusOperationLabels.add("EFFICIENCY");
		}
		
		if (getEgoNetworksOperations().contains(EgoNetworksOperation.RELATIONS)){
			this.censusOperationLabels.add("NETWORK_RELATIONS");
			this.censusOperationLabels.add("CONNECTED_NETWORK_RELATIONS");

			this.censusOperationLabels.add("CENTRAL_ALTERS");
			this.censusOperationLabels.add("CENTRAL_RELATIONS");

			this.censusOperationLabels.add("SDENSITY_PARENT-CHILD");
			this.censusOperationLabels.add("SDENSITY_SPOUSE");
			this.censusOperationLabels.add("SDENSITY_SIBLING");
			this.censusOperationLabels.add("SDENSITY_RELATIVE");
			this.censusOperationLabels.add("SDENSITY_AFFINE");
//			this.censusOperationLabels.add("SDENSITY_EMPLOYMENT");
//			this.censusOperationLabels.add("SDENSITY_RENT");
			
		}
		
		for (RelationClassificationType mainEventType : this.mainRelationClassificationTypes) {
			
			if (getParcoursNetworksOperations().contains(ParcoursNetworksOperation.GENERAL)){
				this.censusOperationLabels.add("SIZE_" + mainEventType);
				this.censusOperationLabels.add("TIES_" + mainEventType);
				this.censusOperationLabels.add("DENSITY_" + mainEventType);
				this.censusOperationLabels.add("DENSITY_NOLOOPS_" + mainEventType);
				this.censusOperationLabels.add("MEANDEGREE_" + mainEventType);
				this.censusOperationLabels.add("MEANDEGREE_NOLOOPS_" + mainEventType);
				this.censusOperationLabels.add("MEANDEGREE_NOLOOPS_NORM_" + mainEventType);
			}

			if (getParcoursNetworksOperations().contains(ParcoursNetworksOperation.CENTRALITY)){
				this.censusOperationLabels.add("MEAN_BETWEENNESS_" + mainEventType);
				this.censusOperationLabels.add("MAX_BETWEENNESS_" + mainEventType);
			}
			
			if (getParcoursNetworksOperations().contains(ParcoursNetworksOperation.RELATIONS)){
				this.censusOperationLabels.add("CENTRAL_ALTERS_" + mainEventType);
			}

			
		}
		

		
		for (RelationClassificationType mainEventType : this.mainRelationClassificationTypes) {
			
			if (getParcoursIntersectionNetworksOperations().contains(ParcoursIntersectionNetworksOperation.GENERAL)){
				this.censusOperationLabels.add("SIZE_" + mainEventType);
				this.censusOperationLabels.add("TIES_" + mainEventType);
				this.censusOperationLabels.add("DENSITY_" + mainEventType);
				this.censusOperationLabels.add("DENSITY_NOLOOPS_" + mainEventType);
				this.censusOperationLabels.add("MEANDEGREE_" + mainEventType);
				this.censusOperationLabels.add("MEANDEGREE_NOLOOPS_" + mainEventType);
				this.censusOperationLabels.add("MEANDEGREE_NOLOOPS_NORM_" + mainEventType);
			}

			if (getParcoursIntersectionNetworksOperations().contains(ParcoursIntersectionNetworksOperation.CENTRALITY)){
				this.censusOperationLabels.add("EGO-BETWEENNESS_" + mainEventType);
				this.censusOperationLabels.add("MEAN_BETWEENNESS_" + mainEventType);
				this.censusOperationLabels.add("MAX_BETWEENNESS_" + mainEventType);
				this.censusOperationLabels.add("ECCENTRICITY_" + mainEventType);
			}
			
			if (getParcoursIntersectionNetworksOperations().contains(ParcoursIntersectionNetworksOperation.COHESION)){
				this.censusOperationLabels.add("NRCOMPONENTS_" + mainEventType);
				this.censusOperationLabels.add("NRISOLATES_" + mainEventType);
				this.censusOperationLabels.add("MAXCOMPONENT_" + mainEventType);
				this.censusOperationLabels.add("NRCOMPONENTS_NORM_" + mainEventType);
				this.censusOperationLabels.add("NRISOLATES_NORM_" + mainEventType);
				this.censusOperationLabels.add("MAXCOMPONENT_NORM_" + mainEventType);
				this.censusOperationLabels.add("CONCENTRATION_" + mainEventType);
				this.censusOperationLabels.add("BROKERAGE_" + mainEventType);
				this.censusOperationLabels.add("EFFICIENT_SIZE_" + mainEventType);
				this.censusOperationLabels.add("EFFICIENCY_" + mainEventType);
			}
			
			if (getParcoursIntersectionNetworksOperations().contains(ParcoursIntersectionNetworksOperation.GENERAL)){
				this.censusOperationLabels.add("NETWORK_RELATIONS_" + mainEventType);
				this.censusOperationLabels.add("CONNECTED_NETWORK_RELATIONS_" + mainEventType);
				this.censusOperationLabels.add("SIMILARITY_" + mainEventType);
			}
			
		}
		
		if (!getTrajectoriesOperations().isEmpty()){

			if (!this.relationClassificationTypes.contains(RelationClassificationType.PLACE)) {
				this.relationClassificationTypes.add(RelationClassificationType.PLACE);
			}
			if (!this.relationClassificationTypes.contains(RelationClassificationType.DISTANCE)) {
				this.relationClassificationTypes.add(RelationClassificationType.DISTANCE);
			}
			
			if (getTrajectoriesOperations().contains(TrajectoriesOperation.DRAW)){
				this.trajectoriesRelationClassificationTypes.add(RelationClassificationType.PLACE);
			}
			
			this.censusOperationLabels.add("NREVENTS");
			this.censusOperationLabels.add("NREXTERNALMOVES");

			for (RelationClassificationType relationClassificationType : this.trajectoriesRelationClassificationTypes) {
				this.censusOperationLabels.add("PROFILE_" + relationClassificationType);
				this.censusOperationLabels.add("SUPPORT_" + relationClassificationType);
				this.censusOperationLabels.add("NRSTATIONS_" + relationClassificationType);
				if (relationClassificationType == RelationClassificationType.PLACE) {
					this.censusOperationLabels.add("CENTERS_" + relationClassificationType);
					this.censusOperationLabels.add("CENTERSNOSTART_" + relationClassificationType);
					this.censusOperationLabels.add("NRCENTERS_" + relationClassificationType);
					this.censusOperationLabels.add("NRCENTERSNOSTART_" + relationClassificationType);
					this.censusOperationLabels.add("NRINTERNALMOVES_" + relationClassificationType);
					this.censusOperationLabels.add("NRDIRECTRETURNS_" + relationClassificationType);
					this.censusOperationLabels.add("NRCYCLES_" + relationClassificationType);
					this.censusOperationLabels.add("NRDIRECTRETURNS_NORM_" + relationClassificationType);
					this.censusOperationLabels.add("NRCYCLES_NORM_" + relationClassificationType);
				}
			}

		}
		
		// Consolidate main event types with event types
		
		for (RelationClassificationType mainEventType : this.mainRelationClassificationTypes) {
			if (!this.relationClassificationTypes.contains(mainEventType)) {
				this.relationClassificationTypes.add(mainEventType);
			}
		}

		for (RelationClassificationType mainEventType : this.trajectoriesRelationClassificationTypes) {
			if (!this.relationClassificationTypes.contains(mainEventType)) {
				this.relationClassificationTypes.add(mainEventType);
			}
		}



		
		// Set Network Titles
		
		this.networkTitles = new ArrayList<String>();
		
		if (!getEgoNetworksOperations().isEmpty()){
			addNetworkTitle("Ego Network");
			addNetworkTitle("Nonmediated Ego Network");
			this.partitionLabels.get("Ego Network").add("EGO-RELATION");
			this.partitionLabels.get("Ego Network").add("DEGREE");
			this.partitionLabels.get("Nonmediated Ego Network").add("BETWEENNESS");
			this.partitionLabels.get("Nonmediated Ego Network").add("EGO-RELATION");
			this.partitionLabels.get("Nonmediated Ego Network").add("DEGREE");
			this.partitionLabels.get("Nonmediated Ego Network").add("COMPONENT");

		// Temporary Solution:

			addNetworkTitle("Parcours Intersection Network");
			this.partitionLabels.get("Parcours Intersection Network").add("EGO-RELATION");
			this.partitionLabels.get("Parcours Intersection Network").add("NREVENTS");
		
			for (RelationClassificationType mainEventType : this.mainRelationClassificationTypes) {
				addNetworkTitle("Parcours Similarity Network_" + mainEventType);
				this.partitionLabels.get("Parcours Similarity Network_" + mainEventType).add("EGO-RELATION");
				this.partitionLabels.get("Parcours Similarity Network_" + mainEventType).add("NRTRANSITIONS");
			}

		}
		
		if (!getParcoursIntersectionNetworksOperations().isEmpty()){
			addNetworkTitle("Parcours Intersection Network");
			this.partitionLabels.get("Parcours Intersection Network").add("EGO-RELATION");
			this.partitionLabels.get("Parcours Intersection Network").add("NREVENTS");
		}
		
		if (!getParcoursSimilarityNetworksOperations().isEmpty()){
			addNetworkTitle("Parcours Similarity Network");
			this.partitionLabels.get("Parcours Similarity Network").add("EGO-RELATION");
			this.partitionLabels.get("Parcours Similarity Network").add("NREVENTS");
		}
		
		if (!getParcoursNetworksOperations().isEmpty()){
			for (RelationClassificationType mainEventType : this.mainRelationClassificationTypes) {
				addNetworkTitle("Parcours Network_" + mainEventType);
				addNetworkTitle("Parcours Similarity Tree_" + mainEventType);
				addNetworkTitle("Parcours Network Fused_" + mainEventType);
//				this.partitionLabels.get("Parcours Network_" + mainEventType).add("BETWEENNESS");
				this.partitionLabels.get("Parcours Network_" + mainEventType).add("DEGREE");
			}
		}
		
		if (getParcoursNetworksOperations().contains(ParcoursNetworksOperation.EXPORT_PARCOURS)){
			addNetworkTitle("Parcours");
			this.partitionLabels.get("Parcours").add("DATE");
			this.partitionLabels.get("Parcours").add("DISTANCE");
		}
		
		if (getParcoursNetworksOperations().contains(ParcoursNetworksOperation.EXPORT_EXTENDED_PARCOURS)){
			addNetworkTitle("Extended Parcours");
			this.partitionLabels.get("Extended Parcours").add("ORDER");
			this.partitionLabels.get("Extended Parcours").add("TYPE");
		}
		
		if (getParcoursNetworksOperations().contains(ParcoursNetworksOperation.EXPORT_MULTIPLE_PARCOURS)){
			addNetworkTitle("Multiple Parcours");
			this.partitionLabels.get("Multiple Parcours").add("DATE");
			this.partitionLabels.get("Multiple Parcours").add("DISTANCE");
		}
		
		if (getParcoursNetworksOperations().contains(ParcoursNetworksOperation.EXPORT_SIMILARY_TREES)){
			for (RelationClassificationType mainEventType : this.mainRelationClassificationTypes) {
				addNetworkTitle("Parcours Similarity Tree_" + mainEventType);
			}
		}
		
		if (!getTrajectoriesOperations().isEmpty()){
			addNetworkTitle("Event Type Network");
			addNetworkTitle("Sequence Type Network");
		}
		
		// Set node statistics criteria;
		
		if (getParcoursNetworksOperations().contains(ParcoursNetworksOperation.EXPORT_PARCOURS_NETWORKS)){
		
			this.nodeStatisticsLabels = new ArrayList<String>();
			this.nodeStatisticsLabels.add("NUMBER");
			this.nodeStatisticsLabels.add("INDEGREE");
			this.nodeStatisticsLabels.add("OUTDEGREE");
			this.nodeStatisticsLabels.add("INSTRENGTH");
			this.nodeStatisticsLabels.add("OUTSTRENGTH");
			this.nodeStatisticsLabels.add("ORIENTATION");
			this.nodeStatisticsLabels.add("BETWEENNESS");
			this.nodeStatisticsLabels.add("MAXINWEIGHT");
			this.nodeStatisticsLabels.add("MAXOUTWEIGHT");
			this.nodeStatisticsLabels.add("MAXPREDECESSOR");
			this.nodeStatisticsLabels.add("MAXSUCCESSOR");
		}
	}
	

	/**
	 * Temporary Method to set default criteria
	 * 
	 * @param censusType
	 */
	public SpaceTimeCriteria(final CensusType censusType) {

		setDefaultCriteria();

		this.censusType = censusType;

		// Set alter attribute label+value [choose from individual attribute
		// censusOperationLabels]
		if (censusType == CensusType.PARCOURSNETWORKS) {
			this.alterFilterAttributeLabel = "INTERV";
			this.alterFilterAttributeValue = "Yes";
		}

		// Set relation attribute types [Choose among values of enum
		// RelationClassificationType]
		switch (censusType) {
			case GENERAL:
				this.relationClassificationTypes.add(RelationClassificationType.PLACE);
				this.relationClassificationTypes.add(RelationClassificationType.MIGRATIONTYPE);
				this.relationClassificationTypes.add(RelationClassificationType.TREES);
				this.relationClassificationTypes.add(RelationClassificationType.CHILDMIGRATIONTYPE);
				this.relationClassificationTypes.add(RelationClassificationType.DISTANCE);
				this.relationClassificationTypes.add(RelationClassificationType.REGION);
			break;
			case EGONETWORKS:
				this.relationClassificationTypes.add(RelationClassificationType.PLACE);
			break;
			case PARCOURSNETWORKS:
				this.relationClassificationTypes.add(RelationClassificationType.PLACE);
			break;
			case PARCOURS:
				this.relationClassificationTypes.add(RelationClassificationType.PLACE);
			break;
		}

		// Set main relation attribute types types [Choose among values of enum
		// RelationClassificationType]
		switch (censusType) {
			case GENERAL:
			break;
			case EGONETWORKS:
			break;
			case PARCOURSNETWORKS:
				this.mainRelationClassificationTypes.add(RelationClassificationType.MIGRATIONTYPE);
				this.mainRelationClassificationTypes.add(RelationClassificationType.TYPEDID);
			break;
			case PARCOURS:
				this.mainRelationClassificationTypes.add(RelationClassificationType.PLACE);
				this.mainRelationClassificationTypes.add(RelationClassificationType.DISTANCE);
				this.mainRelationClassificationTypes.add(RelationClassificationType.TREES);
				this.mainRelationClassificationTypes.add(RelationClassificationType.MIGRATIONTYPE);
			// mainRelationClassificationTypes.add(EventType.COMPONENTS);
			break;
		}

		// Consolidate main event types with event types
		for (RelationClassificationType mainEventType : this.mainRelationClassificationTypes) {
			if (!this.relationClassificationTypes.contains(mainEventType)) {
				this.relationClassificationTypes.add(mainEventType);
			}
		}

		// Set network titles [choose from list]
		this.networkTitles = new ArrayList<String>();
		this.partitionLabels = new HashMap<String, List<String>>();
		switch (censusType) {
			case GENERAL:
			break;
			case EGONETWORKS:
				addNetworkTitle("Ego Network");
				addNetworkTitle("Nonmediated Ego Network");
			break;
			case PARCOURSNETWORKS:
				addNetworkTitle("Parcours");
				addNetworkTitle("Extended Parcours");
				addNetworkTitle("Multiple Parcours");
				for (RelationClassificationType mainEventType : this.mainRelationClassificationTypes) {
					addNetworkTitle("Parcours Network_" + mainEventType);
					addNetworkTitle("Parcours Similarity Tree_" + mainEventType);
				}
			break;
			case PARCOURS:
				addNetworkTitle("Event Type Network");
				addNetworkTitle("Sequence Type Network");
			break;
		}

		// Set partition Labels
		switch (censusType) {
			case GENERAL:
			break;
			case EGONETWORKS:
				this.partitionLabels.get("Ego Network").add("EGO-RELATION");
				this.partitionLabels.get("Ego Network").add("DEGREE");
				this.partitionLabels.get("Nonmediated Ego Network").add("BETWEENNESS");
				this.partitionLabels.get("Nonmediated Ego Network").add("EGO-RELATION");
				this.partitionLabels.get("Nonmediated Ego Network").add("DEGREE");
				this.partitionLabels.get("Nonmediated Ego Network").add("COMPONENT");
			break;
			case PARCOURSNETWORKS:
				this.partitionLabels.get("Parcours").add("DATE");
				this.partitionLabels.get("Parcours").add("DISTANCE");
				this.partitionLabels.get("Multiple Parcours").add("DATE");
				this.partitionLabels.get("Multiple Parcours").add("DISTANCE");
				this.partitionLabels.get("Extended Parcours").add("ORDER");
				this.partitionLabels.get("Extended Parcours").add("TYPE");
			break;
			case PARCOURSINTERSECTIONNETWORKS:
				this.partitionLabels.get("Parcours Intersection Network").add("EGO-RELATION");
			break;
			case PARCOURSSIMILARITYNETWORKS:
				this.partitionLabels.get("Parcours Intersection Network").add("EGO-RELATION");
			break;
			case PARCOURS:
			break;
		}

		// Set ego network and tree network partition censusOperationLabels
		if (censusType == CensusType.PARCOURSNETWORKS) {
			for (RelationClassificationType mainEventType : this.mainRelationClassificationTypes) {
				List<String> egoNetworkPartitionLabels = this.partitionLabels.get("Parcours Network_" + mainEventType);
				egoNetworkPartitionLabels.add("BETWEENNESS");
				egoNetworkPartitionLabels.add("EGO-RELATION");
				egoNetworkPartitionLabels.add("DEGREE");
				egoNetworkPartitionLabels.add("COMPONENT");

				List<String> treePartitionLabels = this.partitionLabels.get("Parcours Similarity Tree_" + mainEventType);
				treePartitionLabels.add("SIZE");
				treePartitionLabels.add("STEP");
				treePartitionLabels.add("EGO-RELATION");
			}
		}

		// Set censusOperationLabels of census operations
		this.censusOperationLabels = new ArrayList<String>();
		switch (censusType) {
			case GENERAL:

				this.censusOperationLabels.add("NREVENTS");
				this.censusOperationLabels.add("NREXTERNALMOVES");

				this.censusOperationLabels.add("AGEFIRST");
				this.censusOperationLabels.add("AGEFIRST_CHILDMIGRATIONTYPE_NOPARENTS");
				this.censusOperationLabels.add("AGEFIRST_DISTANCE_TRANSNATIONAL");
				this.censusOperationLabels.add("MAX_DISTANCE");
				this.censusOperationLabels.add("MEAN_NR_MOVES");
				this.censusOperationLabels.add("SAMESEXALTERS_ALL");
				this.censusOperationLabels.add("SAMEPLACEALTERS_ALL");

				for (String roleName : this.roleNames) {
					this.censusOperationLabels.add("NRALTERS_" + roleName);
					this.censusOperationLabels.add("NRALTERSPEREVENT_" + roleName);
				}

				for (RelationClassificationType type : this.relationClassificationTypes) {
					this.censusOperationLabels.add("EVENTS_" + type);
				}

				this.censusOperationLabels.add("MEAN_COVERAGE");
				this.censusOperationLabels.add("MAX_COVERAGE");

				for (String roleName : this.roleNames) {
					this.censusOperationLabels.add("RELATIONS_" + roleName);
				}

				this.censusOperationLabels.add("MAIN_ALTERS");
				this.censusOperationLabels.add("MAIN_RELATIONS");

			break;

			case EGONETWORKS:

				this.censusOperationLabels.add("NREVENTS");
				this.censusOperationLabels.add("NREXTERNALMOVES");

				this.censusOperationLabels.add("SIZE");
				this.censusOperationLabels.add("TIES");
				this.censusOperationLabels.add("EGO-BETWEENNESS");
				this.censusOperationLabels.add("MEAN_BETWEENNESS");
				this.censusOperationLabels.add("MAX_BETWEENNESS");
				this.censusOperationLabels.add("ECCENTRICITY");
				this.censusOperationLabels.add("DENSITY");
				this.censusOperationLabels.add("DENSITY_NOLOOPS");
				this.censusOperationLabels.add("MEANDEGREE");
				this.censusOperationLabels.add("MEANDEGREE_NOLOOPS");
				this.censusOperationLabels.add("MEANDEGREE_NOLOOPS_NORM");
				this.censusOperationLabels.add("NRCOMPONENTS");
				this.censusOperationLabels.add("NRISOLATES");
				this.censusOperationLabels.add("MAXCOMPONENT");
				this.censusOperationLabels.add("NRCOMPONENTS_NORM");
				this.censusOperationLabels.add("NRISOLATES_NORM");
				this.censusOperationLabels.add("MAXCOMPONENT_NORM");
				this.censusOperationLabels.add("CONCENTRATION");
				this.censusOperationLabels.add("BROKERAGE");
				this.censusOperationLabels.add("EFFICIENT_SIZE");
				this.censusOperationLabels.add("EFFICIENCY");
				this.censusOperationLabels.add("NETWORK_RELATIONS");
				this.censusOperationLabels.add("CONNECTED_NETWORK_RELATIONS");

				this.censusOperationLabels.add("CENTRAL_ALTERS");
				this.censusOperationLabels.add("CENTRAL_RELATIONS");

				this.censusOperationLabels.add("SDENSITY_PARENT-CHILD");
				this.censusOperationLabels.add("SDENSITY_SPOUSE");
				this.censusOperationLabels.add("SDENSITY_SIBLING");
				this.censusOperationLabels.add("SDENSITY_RELATIVE");
				this.censusOperationLabels.add("SDENSITY_AFFINE");
				this.censusOperationLabels.add("SDENSITY_EMPLOYMENT");
				this.censusOperationLabels.add("SDENSITY_RENT");

			break;

			case PARCOURSNETWORKS:

				this.censusOperationLabels.add("NREVENTS");
				this.censusOperationLabels.add("NREXTERNALMOVES");

				for (RelationClassificationType mainEventType : this.mainRelationClassificationTypes) {
					this.censusOperationLabels.add("SIZE_" + mainEventType);
					this.censusOperationLabels.add("TIES_" + mainEventType);
					this.censusOperationLabels.add("EGO-BETWEENNESS_" + mainEventType);
					this.censusOperationLabels.add("MEAN_BETWEENNESS_" + mainEventType);
					this.censusOperationLabels.add("MAX_BETWEENNESS_" + mainEventType);
					this.censusOperationLabels.add("ECCENTRICITY_" + mainEventType);
					this.censusOperationLabels.add("DENSITY_" + mainEventType);
					this.censusOperationLabels.add("DENSITY_NOLOOPS_" + mainEventType);
					this.censusOperationLabels.add("MEANDEGREE_" + mainEventType);
					this.censusOperationLabels.add("MEANDEGREE_NOLOOPS_" + mainEventType);
					this.censusOperationLabels.add("MEANDEGREE_NOLOOPS_NORM_" + mainEventType);
					this.censusOperationLabels.add("NRCOMPONENTS_" + mainEventType);
					this.censusOperationLabels.add("NRISOLATES_" + mainEventType);
					this.censusOperationLabels.add("MAXCOMPONENT_" + mainEventType);
					this.censusOperationLabels.add("NRCOMPONENTS_NORM_" + mainEventType);
					this.censusOperationLabels.add("NRISOLATES_NORM_" + mainEventType);
					this.censusOperationLabels.add("MAXCOMPONENT_NORM_" + mainEventType);
					this.censusOperationLabels.add("CONCENTRATION_" + mainEventType);
					this.censusOperationLabels.add("BROKERAGE_" + mainEventType);
					this.censusOperationLabels.add("EFFICIENT_SIZE_" + mainEventType);
					this.censusOperationLabels.add("EFFICIENCY_" + mainEventType);
					this.censusOperationLabels.add("NETWORK_RELATIONS_" + mainEventType);
					this.censusOperationLabels.add("CONNECTED_NETWORK_RELATIONS_" + mainEventType);
					this.censusOperationLabels.add("SIMILARITY_" + mainEventType);
				}

			break;

			case PARCOURS:
				this.censusOperationLabels.add("NREVENTS");
				this.censusOperationLabels.add("NREXTERNALMOVES");

				for (RelationClassificationType relationClassificationType : this.mainRelationClassificationTypes) {
					this.censusOperationLabels.add("PROFILE_" + relationClassificationType);
					this.censusOperationLabels.add("SUPPORT_" + relationClassificationType);
					if (relationClassificationType == RelationClassificationType.PLACE) {
						this.censusOperationLabels.add("CENTERS_" + relationClassificationType);
						this.censusOperationLabels.add("NRINTERNALMOVES_" + relationClassificationType);
						this.censusOperationLabels.add("NRDIRECTRETURNS_" + relationClassificationType);
						this.censusOperationLabels.add("NRCYCLES_" + relationClassificationType);
						this.censusOperationLabels.add("NRDIRECTRETURNS_NORM_" + relationClassificationType);
						this.censusOperationLabels.add("NRCYCLES_NORM_" + relationClassificationType);
					}
				}

			break;
		}

	}

	public void addNetworkTitle(final String networkTitle) {

		this.networkTitles.add(networkTitle);
		this.partitionLabels.put(networkTitle, new ArrayList<String>());

	}

	public String getAlterFilterAttributeLabel() {
		return this.alterFilterAttributeLabel;
	}

	public String getAlterFilterAttributeValue() {
		return this.alterFilterAttributeValue;
	}

	public String getAlterFilterRoleName() {
		return this.alterFilterRoleName;
	}

	public List<String> getCensusOperationLabels() {
		return this.censusOperationLabels;
	}

	public CensusType getCensusType() {
		return this.censusType;
	}

	public String getChainClassification() {
		return this.chainClassification;
	}

	public String getDateLabel() {
		return this.dateLabel;
	}

	public Integer[] getDates() {
		return this.dates;
	}

	public String getDefaultReferentRoleName() {
		return this.defaultReferentRoleName;
	}

	public List<EgoNetworksOperation> getEgoNetworksOperations() {
		return this.egoNetworksOperations;
	}

	public String getEgoRoleName() {
		return this.egoRoleName;
	}

	public String getEndDateLabel() {
		return this.endDateLabel;
	}

	public String getEndPlaceLabel() {
		return this.endPlaceLabel;
	}

	public List<RelationClassificationType> getEventTypes() {
		return this.relationClassificationTypes;
	}

	public ExpansionMode getExpansionMode() {
		return this.expansionMode;
	}

	public FiliationType getFiliationType() {
		return this.filiationType;
	}

	public Geography getGeography() {
		return this.geography;
	}

	public GeoLevel getLevel() {
		return this.level;
	}

	public String getLocalUnitLabel() {
		return this.localUnitLabel;
	}

	public List<RelationClassificationType> getMainRelationClassificationTypes() {
		return this.mainRelationClassificationTypes;
	}

	public int getMaxAge() {
		return this.maxAge;
	}

	public int getMinAge() {
		return this.minAge;
	}

	public StringList getMinimalPlaceNames() {
		return this.minimalPlaceNames;
	}

	public List<String> getNetworkTitles() {
		return this.networkTitles;
	}

	public List<ParcoursNetworksOperation> getParcoursNetworksOperations() {
		return this.parcoursNetworkdsOperations;
	}

	public List<ParcoursIntersectionNetworksOperation> getParcoursIntersectionNetworksOperations() {
		return this.parcoursIntersectionNetworksOperations;
	}

	public List<ParcoursSimilarityNetworksOperation> getParcoursSimilarityNetworksOperations() {
		return this.parcoursSimilarityNetworksOperations;
	}

	public String getPartitionLabel() {
		return this.individualClassificationType;
	}

	public Map<String, List<String>> getPartitionLabels() {
		return this.partitionLabels;
	}

	public String getPattern() {
		return this.pattern;
	}

	public String getPlaceLabel() {
		return this.placeLabel;
	}

	public List<RelationClassificationType> getRelationClassificationTypes() {
		return this.relationClassificationTypes;
	}

	public String getRelationModelName() {
		return this.relationModelName;
	}

	public List<String> getRelationModelNames() {
		return this.relationModelNames;
	}

	public StringList getRoleNames() {
		return this.roleNames;
	}

	public List<SequenceGeneralStatistics> getSequenceGeneralStatistics() {
		return this.sequenceGeneralStatistics;
	}

	public List<SequenceReportType> getSequenceReportTypes() {
		return this.sequenceReportTypes;
	}

	public List<SliceGeneralStatistics> getSliceGeneralStatistics() {
		return this.sliceGeneralStatistics;
	}

	public List<SliceReportType> getSliceReportTypes() {
		return this.sliceReportTypes;
	}

	public String getStartDateLabel() {
		return this.startDateLabel;
	}

	public String getStartPlaceLabel() {
		return this.startPlaceLabel;
	}

	public List<TrajectoriesOperation> getTrajectoriesOperations() {
		return this.trajectoriesOperations;
	}

	public List<RelationClassificationType> getTrajectoriesRelationClassificationTypes() {
		return this.trajectoriesRelationClassificationTypes;
	}

	public List<RelationClassificationType> getTypes() {
		return this.relationClassificationTypes;
	}

	/**
	 * 
	 * @param criteria
	 * @return
	 */
	public boolean isSequenceAnalysis() {
		boolean result;

		if ((this.dates.length == 0) && (StringUtils.isBlank(this.localUnitLabel))) {
			//
			result = true;
		} else {
			//
			result = false;
		}

		//
		return result;
	}

	public void setAlterAttributeLabel(final String alterAttributeLabel) {
		this.alterFilterAttributeLabel = alterAttributeLabel;
	}

	public void setAlterAttributeValue(final String alterAttributeValue) {
		this.alterFilterAttributeValue = alterAttributeValue;
	}

	public void setAlterFilterRoleName(final String alterFilterRoleName) {
		this.alterFilterRoleName = alterFilterRoleName;
	}

/*	public void setCensusOperationLabels() {

		this.censusOperationLabels = new ArrayList<String>();

		if (!getSequenceGeneralStatistics().isEmpty()) {
			if (!this.relationClassificationTypes.contains(RelationClassificationType.PLACE)) {
				this.relationClassificationTypes.add(RelationClassificationType.PLACE);
			}
			if (!this.relationClassificationTypes.contains(RelationClassificationType.DISTANCE)) {
				this.relationClassificationTypes.add(RelationClassificationType.DISTANCE);
			}
		}

		if (getSequenceGeneralStatistics().contains(SequenceGeneralStatistics.EVENTS)) {

			this.censusOperationLabels.add("NREVENTS");
			this.censusOperationLabels.add("NREXTERNALMOVES");
			this.censusOperationLabels.add("MAX_DISTANCE");
			this.censusOperationLabels.add("MEAN_NR_MOVES");

			for (RelationClassificationType type : this.relationClassificationTypes) {
				this.censusOperationLabels.add("EVENTS_" + type);
			}

		}

		if (getSequenceGeneralStatistics().contains(SequenceGeneralStatistics.AGE)) {
			this.censusOperationLabels.add("AGEFIRST");
			this.censusOperationLabels.add("AGEFIRST_CHILDMIGRATIONTYPE_NOPARENTS");
			this.censusOperationLabels.add("AGEFIRST_DISTANCE_TRANSNATIONAL");
		}

		if (getSequenceGeneralStatistics().contains(SequenceGeneralStatistics.RELATIONS)) {
			this.censusOperationLabels.add("SAMESEXALTERS_ALL");
			this.censusOperationLabels.add("SAMEPLACEALTERS_ALL");

			for (String roleName : this.roleNames) {
				this.censusOperationLabels.add("NRALTERS_" + roleName);
				this.censusOperationLabels.add("NRALTERSPEREVENT_" + roleName);
			}

			this.censusOperationLabels.add("MEAN_COVERAGE");
			this.censusOperationLabels.add("MAX_COVERAGE");

			for (String roleName : this.roleNames) {
				this.censusOperationLabels.add("RELATIONS_" + roleName);
			}

			this.censusOperationLabels.add("MAIN_ALTERS");
			this.censusOperationLabels.add("MAIN_RELATIONS");

		}

				switch (censusType) {

					case EGONETWORKS:

						this.censusOperationLabels.add("NREVENTS");
						this.censusOperationLabels.add("NREXTERNALMOVES");

						this.censusOperationLabels.add("SIZE");
						this.censusOperationLabels.add("TIES");
						this.censusOperationLabels.add("EGO-BETWEENNESS");
						this.censusOperationLabels.add("MEAN_BETWEENNESS");
						this.censusOperationLabels.add("MAX_BETWEENNESS");
						this.censusOperationLabels.add("ECCENTRICITY");
						this.censusOperationLabels.add("DENSITY");
						this.censusOperationLabels.add("DENSITY_NOLOOPS");
						this.censusOperationLabels.add("MEANDEGREE");
						this.censusOperationLabels.add("MEANDEGREE_NOLOOPS");
						this.censusOperationLabels.add("MEANDEGREE_NOLOOPS_NORM");
						this.censusOperationLabels.add("NRCOMPONENTS");
						this.censusOperationLabels.add("NRISOLATES");
						this.censusOperationLabels.add("MAXCOMPONENT");
						this.censusOperationLabels.add("NRCOMPONENTS_NORM");
						this.censusOperationLabels.add("NRISOLATES_NORM");
						this.censusOperationLabels.add("MAXCOMPONENT_NORM");
						this.censusOperationLabels.add("CONCENTRATION");
						this.censusOperationLabels.add("BROKERAGE");
						this.censusOperationLabels.add("EFFICIENT_SIZE");
						this.censusOperationLabels.add("EFFICIENCY");
						this.censusOperationLabels.add("NETWORK_RELATIONS");
						this.censusOperationLabels.add("CONNECTED_NETWORK_RELATIONS");

						this.censusOperationLabels.add("CENTRAL_ALTERS");
						this.censusOperationLabels.add("CENTRAL_RELATIONS");

						this.censusOperationLabels.add("SDENSITY_PARENT-CHILD");
						this.censusOperationLabels.add("SDENSITY_SPOUSE");
						this.censusOperationLabels.add("SDENSITY_SIBLING");
						this.censusOperationLabels.add("SDENSITY_RELATIVE");
						this.censusOperationLabels.add("SDENSITY_AFFINE");
						this.censusOperationLabels.add("SDENSITY_EMPLOYMENT");
						this.censusOperationLabels.add("SDENSITY_RENT");

					break;

					case PARCOURSNETWORKS:

						this.censusOperationLabels.add("NREVENTS");
						this.censusOperationLabels.add("NREXTERNALMOVES");

						for (RelationClassificationType mainEventType : this.mainRelationClassificationTypes) {
							this.censusOperationLabels.add("SIZE_" + mainEventType);
							this.censusOperationLabels.add("TIES_" + mainEventType);
							this.censusOperationLabels.add("EGO-BETWEENNESS_" + mainEventType);
							this.censusOperationLabels.add("MEAN_BETWEENNESS_" + mainEventType);
							this.censusOperationLabels.add("MAX_BETWEENNESS_" + mainEventType);
							this.censusOperationLabels.add("ECCENTRICITY_" + mainEventType);
							this.censusOperationLabels.add("DENSITY_" + mainEventType);
							this.censusOperationLabels.add("DENSITY_NOLOOPS_" + mainEventType);
							this.censusOperationLabels.add("MEANDEGREE_" + mainEventType);
							this.censusOperationLabels.add("MEANDEGREE_NOLOOPS_" + mainEventType);
							this.censusOperationLabels.add("MEANDEGREE_NOLOOPS_NORM_" + mainEventType);
							this.censusOperationLabels.add("NRCOMPONENTS_" + mainEventType);
							this.censusOperationLabels.add("NRISOLATES_" + mainEventType);
							this.censusOperationLabels.add("MAXCOMPONENT_" + mainEventType);
							this.censusOperationLabels.add("NRCOMPONENTS_NORM_" + mainEventType);
							this.censusOperationLabels.add("NRISOLATES_NORM_" + mainEventType);
							this.censusOperationLabels.add("MAXCOMPONENT_NORM_" + mainEventType);
							this.censusOperationLabels.add("CONCENTRATION_" + mainEventType);
							this.censusOperationLabels.add("BROKERAGE_" + mainEventType);
							this.censusOperationLabels.add("EFFICIENT_SIZE_" + mainEventType);
							this.censusOperationLabels.add("EFFICIENCY_" + mainEventType);
							this.censusOperationLabels.add("NETWORK_RELATIONS_" + mainEventType);
							this.censusOperationLabels.add("CONNECTED_NETWORK_RELATIONS_" + mainEventType);
							this.censusOperationLabels.add("SIMILARITY_" + mainEventType);
						}

					break;

					case PARCOURS:
						this.censusOperationLabels.add("NREVENTS");
						this.censusOperationLabels.add("NREXTERNALMOVES");

						for (RelationClassificationType relationClassificationType : this.mainRelationClassificationTypes) {
							this.censusOperationLabels.add("PROFILE_" + relationClassificationType);
							this.censusOperationLabels.add("SUPPORT_" + relationClassificationType);
							if (relationClassificationType == RelationClassificationType.PLACE) {
								this.censusOperationLabels.add("CENTERS_" + relationClassificationType);
								this.censusOperationLabels.add("NRINTERNALMOVES_" + relationClassificationType);
								this.censusOperationLabels.add("NRDIRECTRETURNS_" + relationClassificationType);
								this.censusOperationLabels.add("NRCYCLES_" + relationClassificationType);
								this.censusOperationLabels.add("NRDIRECTRETURNS_NORM_" + relationClassificationType);
								this.censusOperationLabels.add("NRCYCLES_NORM_" + relationClassificationType);
							}
						}

					break;
				}

	}*/

	public void setCensusType(final CensusType censusType) {
		this.censusType = censusType;
	}

	public void setChainClassification(final String chainClassification) {
		this.chainClassification = chainClassification;
	}

	public void setDateLabel(final String dateLabel) {
		this.dateLabel = dateLabel;
	}

	public void setDates(final Integer[] dates) {
		this.dates = dates;
	}

	private void setDefaultCriteria() {

		// Set labels of attribute containing the date [Choose among relation
		// attribute labels]
		this.dateLabel = "DATE";
		this.startDateLabel = "START_DATE";
		this.endDateLabel = "END_DATE";

		// Set labels of attribute concerning the place
		this.placeLabel = "PLACE";
		this.startPlaceLabel = "START_PLACE";
		this.endPlaceLabel = "END_PLACE";

		// Set alter role name [Choose among event role names]
		this.alterFilterRoleName = "ALL";

		// Set event role names [Choose among roles of relationModel that has
		// been chosen as eventModel; in addition: "ALL"]
		this.roleNames = new StringList();
		this.roleNames.add("ALL");
		// *
//		this.roleNames.add("HOST");
//		this.roleNames.add("MIG");

		// Set default roleName of referent [Choose among role names]
		this.defaultReferentRoleName = "HOST";

		// Set age thresholds [Spinner integers with interval 1; Default: min =
		// 0, max = 1000]
		this.minAge = 0;
		this.maxAge = 1000;

		// Set relation model names [Choose among remaining relation model
		// names]
		this.relationModelNames = new ArrayList<String>();
//		this.relationModelNames.add("Apprenticeship");
//		this.relationModelNames.add("Housing");
//		this.relationModelNames.add("Employment");

		// Set relation attribute types [Choose among values of enum
		// RelationClassificationType]
		this.relationClassificationTypes = new ArrayList<RelationClassificationType>();

		// Set main relation attribute types types [Choose among values of enum
		// RelationClassificationType]
		this.mainRelationClassificationTypes = new ArrayList<RelationClassificationType>();

		// Consolidate main event types with event types
		for (RelationClassificationType mainEventType : this.mainRelationClassificationTypes) {
			if (!this.relationClassificationTypes.contains(mainEventType)) {
				this.relationClassificationTypes.add(mainEventType);
			}
		}

		// Set geography criteria [consolidate with new module!]
		this.geography = Geography.getInstance();
		this.level = GeoLevel.TOWN;
		// Set names of minimal places to be considered [write as string
		// separated by comma?]
		if (this.censusType == CensusType.GENERAL) {
			this.minimalPlaceNames = new StringList(Arrays.asList(new String[] { "Afagnan", "Lom�", "Bas-Mono", "Togo", "B�nin", "Ghana", "Nig�ria",
					"C�te d'Ivoire", "Afrique" }));
		} else {
			this.minimalPlaceNames = new StringList();
		}

		// Set chain census criteria
		this.pattern = "4 1";
		this.chainClassification = "LINE";
		// threshold = 0.01;

		// Set statistics criteria
		this.individualClassificationType = "GENDER";

		// Set network titles [choose from list]
		this.networkTitles = new ArrayList<String>();
		this.partitionLabels = new HashMap<String, List<String>>();
		for (String networkTitle : this.networkTitles) {
			this.partitionLabels.put(networkTitle, new ArrayList<String>());
		}
		
		// Set network partition Labels [choose from list]

		// Set censusOperationLabels of census operations
		this.censusOperationLabels = new ArrayList<String>();

		// Set expansion mode and filiation type for extended sequences [choose
		// from values of enums ExpansionMode and FiliationType]
		this.expansionMode = ExpansionMode.RELATED;
		this.filiationType = FiliationType.COGNATIC;
	}

	public void setDefaultReferentRoleName(final String defaultReferentRoleName) {
		this.defaultReferentRoleName = defaultReferentRoleName;
	}

	public void setEgoRoleName(final String egoRoleName) {
		this.egoRoleName = egoRoleName;
	}

	public void setEndDateLabel(final String endDateLabel) {
		this.endDateLabel = endDateLabel;
	}

	public void setEndPlaceLabel(final String endPlaceLabel) {
		this.endPlaceLabel = endPlaceLabel;
	}

	public void setEventTypes(final List<RelationClassificationType> relationClassificationTypes) {
		this.relationClassificationTypes = relationClassificationTypes;
	}

	public void setExpansionMode(final ExpansionMode expansionMode) {
		this.expansionMode = expansionMode;
	}

	public void setFiliationType(final FiliationType filiationType) {
		this.filiationType = filiationType;
	}

	public void setGeography(final Geography geography) {
		this.geography = geography;
	}

	public void setLabels(final List<String> labels) {
		this.censusOperationLabels = labels;
	}

	public void setLevel(final GeoLevel level) {
		this.level = level;
	}

	public void setLocalUnitLabel(final String constantAttributeFilterLabel) {
		this.localUnitLabel = constantAttributeFilterLabel;
	}

	public void setMainEventType(final List<RelationClassificationType> mainEventTypes) {
		this.mainRelationClassificationTypes = mainEventTypes;
	}

	public void setMaxAge(final int maxAge) {
		this.maxAge = maxAge;
	}

	public void setMinAge(final int minAge) {
		this.minAge = minAge;
	}

	public void setPartitionLabel(final String partitionLabel) {
		this.individualClassificationType = partitionLabel;
	}

	public void setPattern(final String pattern) {
		this.pattern = pattern;
	}

	public void setPlaceLabel(final String placeLabel) {
		this.placeLabel = placeLabel;
	}

	public void setRelationClassificationTypes(final List<RelationClassificationType> relationClassificationTypes) {
		this.relationClassificationTypes = relationClassificationTypes;
	}

	public void setRelationModelName(final String relationModelName) {
		this.relationModelName = relationModelName;
	}

	public void setRelationModelNames(final List<String> relationModelNames) {
		this.relationModelNames = relationModelNames;
	}

	public void setRoleNames(final List<String> alterRoleNames) {
		this.roleNames = new StringList(alterRoleNames);
	}

	public void setStartDateLabel(final String startDateLabel) {
		this.startDateLabel = startDateLabel;
	}

	public void setStartPlaceLabel(final String startPlaceLabel) {
		this.startPlaceLabel = startPlaceLabel;
	}

	public void setTypes(final List<RelationClassificationType> types) {
		this.relationClassificationTypes = types;
	}

	public List<String> getNodeStatisticsLabels() {
		return nodeStatisticsLabels;
	}

	public String getGroupAffiliationLabel() {
		return groupAffiliationLabel;
	}

	public void setGroupAffiliationLabel(String groupAffiliationLabel) {
		this.groupAffiliationLabel = groupAffiliationLabel;
	}

	
	
}
