package org.tip.puck.spacetime.workers;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.Set;
import java.util.Stack;
import java.util.TreeMap;

import org.apache.commons.lang3.StringUtils;
import org.tip.puck.PuckException;
import org.tip.puck.geo.GeoLevel;
import org.tip.puck.geo.Geography;
import org.tip.puck.geo.Place;
import org.tip.puck.geo.graphs.GeoNetworkUtils;
import org.tip.puck.geo2.Geography2;
import org.tip.puck.geo2.Place2;
import org.tip.puck.geo2.workers.GeocodingWorker;
import org.tip.puck.graphs.Graph;
import org.tip.puck.graphs.GraphProfile;
import org.tip.puck.graphs.Node;
import org.tip.puck.io.paj.PAJFile;
import org.tip.puck.matrix.MatrixStatistics.Indicator;
import org.tip.puck.matrix.MatrixStatistics.Mode;
import org.tip.puck.net.Attribute;
import org.tip.puck.net.Gender;
import org.tip.puck.net.Individual;
import org.tip.puck.net.Individuals;
import org.tip.puck.net.Net;
import org.tip.puck.net.relations.Relation;
import org.tip.puck.net.relations.Relations;
import org.tip.puck.net.workers.NetUtils;
import org.tip.puck.partitions.Cluster;
import org.tip.puck.partitions.Intervals;
import org.tip.puck.partitions.Partition;
import org.tip.puck.partitions.PartitionCriteria;
import org.tip.puck.partitions.PartitionCriteria.PartitionType;
import org.tip.puck.partitions.PartitionCriteria.SizeFilter;
import org.tip.puck.partitions.PartitionCriteria.ValueFilter;
import org.tip.puck.partitions.PartitionMaker;
import org.tip.puck.report.Report;
import org.tip.puck.report.ReportChart;
import org.tip.puck.report.ReportChart.GraphType;
import org.tip.puck.report.ReportRawData;
import org.tip.puck.report.ReportTable;
import org.tip.puck.segmentation.Segmentation;
import org.tip.puck.spacetime.CorrelationMatrix;
import org.tip.puck.spacetime.EventTriangle;
import org.tip.puck.spacetime.Ordinal;
import org.tip.puck.spacetime.Sequence;
import org.tip.puck.spacetime.Sequences;
import org.tip.puck.spacetime.workers.SpaceTimeCriteria.CensusType;
import org.tip.puck.spacetime.workers.SpaceTimeCriteria.EgoNetworksOperation;
import org.tip.puck.spacetime.workers.SpaceTimeCriteria.ParcoursNetworksOperation;
import org.tip.puck.spacetime.workers.SpaceTimeCriteria.RelationClassificationType;
import org.tip.puck.spacetime.workers.SpaceTimeCriteria.SequenceReportType;
import org.tip.puck.spacetime.workers.SpaceTimeCriteria.TrajectoriesOperation;
import org.tip.puck.statistics.StatisticsReporter;
import org.tip.puck.util.Chronometer;
import org.tip.puck.util.MathUtils;
import org.tip.puck.util.NumberedValues;
import org.tip.puck.util.PuckUtils;
import org.tip.puck.util.ToolBox;
import org.tip.puck.util.Value;

import fr.devinsy.util.StringList;

/**
 * 
 * @author Klaus Hamberger
 * 
 */
public class SequenceReporter {

	private static StringList getStories(final Relation event) {
		StringList result;

		result = new StringList();

		for (Attribute attribute : event.attributes()) {
			String story = "";
			if (attribute.getLabel().contains("NOTE")) {
				String[] label = attribute.getLabel().split("_");
				if (label.length > 1 && StringUtils.isNumeric(label[1])) {
					int id = Integer.parseInt(label[1]);
					Individual indi = event.getIndividuals().getById(id);
					story += indi.signature() + ": ";
				}
				story += attribute.getValue();
				result.appendln(story);
			}
		}

		//
		return result;
	}
	

	/**
	 * 
	 * @param segmentation
	 * @param criteria
	 * @param bundle
	 * @return
	 */
	public static Report reportDiscontinuousItineraries(final Segmentation segmentation, final SpaceTimeCriteria criteria, final ResourceBundle bundle) {
		Report result;

		//
		Chronometer chrono = new Chronometer();

		int errorCount = 0;
		StringList errors = new StringList();

		//
		result = new Report();
		result.setTitle("Discontinuous Itineraries");
		result.setOrigin("Control reporter");
		result.setTarget(segmentation.getLabel());

		for (Sequence itinerary : SequenceMaker.createPersonalSequences(segmentation, criteria).toSortedList()) {
			Sequences partials = SequenceWorker.split(itinerary);
			if (partials.size() > 1) {
				errorCount++;
				errors.appendln(itinerary.getEgo().signature());
				int j = 0;
				for (Sequence partial : partials) {
					if (partial.getEvents().size() != 0 && j > 0) {
						errors.appendln(partial.getFirst() + "\t" + partial.getEvents().get(partial.getFirst()));
					}
					j++;
				}
				errors.appendln();
			}
		}

		//
		errors.add(0, errorCount + " " + Report.translate(bundle, "Discontinuous Itineraries") + "\n");
		result.outputs().append(errors.toString());

		//
		result.setStatus(errorCount);

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;
	}
	
	/**
	 * 
	 * @param segmentation
	 * @param criteria
	 * @param bundle
	 * @return
	 */
	public static Report reportUnknownRelations(final Segmentation segmentation, final SpaceTimeCriteria criteria, final ResourceBundle bundle) {
		Report result;

		//
		Chronometer chrono = new Chronometer();

		int errorCount = 0;
		StringList errors = new StringList();

		//
		result = new Report();
		result.setTitle("Unknown Relations");
		result.setOrigin("Control reporter");
		result.setTarget(segmentation.getLabel());
		
		Map<Individual,Individuals> unknownRelations = new TreeMap<Individual,Individuals>();
				
		for (Sequence itinerary : SequenceMaker.createPersonalSequences(segmentation, criteria).toSortedList()) {
			
			SequenceCensus census = new SequenceCensus(itinerary, criteria);
			
/*			Set<Individual> alters = new HashSet<Individual>();
			for (Relation event : itinerary.getEvents().values()){
				for (Individual alter : event.getIndividuals()){
					alters.add(alter);
				}
			}
			Map<Individual,List<String>> relationsByAlter = NetUtils.getAlterRelations1(itinerary.getEgo(),alters,ToolBox.stringsToInts(criteria.getPattern()),criteria.getRelationModelNames(), criteria.getChainClassification(),null);*/
			
			
			for (Individual alter : census.getAllAlters()){
				if (census.getRelationsByAlter().get(alter).size()==0){
					Individuals unknowns = unknownRelations.get(itinerary.getEgo());
					if (unknowns==null){
						unknowns = new Individuals();
						unknownRelations.put(itinerary.getEgo(), unknowns);
					}
					unknowns.put(alter);
					errorCount++;
				}
			}
		}

		//
		errors.add(0, errorCount + " " + Report.translate(bundle, "Unknown Relations") + "\n");
		for (Individual ego : unknownRelations.keySet()){
			for (Individual alter: unknownRelations.get(ego).toSortedList()){
				errors.add(ego+"\t"+alter+"\n");
			}
		}

		result.outputs().append(errors.toString());

		//
		result.setStatus(errorCount);

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;
	}
	

	/**
	 * 
	 * @param segmentation
	 * @param criteria
	 * @param bundle
	 * @return
	 * @throws PuckException 
	 */
	public static Report reportDiscontinuousBiographies(final Net net, final Segmentation segmentation, final SpaceTimeCriteria criteria, final ResourceBundle bundle) throws PuckException {
		Report result;

		//
		Chronometer chrono = new Chronometer();

		int errorCount = 0;
		StringList errors = new StringList();

		//
		result = new Report();
		result.setTitle("Discontinuous Itineraries");
		result.setOrigin("Control reporter");
		result.setTarget(segmentation.getLabel());

		for (Sequence itinerary : SequenceMaker.createBiographies(net, segmentation, criteria).toSortedList()) {
			Sequences partials = SequenceWorker.split(itinerary);
			if (partials.size() > 1) {
				errorCount++;
				errors.appendln(itinerary.getEgo().signature());
				int j = 0;
				for (Sequence partial : partials) {
					if (partial.getEvents().size() != 0 && j > 0) {
						errors.appendln(partial.getFirst() + "\t" + partial.getEvents().get(partial.getFirst()));
					}
					j++;
				}
				errors.appendln();
			}
		}

		//
		errors.add(0, errorCount + " " + Report.translate(bundle, "Discontinuous Itineraries") + "\n");
		result.outputs().append(errors.toString());

		//
		result.setStatus(errorCount);

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;
	}

	/**
	 * 
	 * @param segmentation
	 * @param criteria
	 * @param bundle
	 * @return
	 */
	public static Report reportMissingTestimonies(final Segmentation segmentation, final MissingTestimoniesCriteria criteria, final ResourceBundle bundle) {
		Report result;

		//
		Chronometer chrono = new Chronometer();

		int errorCount = 0;
		StringList errors = new StringList();

		//
		result = new Report();
		result.setTitle("Missing Testimonies");
		result.setOrigin("Control reporter");
		result.setTarget(segmentation.getLabel());

		for (Relation event : segmentation.getCurrentRelations().getByModelName(criteria.getRelationModelName())) {
			for (Individual witness : event.getIndividuals(criteria.getEgoRoleName())) {
				if (segmentation.getCurrentIndividuals().contains(witness)) {
					boolean missing = true;
					for (Attribute attribute : event.attributes()) {
						if (attribute.getLabel().contains("NOTE")) {
							String signature = witness.getId()+":";
							if (attribute.getValue().contains(signature)){
								missing = false;
								break;
							}
						}
					}
					if (missing) {
						errorCount++;
						errors.appendln(witness.signature() + "\t" + event.getTypedId() + "\t" + event.getName());
					}
				}
			}
		}

		//
		errors.add(0, errorCount + " " + Report.translate(bundle, "Missing Testimonies") + "\n");
		result.outputs().append(errors.toString());

		//
		result.setStatus(errorCount);

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;
	}

	
	/**
	 * 
	 * @param segmentation
	 * @param censusType
	 * @return
	 * @throws PuckException
	 */
	public static Report reportGeneralSequenceCensus(final Segmentation segmentation, SpaceTimeCriteria censusCriteria) throws PuckException {
		Report result;

		//
		Chronometer chrono = new Chronometer();

		result = new Report("General ");
		result.setOrigin("Sequence reporter");

		// Create Reports
		Report overallReport = new Report("Survey");
		Report diagramReport = new Report("Diagrams");
		Report detailReport = new Report("Details");

		// Create Partition charts and tables
		List<ReportChart> charts = new ArrayList<ReportChart>();
		List<ReportTable> tables = new ArrayList<ReportTable>();

		// Get (coherent) itineraries
		Sequences sequences = new Sequences();
		for (Individual ego : segmentation.getCurrentIndividuals().toSortedList()) {
			sequences.addRenumbered(SequenceWorker.getCoherentItinerarySegment(ego, censusCriteria));
		}

		// Proceed census
		SequencesCensus census = new SequencesCensus(sequences, censusCriteria);

		// Make overall report and diagrams
		overallReport.outputs().appendln(
			"Measure\tAverage (Male)\tAverage Pos. (Male)\tMedian (Male)\tMaximum (Male)\tSum (Male)\tAverage (Female)\tAverage Pos. (Female)\tMedian (Female)\tMaximum (Female)\tSum (Female)\tAverage (All)\tAverage Pos. (All)\tMedian (All)\tMaximum (All)\tSum (All)");

		// Set partition criteria
		for (String label : censusCriteria.getCensusOperationLabels()) {

			PartitionCriteria partitionCriteria = new PartitionCriteria(label);
			// partitionCriteria.setValueFilter(ValueFilter.NULL);

			/*			if (label.contains("PROFILE")){
							partitionCriteria.setType(PartitionType.PARTIALIZATION);
						} */

			if (label.equals("NREVENTS") || label.contains("NRSTATIONS")) {
				partitionCriteria.setType(PartitionType.FREE_GROUPING);
				partitionCriteria.setIntervals(PartitionMaker.getIntervals("1 5 10 15 20 25"));
				// partitionCriteria.setCumulationType(CumulationType.DESCENDANT);
			} else if (label.contains("AGEFIRST")) {
				partitionCriteria.setType(PartitionType.SIZED_GROUPING);
				partitionCriteria.setStart(0.);
				partitionCriteria.setSize(5.);
			} else if (label.equals("ECCENTRICITY")) {
				partitionCriteria.setType(PartitionType.SIZED_GROUPING);
				partitionCriteria.setStart(-100.);
				partitionCriteria.setSize(20.);
			} else if (label.contains("COVERAGE") || label.contains("SAME") || label.contains("NORM") || label.contains("DENSITY")
					|| label.contains("BETWEENNESS") || label.contains("EFFICIENCY") || label.contains("CONCENTRATION")) {
				partitionCriteria.setType(PartitionType.SIZED_GROUPING);
				partitionCriteria.setStart(0.);
				partitionCriteria.setSize(20.);
			} else if (label.contains("MEAN") || label.contains("COVERAGE") || label.contains("PEREVENT") || label.contains("BETWEENNESS")
					|| label.contains("BROKERAGE") || label.contains("EFFICIENT_SIZE")) {
				partitionCriteria.setType(PartitionType.SIZED_GROUPING);
				partitionCriteria.setStart(0.);
				partitionCriteria.setSize(1.);
			} else {
				partitionCriteria.setType(PartitionType.RAW);
			}

			ReportChart chart = null;

			if (!label.contains("ALTERS") && !label.contains("PROFILE")&& !label.equals("MEAN_NR_MOVES")) {
				NumberedValues values = census.getValues(label);
				
				Partition<Individual> partition = PartitionMaker.create(label, segmentation.getCurrentIndividuals(), values, partitionCriteria);

				PartitionCriteria splitCriteria = new PartitionCriteria(censusCriteria.getPartitionLabel());
				chart = StatisticsReporter.createPartitionChart(partition, partitionCriteria, splitCriteria);

				if (label.substring(0, 3).equals("AGE")) {

					partitionCriteria.setType(PartitionType.RAW);
					partitionCriteria.setSizeFilter(SizeFilter.HOLES);
					partitionCriteria.setValueFilter(ValueFilter.NULL);

					partition = PartitionMaker.create(label, segmentation.getCurrentIndividuals(), values, partitionCriteria);

					if (partition.maxValue() != null) {
						ReportChart survivalChart = StatisticsReporter.createSurvivalChart(partition, splitCriteria);
						charts.add(survivalChart);
					} else {
						System.err.println(label + " no max value");
					}
				}

				NumberedValues[] genderedValues = PuckUtils.getGenderedNumberedValues(values, segmentation.getCurrentIndividuals());

				overallReport.outputs().append(label + "\t");
				for (int gender = 0; gender < 3; gender++) {
					String sum = "";
					if (label.startsWith("NR")) {
						sum = new Double(genderedValues[gender].sum()).intValue() + "";
					}
					overallReport.outputs().append(
							MathUtils.round(genderedValues[gender].average(), 2) + "\t" + MathUtils.round(genderedValues[gender].averagePositives(), 2) + "\t"
									+ values.median() + "\t" + genderedValues[gender].max() + "\t" + sum + "\t");
				}
				overallReport.outputs().appendln();

			}
			
			if (label.equals("MEAN_NR_MOVES")){
				Map<Value,Double[]> map = census.getMeanNrMoves();
				chart = StatisticsReporter.createMapChart(map, label, new String[]{"MALE","FEMALE"}, GraphType.STACKED_BARS);
			}

			if (chart != null) {
				charts.add(chart);
				ReportTable table = ReportTable.transpose(chart.createReportTableWithSum());
				tables.add(table);
				if (!label.contains("EVENTS_") && !label.contains("RELATIONS")) {
//					tables.add(ReportTable.normalize(table));
				}
			}
		}
		overallReport.outputs().appendln();

		// Create detailed report
		detailReport.outputs().append("Nr\tEgo\tGender");
		for (String partitionLabel : censusCriteria.getCensusOperationLabels()) {
			detailReport.outputs().append("\t" + partitionLabel);
		}
		detailReport.outputs().appendln();

		for (Individual ego : segmentation.getCurrentIndividuals().toSortedList()) {

			if (census.getValues("NREVENTS").get(ego.getId()) != null){
				detailReport.outputs().append(ego.getId() + "\t" + ego + "\t" + ego.getGender());
				for (String label : censusCriteria.getCensusOperationLabels()) {
					detailReport.outputs().append("\t" + census.getValues(label).get(ego.getId()));
				}
				detailReport.outputs().appendln();
			}
		}

		// Manage the number of chart by line.
		for (int chartIndex = 0; chartIndex < charts.size(); chartIndex++) {
			diagramReport.outputs().append(charts.get(chartIndex));
			if (chartIndex % 4 == 3) {
				diagramReport.outputs().appendln();
			}
		}

		// Add chart tables.
		for (ReportTable table : tables) {
			diagramReport.outputs().appendln(table.getTitle());
			diagramReport.outputs().appendln(table);
		}

		// Finalize reports
		result.outputs().append(overallReport);
		result.outputs().append(diagramReport);
		result.outputs().append(detailReport);

		// addPajekData

		Map<String, StringList> pajekBuffers = census.getPajekBuffers();

		for (String title : pajekBuffers.keySet()) {

			StringList pajekBuffer = pajekBuffers.get(title);
			if (pajekBuffer.length() != 0) {
				File targetFile = ToolBox.setExtension(ToolBox.addToName(new File(segmentation.getLabel()), "-" + title), ".paj");
				ReportRawData rawData = new ReportRawData("Export " + title + "s to Pajek", "Pajek", "paj", targetFile);
				rawData.setData(PAJFile.convertToMicrosoftEndOfLine(pajekBuffer.toString()));

				result.outputs().appendln();
				result.outputs().append(rawData);
			}
		}

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;

	}
	

	/**
	 * 
	 * @param segmentation
	 * @param censusType
	 * @return
	 * @throws PuckException
	 */
	public static Report reportEgoNetworksCensus(final Segmentation segmentation, final SpaceTimeCriteria censusCriteria) throws PuckException {
		Report result;

		//
		Chronometer chrono = new Chronometer();

		result = new Report("Ego Networks");
		result.setOrigin("Sequence reporter");

		// Create Reports
		Report overallReport = new Report("Survey");
		Report diagramReport = new Report("Diagrams");
		Report detailReport = new Report("Details");
		Report componentReport = null;
		
		if (censusCriteria.getEgoNetworksOperations().contains(EgoNetworksOperation.COHESION)){
			componentReport = new Report("Components");
		}
		
		// Create Partition charts and tables
		List<ReportChart> charts = new ArrayList<ReportChart>();
		List<ReportTable> tables = new ArrayList<ReportTable>();

		// Get (coherent) itineraries
		Sequences sequences = new Sequences();
		for (Individual ego : segmentation.getCurrentIndividuals().toSortedList()) {
			sequences.addRenumbered(SequenceWorker.getCoherentItinerarySegment(ego, censusCriteria));
		}

		// Proceed census
		SequencesCensus census = new SequencesCensus(sequences, censusCriteria);

		// Make overall report and diagrams
		overallReport
				.outputs()
				.appendln(
						"Measure\tAverage (Male)\tAverage Pos. (Male)\tMedian (Male)\tMaximum (Male)\tSum (Male)\tAverage (Female)\tAverage Pos. (Female)\tMedian (Female)\tMaximum (Female)\tSum (Female)\tAverage (All)\tAverage Pos. (All)\tMedian (All)\tMaximum (All)\tSum (All)");

		// Set partition criteria
		for (String label : censusCriteria.getCensusOperationLabels()) {

			PartitionCriteria partitionCriteria = new PartitionCriteria(label);
			// partitionCriteria.setValueFilter(ValueFilter.NULL);

			/*			if (label.contains("PROFILE")){
							partitionCriteria.setType(PartitionType.PARTIALIZATION);
						} */

			if (label.equals("NREVENTS")) {
				partitionCriteria.setType(PartitionType.RAW);
				// partitionCriteria.setCumulationType(CumulationType.DESCENDANT);
			} else if (label.contains("AGEFIRST")) {
				partitionCriteria.setType(PartitionType.SIZED_GROUPING);
				partitionCriteria.setStart(0.);
				partitionCriteria.setSize(5.);
			} else if (label.equals("ECCENTRICITY")) {
				partitionCriteria.setType(PartitionType.SIZED_GROUPING);
				partitionCriteria.setStart(-100.);
				partitionCriteria.setSize(20.);
			} else if (label.contains("COVERAGE") || label.contains("SAME") || label.contains("NORM") || label.contains("DENSITY")
					|| label.contains("BETWEENNESS") || label.contains("EFFICIENCY") || label.contains("CONCENTRATION")) {
				partitionCriteria.setType(PartitionType.SIZED_GROUPING);
				partitionCriteria.setStart(0.);
				partitionCriteria.setSize(20.);
			} else if (label.contains("MEAN") || label.contains("COVERAGE") || label.contains("PEREVENT") || label.contains("BETWEENNESS")
					|| label.contains("BROKERAGE") || label.contains("EFFICIENT_SIZE")) {
				partitionCriteria.setType(PartitionType.SIZED_GROUPING);
				partitionCriteria.setStart(0.);
				partitionCriteria.setSize(1.);
			} else {
				partitionCriteria.setType(PartitionType.RAW);
			}

			ReportChart chart = null;

			if (!label.contains("ALTERS") && !label.contains("PROFILE")) {
				NumberedValues values = census.getValues(label);

				Partition<Individual> partition = PartitionMaker.create(label, segmentation.getCurrentIndividuals(), values, partitionCriteria);

				PartitionCriteria splitCriteria = new PartitionCriteria(censusCriteria.getPartitionLabel());
				chart = StatisticsReporter.createPartitionChart(partition, partitionCriteria, splitCriteria);

				if (label.substring(0, 3).equals("AGE")) {

					partitionCriteria.setType(PartitionType.RAW);
					partitionCriteria.setSizeFilter(SizeFilter.HOLES);
					partitionCriteria.setValueFilter(ValueFilter.NULL);

					partition = PartitionMaker.create(label, segmentation.getCurrentIndividuals(), values, partitionCriteria);

					if (partition.maxValue() != null) {
						ReportChart survivalChart = StatisticsReporter.createSurvivalChart(partition, splitCriteria);
						charts.add(survivalChart);
					} else {
						System.err.println(label + " no max value");
					}
				}

				NumberedValues[] genderedValues = PuckUtils.getGenderedNumberedValues(values, segmentation.getCurrentIndividuals());

				overallReport.outputs().append(label + "\t");
				for (int gender = 0; gender < 3; gender++) {
					String sum = "";
					if (label.startsWith("NR")) {
						sum = new Double(genderedValues[gender].sum()).intValue() + "";
					}
					overallReport.outputs().append(
							MathUtils.round(genderedValues[gender].average(), 2) + "\t" + MathUtils.round(genderedValues[gender].averagePositives(), 2) + "\t"
									+ values.median() + "\t" + genderedValues[gender].max() + "\t" + sum + "\t");
				}
				overallReport.outputs().appendln();

			}

			if (chart != null) {
				charts.add(chart);
					ReportTable table = ReportTable.transpose(chart.createReportTableWithSum());
					tables.add(table);
					if (!label.contains("EVENTS_") && !label.contains("RELATIONS")) {
//						tables.add(ReportTable.normalize(table));
				}

			}

		}
		overallReport.outputs().appendln();

		// Create detailed report
		detailReport.outputs().append("Nr\tEgo\tGender");
		for (String partitionLabel : censusCriteria.getCensusOperationLabels()) {
				detailReport.outputs().append("\t" + partitionLabel);
		}
		
		Map<String,Map<Value, Double[]>> componentChartMaps = new TreeMap<String, Map<Value, Double[]>>();
		Map<String,Map<Value, Double[]>> componentChartSizeMaps = new TreeMap<String, Map<Value, Double[]>>();
		for (String networkTitle : censusCriteria.getNetworkTitles()){
			componentChartMaps.put(networkTitle,new TreeMap<Value, Double[]>());
			componentChartSizeMaps.put(networkTitle,new TreeMap<Value, Double[]>());
		}

		detailReport.outputs().appendln();
		for (Individual ego : segmentation.getCurrentIndividuals().toSortedList()) {

			if (census.getValues("SIZE").get(ego.getId()) != null) {
				detailReport.outputs().append(ego.getId() + "\t" + ego + "\t" + ego.getGender());
				for (String label : censusCriteria.getCensusOperationLabels()) {
						detailReport.outputs().append("\t" + census.getValues(label).get(ego.getId()));
				}
				detailReport.outputs().appendln();
			}
			
			if (censusCriteria.getEgoNetworksOperations().contains(EgoNetworksOperation.COHESION)){

				for (String networkTitle : censusCriteria.getNetworkTitles()) {
					Map<Integer, Partition<Node<Individual>>> componentsMap = census.getComponents(networkTitle);
					Map<Value,Double[]> componentChartMap = componentChartMaps.get(networkTitle);
					Map<Value,Double[]> componentChartSizeMap = componentChartSizeMaps.get(networkTitle);
					
					if (componentsMap != null) {
						Partition<Node<Individual>> components = componentsMap.get(ego.getId());

						componentReport.outputs().appendln("Components " + networkTitle);
						componentReport.outputs().appendln(ego + "\t" + components.size());
						int i = 1;
						for (Cluster<Node<Individual>> cluster : components.getClusters().toListSortedByValue()) {
							componentReport.outputs().appendln(
									"\t" + i + "\t" + cluster.getValue() + "\t(" + cluster.size() + ")\t" + cluster.getItemsAsString());
							i++;
						}
						componentReport.outputs().appendln();

						for (Value value : components.getValues()) {
							Double[] map = componentChartMap.get(value);
							Double[] map2 = componentChartSizeMap.get(value);
							if (map == null) {
								map = new Double[]{0.,0.,0.};
								componentChartMap.put(value, map);
								map2 = new Double[]{0.,0.,0.};
								componentChartSizeMap.put(value, map2);
							}
							map[ego.getGender().toInt()]+=1.;
							map[2]+=1.;
							map2[ego.getGender().toInt()]+=components.getCluster(value).size();
							map2[2]+=components.getCluster(value).size();
						}
					}
				}
			}
		}
		
		for (String networkTitle : censusCriteria.getNetworkTitles()) {
			
			Map<Value,Double[]> componentChartMap = componentChartMaps.get(networkTitle);
			Map<Value,Double[]> componentChartSizeMap = componentChartSizeMaps.get(networkTitle);
			for (Value value: componentChartMap.keySet()){
				Double[] sums = componentChartSizeMap.get(value);
				for (int i=0;i<3;i++){
					sums[i]=new Double(sums[i]/componentChartMap.get(value)[i]);
				}
			}

			ReportChart componentChart = StatisticsReporter.createMapChart(componentChartMap, "COMPONENTS",new String[]{"MALE","FEMALE"}, GraphType.STACKED_BARS);
			charts.add(componentChart);
			tables.add(ReportTable.transpose(componentChart.createReportTableWithSum()));
			
			ReportChart componentSizeChart = StatisticsReporter.createMapChart(componentChartSizeMap, "COMPONENTS",new String[]{"MALE","FEMALE"}, GraphType.STACKED_BARS);
			charts.add(componentSizeChart);
			tables.add(ReportTable.transpose(componentSizeChart.createReportTableWithSum()));

	/*			if (census.getRelationConnectionMatrix() != null) {
				for (ReportChart chart : census.getRelationConnectionMatrix().getCharts()) {
					charts.add(chart);
				}
				tables.add(census.getRelationConnectionMatrix().getTable("Component Connections"));
			}*/
		}

		// Manage the number of chart by line.
		for (int chartIndex = 0; chartIndex < charts.size(); chartIndex++) {
			diagramReport.outputs().append(charts.get(chartIndex));
			if (chartIndex % 4 == 3) {
				diagramReport.outputs().appendln();
			}
		}

		// Add chart tables.
		for (ReportTable table : tables) {
			diagramReport.outputs().appendln(table.getTitle());
			diagramReport.outputs().appendln(table);
		}

		// Finalize reports
		result.outputs().append(overallReport);
		result.outputs().append(diagramReport);
		result.outputs().append(detailReport);
		
		if (componentReport!=null){
			result.outputs().append(componentReport);
		}

		// addPajekData
		
		if (censusCriteria.getEgoNetworksOperations().contains(EgoNetworksOperation.EXPORT_EGO_NETWORKS)){

			Map<String, StringList> pajekBuffers = census.getPajekBuffers();

			for (String title : pajekBuffers.keySet()) {

				StringList pajekBuffer = pajekBuffers.get(title);
				if (pajekBuffer.length() != 0) {
					File targetFile = ToolBox.setExtension(ToolBox.addToName(new File(segmentation.getLabel()), "-" + title), ".paj");
					ReportRawData rawData = new ReportRawData("Export " + title + "s to Pajek", "Pajek", "paj", targetFile);
					rawData.setData(PAJFile.convertToMicrosoftEndOfLine(pajekBuffer.toString()));

					result.outputs().appendln();
					result.outputs().append(rawData);
				}
			}
		
		}

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;

	}

	/**
	 * 
	 * @param segmentation
	 * @param censusType
	 * @return
	 * @throws PuckException
	 */
	public static Report reportParcoursNetworksCensus(final Segmentation segmentation, final SpaceTimeCriteria censusCriteria) throws PuckException {
		Report result;

		//
		Chronometer chrono = new Chronometer();

		result = new Report("Parcours Networks");
		result.setOrigin("Sequence reporter");

		// Create Reports
		Report overallReport = new Report("Survey");
		Report diagramReport = new Report("Diagrams");
		Report detailReport = new Report("Details");
		Report componentReport = null;
		
		if (censusCriteria.getParcoursNetworksOperations().contains(ParcoursNetworksOperation.COHESION)){
			componentReport = new Report("Components");
		}
		
		// Create Partition charts and tables
		List<ReportChart> charts = new ArrayList<ReportChart>();
		List<ReportTable> tables = new ArrayList<ReportTable>();

		// Get (coherent) itineraries
		Sequences sequences = new Sequences();
		for (Individual ego : segmentation.getCurrentIndividuals().toSortedList()) {
			sequences.addRenumbered(SequenceWorker.getCoherentItinerarySegment(ego, censusCriteria));
		}

		// Proceed census
		SequencesCensus census = new SequencesCensus(sequences, censusCriteria);

		// Make overall report and diagrams
		overallReport
				.outputs()
				.appendln(
						"Measure\tAverage (Male)\tAverage Pos. (Male)\tMedian (Male)\tMaximum (Male)\tSum (Male)\tAverage (Female)\tAverage Pos. (Female)\tMedian (Female)\tMaximum (Female)\tSum (Female)\tAverage (All)\tAverage Pos. (All)\tMedian (All)\tMaximum (All)\tSum (All)");

		// Set partition criteria
		for (String label : censusCriteria.getCensusOperationLabels()) {

			PartitionCriteria partitionCriteria = new PartitionCriteria(label);
			// partitionCriteria.setValueFilter(ValueFilter.NULL);

			/*			if (label.contains("PROFILE")){
							partitionCriteria.setType(PartitionType.PARTIALIZATION);
						} */

			if (label.equals("NREVENTS")) {
				partitionCriteria.setType(PartitionType.RAW);
				// partitionCriteria.setCumulationType(CumulationType.DESCENDANT);
			} else if (label.contains("AGEFIRST")) {
				partitionCriteria.setType(PartitionType.SIZED_GROUPING);
				partitionCriteria.setStart(0.);
				partitionCriteria.setSize(5.);
			} else if (label.equals("ECCENTRICITY")) {
				partitionCriteria.setType(PartitionType.SIZED_GROUPING);
				partitionCriteria.setStart(-100.);
				partitionCriteria.setSize(20.);
			} else if (label.contains("COVERAGE") || label.contains("SAME") || label.contains("NORM") || label.contains("DENSITY")
					|| label.contains("BETWEENNESS") || label.contains("EFFICIENCY") || label.contains("CONCENTRATION")) {
				partitionCriteria.setType(PartitionType.SIZED_GROUPING);
				partitionCriteria.setStart(0.);
				partitionCriteria.setSize(20.);
			} else if (label.contains("MEAN") || label.contains("COVERAGE") || label.contains("PEREVENT") || label.contains("BETWEENNESS")
					|| label.contains("BROKERAGE") || label.contains("EFFICIENT_SIZE")) {
				partitionCriteria.setType(PartitionType.SIZED_GROUPING);
				partitionCriteria.setStart(0.);
				partitionCriteria.setSize(1.);
			} else {
				partitionCriteria.setType(PartitionType.RAW);
			}

			ReportChart chart = null;

			if (label.contains("SIMILARITY")) {
				RelationClassificationType relationClassificationType = RelationClassificationType.valueOf(label.substring(label.lastIndexOf("_") + 1));
				Map<Value, Double[]> similaritiesMap = census.getSimilaritiesMap(relationClassificationType);
				chart = StatisticsReporter.createMapChart(similaritiesMap, label, new String[] { "HH", "FH", "HF", "FF", "All" }, GraphType.LINES);

				for (Value key : similaritiesMap.keySet()) {
					overallReport.outputs().appendln(label + "_" + key + "\t" + MathUtils.percent(similaritiesMap.get(key)[4], 100));
				}

			} else if (!label.contains("ALTERS") && !label.contains("PROFILE")) {
				NumberedValues values = census.getValues(label);

				Partition<Individual> partition = PartitionMaker.create(label, segmentation.getCurrentIndividuals(), values, partitionCriteria);

				PartitionCriteria splitCriteria = new PartitionCriteria(censusCriteria.getPartitionLabel());
				chart = StatisticsReporter.createPartitionChart(partition, partitionCriteria, splitCriteria);

				if (label.substring(0, 3).equals("AGE")) {

					partitionCriteria.setType(PartitionType.RAW);
					partitionCriteria.setSizeFilter(SizeFilter.HOLES);
					partitionCriteria.setValueFilter(ValueFilter.NULL);

					partition = PartitionMaker.create(label, segmentation.getCurrentIndividuals(), values, partitionCriteria);

					if (partition.maxValue() != null) {
						ReportChart survivalChart = StatisticsReporter.createSurvivalChart(partition, splitCriteria);
						charts.add(survivalChart);
					} else {
						System.err.println(label + " no max value");
					}
				}

				NumberedValues[] genderedValues = PuckUtils.getGenderedNumberedValues(values, segmentation.getCurrentIndividuals());

				overallReport.outputs().append(label + "\t");
				for (int gender = 0; gender < 3; gender++) {
					String sum = "";
					if (label.startsWith("NR")) {
						sum = new Double(genderedValues[gender].sum()).intValue() + "";
					}
					overallReport.outputs().append(
							MathUtils.round(genderedValues[gender].average(), 2) + "\t" + MathUtils.round(genderedValues[gender].averagePositives(), 2) + "\t"
									+ values.median() + "\t" + genderedValues[gender].max() + "\t" + sum + "\t");
				}
				overallReport.outputs().appendln();

			}

			if (chart != null) {
				charts.add(chart);
				if (label.equals("SIMILARITY")) {
					tables.add(ReportTable.transpose(chart.createReportTable()));
				} else {
					ReportTable table = ReportTable.transpose(chart.createReportTableWithSum());
					tables.add(table);
					if (!label.contains("EVENTS_") && !label.contains("RELATIONS")) {
						tables.add(ReportTable.normalize(table));
					}
				}

			}

		}
		overallReport.outputs().appendln();
		
		// Add parcours network statistics
		
		Map<String,Map<String,Map<String,Value>>> parcoursNetworkStatistics = census.getParcoursNetworkStatistics();
		
		List<String> clusterValues = new ArrayList<String>(parcoursNetworkStatistics.keySet());
		
		if (parcoursNetworkStatistics!=null){
			overallReport.outputs().appendln("Parcours Network Statistics");

			String headLine = null;
			boolean first = true;
			
			for (String nodeLabel : parcoursNetworkStatistics.get("Total").keySet()){
				String line = nodeLabel;
				headLine = "Node";
				
				for (String genderLabel : clusterValues){
					Map<String,Value> statistics = parcoursNetworkStatistics.get(genderLabel).get(nodeLabel);
					for (String statisticsLabel : censusCriteria.getNodeStatisticsLabels()){
						if (first){
							headLine+="\t"+statisticsLabel+" ("+genderLabel+")";
						}
						if (statistics!=null){
							line+="\t"+statistics.get(statisticsLabel);
						} else {
							line+="\t";
						}
					}
				}
				if (first){
					overallReport.outputs().appendln(headLine);
					first = false;
				}
				overallReport.outputs().appendln(line);
			}
		}

		// Create detailed report
		detailReport.outputs().append("Nr\tEgo\tGender");
		for (String partitionLabel : censusCriteria.getCensusOperationLabels()) {
			if (partitionLabel.contains("SIMILARITY")) {
				RelationClassificationType relationClassificationType = RelationClassificationType.valueOf(partitionLabel.substring(partitionLabel
						.lastIndexOf("_") + 1));
				detailReport.outputs().append(
						"\tSIMILARITY_PARENT_" + relationClassificationType + "\tSIMILARITY_CHILD_" + relationClassificationType + "\tSIMILARITY_SIBLING_"
								+ relationClassificationType + "\tSIMILARITY_SPOUSE_" + relationClassificationType);
			} else {
				detailReport.outputs().append("\t" + partitionLabel);
			}
		}

		Map<String,Map<Value, Double[]>> componentChartMaps = new TreeMap<String, Map<Value, Double[]>>();
		for (String networkTitle : censusCriteria.getNetworkTitles()){
			componentChartMaps.put(networkTitle,new TreeMap<Value, Double[]>());
		}

		detailReport.outputs().appendln();
		for (Individual ego : segmentation.getCurrentIndividuals().toSortedList()) {

			if ((census.getValues("NREVENTS").get(ego.getId()) != null)){
				detailReport.outputs().append(ego.getId() + "\t" + ego + "\t" + ego.getGender());
				for (String label : censusCriteria.getCensusOperationLabels()) {
					if (label.contains("SIMILARITY")) {
						Value value = census.getValues(label).get(ego.getId());
						Map<Value, Double[]> indiSimilaritiesMap = value.mapValue();
						String[] keys = new String[] { "PARENT", "CHILD", "SIBLING", "SPOUSE" };
						for (String key : keys) {
							Double[] sim = indiSimilaritiesMap.get(new Value(key));
							if (sim != null) {
								detailReport.outputs().append("\t" + MathUtils.round(sim[4], 2));
							}
						}
					} else {
						detailReport.outputs().append("\t" + census.getValues(label).get(ego.getId()));
					}
				}
				detailReport.outputs().appendln();
			}

			if (componentReport!=null){

				for (String networkTitle : censusCriteria.getNetworkTitles()) {
					Map<Integer, Partition<Node<Individual>>> componentsMap = census.getComponents(networkTitle);
					Map<Value,Double[]> componentChartMap = componentChartMaps.get(networkTitle);
					if (componentsMap != null) {
						Partition<Node<Individual>> components = componentsMap.get(ego.getId());

						componentReport.outputs().appendln("Components " + networkTitle);
						componentReport.outputs().appendln(ego + "\t" + components.size());
						int i = 1;
						for (Cluster<Node<Individual>> cluster : components.getClusters().toListSortedByValue()) {
							componentReport.outputs().appendln(
									"\t" + i + "\t" + cluster.getValue() + "\t(" + cluster.size() + ")\t" + cluster.getItemsAsString());
							i++;
						}
						componentReport.outputs().appendln();

						for (Value value : components.getValues()) {
							Double[] map = componentChartMap.get(value);
							if (map == null) {
								map = new Double[]{0.,0.,0.};
								componentChartMap.put(value, map);
							}
							map[ego.getGender().toInt()]+=+ 1.;
							map[2]+=+ 1.;
						}
					}
				}
			}
		}

		for (String networkTitle : censusCriteria.getNetworkTitles()) {
			ReportChart componentChart = StatisticsReporter.createMapChart(componentChartMaps.get(networkTitle),"COMPONENTS", new String[]{"MALE","FEMALE"}, GraphType.STACKED_BARS);
			charts.add(componentChart);
			tables.add(ReportTable.transpose(componentChart.createReportTableWithSum()));

			if (census.getRelationConnectionMatrix() != null) {
				for (ReportChart chart : census.getRelationConnectionMatrix().getCharts()) {
					charts.add(chart);
				}
				tables.add(census.getRelationConnectionMatrix().getTable("Component Connections"));
			}
		}
		

		// Manage the number of chart by line.
		for (int chartIndex = 0; chartIndex < charts.size(); chartIndex++) {
			diagramReport.outputs().append(charts.get(chartIndex));
			if (chartIndex % 4 == 3) {
				diagramReport.outputs().appendln();
			}
		}

		// Add chart tables.
		for (ReportTable table : tables) {
			diagramReport.outputs().appendln(table.getTitle());
			diagramReport.outputs().appendln(table);
		}

		// Finalize reports
		result.outputs().append(overallReport);
		result.outputs().append(diagramReport);
		result.outputs().append(detailReport);

			result.outputs().append(componentReport);

		// addPajekData
		
		Map<String, StringList> pajekBuffers = census.getPajekBuffers();

		for (String title : pajekBuffers.keySet()) {

			StringList pajekBuffer = pajekBuffers.get(title);
			if (pajekBuffer.length() != 0) {
				File targetFile = ToolBox.setExtension(ToolBox.addToName(new File(segmentation.getLabel()), "-" + title), ".paj");
				ReportRawData rawData = new ReportRawData("Export " + title + "s to Pajek", "Pajek", "paj", targetFile);
				rawData.setData(PAJFile.convertToMicrosoftEndOfLine(pajekBuffer.toString()));

				result.outputs().appendln();
				result.outputs().append(rawData);
			}
		}

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;

	}

	/**
	 * 
	 * @param segmentation
	 * @param censusType
	 * @return
	 * @throws PuckException
	 */
	public static Report reportTrajectoriesCensus(final Segmentation segmentation, final SpaceTimeCriteria censusCriteria) throws PuckException {
		Report result;

		//
		Chronometer chrono = new Chronometer();

		result = new Report("Trajectories");
		result.setOrigin("Sequence reporter");

		// Create Reports
		Report overallReport = new Report("Survey");
		Report diagramReport = new Report("Diagrams");
		Report detailReport = new Report("Details");
		Report treeReport = null;
				
		if (censusCriteria.getTrajectoriesOperations().contains(TrajectoriesOperation.LIST_TREES)){
			treeReport = new Report("Trees");
			
		}


		// Create Partition charts and tables
		List<ReportChart> charts = new ArrayList<ReportChart>();
		List<ReportTable> tables = new ArrayList<ReportTable>();

		// Get (coherent) itineraries
		Sequences sequences = new Sequences();
		for (Individual ego : segmentation.getCurrentIndividuals().toSortedList()) {
			sequences.addRenumbered(SequenceWorker.getCoherentItinerarySegment(ego, censusCriteria));
		}

		// Proceed census
		SequencesCensus census = new SequencesCensus(sequences, censusCriteria);

		// Make overall report and diagrams
		overallReport
				.outputs()
				.appendln(
						"Measure\tAverage (Male)\tAverage Pos. (Male)\tMedian (Male)\tMaximum (Male)\tSum (Male)\tAverage (Female)\tAverage Pos. (Female)\tMedian (Female)\tMaximum (Female)\tSum (Female)\tAverage (All)\tAverage Pos. (All)\tMedian (All)\tMaximum (All)\tSum (All)");

		// Set partition criteria
		for (String label : censusCriteria.getCensusOperationLabels()) {

			PartitionCriteria partitionCriteria = new PartitionCriteria(label);
			// partitionCriteria.setValueFilter(ValueFilter.NULL);

			/*			if (label.contains("PROFILE")){
							partitionCriteria.setType(PartitionType.PARTIALIZATION);
						} */

			if (label.equals("NREVENTS") || label.contains("NRSTATIONS")) {
				partitionCriteria.setType(PartitionType.RAW);
//				partitionCriteria.setType(PartitionType.FREE_GROUPING);
//				partitionCriteria.setIntervals(PartitionMaker.getIntervals("1 5 10 15 20 25"));
				// partitionCriteria.setCumulationType(CumulationType.DESCENDANT);
			} else if (label.contains("AGEFIRST")) {
				partitionCriteria.setType(PartitionType.SIZED_GROUPING);
				partitionCriteria.setStart(0.);
				partitionCriteria.setSize(5.);
			} else if (label.equals("ECCENTRICITY")) {
				partitionCriteria.setType(PartitionType.SIZED_GROUPING);
				partitionCriteria.setStart(-100.);
				partitionCriteria.setSize(20.);
			} else if (label.contains("COVERAGE") || label.contains("SAME") || label.contains("NORM") || label.contains("DENSITY")
					|| label.contains("BETWEENNESS") || label.contains("EFFICIENCY") || label.contains("CONCENTRATION")) {
				partitionCriteria.setType(PartitionType.SIZED_GROUPING);
				partitionCriteria.setStart(0.);
				partitionCriteria.setSize(20.);
			} else if (label.contains("MEAN") || label.contains("COVERAGE") || label.contains("PEREVENT") || label.contains("BETWEENNESS")
					|| label.contains("BROKERAGE") || label.contains("EFFICIENT_SIZE")) {
				partitionCriteria.setType(PartitionType.SIZED_GROUPING);
				partitionCriteria.setStart(0.);
				partitionCriteria.setSize(1.);
			} else {
				partitionCriteria.setType(PartitionType.RAW);
			}

			ReportChart chart = null;

			if (!label.contains("ALTERS") && !label.contains("PROFILE")) {
				NumberedValues values = census.getValues(label);

				Partition<Individual> partition = PartitionMaker.create(label, segmentation.getCurrentIndividuals(), values, partitionCriteria);

				PartitionCriteria splitCriteria = new PartitionCriteria(censusCriteria.getPartitionLabel());
				chart = StatisticsReporter.createPartitionChart(partition, partitionCriteria, splitCriteria);

				if (label.substring(0, 3).equals("AGE")) {

					partitionCriteria.setType(PartitionType.RAW);
					partitionCriteria.setSizeFilter(SizeFilter.HOLES);
					partitionCriteria.setValueFilter(ValueFilter.NULL);

					partition = PartitionMaker.create(label, segmentation.getCurrentIndividuals(), values, partitionCriteria);

					if (partition.maxValue() != null) {
						ReportChart survivalChart = StatisticsReporter.createSurvivalChart(partition, splitCriteria);
						charts.add(survivalChart);
					} else {
						System.err.println(label + " no max value");
					}
				}

				NumberedValues[] genderedValues = PuckUtils.getGenderedNumberedValues(values, segmentation.getCurrentIndividuals());

				overallReport.outputs().append(label + "\t");
				for (int gender = 0; gender < 3; gender++) {
					String sum = "";
					if (label.startsWith("NR")) {
						sum = new Double(genderedValues[gender].sum()).intValue() + "";
					}
					overallReport.outputs().append(
							MathUtils.round(genderedValues[gender].average(), 2) + "\t" + MathUtils.round(genderedValues[gender].averagePositives(), 2) + "\t"
									+ values.median() + "\t" + genderedValues[gender].max() + "\t" + sum + "\t");
				}
				overallReport.outputs().appendln();

			}

			if (chart != null) {
				charts.add(chart);
					ReportTable table = ReportTable.transpose(chart.createReportTableWithSum());
					tables.add(table);
					if (!label.contains("EVENTS_") && !label.contains("RELATIONS")) {
						tables.add(ReportTable.normalize(table));
					}

			}

		}
		overallReport.outputs().appendln();

		// Create detailed report
		
		Map<String, StringList> rStateBuffers = new TreeMap<String,StringList>();
		Map<String, StringList> rEventBuffers = new TreeMap<String,StringList>();
		Map<String, Integer> rSizes = new TreeMap<String,Integer>();
		
		NumberedValues dateProfiles = census.getValues("PROFILE_"+censusCriteria.getDateLabel());
		NumberedValues ageProfiles = census.getValues("PROFILE_AGE");
		
		detailReport.outputs().append("Nr\tEgo\tGender");
		for (String partitionLabel : censusCriteria.getCensusOperationLabels()) {
				detailReport.outputs().append("\t" + partitionLabel);
				
				if (partitionLabel.contains("PROFILE")){
					rStateBuffers.put(partitionLabel.substring(8),new StringList());
					rSizes.put(partitionLabel.substring(8),0);
					StringList rEventBuffer = new StringList();
					rEventBuffer.appendln("Id\tStep\tDate\tAge\t"+partitionLabel.substring(8));
					rEventBuffers.put(partitionLabel.substring(8),rEventBuffer);
				}
		}

		detailReport.outputs().appendln();
		for (Individual ego : segmentation.getCurrentIndividuals().toSortedList()) {

			if (census.getValues("NREVENTS").get(ego.getId()) != null) {
				
				detailReport.outputs().append(ego.getId() + "\t" + ego + "\t" + ego.getGender());
				List<String> dates = (List<String>)dateProfiles.get(ego.getId()).listValue();
				List<String> ages = (List<String>)ageProfiles.get(ego.getId()).listValue();
				
				for (String label : censusCriteria.getCensusOperationLabels()) {
					Value value = census.getValues(label).get(ego.getId());
						detailReport.outputs().append("\t" + value);
						
						if (label.contains("PROFILE")){
							StringList rStateBuffer = rStateBuffers.get(label.substring(8));
							StringList rEventBuffer = rEventBuffers.get(label.substring(8));

							int rSize = rSizes.get(label.substring(8));
									
							String rStateLine = ego.getId() + "\t" + ego + "\t" + ego.getGender();
							List<String> rItems = (List<String>)value.listValue();
							
							if (rItems.size()>rSize){
								rSizes.put(label.substring(8), rItems.size());
							}
							int time = 0;
							for (String rItem: rItems){
								rStateLine += "\t"+rItem;
								rEventBuffer.appendln(ego.getId()+"\t"+time+"\t"+dates.get(time)+"\t"+ages.get(time)+"\t"+rItem);
								time++;
							}
							rStateBuffer.append(rStateLine);
						}
				}
				detailReport.outputs().appendln();
			}


		// SequenceAnalysis

			for (RelationClassificationType relationClassificationType : censusCriteria.getMainRelationClassificationTypes()) {

				if (censusCriteria.getNetworkTitles().contains("Event Type Network")) {

					CorrelationMatrix eventSequenceMatrix = census.getEventSequenceMatrix(relationClassificationType.toString());

					if (eventSequenceMatrix != null) {
						for (ReportChart chart : eventSequenceMatrix.getCharts()) {
							charts.add(chart);
						}
						tables.add(eventSequenceMatrix.getTable("Event Type Sequences"));
						
					}

					overallReport.outputs().appendln();
					overallReport.outputs().appendln("Sequence Network Statistics " + relationClassificationType);
					overallReport.outputs().appendln("\tDensity\tInertia\t(Divergence)\tConcentration\t(Divergence)\tSymmetry\t(Divergence)\tCentral nodes");

					for (Gender gender : Gender.values()) {
						GraphProfile<Cluster<String>> profile = eventSequenceMatrix.getProfile(gender);

						String centralReferents = "";
						for (Cluster<String> centralReferent : profile.getCentralReferents()) {
							centralReferents += centralReferent.getValue() + " ";
						}
						double maxBetweenness = profile.getMaxBetweenness();
						double density = profile.density();
						double endo = MathUtils.round(profile.getStatistics(Indicator.LOOPS, Mode.NORMALIZED), 2);
						double endoExp = MathUtils.round(profile.getStatistics(Indicator.LOOPS, Mode.DIVERGENCE_NORMALIZED), 2);
						double conc = MathUtils.round(profile.getStatistics(Indicator.CONCENTRATION, Mode.SIMPLE), 2);
						double concExp = MathUtils.round(profile.getStatistics(Indicator.CONCENTRATION, Mode.DIVERGENCE), 2);
						double sym = MathUtils.round(profile.getStatistics(Indicator.SYMMETRY, Mode.SIMPLE), 2);
						double symExp = MathUtils.round(profile.getStatistics(Indicator.SYMMETRY, Mode.DIVERGENCE), 2);

						overallReport.outputs().appendln(
								gender + "\t" + density + "\t" + endo + "\t" + endoExp + "\t" + conc + "\t" + concExp + "\t" + sym + "\t" + symExp + "\t"
										+ centralReferents + "(" + maxBetweenness + ") betweenness centrality");
					}
					overallReport.outputs().appendln();

				}

				if (censusCriteria.getNetworkTitles().contains("Sequence Type Network")) {

					CorrelationMatrix subSequenceMatrix = census.getSubSequenceMatrix(relationClassificationType.toString());

					if (subSequenceMatrix != null) {
						charts.add(subSequenceMatrix.getRamificationChart());
					}
				}

				// reportSequencePartition(overallReport,"",census.getEventPartition(eventTypeName),census.getEventPairPartition(eventTypeName),census.nrSequences(),census.nrEvents());
				// reportSequenceTree(treeReport,"",census.getSequenceTypeNetwork(eventTypeName));

				/*				int maxPositions = census.getNrValues(eventType,eventType.toString());
										
								ReportChart diversityChart = StatisticsReporter.createDiversityChart(census.getDepthPartition(eventType.toString()), maxPositions);
								charts.add(diversityChart);*/
			}
		}
		

		// Manage the number of chart by line.
		for (int chartIndex = 0; chartIndex < charts.size(); chartIndex++) {
			diagramReport.outputs().append(charts.get(chartIndex));
			if (chartIndex % 4 == 3) {
				diagramReport.outputs().appendln();
			}
		}

		// Add chart tables.
		for (ReportTable table : tables) {
			diagramReport.outputs().appendln(table.getTitle());
			diagramReport.outputs().appendln(table);
		}

		// Finalize reports
		result.outputs().append(overallReport);
		result.outputs().append(diagramReport);
		result.outputs().append(detailReport);
		if (treeReport!=null){
			result.outputs().append(treeReport);
		}

		// addPajekData

		if (censusCriteria.getTrajectoriesOperations().contains(TrajectoriesOperation.EXPORT_EVENT_TYPE_NETWORK) || censusCriteria.getTrajectoriesOperations().contains(TrajectoriesOperation.EXPORT_SEQUENCE_TYPE_NETWORK)){
			
			Map<String, StringList> pajekBuffers = census.getPajekBuffers();

			for (String title : pajekBuffers.keySet()) {
				
				StringList pajekBuffer = pajekBuffers.get(title);
				if (pajekBuffer.length() != 0) {
										
					File targetFile = ToolBox.setExtension(ToolBox.addToName(new File(segmentation.getLabel()), "-" + title), ".paj");
					ReportRawData rawData = new ReportRawData("Export " + title + "s to Pajek", "Pajek", "paj", targetFile);
					rawData.setData(PAJFile.convertToMicrosoftEndOfLine(pajekBuffer.toString()));

					result.outputs().appendln();
					result.outputs().append(rawData);
				}
			}
		}
		
		// addRData

		if (censusCriteria.getTrajectoriesOperations().contains(TrajectoriesOperation.GENERAL)){
			
			for (String title : rStateBuffers.keySet()) {
				
				StringList rStateBuffer = rStateBuffers.get(title);
				StringList rStateBufferCompleted = new StringList();
				
				int rSize = rSizes.get(title);
				String rStateHeadLine = "Id\tName\tGender";
				for (int i=0;i<rSize;i++){
					rStateHeadLine+="\t"+title+i;
				}
				rStateBufferCompleted.appendln(rStateHeadLine);
				for (String rStateLine : rStateBuffer){
					String rStateLineCompleted = "";
					String[] rStateItems = rStateLine.split("\t");
					for (int i=0;i<3+rSize;i++){
						if (i<rStateItems.length){
							rStateLineCompleted+=rStateItems[i]+"\t";
						} else {
							rStateLineCompleted+="NA\t";
						}
					}
					rStateBufferCompleted.appendln(rStateLineCompleted);
				}
				
				if (rStateBuffer.length() != 0) {
					
					File targetFile = ToolBox.setExtension(ToolBox.addToName(new File(segmentation.getLabel()), "-States-" + title), ".txt");
					ReportRawData rawData = new ReportRawData("Export " + title + "s to RData (States)", "Text", "txt", targetFile);
					rawData.setData(PAJFile.convertToMicrosoftEndOfLine(rStateBufferCompleted.toString()));

					result.outputs().appendln();
					result.outputs().append(rawData);
				}
				
				StringList rEventBuffer = rEventBuffers.get(title);
				if (rEventBuffer.length() != 0) {
					
					File targetFile = ToolBox.setExtension(ToolBox.addToName(new File(segmentation.getLabel()), "-Events-" + title), ".txt");
					ReportRawData rawData = new ReportRawData("Export " + title + "s to RData (Events)", "Text", "txt", targetFile);
					rawData.setData(PAJFile.convertToMicrosoftEndOfLine(rEventBuffer.toString()));

					result.outputs().appendln();
					result.outputs().append(rawData);
				}
			}
		}
		

		// draw graph
		
		
		
		
		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;

	}
	
	public static Graph<Place> getTrajectoryGraph (final Segmentation segmentation, final SpaceTimeCriteria censusCriteria) throws PuckException {
		Graph<Place> result;
		
		result = null;

		// Create Partition charts and tables
		List<ReportChart> charts = new ArrayList<ReportChart>();
		List<ReportTable> tables = new ArrayList<ReportTable>();

		// Get (coherent) itineraries
		Sequences sequences = new Sequences();
		for (Individual ego : segmentation.getCurrentIndividuals().toSortedList()) {
			sequences.addRenumbered(SequenceWorker.getCoherentItinerarySegment(ego, censusCriteria));
		}

		// Proceed census
		SequencesCensus census = new SequencesCensus(sequences, censusCriteria);


		if (censusCriteria.getTrajectoriesOperations().contains(TrajectoriesOperation.DRAW)){
			
			CorrelationMatrix eventSequenceMatrix = census.getEventSequenceMatrix("PLACE");
			Graph<Cluster<String>> drawGraph = eventSequenceMatrix.getSequenceNetwork("PLACE", Gender.UNKNOWN);
			
			result = GeoNetworkUtils.createGeoNetwork(drawGraph, censusCriteria.getLevel());

		}
		//
		return result;


	}

    public static Graph<Place2> getTrajectoryGraph2 (Geography2 geography, final Segmentation segmentation, final SpaceTimeCriteria censusCriteria) throws PuckException {
        Graph<Place2> result;
        
        result = null;

        // Create Partition charts and tables
        List<ReportChart> charts = new ArrayList<ReportChart>();
        List<ReportTable> tables = new ArrayList<ReportTable>();

        // Get (coherent) itineraries
        Sequences sequences = new Sequences();
        for (Individual ego : segmentation.getCurrentIndividuals().toSortedList()) {
            sequences.addRenumbered(SequenceWorker.getCoherentItinerarySegment(ego, censusCriteria));
        }

        // Proceed census
        SequencesCensus census = new SequencesCensus(sequences, censusCriteria);


        if (censusCriteria.getTrajectoriesOperations().contains(TrajectoriesOperation.DRAW)){
            
            CorrelationMatrix eventSequenceMatrix = census.getEventSequenceMatrix("PLACE");
            Graph<Cluster<String>> drawGraph = eventSequenceMatrix.getSequenceNetwork("PLACE", Gender.UNKNOWN);
            
            Graph<String> placeNameGraph = GeoNetworkUtils.createGeoNetwork2(drawGraph, censusCriteria.getLevel());
            
            result = GeocodingWorker.geocodeGraph(geography, placeNameGraph);

        }
        //
        return result;


    }



	/**
	 * 
	 * @param segmentation
	 * @param censusType
	 * @return
	 * @throws PuckException
	 */
	public static Report reportSequenceCensus(final Segmentation segmentation, final CensusType censusType) throws PuckException {
		Report result;

		//
		Chronometer chrono = new Chronometer();

		result = new Report(censusType.toString());
		result.setOrigin("Sequence reporter");

		// Create Reports
		Report overallReport = new Report("Survey");
		Report diagramReport = new Report("Diagrams");
		Report detailReport = new Report("Details");
		Report componentReport = new Report("Components");
		Report treeReport = new Report("Trees");

		// Create Partition charts and tables
		List<ReportChart> charts = new ArrayList<ReportChart>();
		List<ReportTable> tables = new ArrayList<ReportTable>();

		// Set census criteria
		SpaceTimeCriteria censusCriteria = new SpaceTimeCriteria(censusType);

		// Get (coherent) itineraries
		Sequences sequences = new Sequences();
		for (Individual ego : segmentation.getCurrentIndividuals().toSortedList()) {
			sequences.addRenumbered(SequenceWorker.getCoherentItinerarySegment(ego, censusCriteria));
		}

		// Proceed census
		SequencesCensus census = new SequencesCensus(sequences, censusCriteria);

		// Make overall report and diagrams
		// Generalize for all types of partitions (not only male and female)
		overallReport
				.outputs()
				.appendln(
						"Measure\tAverage (Male)\tAverage Pos. (Male)\tMedian (Male)\tMaximum (Male)\tSum (Male)\tAverage (Female)\tAverage Pos. (Female)\tMedian (Female)\tMaximum (Female)\tSum (Female)\tAverage (All)\tAverage Pos. (All)\tMedian (All)\tMaximum (All)\tSum (All)");

		// Set partition criteria
		for (String label : censusCriteria.getCensusOperationLabels()) {

			PartitionCriteria partitionCriteria = new PartitionCriteria(label);
			// partitionCriteria.setValueFilter(ValueFilter.NULL);

			/*			if (label.contains("PROFILE")){
							partitionCriteria.setType(PartitionType.PARTIALIZATION);
						} */

			if (label.equals("NREVENTS")) {
				partitionCriteria.setType(PartitionType.RAW);
				// partitionCriteria.setCumulationType(CumulationType.DESCENDANT);
			} else if (label.contains("AGEFIRST")) {
				partitionCriteria.setType(PartitionType.SIZED_GROUPING);
				partitionCriteria.setStart(0.);
				partitionCriteria.setSize(5.);
			} else if (label.equals("ECCENTRICITY")) {
				partitionCriteria.setType(PartitionType.SIZED_GROUPING);
				partitionCriteria.setStart(-100.);
				partitionCriteria.setSize(20.);
			} else if (label.contains("COVERAGE") || label.contains("SAME") || label.contains("NORM") || label.contains("DENSITY")
					|| label.contains("BETWEENNESS") || label.contains("EFFICIENCY") || label.contains("CONCENTRATION")) {
				partitionCriteria.setType(PartitionType.SIZED_GROUPING);
				partitionCriteria.setStart(0.);
				partitionCriteria.setSize(20.);
			} else if (label.contains("MEAN") || label.contains("COVERAGE") || label.contains("PEREVENT") || label.contains("BETWEENNESS")
					|| label.contains("BROKERAGE") || label.contains("EFFICIENT_SIZE")) {
				partitionCriteria.setType(PartitionType.SIZED_GROUPING);
				partitionCriteria.setStart(0.);
				partitionCriteria.setSize(1.);
			} else {
				partitionCriteria.setType(PartitionType.RAW);
			}

			ReportChart chart = null;

			if (label.contains("SIMILARITY")) {
				RelationClassificationType relationClassificationType = RelationClassificationType.valueOf(label.substring(label.lastIndexOf("_") + 1));
				Map<Value, Double[]> similaritiesMap = census.getSimilaritiesMap(relationClassificationType);
				chart = StatisticsReporter.createMapChart(similaritiesMap, label, new String[] { "HH", "FH", "HF", "FF", "All" }, GraphType.LINES);

				for (Value key : similaritiesMap.keySet()) {
					overallReport.outputs().appendln(label + "_" + key + "\t" + MathUtils.percent(similaritiesMap.get(key)[4], 100));
				}

			} else if (!label.contains("ALTERS") && !label.contains("PROFILE")) {
				NumberedValues values = census.getValues(label);

				Partition<Individual> partition = PartitionMaker.create(label, segmentation.getCurrentIndividuals(), values, partitionCriteria);

				PartitionCriteria splitCriteria = new PartitionCriteria(censusCriteria.getPartitionLabel());
				chart = StatisticsReporter.createPartitionChart(partition, partitionCriteria, splitCriteria);

				if (label.substring(0, 3).equals("AGE")) {

					partitionCriteria.setType(PartitionType.RAW);
					partitionCriteria.setSizeFilter(SizeFilter.HOLES);
					partitionCriteria.setValueFilter(ValueFilter.NULL);

					partition = PartitionMaker.create(label, segmentation.getCurrentIndividuals(), values, partitionCriteria);

					if (partition.maxValue() != null) {
						ReportChart survivalChart = StatisticsReporter.createSurvivalChart(partition, splitCriteria);
						charts.add(survivalChart);
					} else {
						System.err.println(label + " no max value");
					}
				}

				NumberedValues[] genderedValues = PuckUtils.getGenderedNumberedValues(values, segmentation.getCurrentIndividuals());

				overallReport.outputs().append(label + "\t");
				for (int gender = 0; gender < 3; gender++) {
					String sum = "";
					if (label.startsWith("NR")) {
						sum = new Double(genderedValues[gender].sum()).intValue() + "";
					}
					overallReport.outputs().append(
							MathUtils.round(genderedValues[gender].average(), 2) + "\t" + MathUtils.round(genderedValues[gender].averagePositives(), 2) + "\t"
									+ values.median() + "\t" + genderedValues[gender].max() + "\t" + sum + "\t");
				}
				overallReport.outputs().appendln();

			}

			if (chart != null) {
				charts.add(chart);
				if (label.equals("SIMILARITY")) {
					tables.add(ReportTable.transpose(chart.createReportTable()));
				} else {
					ReportTable table = ReportTable.transpose(chart.createReportTableWithSum());
					tables.add(table);
					if (!label.contains("EVENTS_") && !label.contains("RELATIONS")) {
						tables.add(ReportTable.normalize(table));
					}
				}

			}

		}
		overallReport.outputs().appendln();
		
		// Add parcours network statistics
		
		Map<String,Map<String,Map<String,Value>>> parcoursNetworkStatistics = census.getParcoursNetworkStatistics();
		
		if (parcoursNetworkStatistics!=null){
			overallReport.outputs().appendln("Parcours Network Statistics");
			
			for (String nodeLabel : parcoursNetworkStatistics.get("Total").keySet()){
				String line = nodeLabel;
				for (String statisticsLabel : parcoursNetworkStatistics.get("Total").get(nodeLabel).keySet()){
					for (String genderLabel : parcoursNetworkStatistics.keySet()){
						line+="\t"+parcoursNetworkStatistics.get(genderLabel).get(nodeLabel).get(statisticsLabel);
					}
				}
				overallReport.outputs().appendln(line);
			}
		}
		
		// Create detailed report
		detailReport.outputs().append("Nr\tEgo\tGender");
		for (String partitionLabel : censusCriteria.getCensusOperationLabels()) {
			if (partitionLabel.contains("SIMILARITY")) {
				RelationClassificationType relationClassificationType = RelationClassificationType.valueOf(partitionLabel.substring(partitionLabel
						.lastIndexOf("_") + 1));
				detailReport.outputs().append(
						"\tSIMILARITY_PARENT_" + relationClassificationType + "\tSIMILARITY_CHILD_" + relationClassificationType + "\tSIMILARITY_SIBLING_"
								+ relationClassificationType + "\tSIMILARITY_SPOUSE_" + relationClassificationType);
			} else {
				detailReport.outputs().append("\t" + partitionLabel);
			}
		}

		Map<Value, Double[]> componentChartMap = new TreeMap<Value, Double[]>();

		detailReport.outputs().appendln();
		for (Individual ego : segmentation.getCurrentIndividuals().toSortedList()) {

			if ((((censusType == CensusType.GENERAL) || (censusType == CensusType.PARCOURS) || (censusType == CensusType.PARCOURSNETWORKS)) && (census
					.getValues("NREVENTS").get(ego.getId()) != null))
					|| ((censusType == CensusType.EGONETWORKS) && (census.getValues("SIZE").get(ego.getId()) != null))) {
				detailReport.outputs().append(ego.getId() + "\t" + ego + "\t" + ego.getGender());
				for (String label : censusCriteria.getCensusOperationLabels()) {
					if (label.contains("SIMILARITY")) {
						Value value = census.getValues(label).get(ego.getId());
						Map<Value, Double[]> indiSimilaritiesMap = value.mapValue();
						String[] keys = new String[] { "PARENT", "CHILD", "SIBLING", "SPOUSE" };
						for (String key : keys) {
							Double[] sim = indiSimilaritiesMap.get(new Value(key));
							if (sim != null) {
								detailReport.outputs().append("\t" + MathUtils.round(sim[4], 2));
							}
						}
					} else {
						detailReport.outputs().append("\t" + census.getValues(label).get(ego.getId()));
					}
				}
				detailReport.outputs().appendln();
			}

			if ((censusType == CensusType.EGONETWORKS || censusType == CensusType.PARCOURSNETWORKS)) {

				for (String networkTitle : censusCriteria.getNetworkTitles()) {
					Map<Integer, Partition<Node<Individual>>> componentsMap = census.getComponents(networkTitle);
					if (componentsMap != null) {
						Partition<Node<Individual>> components = componentsMap.get(ego.getId());

						componentReport.outputs().appendln("Components " + networkTitle);
						componentReport.outputs().appendln(ego + "\t" + components.size());
						int i = 1;
						for (Cluster<Node<Individual>> cluster : components.getClusters().toListSortedByValue()) {
							componentReport.outputs().appendln(
									"\t" + i + "\t" + cluster.getValue() + "\t(" + cluster.size() + ")\t" + cluster.getItemsAsString());
							i++;
						}
						componentReport.outputs().appendln();

						for (Value value : components.getValues()) {
							Double[] map = componentChartMap.get(value);
							if (map == null) {
								map = new Double[]{0.,0.,0.};
								componentChartMap.put(value, map);
							}
							map[ego.getGender().toInt()]+=+ 1.;
							map[2]+=+ 1.;
						}
					}
				}
			}
		}

		if ((censusType == CensusType.EGONETWORKS || censusType == CensusType.PARCOURSNETWORKS)) {
			ReportChart componentChart = StatisticsReporter.createMapChart(componentChartMap,"COMPONENTS",new String[]{"MALE","FEMALE"}, GraphType.STACKED_BARS);
			charts.add(componentChart);
			tables.add(ReportTable.transpose(componentChart.createReportTableWithSum()));

			if (census.getRelationConnectionMatrix() != null) {
				for (ReportChart chart : census.getRelationConnectionMatrix().getCharts()) {
					charts.add(chart);
				}
				tables.add(census.getRelationConnectionMatrix().getTable("Component Connections"));
			}

		}

		// SequenceAnalysis

		if (censusType == CensusType.PARCOURS) {

			for (RelationClassificationType relationClassificationType : censusCriteria.getMainRelationClassificationTypes()) {

				if (censusCriteria.getNetworkTitles().contains("Event Type Network")) {

					CorrelationMatrix eventSequenceMatrix = census.getEventSequenceMatrix(relationClassificationType.toString());

					if (eventSequenceMatrix != null) {
						for (ReportChart chart : eventSequenceMatrix.getCharts()) {
							charts.add(chart);
						}
						tables.add(eventSequenceMatrix.getTable("Event Type Sequences"));
					}

					overallReport.outputs().appendln();
					overallReport.outputs().appendln("Sequence Network Statistics " + relationClassificationType);
					overallReport.outputs().appendln("\tDensity\tInertia\t(Divergence)\tConcentration\t(Divergence)\tSymmetry\t(Divergence)\tCentral nodes");

					for (Gender gender : Gender.values()) {
						GraphProfile<Cluster<String>> profile = eventSequenceMatrix.getProfile(gender);

						String centralReferents = "";
						for (Cluster<String> centralReferent : profile.getCentralReferents()) {
							centralReferents += centralReferent.getValue() + " ";
						}
						double maxBetweenness = profile.getMaxBetweenness();
						double density = profile.density();
						double endo = MathUtils.round(profile.getStatistics(Indicator.LOOPS, Mode.NORMALIZED), 2);
						double endoExp = MathUtils.round(profile.getStatistics(Indicator.LOOPS, Mode.DIVERGENCE_NORMALIZED), 2);
						double conc = MathUtils.round(profile.getStatistics(Indicator.CONCENTRATION, Mode.SIMPLE), 2);
						double concExp = MathUtils.round(profile.getStatistics(Indicator.CONCENTRATION, Mode.DIVERGENCE), 2);
						double sym = MathUtils.round(profile.getStatistics(Indicator.SYMMETRY, Mode.SIMPLE), 2);
						double symExp = MathUtils.round(profile.getStatistics(Indicator.SYMMETRY, Mode.DIVERGENCE), 2);

						overallReport.outputs().appendln(
								gender + "\t" + density + "\t" + endo + "\t" + endoExp + "\t" + conc + "\t" + concExp + "\t" + sym + "\t" + symExp + "\t"
										+ centralReferents + "(" + maxBetweenness + ") betweenness centrality");
					}
					overallReport.outputs().appendln();

				}

				if (censusCriteria.getNetworkTitles().contains("Sequence Type Network")) {

					CorrelationMatrix subSequenceMatrix = census.getSubSequenceMatrix(relationClassificationType.toString());

					if (subSequenceMatrix != null) {
						charts.add(subSequenceMatrix.getRamificationChart());
					}
				}

				// reportSequencePartition(overallReport,"",census.getEventPartition(eventTypeName),census.getEventPairPartition(eventTypeName),census.nrSequences(),census.nrEvents());
				// reportSequenceTree(treeReport,"",census.getSequenceTypeNetwork(eventTypeName));

				/*				int maxPositions = census.getNrValues(eventType,eventType.toString());
										
								ReportChart diversityChart = StatisticsReporter.createDiversityChart(census.getDepthPartition(eventType.toString()), maxPositions);
								charts.add(diversityChart);*/
			}
		}

		// Manage the number of chart by line.
		for (int chartIndex = 0; chartIndex < charts.size(); chartIndex++) {
			diagramReport.outputs().append(charts.get(chartIndex));
			if (chartIndex % 4 == 3) {
				diagramReport.outputs().appendln();
			}
		}

		// Add chart tables.
		for (ReportTable table : tables) {
			diagramReport.outputs().appendln(table.getTitle());
			diagramReport.outputs().appendln(table);
		}

		// Finalize reports
		result.outputs().append(overallReport);
		result.outputs().append(diagramReport);
		result.outputs().append(detailReport);

		if (censusType == CensusType.EGONETWORKS || censusType == CensusType.PARCOURSNETWORKS) {
			result.outputs().append(componentReport);
		}
		if (censusType == CensusType.PARCOURS) {
			result.outputs().append(treeReport);
		}

		// addPajekData

		Map<String, StringList> pajekBuffers = census.getPajekBuffers();

		for (String title : pajekBuffers.keySet()) {

			StringList pajekBuffer = pajekBuffers.get(title);
			if (pajekBuffer.length() != 0) {
				File targetFile = ToolBox.setExtension(ToolBox.addToName(new File(segmentation.getLabel()), "-" + title), ".paj");
				ReportRawData rawData = new ReportRawData("Export " + title + "s to Pajek", "Pajek", "paj", targetFile);
				rawData.setData(PAJFile.convertToMicrosoftEndOfLine(pajekBuffer.toString()));

				result.outputs().appendln();
				result.outputs().append(rawData);
			}
		}

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;

	}


	/**
	 * 
	 * @param net
	 * @param segmentation
	 * @param censusCriteria
	 * @return
	 * @throws PuckException
	 */
	public static Report reportSequences(final Net net, final Segmentation segmentation, final SpaceTimeCriteria censusCriteria) throws PuckException {
		Report result;

		String spouseFilterLabel = "INTERV";

		//
		Chronometer chrono = new Chronometer();

		result = new Report("Sequence Report");
		result.setOrigin("Sequence reporter");

		// Initialize reports
		
		Report surveyReport = null;
		Report detailedReport = null;
		Report actorEventTableReport = null;
		Report interactionTableReport = null;
		Report biographyReport = null;
		Report extendedBiographyReport = null;
		
		List<Report> reports = new ArrayList<Report>();

		if (censusCriteria.getSequenceReportTypes().contains(SequenceReportType.ITINERARIES_SURVEY)){
			surveyReport = new Report("Survey");
			reports.add(surveyReport);
		}
		
		if (censusCriteria.getSequenceReportTypes().contains(SequenceReportType.ITINERARIES_DETAILS)){
			detailedReport = new Report("Details");
			reports.add(detailedReport);
		}
		
		if (censusCriteria.getSequenceReportTypes().contains(SequenceReportType.ACTOR_EVENT_TABLES)){
			actorEventTableReport = new Report("Actor-Event tables");
			reports.add(actorEventTableReport);
		}
		
		if (censusCriteria.getSequenceReportTypes().contains(SequenceReportType.INTERACTION_TABLES)){
			interactionTableReport = new Report("Interaction tables");
			reports.add(interactionTableReport);
		}

		// Initialize geography
		
		Geography geography = censusCriteria.getGeography();
		
		// Create sequences

		Sequences sequences = new Sequences();

		for (Individual ego : segmentation.getCurrentIndividuals().toSortedList()) {

			if (surveyReport!=null){
				surveyReport.outputs().appendln(ego.signature());
				surveyReport.outputs().appendln();
			}
			
			if (detailedReport!=null){
				detailedReport.outputs().appendln(ego.signature());
				detailedReport.outputs().appendln();
			}

			Sequence sequence = SequenceMaker.createPersonalSequence(ego, censusCriteria);
			sequences.add(sequence);

			// Move to appropriate census!
			for (Individual spouse : ego.spouses()) {
				if (spouse.getAttributeValue(spouseFilterLabel) != null) {
					// surveyReport.outputs().appendln("Common events with spouse ("+spouse+"): "+SequenceWorker.getCommonEvents(ego,spouse,censusCriteria.getRelationModelName()).size());
				}
			}

			// Add role tables

			if (actorEventTableReport!=null){
				actorEventTableReport.outputs().append(sequence.roleTable());
			}
			
			if (interactionTableReport!=null){
				interactionTableReport.outputs().append(sequence.interactionTable());
			}

			// Create coherent subsequences

			Sequences subSequences = SequenceWorker.split(sequence);

			for (Sequence subSequence : subSequences) {

				if (subSequences.size() > 1) {
					if (surveyReport!=null){
						surveyReport.outputs().appendln(subSequence.getId());
					}
					if (detailedReport!=null){
						detailedReport.outputs().appendln(subSequence.getId());
					}
				}

				for (Ordinal key : subSequence.getEvents().keySet()) {

					Relation event = subSequence.getEvents().get(key);
					
					Place start = geography.getByHomonym(event.getAttributeValue("START_PLACE"));
					Place end = geography.getByHomonym(event.getAttributeValue("END_PLACE"));
					Place ancestor = geography.getCommonAncestor(start, end);
					GeoLevel commonLevel = null;
					String commonPlaceName = null;
					String startPlaceLabel = censusCriteria.getStartPlaceLabel();
					String endPlaceLabel = censusCriteria.getEndPlaceLabel();
					
					
					if (ancestor != null) {
						commonLevel = ancestor.getLevel();
						commonPlaceName = ancestor.getName();
					}

					String order = SequenceWorker.order(event, ego);

					if (surveyReport!=null){
						surveyReport.outputs().appendln(
								key + "\t" + order + "\t"+key.getYear()+"\t(" + subSequence.getAge(key.getYear()) + ")\t" + event.getTypedId() + "\t"
										+ event.getAttributeValue(startPlaceLabel) + "\t" + event.getAttributeValue(endPlaceLabel) + "\t" + commonLevel + "\t("
										+ commonPlaceName + ")\t");
					}
					
					if (detailedReport!=null){
						detailedReport.outputs().appendln(
								key + "\t" + order  + "\t"+key.getYear()+ "\t(" + subSequence.getAge(key.getYear()) + ")\t" + event.getTypedId() + "\t"
										+ event.getAttributeValue(startPlaceLabel) + "\t" + event.getAttributeValue(endPlaceLabel) + "\t" + commonLevel + "\t("
										+ commonPlaceName + ")\t");
						detailedReport.outputs().appendln();
						detailedReport.outputs().appendln(getStories(event));
					}

				}
				
				for (Report report : reports){
					report.outputs().appendln();
				}
			}
		}

		// Create sequences including life events

		if (censusCriteria.getSequenceReportTypes().contains(SequenceReportType.BIOGRAPHIES)){
			
			biographyReport = new Report("Biographies");
			reports.add(biographyReport);

			for (Sequence biography : SequenceMaker.createBiographies(net, segmentation, censusCriteria).toSortedList()) {

				Individual ego = segmentation.getCurrentIndividuals().getById(biography.getId());

				biographyReport.outputs().appendln(ego.signature());
				for (Ordinal key : biography.getEvents().keySet()) {
					Relation event = biography.getEvent(key);
					biographyReport.outputs().appendln(
							key + "\t" + event.getRoles(ego).toString() + " (" + biography.getAge(key.getYear()) + ")\t" + event.getName() + "\t"
									+ SequenceCensus.getEgoRolePartners(event, censusCriteria,ego));
				}
				biographyReport.outputs().appendln();
			}
		}
		
		// Create extended biographies

		if (censusCriteria.getSequenceReportTypes().contains(SequenceReportType.EXTENDED_BIOGRAPHIES)){

			extendedBiographyReport = new Report("Extended biographies");
			reports.add(extendedBiographyReport);

			for (Sequence extendedBiography : SequenceMaker.createExtendedBiographies(net, segmentation, censusCriteria).toSortedList()) {

				Individual ego = segmentation.getCurrentIndividuals().getById(extendedBiography.getId());

				extendedBiographyReport.outputs().appendln(ego.signature());
				for (Ordinal key : extendedBiography.getEvents().keySet()) {
					Relation event = extendedBiography.getEvent(key);
					extendedBiographyReport.outputs().appendln(
							key + "\t" + event.getRoles(ego).toString() + " (" + extendedBiography.getAge(key.getYear()) + ")\t" + event.getName() + " "
									+ SequenceCensus.getEgoRolePartners(event, censusCriteria,ego));
				}
				extendedBiographyReport.outputs().appendln();
			}
		}
		
		// Create Pajek File

		if (censusCriteria.getSequenceReportTypes().contains(SequenceReportType.EXPORT_RELATION_GRAPH)){
			
			Graph<Individual> graph = NetUtils.createRelationGraph(segmentation, censusCriteria.getRelationModelName());
			List<String> partitionLabels = new ArrayList<String>();

			StringList pajekBuffer = new StringList();
			pajekBuffer.addAll(PuckUtils.writePajekNetwork(graph, partitionLabels));
			pajekBuffer.appendln();


			if (pajekBuffer.length() != 0) {
				File targetFile = ToolBox.setExtension(ToolBox.addToName(new File(segmentation.getLabel()), "-Relation Network"), ".paj");
				ReportRawData rawData = new ReportRawData("Export Relation Network to Pajek", "Pajek", "paj", targetFile);
				rawData.setData(PAJFile.convertToMicrosoftEndOfLine(pajekBuffer.toString()));

				result.outputs().appendln();
				result.outputs().append(rawData);
			}
		}

		for (Report report : reports){
			result.outputs().append(report);
		}

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;
	}

	/**
	 * 
	 * @param report
	 * @param value
	 * @param sequenceTypeNetwork
	 */
	private static void reportSequenceTree(final Report report, final String value, final Graph<Cluster<String>> sequenceTypeNetwork) {

		report.outputs().appendln("Sequence Tree " + value);
		Node<Cluster<String>> start = sequenceTypeNetwork.getNode(1);
		Stack<Node<Cluster<String>>> stack = new Stack<Node<Cluster<String>>>();
		report.outputs().appendln(start.getReferent() + "\t" + start.getReferent().size());
		stack.push(start);

		while (!stack.isEmpty()) {
			Node<Cluster<String>> node = stack.pop();
			for (Node<Cluster<String>> next : node.getOutNodes().toListSortedByLabel()) {
				report.outputs().appendln(next.getReferent() + "\t" + next.getReferent().size());
				stack.push(next);
			}
		}
		report.outputs().appendln();

	}


	/**
	 * 
	 * @param report
	 * @param segmentation
	 * @throws PuckException
	 */
	private static void reportTriangles(final Report report, final Segmentation segmentation) throws PuckException {

		Map<Integer, Partition<EventTriangle>> trianglesMap = new HashMap<Integer, Partition<EventTriangle>>();
		Partition<EventTriangle> allTriangles = new Partition<EventTriangle>();
		Partition<Individual> triangleTypes = new Partition<Individual>();

		// Get triangles

		for (Individual ego : segmentation.getCurrentIndividuals()) {
			Individuals individuals = ego.getRelated("Migevent");
			individuals.add(ego);
			Partition<EventTriangle> triangles = SequenceWorker.getTriangles(individuals, "Migevent");
			allTriangles.add(triangles);
			trianglesMap.put(ego.getId(), triangles);
			for (Cluster<EventTriangle> cluster : triangles.getClusters()) {
				triangleTypes.put(ego, cluster.getValue());
			}
		}

		// Report triangles
		Report trianglesReport = new Report("Triangles");
		trianglesReport.outputs().appendln("Type\tnrTriangles\tnrEgoNetworks");

		for (Cluster<EventTriangle> cluster : allTriangles.getClusters().toListSortedByValue()) {
			trianglesReport.outputs().appendln(cluster.getValue() + "\t" + cluster.size() + "\t" + triangleTypes.getCluster(cluster.getValue()).size());
		}
		trianglesReport.outputs().appendln();

		PartitionCriteria partitionCriteria = new PartitionCriteria("Triangles");
		ReportChart chart5 = StatisticsReporter.createPartitionChart(allTriangles, partitionCriteria, null);
		trianglesReport.outputs().appendln(chart5);

		for (Individual ego : segmentation.getCurrentIndividuals().toSortedList()) {
			Partition<EventTriangle> triangles = trianglesMap.get(ego.getId());
			trianglesReport.outputs().appendln(ego + "\t" + triangles.size() + " types");
			for (Cluster<EventTriangle> cluster : triangles.getClusters().toListSortedByValue()) {
				trianglesReport.outputs().appendln(cluster.getValue() + "\t" + cluster.size());
				for (EventTriangle triangle : cluster.getItems()) {
					trianglesReport.outputs().appendln(triangle.getEventPattern());
				}
				trianglesReport.outputs().appendln();
			}
			trianglesReport.outputs().appendln();
		}
	}

	// Should be placed into control reporter (not specific for sequences)
	/**
	 * @param segmentation
	 * @param bundle
	 * @return
	 */
	public static Report reportUnknownPlaces(final Segmentation segmentation, final ResourceBundle bundle) {
		Report result;

		UnknownPlacesCriteria criteria = new UnknownPlacesCriteria();
		criteria.setIncludedIndividual(true);
		criteria.setIncludedAllRelations(true);

		result = reportUnknownPlaces(segmentation, criteria, bundle);

		//
		return result;
	}
	

	
	// Should be placed into control reporter (not specific for sequences)
	/**
	 * @param segmentation
	 * @param bundle
	 * @return
	 */
	public static Report reportUncodedPlaces(final Segmentation segmentation, final ResourceBundle bundle) {
		Report result;

		UnknownPlacesCriteria criteria = new UnknownPlacesCriteria();
		criteria.setIncludedIndividual(true);
		criteria.setIncludedAllRelations(true);

		result = reportUncodedPlaces(segmentation, criteria, bundle);

		//
		return result;
	}

	/**
	 * @param segmentation
	 * @param bundle
	 * @return
	 */
	public static Report reportUnknownPlaces(final Segmentation segmentation, final UnknownPlacesCriteria criteria, final ResourceBundle bundle) {
		Report result;

		//
		Chronometer chrono = new Chronometer();

		Geography geography = Geography.getInstance();

		Partition<String> unknownPlaces = new Partition<String>();

		int errorCount = 0;
		StringList errors = new StringList();

		//
		if (criteria.isIncludedIndividual()) {
			//
			for (Individual individual : segmentation.getCurrentIndividuals()) {
				for (Attribute attribute : individual.attributes()) {
					if (attribute.getLabel().contains("PLAC")) {
						if (geography.getByHomonym(attribute.getValue()) == null) {
							unknownPlaces.put(attribute.getValue(), new Value(attribute.getLabel()));
							errorCount++;
						}
					}
				}
			}
		}

		//
		Relations relations;
		if (criteria.isIncludedAllRelations()) {
			//
			relations = segmentation.getCurrentRelations();

		} else if (criteria.getRelationNames().isEmpty()) {
			//
			relations = new Relations();

		} else {
			//
			relations = segmentation.getCurrentRelations().getByModelNames(criteria.getRelationNames());
		}

		for (Relation relation : relations) {
			for (Attribute attribute : relation.attributes()) {
				if (attribute.getLabel().contains("PLAC")) {
					if (geography.getByHomonym(attribute.getValue()) == null) {
						unknownPlaces.put(attribute.getValue(), new Value(attribute.getLabel()));
						errorCount++;
					}
				}
			}
		}

		for (Cluster<String> cluster : unknownPlaces.getClusters()) {
			errors.appendln(cluster.getValue().toString());
			for (String placeName : cluster.getItems()) {
				errors.appendln("\t" + placeName);
			}
			errors.appendln();
		}

		//
		result = new Report();
		result.setTitle("Unknown Places");
		result.setOrigin("Control reporter");
		result.setTarget(segmentation.getLabel());

		//
		errors.add(0, errorCount + " " + Report.translate(bundle, "Unknown Places") + "\n");
		result.outputs().append(errors.toString());

		//
		result.setStatus(errorCount);

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;
	}

	/**
	 * @param segmentation
	 * @param bundle
	 * @return
	 */
	public static Report reportUncodedPlaces(final Segmentation segmentation, final UnknownPlacesCriteria criteria, final ResourceBundle bundle) {
		Report result;

		//
		Chronometer chrono = new Chronometer();

		Geography geography = Geography.getInstance();
		
		Partition<Place> uncodedPlaces = new Partition<Place>();

		int errorCount = 0;
		StringList errors = new StringList();

		//
		if (criteria.isIncludedIndividual()) {
			//
			for (Individual individual : segmentation.getCurrentIndividuals()) {
				for (Attribute attribute : individual.attributes()) {
					if (attribute.getLabel().contains("PLAC")) {
						Place place = geography.getByHomonym(attribute.getValue());
						if (place != null && place.getCoordinate()==null) {
							uncodedPlaces.put(place,new Value(place.getLevel()));
							Place sup = place.getSup();
							while (sup.getCoordinate()==null){
								uncodedPlaces.put(sup, new Value (sup.getLevel()));
								sup = sup.getSup();
							}
						}
					}
				}
			}
		}

		//
		Relations relations;
		if (criteria.isIncludedAllRelations()) {
			//
			relations = segmentation.getCurrentRelations();

		} else if (criteria.getRelationNames().isEmpty()) {
			//
			relations = new Relations();

		} else {
			//
			relations = segmentation.getCurrentRelations().getByModelNames(criteria.getRelationNames());
		}

		for (Relation relation : relations) {
			for (Attribute attribute : relation.attributes()) {
				if (attribute.getLabel().contains("PLAC")) {
					Place place = geography.getByHomonym(attribute.getValue());
					if (place != null && place.getCoordinate()==null) {
						uncodedPlaces.put(place,new Value(place.getLevel()));
						Place sup = place.getSup();
						while (sup.getCoordinate()==null){
							uncodedPlaces.put(sup, new Value (sup.getLevel()));
							sup = sup.getSup();
						}
					}
				}
			}
		}
				

		for (Cluster<Place> cluster : uncodedPlaces.getClusters().toListSortedByValue()) {
			errors.appendln(cluster.getValue().toString()+" "+cluster.size());
			errorCount+= cluster.size();
			List<String> toponyms = new ArrayList<String>();
			for (Place place : cluster.getItems()){
				toponyms.add(place.getToponym());
			}
			Collections.sort(toponyms);
			for (String toponym : toponyms){
				errors.appendln("\t" + toponym);
			}
			errors.appendln();
		}

		//
		result = new Report();
		result.setTitle("Uncoded Places");
		result.setOrigin("Control reporter");
		result.setTarget(segmentation.getLabel());

		//
		errors.add(0, errorCount + " " + Report.translate(bundle, "Uncoded Places") + "\n");
		result.outputs().append(errors.toString());

		//
		result.setStatus(errorCount);

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;
	}
	/**
	 * 
	 * @param cluster
	 * @return
	 */
	private static int sequenceNumber(final Cluster<String> cluster) {
		int result;

		Set<String> set = new HashSet<String>();

		for (String string : cluster.getItems()) {
			set.add(string.split("\\s")[0]);
		}

		result = set.size();

		//
		return result;
	}

}
