package org.tip.puck.net.relations.workers;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.tip.puck.PuckException;
import org.tip.puck.census.chains.Chain;
import org.tip.puck.census.workers.CensusCriteria;
import org.tip.puck.census.workers.CircuitFinder;
import org.tip.puck.graphs.Graph;
import org.tip.puck.graphs.Node;
import org.tip.puck.graphs.workers.GraphUtils;
import org.tip.puck.net.Individual;
import org.tip.puck.net.Individuals;
import org.tip.puck.net.Net;
import org.tip.puck.net.relations.Actor;
import org.tip.puck.net.relations.Actors;
import org.tip.puck.net.relations.Relation;
import org.tip.puck.net.relations.RelationModel;
import org.tip.puck.net.relations.Relations;
import org.tip.puck.net.relations.Role;
import org.tip.puck.net.workers.IndividualValuator;
import org.tip.puck.net.workers.NetUtils;
import org.tip.puck.partitions.Cluster;
import org.tip.puck.partitions.Partition;
import org.tip.puck.segmentation.Segmentation;
import org.tip.puck.spacetime.workers.SequenceCensus;
import org.tip.puck.util.MathUtils;
import org.tip.puck.util.ToolBox;

import fr.devinsy.util.StringList;

/**
 * @author Klaus Hamberger
 * @author TIP
 */
public class RelationWorker {

	/**
	 * 
	 * @param net
	 * @param criteria
	 */
	public static void addChildRoles(final Net net, final AddChildRolesCriteria criteria) {
		//
		if ((net != null) && (criteria != null)) {
			//
			RelationModel model = net.relationModels().getByName(criteria.getRelationModelName());
			String egoRoleName = criteria.getEgoRoleName();
			StringList roleNames = criteria.getAlterRoleNames();
			int maxAge = criteria.getMaxAge();
			String dateLabel = criteria.getDateLabel();

			if (model != null) {

				model.roles().add(new Role(egoRoleName + "_CHILD", 0));
				Relations relations = net.relations().getByModel(model);

				for (Relation event : relations) {
					if (!SequenceCensus.isBirth(event) && !SequenceCensus.isDeath(event)) {
						Integer eventYear = IndividualValuator.extractYearAsInt(event.getAttributeValue(dateLabel));
						if (eventYear != null) {
							for (Actor actor : event.actors().toList()) {
								if (actor.getRole().getName().equals(egoRoleName)) {
									Individual ego = actor.getIndividual();
									int age = IndividualValuator.ageAtYear(ego, eventYear);
									if (age > -1 && age < maxAge && event.getRoleNames(ego).contains(egoRoleName) && !SequenceCensus.isBirth(event)
											&& !SequenceCensus.isDeath(event)) {
										for (String alterRoleName : roleNames) {
											if (!event.getIndividuals(alterRoleName).contains(ego.getFather())
													&& !event.getIndividuals(alterRoleName).contains(ego.getMother())) {
												event.actors().add(new Actor(ego, new Role(egoRoleName + "_CHILD")));
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}

	/**
	 * 
	 * @param ancestorList
	 * @param referent
	 * @param ascendants
	 * @param relation
	 * @param treeType
	 * @param maxDegrees
	 * @return
	 */
	private static Set<List<String>> getLinkChains(final List<String> ancestorList, final Actor referent, final Actors ascendants, final Relation relation,
			final String treeType, final int[] maxDegrees) {
		Set<List<String>> result;

		result = new HashSet<List<String>>();

		Actors dependants = relation.getDependants(referent);
		ascendants.add(referent);

		if (dependants.size() > 0) {
			for (Actor descendant : dependants) {
				if (ascendants.contains(descendant)) {
					System.err.println("Cyclic referent structure : " + referent + " for " + descendant + " " + referent + " in " + relation);
				} else {
					List<String> list = new ArrayList<String>();
					list.addAll(ancestorList);

					Individual individual = descendant.getIndividual();
					if (treeType.equals("GENDER")) {
						list.add(individual.getGender().toChar() + "");
					} else if (treeType.equals("ID")) {
						list.add(individual.getId() + "");
					} else if (treeType.equals("KIN")) {
						list.add(individual.getGender().toChar() + "");
					}

					for (List<String> chain : getLinkChains(list, descendant, ascendants, relation, treeType, maxDegrees)) {
						result.add(chain);
					}
				}
			}
		} else {
			result.add(ancestorList);
		}

		//
		return result;
	}

	/**
	 * 
	 * @param relation
	 * @param treeType
	 * @param pattern
	 * @return
	 */
	public static Set<List<String>> getLinkChains(final Relation relation, final String treeType, final String pattern) {
		Set<List<String>> result;

		result = new HashSet<List<String>>();

		int[] maxDegrees = ToolBox.stringsToInts(pattern);

		for (Actor actor : relation.actors()) {
			Individual referent = actor.getReferent();
			if (referent == null) {
				List<String> ancestorList = new ArrayList<String>();
				Individual individual = actor.getIndividual();
				if (treeType.equals("GENDER")) {
					ancestorList.add(individual.getGender().toChar() + "");
				} else if (treeType.equals("ID")) {
					ancestorList.add(individual.getId() + "");
				} else if (treeType.equals("KIN")) {
					ancestorList.add(individual.getGender().toChar() + "");
				}

				for (List<String> chain : getLinkChains(ancestorList, actor, new Actors(), relation, treeType, maxDegrees)) {
					result.add(chain);
				}
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @param relation
	 * @param treeType
	 * @param pattern
	 * @return
	 */
	public static String getLinkChainsAsString(final Relation relation, final String treeType, final String pattern) {
		String result;

		result = "";

		Set<List<String>> chains = getLinkChains(relation, treeType, pattern);
		List<String> chainsAsStrings = new ArrayList<String>();
		for (List<String> chain : chains) {
			chainsAsStrings.add(chain.toString());
		}
		Collections.sort(chainsAsStrings);

		result = chainsAsStrings.toString();
		//
		return result;
	}

	/**
	 * 
	 * @param referent
	 * @param ascendants
	 * @param relation
	 * @param reduced
	 * @param treeType
	 * @param maxDegrees
	 * @return
	 */
	private static String getLinkTree(final Actor referent, final Individuals ascendants, final Relation relation, final boolean reduced, final String treeType,
			final int[] maxDegrees) {
		String result;

		result = "";

		Actors dependants = relation.getDependants(referent);
		ascendants.add(referent.getIndividual());
		
		if (dependants.size() > 0) {
			result = "[";
			List<String> list = new ArrayList<String>();
			for (Actor descendant : dependants.toSortedList()) {
				if (ascendants.contains(descendant.getIndividual())) {
					System.err.println("Cyclic referent structure : " + referent + " for " + descendant + " " + referent + " in " + relation);
				} else {
					String item = getLinkTree(descendant, ascendants, relation, reduced, treeType, maxDegrees);
					if (treeType.equals("GENDER")) {
						item = descendant.getIndividual().getGender().toChar() + item;
					} else if (treeType.equals("ID")) {
						item = descendant.getIndividual().getId() + " " + item;
					} else if (treeType.equals("KIN")) {
						item = NetUtils.getAlterRole(descendant.getIndividual(), referent.getIndividual(), maxDegrees, null) + " " + item;
					}
					if (!reduced || !list.contains(item)) {
						list.add(item);
					}
				}
			}
//			Collections.sort(list);
			for (String item : list) {
				result += item;
			}
			result += "]";
		}
		//
		return result;
	}

	/**
	 * 
	 * @param relation
	 * @param reduced
	 * @param treeType
	 * @param pattern
	 * @return
	 */
	public static String getLinkTrees(final Relation relation, final boolean reduced, final String treeType, final String pattern) {
		String result;

		result = "";

		int[] maxDegrees = ToolBox.stringsToInts(pattern);

		List<String> list = new ArrayList<String>();
		
		if (relation != null){
			for (Actor actor : relation.actors().toSortedList()) {
				Individual individual = actor.getIndividual();
				Individual referent = actor.getReferent();
				if (referent == null) {
					String item = getLinkTree(actor, new Individuals(), relation, reduced, treeType, maxDegrees);
					if (treeType.equals("GENDER")) {
						item = individual.getGender().toChar() + item;
					} else if (treeType.equals("ID")) {
						item = individual.getId() + " " + item;
					} else if (treeType.equals("KIN")) {
						item = individual.getGender().toChar() + " " + item;
					}
					if (!reduced || !list.contains(item)) {
						list.add(item);
					}
				}
			}
		}
//		Collections.sort(list);
		for (String item : list) {
			result += item + " ";
		}

		//
		return result;
	}

	/**
	 * 
	 * @param relation
	 * @return
	 */
	public static Graph<Individual> getReferentGraph(final Relation relation) {
		Graph<Individual> result;

		result = new Graph<Individual>();
		
		if (relation!=null){
			
			for (Actor actor : relation.actors()) {
				Individual ego = actor.getIndividual();
				Individual alter = actor.getReferent();
				if (ego!=null && alter!=null){
					result.addArc(ego, alter,1);
				} else if (ego!=null){
					result.addNode(ego);
				}
			}
			result.setLabel(relation.getName());

		}
		
		//
		return result;
	}
	
	public static Map<String,Object> getAllKinCensus (final Segmentation segmentation, final Relation relation, final String pattern, final CensusCriteria censusCriteria){
		Map<String,Object> result;
		
		result = new HashMap<String,Object>();
		
		if (relation!=null){
			
			Segmentation domain = new Segmentation(relation.getIndividuals(),segmentation.getAllFamilies(),segmentation.getAllRelations());

			try {
				
				CircuitFinder finder = new CircuitFinder(domain,censusCriteria);
				finder.findCircuits();

				for (Cluster<Chain> cluster : finder.getCircuits().getClusters()){
					result.put(cluster.getValue()+"", new Integer(cluster.size()));
				}

			} catch (PuckException e) {
				e.printStackTrace();
			}
		}
		//
		return result;
		
	}
	
	public static Actor getClosestHomologue (Relation relation, Actor actor, Integer date, String dateLabel, String direction){
		Actor result;
		
		result = null;
		
		if (relation!=null && actor!=null && date!=null) {

			Relation closestRelation = null;
			Integer closestDate = null;
			
			for (Relation otherRelation : actor.getIndividual().relations().getByModel(relation.getModel())){
				if (!otherRelation.equals(relation)){
					String dateString = otherRelation.getAttributeValue(dateLabel);
					if (dateString!=null){
						Integer otherDate = Integer.parseInt(dateString);
						if (otherDate!=null && ((direction.equals("OUT") && otherDate > date && (closestDate == null || otherDate < closestDate)) || ((direction.equals("IN") && otherDate < date && (closestDate == null || otherDate > closestDate))))){
							closestRelation = otherRelation;
							closestDate = otherDate;
						}
					}
				}
			}
			
			if (closestRelation != null){
				result = closestRelation.actors().get(actor.getIndividual().getId(), actor.getRole());
			}

		}
		
		//
		return result;
		
	}
		

	public static Map<String,Object> getReferentChainCensus (final Relation relation, final String affiliationLabel){
		Map<String,Object> result;
		
		result = new HashMap<String,Object>();
		
		if (relation!=null){
			
			for (Actor actor : relation.actors().toSortedList()) {
				
				String chain = getReferentChain(actor, affiliationLabel, relation);
						
				Object count = result.get(chain);
				if (count == null){
					result.put(chain, 1.);
				} else {
					result.put(chain, ((Double)result.get(chain))+1.);
				}
			}
		}
		
		//
		return result;
	}
	
	public static String getReferentChain(final Actor actor, final String affiliationLabel, final Relation relation){
		String chain;
		
		Individual ego = actor.getIndividual();
		Individual alter = actor.getReferent();
		Individuals previous = new Individuals();
		previous.add(ego);
		
		chain = "";
		
		if (alter==null){
			String status = getHouseStatus(ego,affiliationLabel, relation);
			if (status==null){
				chain = "[]";
			} else if (status.equals("OWNER")){
				chain = "-";
			} else if (status.equals("OWNER_SPOUSE")){
				chain = "[H]";
			}
		}

		while (alter!=null){
			if (previous.contains(alter)){
				System.err.println("Cyclic referent chain: "+previous);
				break;
			}
			chain += alter.getGender().toChar();
			Actor alterActor = relation.actors().get(alter.getId(),actor.getRole());
			if (alterActor !=null){
				previous.add(alter);
				ego = alter; 
				alter = alterActor.getReferent();
				if (alter==null){
					String status = getHouseStatus(ego,affiliationLabel, relation);
					if (status==null){
						chain += "[]";
					} else if (status.equals("OWNER_SPOUSE")){
						chain += "[H]";
					}
				}
			} else {
				System.err.println("Nonresident referent for "+relation+"\t"+alter);
				alter = null;
			}
		}

		//
		return chain;
	}
	
	private static String getHouseStatus(final Individual ego, final String affiliationLabel, final Relation relation){
		String result;
		
		String egoAttributeValue = ego.getAttributeValue(affiliationLabel);
		String relationAttributeValue = relation.getAttributeValue(affiliationLabel);

		result = null;
		
		if (egoAttributeValue!=null && egoAttributeValue.equals(relationAttributeValue)){
			result = "OWNER";
		} else {
			for (Individual spouse : ego.spouses()){
				String spouseAttributeValue = spouse.getAttributeValue(affiliationLabel);
				if (spouseAttributeValue!=null && spouseAttributeValue.equals(relationAttributeValue)){
					result = "OWNER_SPOUSE";
					break;
				}
			}
		}

		//
		return result;
	}
	
	public static String getReferentRole(final Actor actor, final String pattern, final String affiliationLabel,Relation relation){
		String result;

		int[] maxDegrees = ToolBox.stringsToInts(pattern);

		Individual ego = actor.getIndividual();
		Individual alter = actor.getReferent();
		result = "NONE";
		
		if (alter==null){
			
			result = getHouseStatus(ego, affiliationLabel, relation);
			
		} else {
			
			result = NetUtils.getAlterRole(ego, alter, maxDegrees, null);
			
		}

		
		//
		return result;
	}

	
	public static Map<String,Object> getReferentKinCensus (final Relation relation, final String pattern, final String affiliationLabel){
		Map<String,Object> result;
		
		result = new HashMap<String,Object>();
		
		if (relation!=null){

			for (Actor actor : relation.actors().toSortedList()) {
				
				String alterRole = getReferentRole(actor,pattern, affiliationLabel, relation);
				
				Object count = result.get(alterRole);
				if (count == null){
					result.put(alterRole, 1.);
				} else {
					result.put(alterRole, (Double)count + 1.);
				}
			}
			
			for (String key : result.keySet()){
				result.put(key, MathUtils.percent((Double)result.get(key),relation.actors().size()));
			}
		}
		
		
		
		//
		return result;
	}
	
	public static Map<String,Object> getStatistics (final Relation relation, final List<String> indicators, final String pattern) {
		Map<String,Object> result;
		
		result = new HashMap<String,Object>();
		
		Graph<Individual> graph = RelationWorker.getReferentGraph(relation);
		
		for (String indicator : indicators){
			
			if (indicator.equals("GRAPH")){
				
				result.put(indicator, graph);
				
			} else if (indicator.equals("SIZE")){
				
				result.put(indicator, graph.nodeCount());
				
			} else if (indicator.equals("MAXDEPTH")){
				
				result.put(indicator, GraphUtils.getMaxDepth(graph));
				
			} else if (indicator.equals("MEANDEPTH")){
				
				result.put(indicator, MathUtils.round(GraphUtils.getMeanDepth(graph),2));
				
			} else if (indicator.equals("MEANINDEGREE")){
				
				result.put(indicator, MathUtils.round(GraphUtils.meanInDegree(graph),2));
				
			} else if (indicator.equals("DIAMETER")){
				
				result.put(indicator, GraphUtils.getTreeDiameter(graph));
				
			} else if (indicator.contains("COMPONENT") || indicator.equals("CONCENTRATION")){
				
				Partition<Node<Individual>> components = GraphUtils.components(graph);
				
				if (indicator.equals("NRCOMPONENTS")){
					
					result.put(indicator, components.size());

				} else if (indicator.equals("MAXCOMPONENT")){
					
					result.put(indicator, components.maxClusterSize());

				} else if (indicator.equals("CONCENTRATION")){
					
					result.put(indicator, components.concentration());
					
				}
				
			} else if (indicator.equals("TREES_BY_ID")){
				
				result.put(indicator, getLinkTrees(relation, false, "ID", null));
				
			} else if (indicator.equals("TREES_BY_GENDER")){
				
				result.put(indicator, getLinkTrees(relation, false, "GENDER", null));
				
			} else if (indicator.equals("TREES_BY_KIN")){
				
				result.put(indicator, getLinkTrees(relation, false, "KIN", pattern));
			}
		}
		
		
		//
		return result;
	}



}
