package org.tip.puck.geo;

import java.io.File;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.tip.puck.PuckException;
import org.tip.puck.geo.io.PlaceFile;
import org.tip.puck.graphs.Graph;
import org.tip.puck.net.Attribute;
import org.tip.puck.net.Attributes;
import org.tip.puck.net.Individual;
import org.tip.puck.net.relations.Relation;
import org.tip.puck.partitions.Partition;
import org.tip.puck.util.Value;

import fr.devinsy.util.StringList;

/**
 * @author Klaus Hamberger
 * @author TIP
 */
public class Geography {

	private static Geography geography;

	private Map<GeoLevel, Map<String, Place>> levels;
	private Map<String, String> homonyms;
	private Map<String, Map<GeoLevel, Place>> sups;
//	private Attributes attributes;
	private Map<String,Place> toponyms;

	/**
	 * 
	 */
	public Geography() {
		this.levels = new HashMap<GeoLevel, Map<String, Place>>();
		this.sups = new HashMap<String, Map<GeoLevel, Place>>();
		for (GeoLevel level : GeoLevel.values()) {
			this.levels.put(level, new HashMap<String, Place>());
		}
		this.homonyms = new HashMap<String, String>();
		this.toponyms = new HashMap<String, Place>();
	}
	
/*	public Attributes attributes(){
		return attributes;
	}*/
	
	public Places getPlaces (){
		Places result;
		
		result = new Places();
		
		for (GeoLevel level : levels.keySet()){
			if (level!=GeoLevel.INTERCONTINENTAL){
				for (Place place : levels.get(level).values()){
					result.put(place.getId(), place);
				}
			}
		}
		//
		return result;
	}
	
	public Map<String, StringList> homonymLists(){
		Map<String, StringList> result;
		
		result = new HashMap<String,StringList>();
		
		for (String homonym : homonyms.keySet()){
			String id = homonyms.get(homonym);
			StringList list = result.get(id);
			if (list==null){
				list = new StringList();
				result.put(id, list);
			}
			list.append(homonym);
		}
		
		for (StringList list : result.values()){
			list.sort();
		}
		
		//
		return result;
	}

	/**
	 * 
	 * @param level
	 * @param id
	 * @return
	 */
	public Place get(final GeoLevel level, final String id) {
		return this.levels.get(level).get(id);
	}

	/**
	 * 
	 * @param id
	 * @return
	 */
	private Place get(final String id) {
		Place result;

		result = null;

		for (GeoLevel level : GeoLevel.values()) {
			result = this.levels.get(level).get(id);
			if (result != null) {
				break;
			}
		}

		return result;
	}

	/**
	 * 
	 * @param homonym
	 * @param minLevel
	 * @return
	 */
 /*	public Place get(final String homonym, final GeoLevel minLevel) {
		Place result;

		result = getByHomonym(homonym);

		if (result != null && minLevel != null && minLevel.compareTo(result.level) < 0) {
			result = getPlace(homonym, minLevel);
		}

		//
		return result;

	}*/
	
	public Place getByToponym (String toponym){
		return toponyms.get(toponym);
	}
	
	public String getToponym(final String homonym){
		String result;
		
		Place place = getByHomonym(homonym);
		if (place == null){
			result = null;
		} else {
			result = place.getToponym();
		}
		//
		return result;
	}
	
	public String getToponym(final String homonym, final String parameter){
		String result;
		
		Place place = getPlace(homonym, parameter);
		if (place == null){
			result = null;
		} else {
			result = place.getToponym();
		}
		//
		return result;
	}
	
	public void putUncodedPlaces (Partition<Place> uncodedPlaces, String placeName){
		Place place = getByHomonym(placeName);
		if (place != null && place.getCoordinate()==null) {
			uncodedPlaces.put(place,new Value(place.getLevel()));
			Place sup = place.getSup();
			while (sup.getCoordinate()==null){
				uncodedPlaces.put(sup, new Value (sup.getLevel()));
				sup = sup.getSup();
			}
		}
	}

	/**
	 * 
	 * @param homonym
	 * @return
	 */
	public Place getByHomonym(final String homonym) {
		Place result;
		
		if (homonym==null){
			result = null;
		} else {
			result = get(this.homonyms.get(homonym));
			if (result == null) {
				System.err.println("missing geography entry:\t" + homonym);
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @param first
	 * @param second
	 * @return
	 */
	public Place getCommonAncestor(final Place first, final Place second) {
		Place result;

		result = null;

		if (first == second) {
			result = first;
		} else if (first != null && second != null) {
			int comp = first.compareByLevel(second);
			if (comp == 0) {
				result = getCommonAncestor(first.getSup(), second.getSup());
			} else if (comp < 0) {
				result = getCommonAncestor(first, second.getSup());
			} else if (comp > 0) {
				result = getCommonAncestor(first.getSup(), second);
			}
		}
		//
		return result;
	}
	
	/**
	 * @param geography
	 * @param event
	 * @return
	 */
	public GeoLevel getDistance (final String startName, final String endName){
		GeoLevel result;
		
		Place start = getByHomonym(startName);
		Place end = getByHomonym(endName);

		if (start != null && end != null){
			Place commonAncestor = getCommonAncestor(start, end);
			result = commonAncestor.getLevel();
		} else {
			result = null;
		}
		
		//
		return result;
	}

	/**
	 * 
	 * @param homonym
	 * @param level
	 * @return
	 */
	public Place getPlace(final String homonym, final GeoLevel level) {
		Place result;

		if (getByHomonym(homonym) == null) {
			result = null;
//			System.err.println("Place not known: " + homonym);
		} else {
			result = getByHomonym(homonym).atLevel(level);
		}

		//
		return result;
	}

	/**
	 * 
	 * @param homonym
	 * @param parameter
	 * @return
	 */
	public Place getPlace(final String homonym, final String parameter) {
		Place result;

		try {
			result = getPlace(homonym, GeoLevel.valueOf(parameter));
		} catch (IllegalArgumentException iae) {
			result = getPlaceByCenter(homonym, parameter);
		}

		//
		return result;
	}

	/**
	 * @param homonym
	 * @return
	 */
	private Place getPlaceByCenter(final String homonym, final String center) {
		Place result;
		
		GeoLevel centerLevel = getByHomonym(center).getLevel();

		result = null;
		if (homonym.equals(center)) {
			result = getByHomonym(homonym);
		} else {
			for (GeoLevel level : GeoLevel.values()) {
				Place preResult = getPlace(homonym, level);
				if (preResult == null){
					break;
				} else {
					result = preResult;
					if (level.ordinal()>centerLevel.ordinal() || level == GeoLevel.HOMONYM || (result != null && !result.equals(getPlace(center, level)))) {
						break;
					}
				}
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @param homonym
	 * @param placenames
	 * @return
	 */
	public Place getSup(final String homonym, final List<String> placenames) {
		Place result;

		if (getByHomonym(homonym) == null) {
			result = null;
		} else {
			result = getByHomonym(homonym).getSup(placenames);
		}

		//
		return result;
	}

	/**
	 * 
	 * @param level
	 * @return
	 */
	public Graph<Place> graph(final GeoLevel level) {
		Graph<Place> result;

		result = new Graph<Place>();

		for (Place place : places(level)) {
			result.addNode(place);
		}
		//
		return result;

	}

	/**
	 * 
	 * @param level
	 * @return
	 */
	private Places places(final GeoLevel level) {
		Places result;

		result = new Places();

		Map<String, Place> placeMap = this.levels.get(level);
		if (placeMap != null) {
			for (Place place : placeMap.values()) {
				result.put(place.getId(), place);
			}
		}
		//
		return result;
	}

	/**
	 * 
	 * @param place
	 */
	public void put(final Place place) {
		this.levels.get(place.getLevel()).put(place.getId(), place);
		this.toponyms.put(place.getToponym(), place);
	}

	/**
	 * 
	 * @param homonym
	 * @param place
	 */
	public void put(final String homonym, final Place place) {
		this.homonyms.put(homonym, place.getId());
	}

	/**
	 * 
	 */
/*	public void reportGeography() {
		for (GeoLevel level : GeoLevel.values()) {
			System.out.println(level);
			for (Place place : this.levels.get(level).values()) {
				System.out.println("\t" + place.getId());
			}
		}
	}*/

	/**
	 * 
	 * @return
	 */
	public static Geography getInstance() {
		if (geography == null) {
			try {
				geography = PlaceFile.loadGeoData(new File("src/org/tip/puck/geo/data/togo.txt"));
			} catch (PuckException e) {
				geography = null;
			}
		}
		//
		return geography;
	}

	/**
	 * 
	 * @param geography
	 */
	/*	public static void setGeography(final Geography geography) {
		Geography.geography = geography;
	}*/
	
	public void updateDistricts(){
		for (Place district : levels.get(GeoLevel.TOWNSHIP).values()){
			if (district.getCoordinate()==null){
				String townId = homonyms.get(district.getName());
				Place town = levels.get(GeoLevel.TOWN).get(townId);
				if (town!=null){
					district.setCoordinate(town.getCoordinate());
				}
			}
		}
		
	}
	
	
}
