package org.tip.puck.spacetime.workers;

import java.util.ResourceBundle;

import org.tip.puck.PuckException;
import org.tip.puck.census.workers.CensusCriteria;
import org.tip.puck.census.workers.CensusReporter;
import org.tip.puck.net.Net;
import org.tip.puck.net.relations.Relations;
import org.tip.puck.net.relations.workers.RelationReporter;
import org.tip.puck.net.workers.AttributeDescriptors;
import org.tip.puck.net.workers.AttributeWorker;
import org.tip.puck.partitions.PartitionCriteria;
import org.tip.puck.report.Report;
import org.tip.puck.report.ReportList;
import org.tip.puck.segmentation.Segmentation;
import org.tip.puck.spacetime.Sequence;
import org.tip.puck.spacetime.workers.SpaceTimeCriteria.ReferentNetworkOperation;
import org.tip.puck.spacetime.workers.SpaceTimeCriteria.SliceGeneralStatistics;
import org.tip.puck.statistics.StatisticsCriteria;

import fr.devinsy.util.StringList;

/**
 * 
 * @author Klaus Hamberger
 * @author TIP
 */
public class SpaceTimeReporter {

	/**
	 * Builds many report from a large criteria object:
	 * 
	 * <ul>
	 * <li>Report - Sequence Report</li>
	 * <li>Report - Slice Report</li>
	 * <li>Analysis - Sequences - General Sequence Statistics</li>
	 * <li>Analysis - Sequences - Trajectories</li>
	 * <li>Analysis - Sequences - Ego-Networks</li>
	 * <li>Analysis - Sequences - Parcours Networks</li>
	 * <li>Analysis - Slices - General Slices Statistics</li>
	 * <li>Analysis - Slices - Trajectories</li>
	 * <li>Analysis - Slices - EgoNetworks</li>
	 * </ul>
	 * 
	 * @param criteria
	 * @return
	 * @throws PuckException
	 */
	public static ReportList reportHugeAnalysis(final Net net, final Segmentation segmentation, final SpaceTimeCriteria criteria, final ResourceBundle bundle)
			throws PuckException {
		ReportList result;

		result = new ReportList();

		if (criteria != null) {
			//
			StringList relationModelNames = net.relationModels().sortedNameList();
			AttributeDescriptors attributeDescriptors = AttributeWorker.getExogenousAttributeDescriptors(segmentation, null);

			//
			if (criteria.isSequenceAnalysis()) {
				//
				
				// Reports
				if (!criteria.getSequenceReportTypes().isEmpty()){
					
					Report sequenceReport = SequenceReporter.reportSequences(net, segmentation, criteria);
					result.add(sequenceReport);

				}
				
				if (!criteria.getSequenceGeneralStatistics().isEmpty()){
					Report sequenceCensusReport = SequenceReporter.reportGeneralSequenceCensus(segmentation, criteria);
					result.add(sequenceCensusReport);
				}
				
				if (!criteria.getEgoNetworksOperations().isEmpty()){
					Report egoNetworkReport = SequenceReporter.reportEgoNetworksCensus(segmentation, criteria);
					result.add(egoNetworkReport);
				}
				
				if (!criteria.getParcoursNetworksOperations().isEmpty()){
					Report parcoursNetworkReport = SequenceReporter.reportParcoursNetworksCensus(segmentation, criteria);
					result.add(parcoursNetworkReport);
				}
				
				if (!criteria.getTrajectoriesOperations().isEmpty()){
					Report trajectoriesReport = SequenceReporter.reportTrajectoriesCensus(segmentation, criteria);
					result.add(trajectoriesReport);
				}
			} else {
				//
				
				StatisticsCriteria statisticsCriteria = new StatisticsCriteria();

				PartitionCriteria placeCriteria = PartitionCriteria.createRaw("BIRT_PLACE");
				placeCriteria.setLabelParameter("Bas-Mono");

				statisticsCriteria.getPartitionCriterias().add(placeCriteria);
				statisticsCriteria.getPartitionCriterias().add(PartitionCriteria.createSizedGrouping("AGE", null, 0., 20., null));
				statisticsCriteria.getPartitionCriterias().add(PartitionCriteria.createRaw("QUARTER"));
				statisticsCriteria.getPartitionCriterias().add(PartitionCriteria.createRaw("PATRIL"));
				statisticsCriteria.getPartitionCriterias().add(PartitionCriteria.createRaw("OCCUPATION"));
				statisticsCriteria.getPartitionCriterias().add(PartitionCriteria.createRaw("MATRISTATUS"));
				statisticsCriteria.getPartitionCriterias().add(PartitionCriteria.createRaw("REFERENT"));
				statisticsCriteria.setPlaceParameter("Bas-Mono");
				
				Sequence<Relations> slices = SequenceMaker.createRelationSetSequence(segmentation, criteria);
				
				if (!criteria.getSliceReportTypes().isEmpty()){
					Report sliceReport = SequenceReporter.reportMembers(slices, criteria);
					result.add(sliceReport);
				}
				
				if (criteria.getSliceGeneralStatistics().contains(SliceGeneralStatistics.STATICS)){
					Report sliceStaticsReport = SequenceReporter.reportIndividualStatics(segmentation, slices, criteria, statisticsCriteria);
					result.add(sliceStaticsReport);
				}
				
				if (criteria.getSliceGeneralStatistics().contains(SliceGeneralStatistics.DYNAMICS)){
					Report sliceDynamicsReport = SequenceReporter.reportIndividualDynamics(segmentation, slices, criteria, statisticsCriteria);
					result.add(sliceDynamicsReport);
				}
				
				if (!criteria.getEgoNetworksOperations().isEmpty()){
					Report egoNetworkReport = SequenceReporter.reportEgoNetworks(slices, statisticsCriteria, criteria);
					result.add(egoNetworkReport);
				}
				
				if (!criteria.getTrajectoriesOperations().isEmpty()){
					Report trajectoriesReport = SequenceReporter.reportGeneralSequenceCensus(slices, statisticsCriteria, criteria);
					result.add(trajectoriesReport);

				}
				

				
				
				
			}
		}

		//
		return result;
	}

	public static ReportList reportReferentNetworkAnalysis(final Segmentation segmentation, final SpaceTimeCriteria criteria)
			throws PuckException {
		ReportList result;
		
		result = new ReportList();
		
		if (criteria != null) {
			// Referent Trees.
			
			if (criteria.getReferentNetworkOperations().contains(ReferentNetworkOperation.REFERENT_TREES)){
				boolean reducedTrees = false;
				Report report = SequenceReporter.reportTreeStructure(segmentation, criteria, reducedTrees);
				result.add(report);
			}

			// Itineraries.
			if (criteria.getReferentNetworkOperations().contains(ReferentNetworkOperation.ITINERARIES)){
				Report report2 = SequenceReporter.reportItineraries(segmentation, criteria);
				result.add(report2);
			}

			// Kinship Circuits.
			if (criteria.getReferentNetworkOperations().contains(ReferentNetworkOperation.CIRCUIT_CENSUS)){
				for (Report report3 : RelationReporter.reportRelationCensus(segmentation, criteria)) {
					result.add(report3);
				}
			}

			// Differential Circuit Census.
			if (criteria.getReferentNetworkOperations().contains(ReferentNetworkOperation.DIFFERENTIAL_CENSUS)){
				CensusCriteria censusCriteria = new CensusCriteria();

				censusCriteria.setPattern(criteria.getPattern());
				censusCriteria.setChainClassification(criteria.getChainClassification());
				censusCriteria.setRelationAttributeLabel(criteria.getLocalUnitLabel());

				Integer[] dates = criteria.getDates();
				if (dates == null) {
					Report report = CensusReporter.reportDifferentialCensus(segmentation, criteria, null,
							censusCriteria);
					result.add(report);
				} else {
					for (Integer date : dates) {
						Report report = CensusReporter.reportDifferentialCensus(segmentation, criteria, date,
								censusCriteria);
						result.add(report);
					}
				}
			}
		}
		
		//
		return result;
		
	}

}
