package org.tip.puck.spacetime.workers;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.TreeMap;

import org.tip.puck.PuckException;
import org.tip.puck.matrix.Matrix;
import org.tip.puck.net.Individual;
import org.tip.puck.net.Individuals;
import org.tip.puck.net.Populatable;
import org.tip.puck.net.relations.Actor;
import org.tip.puck.net.relations.Relation;
import org.tip.puck.net.relations.Relations;
import org.tip.puck.net.relations.workers.RelationWorker;
import org.tip.puck.net.workers.IndividualValuator;
import org.tip.puck.net.workers.NetUtils;
import org.tip.puck.net.workers.RelationValuator;
import org.tip.puck.partitions.Partition;
import org.tip.puck.partitions.PartitionCriteria;
import org.tip.puck.partitions.PartitionMaker;
import org.tip.puck.segmentation.Segmentation;
import org.tip.puck.spacetime.Ordinal;
import org.tip.puck.spacetime.Sequence;
import org.tip.puck.spacetime.Sequenceable;
import org.tip.puck.spacetime.Organizable;
import org.tip.puck.statistics.StatisticsCriteria;
import org.tip.puck.util.MathUtils;
import org.tip.puck.util.ToolBox;
import org.tip.puck.util.Value;

public class SequenceStatistics<S extends Sequenceable<E>,E>  {
	
	Segmentation segmentation;
	List<Ordinal> times;

	Organizable<S> sequences;
	S singleSequence;
	
	Map<S,Map<Ordinal,Map<String,Value>>> valuesBySequences;
	Map<Individual,Map<Ordinal,Map<String,Value>>> valuesByIndividuals;
	
	Map<Ordinal,Map<String,Double>> sumsOverSequences;
	Map<Ordinal,Map<String,Double>> sumsOverIndividuals;

	List<String> indicators;
	
	String pattern;
	String affiliationLabel;
	
	public SequenceStatistics(Segmentation segmentation, S singleSequence, SpaceTimeCriteria criteria){
		
		this.times = Ordinal.getOrdinals(criteria.getDates());
		this.segmentation = segmentation;
		this.singleSequence = singleSequence;
		
	}

	
	public SequenceStatistics(Segmentation segmentation, Organizable<S> sequences, SpaceTimeCriteria criteria, List<String> indicators){
		
		this.times = Ordinal.getOrdinals(criteria.getDates());
		this.pattern = criteria.getPattern();
		
		this.indicators = indicators;
		
		// Initialize valuesBySequences
		
		this.sequences = sequences;

		this.valuesBySequences = new TreeMap<S,Map<Ordinal,Map<String,Value>>>();
		for (S sequence : sequences.toSortedList()){
			valuesBySequences.put(sequence, new TreeMap<Ordinal,Map<String,Value>>());
			for (Ordinal time : times){
				valuesBySequences.get(sequence).put(time, new TreeMap<String,Value>());
			}
		}
		
		this.sumsOverSequences = new TreeMap<Ordinal,Map<String,Double>>();
		for (Ordinal time : times){
			sumsOverSequences.put(time, new TreeMap<String,Double>());
		}

		
		// Initialize valuesByIndividuals
		
		this.segmentation = segmentation;

		this.valuesByIndividuals = new TreeMap<Individual,Map<Ordinal,Map<String,Value>>>();
		for (Individual member : sequences.getIndividuals(segmentation).toSortedList()){
			valuesByIndividuals.put(member, new TreeMap<Ordinal,Map<String,Value>>());
			for (Ordinal time : times){
				valuesByIndividuals.get(member).put(time, new HashMap<String,Value>());
			}
		}
		
		this.sumsOverIndividuals = new TreeMap<Ordinal,Map<String,Double>>();
		for (Ordinal time : times){
			sumsOverIndividuals.put(time, new TreeMap<String,Double>());
		}
		
	}
	
	public Map<Ordinal,Relation> completeRelations(Individual individual, String relationModelName, String dateLabel, String egoRoleName, String startDateLabel, String endDateLabel){
		Map<Ordinal,Relation> result;
		
		result = new TreeMap<Ordinal,Relation>();
				
		Ordinal previousTime = null;
		
		for (Ordinal time : times){
			
			// Warning! supposes that there is only one relation by year...
			Relation relation = individual.relations().getByModelName(relationModelName).getByTime(dateLabel, time.getYear()).getFirst();
			
			if (relation != null){
				
				result.put(time,relation);
				
				if (previousTime!=null && result.get(previousTime)==null){
					Relation previousRelation = individual.relations().getByModelName(relationModelName).getPredecessors(relation, individual, egoRoleName, dateLabel, startDateLabel, endDateLabel, previousTime.getYear()).getFirst();
					if (previousRelation!=null){
						result.put(previousTime, previousRelation);
					} 
				}
			}
			previousTime = time;
		}
		//
		return result;
	}
	
	public void putMemberValues(SpaceTimeCriteria criteria){
		
		for (Individual member : sequences.getIndividuals(segmentation).toSortedList()){
			
			Map<Ordinal,Map<String,Value>> memberValues = valuesByIndividuals.get(member);
			Map<Ordinal,Relation> completeRelations = completeRelations(member, criteria.getRelationModelName(),criteria.getDateLabel(),criteria.getEgoRoleName(),criteria.getStartDateLabel(),criteria.getEndDateLabel());
			
			for (Ordinal time : times){
				
				Map<String,Value> valueMap = new HashMap<String,Value>();
				Value relationValue = IndividualValuator.get(member,completeRelations.get(time),null);

				// Update life Status
				
				String lifeStatus = IndividualValuator.lifeStatusAtYear(member, time.getYear());
				
				if (relationValue==null){
					if (time.getYear().toString().equals(member.getAttributeValue("BIRT_DATE"))){
						lifeStatus = "UNBORN";
					} else if (time.getYear().toString().equals(member.getAttributeValue("DEAT_DATE"))){
						lifeStatus = "DEAD";
					}
				}
				valueMap.put("LIFE_STATUS", new Value(lifeStatus));
				
				// Set values // Generalize for individual-, relation- and actorvaluators...
				
				for (String indicator : indicators){
					if (relationValue!=null){
						
						Relation relation = relationValue.relationValue();
						
						if (indicator.equals("PLACE")){
							
							String placeValue = relation.getAttributeValue(criteria.getLocalUnitLabel());
							if (placeValue==null){
								placeValue = relation.getAttributeValue(criteria.getPlaceLabel());
							}
							valueMap.put(indicator, Value.valueOf(placeValue));
							
						} else {
							
							Actor actor = relation.getActor(member, criteria.getEgoRoleName());
							if (indicator.equals("REFERENT")){
								valueMap.put(indicator, Value.valueOf(actor.getReferent()));
							} else if (indicator.equals("REFERENT_KIN")){
								valueMap.put(indicator, Value.valueOf(RelationWorker.getReferentRole(actor, pattern, affiliationLabel, relation)));
							} else if (indicator.equals("REFERENT_CHAIN")){
								valueMap.put(indicator, Value.valueOf(RelationWorker.getReferentChainGenderString(actor, affiliationLabel, relation)));
							} else if (indicator.equals("REFERENT_KIN_TYPE")){
								valueMap.put(indicator, Value.valueOf(RelationWorker.getReferentRoleShort(actor, pattern, affiliationLabel, relation)));
							} else if (indicator.equals("REFERENT_CHAIN_TYPE")){
								valueMap.put(indicator, Value.valueOf(RelationWorker.getReferentChainNumber(actor, relation)));
							}
						}
					}
				}
				memberValues.put(time, valueMap);
			}
		}
	}
	
	public <V> Value getValue(V item, Ordinal time, String indicator){
		Value result;
		
		result = null;
		
		Map<Ordinal,Map<String,Value>> itemValues = null;
		
		if (item instanceof Sequence){
			itemValues = valuesBySequences.get(item);
		} else if (item instanceof Individual){
			itemValues = valuesByIndividuals.get(item);
		}
		
		if (itemValues != null){
			
			Map<String,Value> stationValues = itemValues.get(time);
			
			if (stationValues != null){
				
				result = stationValues.get(indicator);
			}
		}
		
		//
		return result;
		
	}

	
/*	public Value getBySequence(S sequence, Ordinal time, String indicator){
		Value result;
		
		result = null;
		
		Map<Ordinal,Map<String,Value>> sequenceValues = valuesBySequences.get(sequence);
		
		if (sequenceValues != null){
			
			Map<String,Value> stationValues = sequenceValues.get(time);
			
			if (stationValues != null){
				
				result = stationValues.get(indicator);
			}
		}
		
		//
		return result;
		
	}
	
	public Value getByIndividual(Individual individual, Ordinal time, String indicator){
		Value result;
		
		result = null;
		
		Map<Ordinal,Map<String,Value>> individualValues = valuesByIndividuals.get(individual);
		
		if (individualValues != null){
			
			Map<String,Value> stationValues = individualValues.get(time);
			
			if (stationValues != null){
				
				result = stationValues.get(indicator);
			}
		}
		
		//
		return result;
	}*/
	
	public Double sumOverSequences (Ordinal time, String indicator) {
		Double result;
		
		Double sum = sumsOverSequences.get(time).get(indicator);
		
		if (sum == null){
			result = 0.;
		} else {
			result = sum;
		}
		
		//
		return result;
	}
	
	public Double sumOverIndividuals (Ordinal time, String indicator) {
		Double result;
		
		Double sum = sumsOverIndividuals.get(time).get(indicator);
		
		if (sum == null){
			result = 0.;
		} else {
			result = sum;
		}
		
		//
		return result;
	}
	
	public Double meanOverSequences (Ordinal time, String indicator){
		Double result;
		
		result = sumOverSequences(time, indicator)/new Double(valuesBySequences.size());

		//
		return result;
	}
	
	public Double meanOverIndividuals (Individual individual, Ordinal time, String indicator){
		Double result;
		
		result = sumOverSequences(time, indicator)/new Double(valuesByIndividuals.size());

		//
		return result;
	}
		
	public void put(S sequence, Ordinal time, String indicator, Value value){
		
		Map<Ordinal,Map<String,Value>> sequenceValues = valuesBySequences.get(sequence);
		
		if (sequenceValues != null){
			
			Map<String,Value> stationValues = sequenceValues.get(time);
			
			if (stationValues != null){
				
				stationValues.put(indicator,value);
				
				if (value.isNumber()){
					
					sumsOverSequences.get(time).put(indicator,sumOverSequences(time,indicator) + new Value(value).doubleValue());
					
				}
				
			}
		}
	}
	
	public void putSequenceValues(){
		
		for (S sequence : sequences.toSortedList()){
			
//			Map<Ordinal,Map<String,Object>> sequenceValues = valuesBySequences.get(sequence);
			
			for (Ordinal time : times){
				
				E station = sequence.getStation(time);
				
				if (station!=null) {

					Map<String,Value> statistics = getStatistics(station,indicators,pattern);
					
					for (String indicator : indicators){
						
						Value value = statistics.get(indicator);
						put(sequence, time, indicator, value);
					}
					
//					sequenceValues.get(time).putAll(statistics);
				}
			}
		}
	}
	
	/**
	 * @param station
	 * @param indicators
	 * @param pattern
	 * @return
	 */
	public Map<String,Value> getStatistics (final E station, final List<String> indicators, final String pattern){
		Map<String,Value> result;
				
		result = null;
		
		if (station instanceof Relation){
			result = RelationWorker.getStatistics((Relation)station, indicators, pattern);
		}
		//
		return result;
	}
	

	public List<String> indicators() {
		return indicators;
	}
	
	public String getTrend(S sequence, String indicator){
		String result;
		
		result = null;
		Value lastValue = null;
		
		for (Ordinal time : times){
			
			Value value = getValue(sequence,time,indicator);
//			Value value = getBySequence(sequence,time,indicator);

			if (lastValue != null && value != null){

				if (lastValue.isNotNumber() || value.isNotNumber()){
					
					break;
					
				} else {
											
					int comp = ((Comparable<Value>)value).compareTo(lastValue);
					String trend = null;
					
					if (comp < 0){
						trend = "DECLINING";
					} else if (comp > 0) {
						trend = "AUGMENTING";
					} else if (comp == 0){
						trend = "CONSTANT";
					}
					
					if (result == null || result.equals("CONSTANT")){
						result = trend;
					} else if (!trend.equals("CONSTANT") && !trend.equals(result)){
						result = "VARIABLE";
						break;
					} 
				}
			}

			lastValue = value;
				
		}
		
		//
		return result;
	}
	
	public String getMeanTrend (String indicator){
		String result;
		
		result = "";
		
		Map<String,Double> trendCounts = new HashMap<String,Double>();
		
		for (S sequence : sequences){
			
			String trend = getTrend(sequence,indicator);
			Double count = trendCounts.get(trend);
			if (count == null){
				trendCounts.put(trend,1.);
			} else {
				trendCounts.put(trend,count+1.);
			}
		}
		
		for (String trend : trendCounts.keySet()){
			trendCounts.put(trend,MathUtils.percent(trendCounts.get(trend), new Double(sequences.size())));
		}
		
	    List<Entry<String,Double>> sortedEntries = new ArrayList<Entry<String,Double>>(trendCounts.entrySet());

	    Collections.sort(sortedEntries, 
	            new Comparator<Entry<String,Double>>() {
	                @Override
	                public int compare(Entry<String,Double> e1, Entry<String,Double> e2) {
	                    return e2.getValue().compareTo(e1.getValue());
	                }
	            }
	    );

		for (Entry<String,Double> entry: sortedEntries){
			result += entry.getKey()+" "+entry.getValue()+" ";
		}
		
		//
		return result;
	}

	public <V> Map<Value,Double> getMeanValueFrequencies(Map<Ordinal,Partition<V>> census, Organizable<V> sliceables){
		Map<Value,Double> result;
		
		result = new TreeMap<Value,Double>();

		for (V sliceable : sliceables){
			
			for (Ordinal time: times){
				
				Value value = census.get(time).getValue(sliceable);

				if (value!=null){
					Double count = result.get(value);
					if (count==null){
						count = 1.;
					} else {
						count += 1.;
					}
					result.put(value,count);
				}
			}
		}
		
		for (Value value : result.keySet()){
			if (result.get(value)==null){
				result.put(value,result.get(value)/new Double(times.size()));
			}
		}
		//
		return result;
	}
	
	public <V> Matrix getTransitionMatrix(Map<Ordinal,Partition<V>> census, Organizable<V> sliceables){
		Matrix result;
		
		Map<String,Map<String,Integer>> transitionMap = new TreeMap<String,Map<String,Integer>>();
		List<String> values = new ArrayList<String>();
		
		for (V sliceable : sliceables){
			
			for (int i=1;i<times.size();i++){
				
				Value object1 = census.get(times.get(i-1)).getValue(sliceable);
				Value object2 = census.get(times.get(i)).getValue(sliceable);
				
				if (object1!=null && object2!=null){

					String value1 = object1.toString();
					String value2 = object2.toString();
					
					if (!values.contains(value1)){
						values.add(value1);
					}
					if (!values.contains(value2)){
						values.add(value2);
					}
					
					Map<String,Integer> targetMap = transitionMap.get(value1);
					if (targetMap==null){
						targetMap = new TreeMap<String,Integer>();
						transitionMap.put(value1,targetMap);
					}

					Integer count = targetMap.get(value2);
					if (count==null){
						count = 1;
					} else {
						count += 1;
					}
					targetMap.put(value2,count);
				}
			}
		}
		
		Collections.sort(values);
		String[] labels = new String[values.size()];
		for (int i=0;i<values.size();i++){
			labels[i] = values.get(i)+"";
		}
		
		result = new Matrix(values.size(),values.size());
		result.setRowLabels(labels);
		result.setColLabels(labels);
		
		for (Object value1 : transitionMap.keySet()){
			
			Map<String,Integer> targetMap = transitionMap.get(value1);

			for (Object value2 : targetMap.keySet()){
				
				result.augment(values.indexOf(value1), values.indexOf(value2), targetMap.get(value2));

			}
		}
		
		//
		return result;
	}
	
	public Map<Ordinal,Partition<Individual>> getDatedIndividualCensus (Segmentation segmentation, String censusType){
		Map<Ordinal,Partition<Individual>> result;
		
		result = new TreeMap<Ordinal,Partition<Individual>>();
		for (Ordinal time : times){
			result.put(time, new Partition<Individual>());
		}
		
		for (Individual member : sequences.getIndividuals(segmentation)){
			
			for (Ordinal time : times){
				
				Value value = getValue(member, time, censusType);
//				Value value = getByIndividual(member, time, censusType);

				if (value!=null){

					result.get(time).put(member, value);

				}
			}
		}
		//
		return result;
	}
	
	private Individuals getIndividuals(E station){
		Individuals result;
		
		if (station instanceof Populatable){
			result = ((Populatable)station).getIndividuals();
		} else {
			result = null;
		}
		//
		return result;
	}
	
	// Unchecked casts
	public Map<String,Map<Ordinal,Partition<Individual>>> getDynamicIndividualCensus (final SpaceTimeCriteria spaceTimeCriteria, final StatisticsCriteria statisticsCriteria){
		Map<String,Map<Ordinal,Partition<Individual>>> result;
		
		result = new TreeMap<String,Map<Ordinal,Partition<Individual>>>();
		
		result.put("MIGRATIONS", new TreeMap<Ordinal,Partition<Individual>>());
		result.put("DESTINATIONS", new TreeMap<Ordinal,Partition<Individual>>());
		result.put("ORIGINS", new TreeMap<Ordinal,Partition<Individual>>());
		
		for (int i=0;i<times.size()-1;i++){
			
			Ordinal startTime = times.get(i);
			Ordinal endTime = times.get(i+1);

			Relations filteredStartSpace = (Relations)singleSequence.getStation(startTime);
			Relations filteredEndSpace = (Relations)singleSequence.getStation(endTime);

			Relations totalStartSpace = segmentation.getAllRelations().getByTime(spaceTimeCriteria.getDateLabel(), startTime.getYear());
			Relations totalEndSpace = segmentation.getAllRelations().getByTime(spaceTimeCriteria.getDateLabel(), endTime.getYear());

			Partition<Individual> migrations = new Partition<Individual>();
			migrations.setLabel("Migrations "+startTime+"/"+endTime);
			result.get("MIGRATIONS").put(endTime, migrations);

			Partition<Individual> destinations = new Partition<Individual>();
			destinations.setLabel("Destinations "+startTime+"/"+endTime);
			result.get("DESTINATIONS").put(startTime, destinations);
			
			Partition<Individual> origins = new Partition<Individual>();
			origins.setLabel("Origins "+startTime+"/"+endTime);
			result.get("ORIGINS").put(endTime, origins);


//			Individuals totalStartPopulation = totalStartSpace.getIndividuals();
//			Individuals filteredStartPopulation = filteredStartSpace.getIndividuals();
//			Individuals totalEndPopulation = totalEndSpace.getIndividuals();
//			Individuals filteredEndPopulation = filteredEndSpace.getIndividuals();
			
			// Forward 
			
			for (Individual individual : filteredStartSpace.getIndividuals().toSortedList()){
				if (IndividualValuator.lifeStatusAtYear(individual, endTime.getYear()).equals("DEAD")){
					migrations.put(individual, new Value("DIED"));
//				} else if (!totalEndPopulation.contains(individual)){
				} else if (totalEndSpace.getByIndividual(individual).isEmpty()){
					migrations.put(individual, new Value("UNKNOWN DESTINATION"));
//				} else if (!filteredEndPopulation.contains(individual)){
				} else if (filteredEndSpace.getByIndividual(individual).isEmpty()){
					migrations.put(individual, new Value("LEFT"));
					for (Relation destination : totalEndSpace.getByIndividual(individual)){
						destinations.put(individual,RelationValuator.get(destination,"PLACE",statisticsCriteria.getPlaceParameter()));
					}
				} else {
					Relation start = filteredStartSpace.getByIndividual(individual).getFirst();
					Relation end = filteredEndSpace.getByIndividual(individual).getFirst();
					String startUnit = start.getAttributeValue(singleSequence.idLabel());
					if (startUnit==null){
						startUnit = start.getAttributeValue(spaceTimeCriteria.getPlaceLabel());
					}
					String endUnit = end.getAttributeValue(singleSequence.idLabel());
					if (endUnit==null){
						endUnit = start.getAttributeValue(spaceTimeCriteria.getPlaceLabel());
					}
					if (!startUnit.equals(endUnit)){
						migrations.put(individual, new Value("INTERNAL CHANGE"));
					} else {
						migrations.put(individual, new Value("UNCHANGED"));
					}
				}
			}
			
			// Backward 
			
			for (Individual individual : filteredEndSpace.getIndividuals().toSortedList()){
				if (IndividualValuator.lifeStatusAtYear(individual, startTime.getYear()).equals("UNBORN")){
					migrations.put(individual, new Value("NEWBORN"));
//				} else if (!totalStartPopulation.contains(individual)){
				} else if (totalStartSpace.getByIndividual(individual).isEmpty()){
					migrations.put(individual, new Value("UNKNOWN ORIGIN"));
//				} else if (!filteredStartPopulation.contains(individual)){
				} else if (filteredStartSpace.getByIndividual(individual).isEmpty()){
					migrations.put(individual, new Value("ENTERED"));
					for (Relation origin : totalStartSpace.getByIndividual(individual)){
						origins.put(individual,RelationValuator.get(origin,"PLACE",statisticsCriteria.getPlaceParameter()));
					}
/*				} else if (!filteredStartSpace.getByIndividual(individual).equals(filteredEndSpace.getByIndividual(individual))){
//						partition.put(individual, new Value("INTERNAL CHANGE2"));
				} else {
//						partition.put(individual, new Value("UNCHANGED2"));*/
				}
			}
		}
		//
		return result;
	}
	
	public Map<Ordinal,Partition<Individual>> getDatedIndividualCensus (final SpaceTimeCriteria spaceTimeCriteria, final PartitionCriteria partitionCriteria) throws PuckException{
		Map<Ordinal,Partition<Individual>> result;
		
		result = new TreeMap<Ordinal,Partition<Individual>>();
		
		for (Ordinal time : times){

			E relations = singleSequence.getStation(time);
			String label = spaceTimeCriteria.getRelationModelName()+" "+time;
			Partition<Individual> partition = new Partition<Individual>();
			
			if (partitionCriteria.getLabel().equals("REFERENT")){
				
				partitionCriteria.setLabelParameter(spaceTimeCriteria.getRelationModelName()+" "+spaceTimeCriteria.getEgoRoleName()+" "+time);
				Partition<Individual> prePartition = PartitionMaker.create(label, getIndividuals(relations), (Relations)relations, partitionCriteria);
				
				for (Individual ego : prePartition.getItemsAsList()){
					Value alterId = prePartition.getValue(ego);
					if (alterId!=null){
						List<String> alterRoles = NetUtils.getAlterRoles(ego, segmentation.getAllIndividuals().getById(alterId.intValue()), ToolBox.stringsToInts(spaceTimeCriteria.getPattern()), spaceTimeCriteria.getRelationModelNames(), spaceTimeCriteria.getChainClassification(), null, null);
						Collections.sort(alterRoles);
						partition.put(ego, new Value(alterRoles.toString()));
					}
				}
				
				
			} else {
				if (partitionCriteria.getLabel().equals("AGE") || partitionCriteria.getLabel().equals("MATRISTATUS")|| partitionCriteria.getLabel().equals("OCCUPATION")){
					partitionCriteria.setLabelParameter(time+"");
				} 
				partition = PartitionMaker.create(label, getIndividuals(relations), (Relations)relations, partitionCriteria);
			}
			//
			result.put(time, partition);
		}
		//
		return result;
	}
	
	public Map<Ordinal,Partition<S>>  getDatedSequenceCensus(String censusType, SpaceTimeCriteria criteria){
		Map<Ordinal,Partition<S>> result;
		
		result = new TreeMap<Ordinal,Partition<S>>();
		for (Ordinal time : times){
			result.put(time, new Partition<S>());
		}
				
		for (S sequence : valuesBySequences.keySet()){
				
			Map<Ordinal,Map<String,Value>> sequenceValues = valuesBySequences.get(sequence);
				
			for (Ordinal time : times){
					
				E station = sequence.getStation(time);
				Map<String,Value> map = null;
				
				if (station instanceof Relation){
					map = RelationValuator.get((Relation)station, censusType, segmentation, criteria).mapValue();
				}
				
				if (map!=null){
//					RelationWorker.getReferentKinCensus(relation, pattern, affiliationLabel);
					sequenceValues.get(time).putAll(map);
						
					for (String indicator : sequenceValues.get(time).keySet()){
						if (!indicators.contains(indicator)){
							indicators.add(indicator);
						}
					}
						
					Partition<S> partition = result.get(time);
					Value value = map.get("Types");
					if (value!=null){
						partition.put(sequence, new Value(value));
					}
				}
			}
		}
		
		return result;
	}
	
	public <V> Partition<V> getSequenceCensus (Map<Ordinal,Partition<V>> datedPartition, Organizable<V> sliceables){
		Partition<V> result;
		
		result = new Partition<V>();
		
		for (V member : sliceables){
			
			Sequence<Value> sequence = new Sequence<Value>();
			
			for (Ordinal time: times){
				
				sequence.put(time, datedPartition.get(time).getValue(member));
			}
			result.put(member, new Value(sequence.toValueString()));
		}
		//
		return result;
	}
	
	public Map<E,E> adjacentStations (String direction){
		Map<E,E> result;
		
		result = new HashMap<E,E>();
		
		Ordinal former = null;
		
		for (Ordinal later : times){
			
			if (former!=null){
				
				Ordinal current = null;
				if (direction.equals("OUT")){
					current = former;
				} else if (direction.equals("IN")){
					current = later;
				}
				
				for (S house : sequences){
					
					E currentRelation = house.getStation(current);
					
					if (currentRelation != null){
						if (direction.equals("OUT")){
							result.put(currentRelation,  house.getStation(later));
						} else if (direction.equals("IN")){
							result.put(currentRelation, house.getStation(former));
						}
					}
				}
			}
			former = later;
		}
		
		//
		return result;
	}
	
	// Temporary solution (unchecked casts, should be in other class)
	public Partition<String> getFlows (String direction, String dateLabel){
		Partition<String> result;
		
		result = new Partition<String>();
		
		int[] maxDegrees = ToolBox.stringsToInts(pattern);
		
		Map<Relation,Relation> adjacentStations = (Map<Relation,Relation>)adjacentStations(direction);
		
		for (Relation currentRelation : adjacentStations.keySet()){
			
			Relation otherRelation = adjacentStations.get(currentRelation);

			for (Actor actor : currentRelation.getDifferentwActors(otherRelation)){
				
				Individual referent = actor.getReferent();
				String link = "UNKNOWN";
				if (referent!=null){
					link = 	NetUtils.getAlterRole(actor.getIndividual(), referent, maxDegrees, null);
				}
				Individual otherReferent = null;
				Actor otherActor = RelationWorker.getClosestHomologue(currentRelation, actor, dateLabel,direction);
				if (otherActor!=null){
					otherReferent = otherActor.getReferent();
				} else {
//					System.err.println("Missing homologue "+actor+" "+direction+" "+year);
				}
				
				String otherLink = "UNKNOWN";
				if (otherReferent!=null){
					otherLink = NetUtils.getAlterRole(actor.getIndividual(), otherReferent, maxDegrees, null);
				} 
				
				String change = null;
				if (referent!=null && referent.equals(otherReferent)){
					change = "IDENTICAL";
				} else if (direction.equals("OUT")){
					change = link+">"+otherLink;
				} else if (direction.equals("IN")){
					change = otherLink+">"+link;
				}
				System.out.println(currentRelation+" "+otherRelation+" "+actor);
				result.put(currentRelation+"\t"+currentRelation.getTime(dateLabel)+"\t"+direction+"\t"+actor.getIndividual()+"\t"+referent+"\t"+otherReferent, new Value(change));
			}
		}

		//
		return result;
	}


}
