package org.tip.puck.spacetime;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.tip.puck.geo.GeoLevel;
import org.tip.puck.geo.Geography;
import org.tip.puck.net.FiliationType;
import org.tip.puck.net.workers.ExpansionMode;

/**
 * 
 * @author Klaus Hamberger
 *
 */
public class SpaceTimeCriteria {
	
	public enum RelationClassificationType {
		TYPEDID,
		HOST,
		MIG,
		HOSTMIG,
		MIGRATIONTYPE,
		CHILDMIGRATIONTYPE,
		TREES,
		DISTANCE,
		PLACE,
		REGION,
		MOVEMENT,
		TURNOVER,
		COMPONENTS
	}
	
	public enum CensusType {
		GENERAL,
		EGONETWORKS,
		PARCOURSNETWORKS,
		PARCOURS;
		
		public String toString(){
			String result;
			
			switch (this){
			case GENERAL:
				result = "General";
				break;
			case EGONETWORKS:
				result = "Ego Networks";
				break;
			case PARCOURSNETWORKS:
				result = "Parcours Networks";
				break;
			case PARCOURS:
				result = "Parcours";
				break;
			default:
				result = null;
			}
			//
			return result;
			
		}
	}

	// General criteria for space time analysis
	
	private String relationModelName;
	
	// Attribute labels
	
	private String dateLabel;
	private String startDateLabel;
	private String endDateLabel;
	private String placeLabel;
	private String startPlaceLabel;
	private String endPlaceLabel;
	private String localUnitLabel;
	
	// Role names
	
	private String egoRoleName;
	private String defaultReferentRoleName;
	private List<String> roleNames; 
	
	// Time criteria
	
	private int maxAge;
	private int minAge;
	private Integer[] dates;

	// Geography criteria [temporary]
	
	private Geography geography;
	private GeoLevel level;
	private List<String> minimalPlaceNames;
	
	// Parameters for sequence expansion
	
	private ExpansionMode expansionMode;
	private FiliationType filiationType;
	
	// General census criteria
	
	private CensusType censusType;
	
	// Circuit census criteria

	private String pattern;
	private String chainClassification;
	private List<String> relationModelNames; 

	// Partition criteria

	private String individualClassificationType;
	private List<RelationClassificationType> relationClassificationTypes;
	private List<RelationClassificationType> mainRelationClassificationTypes; // for parcours networks

	// Census criteria
	
	private List<String> censusOperationLabels;
	private List<String> networkTitles;
	private Map<String,List<String>> partitionLabels;

//	private double threshold;
	
	// Filter for alters in ego and parcours networks
	
	private String alterFilterRoleName; 
	private String alterFilterAttributeLabel; 
	private String alterFilterAttributeValue; 

	private void setDefaultCriteria(){
	
		// *Set event model name [Choose among possible relation model names]
		relationModelName = "MIGEVENT";
		
		// Set labels of attribute containing the date [Choose among relation attribute labels]
		dateLabel = "DATE";
		startDateLabel = "START_DATE";
		endDateLabel = "END_DATE";
		
		// Set labels of attribute concerning the place
		placeLabel = "PLACE";
		startPlaceLabel = "START_PLACE";
		endPlaceLabel = "END_PLACE";
		//*
		localUnitLabel = "HOUSE"; 

		// *Set ego role name [Choose among event role names]
		egoRoleName = "MIG";
		
		// Set alter role name [Choose among event role names]
		alterFilterRoleName = "ALL";
		
		// Set event role names [Choose among roles of relationModel that has been chosen as eventModel; in addition: "ALL"]
		roleNames = new ArrayList<String>();
		roleNames.add("ALL");
		//*
		roleNames.add("HOST");
		roleNames.add("MIG");
		
		// Set default roleName of referent [Choose among role names]
		defaultReferentRoleName = "HOST";
		
		// Set age thresholds [Spinner integers with interval 1; Default: min = 0, max = 1000]
		minAge = 0;
		maxAge = 1000;
		
		// *Set dates
		dates = new Integer[]{2004,2009,2014};
		
		// Set relation model names [Choose among remaining relation model names]
		relationModelNames = new ArrayList<String>();
		relationModelNames.add("Apprenticeship");
		relationModelNames.add("Housing");
		relationModelNames.add("Employment");
		
		// Set relation attribute types [Choose among values of enum RelationClassificationType]
		relationClassificationTypes = new ArrayList<RelationClassificationType>();

		// Set main relation attribute types types [Choose among values of enum RelationClassificationType] 
		mainRelationClassificationTypes = new ArrayList<RelationClassificationType>();

		// Consolidate main event types with event types
		for (RelationClassificationType mainEventType : mainRelationClassificationTypes){
			if (!relationClassificationTypes.contains(mainEventType)){
				relationClassificationTypes.add(mainEventType);
			}
		}
		
		// Set geography criteria [consolidate with new module!]
		geography = Geography.getInstance();
		level = GeoLevel.TOWN;
		// Set names of minimal places to be considered [write as string separated by comma?]
		if (censusType == CensusType.GENERAL){
			minimalPlaceNames = Arrays.asList(new String[]{"Afagnan","Lom","Bas-Mono","Togo","Bnin","Ghana","Nigria","Cte d'Ivoire","Afrique"});
		}
		
		// Set chain census criteria
		pattern = "3 1";
		chainClassification = "LINE";
//		threshold = 0.01;
		
		// Set statistics criteria
		individualClassificationType = "GENDER";
		
		// Set network titles [choose from list]
		networkTitles = new ArrayList<String>();
		partitionLabels = new HashMap<String,List<String>>();
		for (String networkTitle : networkTitles){
			partitionLabels.put(networkTitle,new ArrayList<String>());
		}

		// Set network partition censusOperationLabels [choose from list]

		// Set censusOperationLabels of census operations
		censusOperationLabels = new ArrayList<String>();
		
		// Set expansion mode and filiation type for extended sequences [choose from values of enums ExpansionMode and FiliationType]
		expansionMode = ExpansionMode.RELATED;
		filiationType = FiliationType.COGNATIC;



	}

	public void setRelationClassificationTypes(
			List<RelationClassificationType> relationClassificationTypes) {
		this.relationClassificationTypes = relationClassificationTypes;
	}

	public List<RelationClassificationType> getRelationClassificationTypes() {
		return relationClassificationTypes;
	}

	public SpaceTimeCriteria (){

		setDefaultCriteria();

	}
	

	/**
	 * Temporary Method to set default criteria 
	 * @param censusType
	 */
	public SpaceTimeCriteria (CensusType censusType){
		
		setDefaultCriteria();
		this.censusType = censusType;
		
		// Set alter attribute label+value [choose from individual attribute censusOperationLabels]
		if (censusType == CensusType.PARCOURSNETWORKS){
			alterFilterAttributeLabel = "INTERV";
			alterFilterAttributeValue = "Yes";
		}

		// Set relation attribute types [Choose among values of enum RelationClassificationType]
		switch (censusType){
		case GENERAL:
			relationClassificationTypes.add(RelationClassificationType.PLACE);
			relationClassificationTypes.add(RelationClassificationType.MIGRATIONTYPE);
			relationClassificationTypes.add(RelationClassificationType.TREES);
			relationClassificationTypes.add(RelationClassificationType.CHILDMIGRATIONTYPE);
			relationClassificationTypes.add(RelationClassificationType.DISTANCE);
			relationClassificationTypes.add(RelationClassificationType.REGION);
			break;
		case EGONETWORKS:
			relationClassificationTypes.add(RelationClassificationType.PLACE);
			break;
		case PARCOURSNETWORKS:
			relationClassificationTypes.add(RelationClassificationType.PLACE);
			break;
		case PARCOURS:
			relationClassificationTypes.add(RelationClassificationType.PLACE);
			break;
		}

		// Set main relation attribute types types [Choose among values of enum RelationClassificationType] 
		switch (censusType){
		case GENERAL:
			break;
		case EGONETWORKS:
			break;
		case PARCOURSNETWORKS:
			mainRelationClassificationTypes.add(RelationClassificationType.MIGRATIONTYPE);
			mainRelationClassificationTypes.add(RelationClassificationType.TYPEDID);
			break;
		case PARCOURS:
			mainRelationClassificationTypes.add(RelationClassificationType.PLACE);
			mainRelationClassificationTypes.add(RelationClassificationType.DISTANCE);
			mainRelationClassificationTypes.add(RelationClassificationType.TREES);
			mainRelationClassificationTypes.add(RelationClassificationType.MIGRATIONTYPE);
//			mainRelationClassificationTypes.add(EventType.COMPONENTS);
			break;
		}
		
		// Consolidate main event types with event types
		for (RelationClassificationType mainEventType : mainRelationClassificationTypes){
			if (!relationClassificationTypes.contains(mainEventType)){
				relationClassificationTypes.add(mainEventType);
			}
		}
		
		// Set network titles [choose from list]
		networkTitles = new ArrayList<String>();
		partitionLabels = new HashMap<String,List<String>>();
		switch (censusType){
		case GENERAL:
			break;
		case EGONETWORKS:
			addNetworkTitle("Ego Network");
			addNetworkTitle("Nonmediated Ego Network");
			break;
		case PARCOURSNETWORKS:
			addNetworkTitle("Parcours");
			addNetworkTitle("Extended Parcours");
			addNetworkTitle("Multiple Parcours");
			for (RelationClassificationType mainEventType : mainRelationClassificationTypes){
				addNetworkTitle("Parcours Network_"+mainEventType);
				addNetworkTitle("Parcours Similarity Tree_"+mainEventType);
			}
			break;
		case PARCOURS:
			addNetworkTitle("Event Type Network");
			addNetworkTitle("Sequence Type Network");
			break;
		}
		
		// Set partition censusOperationLabels
		switch (censusType){
		case GENERAL:
			break;
		case EGONETWORKS:
			partitionLabels.get("Ego Network").add("EGO-RELATION");
			partitionLabels.get("Ego Network").add("DEGREE");
			partitionLabels.get("Nonmediated Ego Network").add("BETWEENNESS");
			partitionLabels.get("Nonmediated Ego Network").add("EGO-RELATION");
			partitionLabels.get("Nonmediated Ego Network").add("DEGREE");
			partitionLabels.get("Nonmediated Ego Network").add("COMPONENT");
			break;
		case PARCOURSNETWORKS:
			partitionLabels.get("Parcours").add("DATE");
			partitionLabels.get("Parcours").add("DISTANCE");
			partitionLabels.get("Multiple Parcours").add("DATE");
			partitionLabels.get("Multiple Parcours").add("DISTANCE");
			partitionLabels.get("Extended Parcours").add("ORDER");
			partitionLabels.get("Extended Parcours").add("TYPE");
			break;
		case PARCOURS:
			break;
		}
		
		// Set ego network and tree network partition censusOperationLabels
		if (censusType == CensusType.PARCOURSNETWORKS){
			for (RelationClassificationType mainEventType : mainRelationClassificationTypes){
				List<String> egoNetworkPartitionLabels = partitionLabels.get("Parcours Network_"+mainEventType);
				egoNetworkPartitionLabels.add("BETWEENNESS");
				egoNetworkPartitionLabels.add("EGO-RELATION");
				egoNetworkPartitionLabels.add("DEGREE");
				egoNetworkPartitionLabels.add("COMPONENT");
				
				List<String> treePartitionLabels = partitionLabels.get("Parcours Similarity Tree_"+mainEventType);
				treePartitionLabels.add("SIZE");
				treePartitionLabels.add("STEP");
				treePartitionLabels.add("EGO-RELATION");
			}
		}
		
		// Set censusOperationLabels of census operations
		censusOperationLabels = new ArrayList<String>();
		switch (censusType){
		case GENERAL:
			
			censusOperationLabels.add("NREVENTS");
			censusOperationLabels.add("NREXTERNALMOVES");
			
			censusOperationLabels.add("AGEFIRST");
			censusOperationLabels.add("AGEFIRST_CHILDMIGRATIONTYPE_NOPARENTS");
			censusOperationLabels.add("AGEFIRST_DISTANCE_TRANSNATIONAL");
			censusOperationLabels.add("MAX_DISTANCE");
			censusOperationLabels.add("SAMESEXALTERS_ALL");
			censusOperationLabels.add("SAMEPLACEALTERS_ALL");
			
			for (String roleName : roleNames){
				censusOperationLabels.add("NRALTERS_"+roleName);
				censusOperationLabels.add("NRALTERSPEREVENT_"+roleName);
			}
			
			for (RelationClassificationType type : relationClassificationTypes){
				censusOperationLabels.add("EVENTS_"+type);
			}
			
			censusOperationLabels.add("MEAN_COVERAGE");
			censusOperationLabels.add("MAX_COVERAGE");
			
			for (String roleName : roleNames){
				censusOperationLabels.add("RELATIONS_"+roleName);
			}
			
			censusOperationLabels.add("MAIN_ALTERS");
			censusOperationLabels.add("MAIN_RELATIONS");
			
			break;
			
		case EGONETWORKS:

			censusOperationLabels.add("NREVENTS");
			censusOperationLabels.add("NREXTERNALMOVES");

			censusOperationLabels.add("SIZE");
			censusOperationLabels.add("TIES");
			censusOperationLabels.add("EGO-BETWEENNESS");
			censusOperationLabels.add("MEAN_BETWEENNESS");
			censusOperationLabels.add("MAX_BETWEENNESS");
			censusOperationLabels.add("ECCENTRICITY");
			censusOperationLabels.add("DENSITY");
			censusOperationLabels.add("DENSITY_NOLOOPS");
			censusOperationLabels.add("MEANDEGREE");
			censusOperationLabels.add("MEANDEGREE_NOLOOPS");
			censusOperationLabels.add("MEANDEGREE_NOLOOPS_NORM");
			censusOperationLabels.add("NRCOMPONENTS");
			censusOperationLabels.add("NRISOLATES");
			censusOperationLabels.add("MAXCOMPONENT");
			censusOperationLabels.add("NRCOMPONENTS_NORM");
			censusOperationLabels.add("NRISOLATES_NORM");
			censusOperationLabels.add("MAXCOMPONENT_NORM");
			censusOperationLabels.add("CONCENTRATION");
			censusOperationLabels.add("BROKERAGE");
			censusOperationLabels.add("EFFICIENT_SIZE");
			censusOperationLabels.add("EFFICIENCY");
			censusOperationLabels.add("NETWORK_RELATIONS");
			censusOperationLabels.add("CONNECTED_NETWORK_RELATIONS");
			
			censusOperationLabels.add("CENTRAL_ALTERS");
			censusOperationLabels.add("CENTRAL_RELATIONS");

			censusOperationLabels.add("SDENSITY_PARENT-CHILD");
			censusOperationLabels.add("SDENSITY_SPOUSE");
			censusOperationLabels.add("SDENSITY_SIBLING");
			censusOperationLabels.add("SDENSITY_RELATIVE");
			censusOperationLabels.add("SDENSITY_AFFINE");
			censusOperationLabels.add("SDENSITY_EMPLOYMENT");
			censusOperationLabels.add("SDENSITY_RENT");

			break;
			
		case PARCOURSNETWORKS:
			
			censusOperationLabels.add("NREVENTS");
			censusOperationLabels.add("NREXTERNALMOVES");
			
			for (RelationClassificationType mainEventType : mainRelationClassificationTypes){
				censusOperationLabels.add("SIZE_"+mainEventType);
				censusOperationLabels.add("TIES_"+mainEventType);
				censusOperationLabels.add("EGO-BETWEENNESS_"+mainEventType);
				censusOperationLabels.add("MEAN_BETWEENNESS_"+mainEventType);
				censusOperationLabels.add("MAX_BETWEENNESS_"+mainEventType);
				censusOperationLabels.add("ECCENTRICITY_"+mainEventType);
				censusOperationLabels.add("DENSITY_"+mainEventType);
				censusOperationLabels.add("DENSITY_NOLOOPS_"+mainEventType);
				censusOperationLabels.add("MEANDEGREE_"+mainEventType);
				censusOperationLabels.add("MEANDEGREE_NOLOOPS_"+mainEventType);
				censusOperationLabels.add("MEANDEGREE_NOLOOPS_NORM_"+mainEventType);
				censusOperationLabels.add("NRCOMPONENTS_"+mainEventType);
				censusOperationLabels.add("NRISOLATES_"+mainEventType);
				censusOperationLabels.add("MAXCOMPONENT_"+mainEventType);
				censusOperationLabels.add("NRCOMPONENTS_NORM_"+mainEventType);
				censusOperationLabels.add("NRISOLATES_NORM_"+mainEventType);
				censusOperationLabels.add("MAXCOMPONENT_NORM_"+mainEventType);
				censusOperationLabels.add("CONCENTRATION_"+mainEventType);
				censusOperationLabels.add("BROKERAGE_"+mainEventType);
				censusOperationLabels.add("EFFICIENT_SIZE_"+mainEventType);
				censusOperationLabels.add("EFFICIENCY_"+mainEventType);
				censusOperationLabels.add("NETWORK_RELATIONS_"+mainEventType);
				censusOperationLabels.add("CONNECTED_NETWORK_RELATIONS_"+mainEventType);
				censusOperationLabels.add("SIMILARITY_"+mainEventType);
			}

			break;
			
		case PARCOURS:
			censusOperationLabels.add("NREVENTS");
			censusOperationLabels.add("NREXTERNALMOVES");
			
			for (RelationClassificationType relationClassificationType : mainRelationClassificationTypes){
				censusOperationLabels.add("PROFILE_"+relationClassificationType);
				censusOperationLabels.add("SUPPORT_"+relationClassificationType);
				if (relationClassificationType == RelationClassificationType.PLACE){
					censusOperationLabels.add("CENTERS_"+relationClassificationType);
					censusOperationLabels.add("NRINTERNALMOVES_"+relationClassificationType);
					censusOperationLabels.add("NRDIRECTRETURNS_"+relationClassificationType);
					censusOperationLabels.add("NRCYCLES_"+relationClassificationType);
					censusOperationLabels.add("NRDIRECTRETURNS_NORM_"+relationClassificationType);
					censusOperationLabels.add("NRCYCLES_NORM_"+relationClassificationType);
				}
			}

			break;
		}

	}

	

	
	public void addNetworkTitle (String networkTitle){
		
		networkTitles.add(networkTitle);
		partitionLabels.put(networkTitle,new ArrayList<String>());
		
	}

	public String getEgoRoleName() {
		return egoRoleName;
	}

	public void setEgoRoleName(String egoRoleName) {
		this.egoRoleName = egoRoleName;
	}

	public int getMaxAge() {
		return maxAge;
	}

	public void setMaxAge(int maxAge) {
		this.maxAge = maxAge;
	}

	public int getMinAge() {
		return minAge;
	}

	public void setMinAge(int minAge) {
		this.minAge = minAge;
	}

	public String getRelationModelName() {
		return relationModelName;
	}

	public void setRelationModelName(String relationModelName) {
		this.relationModelName = relationModelName;
	}

	public GeoLevel getLevel() {
		return level;
	}
	public List<String> getCensusOperationLabels() {
		return censusOperationLabels;
	}

	public void setLabels(List<String> labels) {
		this.censusOperationLabels = labels;
	}

	public void setLevel(GeoLevel level) {
		this.level = level;
	}
	public String getPartitionLabel() {
		return individualClassificationType;
	}
	public void setPartitionLabel(String partitionLabel) {
		this.individualClassificationType = partitionLabel;
	}

	public List<String> getRelationModelNames() {
		return relationModelNames;
	}

	public String getAlterFilterRoleName() {
		return alterFilterRoleName;
	}

	public void setAlterFilterRoleName(String alterFilterRoleName) {
		this.alterFilterRoleName = alterFilterRoleName;
	}

	public Geography getGeography() {
		return geography;
	}

	public List<RelationClassificationType> getMainRelationClassificationTypes() {
		return mainRelationClassificationTypes;
	}

	public void setMainEventType(List<RelationClassificationType> mainEventTypes) {
		this.mainRelationClassificationTypes = mainEventTypes;
	}

	public void setGeography(Geography geography) {
		this.geography = geography;
	}

	public void setRelationModelNames(List<String> relationModelNames) {
		this.relationModelNames = relationModelNames;
	}

	public List<String> getRoleNames() {
		return roleNames;
	}

	public void setRoleNames(List<String> alterRoleNames) {
		this.roleNames = alterRoleNames;
	}

	public List<RelationClassificationType> getTypes() {
		return relationClassificationTypes;
	}

	public void setTypes(List<RelationClassificationType> types) {
		this.relationClassificationTypes = types;
	}

	public List<String> getNetworkTitles() {
		return networkTitles;
	}

	public String getAlterFilterAttributeLabel() {
		return alterFilterAttributeLabel;
	}

	public void setAlterAttributeLabel(String alterAttributeLabel) {
		this.alterFilterAttributeLabel = alterAttributeLabel;
	}

	public String getAlterFilterAttributeValue() {
		return alterFilterAttributeValue;
	}

	public void setAlterAttributeValue(String alterAttributeValue) {
		this.alterFilterAttributeValue = alterAttributeValue;
	}

	public Map<String, List<String>> getPartitionLabels() {
		return partitionLabels;
	}

	public List<String> getMinimalPlaceNames() {
		return minimalPlaceNames;
	}
	
	public void setPlaceNames(List<String> placeNames) {
		this.minimalPlaceNames = placeNames;
	}

	public String getPattern() {
		return pattern;
	}

	public void setPattern(String pattern) {
		this.pattern = pattern;
	}

	public String getDefaultReferentRoleName() {
		return defaultReferentRoleName;
	}

	public void setDefaultReferentRoleName(String defaultReferentRoleName) {
		this.defaultReferentRoleName = defaultReferentRoleName;
	}

	public void setCensusType(CensusType censusType) {
		this.censusType = censusType;
	}

	public CensusType getCensusType() {
		return censusType;
	}

	public List<RelationClassificationType> getEventTypes() {
		return relationClassificationTypes;
	}

	public void setEventTypes(List<RelationClassificationType> relationClassificationTypes) {
		this.relationClassificationTypes = relationClassificationTypes;
	}

	public ExpansionMode getExpansionMode() {
		return expansionMode;
	}

	public void setExpansionMode(ExpansionMode expansionMode) {
		this.expansionMode = expansionMode;
	}

	public FiliationType getFiliationType() {
		return filiationType;
	}

	public void setFiliationType(FiliationType filiationType) {
		this.filiationType = filiationType;
	}

	public String getDateLabel() {
		return dateLabel;
	}

	public void setDateLabel(String dateLabel) {
		this.dateLabel = dateLabel;
	}

	public String getStartPlaceLabel() {
		return startPlaceLabel;
	}

	public void setStartPlaceLabel(String startPlaceLabel) {
		this.startPlaceLabel = startPlaceLabel;
	}

	public String getEndPlaceLabel() {
		return endPlaceLabel;
	}

	public void setEndPlaceLabel(String endPlaceLabel) {
		this.endPlaceLabel = endPlaceLabel;
	}
	
	public String getChainClassification() {
		return chainClassification;
	}

	public void setChainClassification(String chainClassification) {
		this.chainClassification = chainClassification;
	}

	public Integer[] getDates() {
		return dates;
	}

	public void setDates(Integer[] dates) {
		this.dates = dates;
	}

	public String getLocalUnitLabel() {
		return localUnitLabel;
	}

	public void setLocalUnitLabel(String constantAttributeFilterLabel) {
		this.localUnitLabel = constantAttributeFilterLabel;
	}

	public String getStartDateLabel() {
		return startDateLabel;
	}

	public void setStartDateLabel(String startDateLabel) {
		this.startDateLabel = startDateLabel;
	}

	public String getPlaceLabel() {
		return placeLabel;
	}

	public void setPlaceLabel(String placeLabel) {
		this.placeLabel = placeLabel;
	}

	public String getEndDateLabel() {
		return endDateLabel;
	}

	public void setEndDateLabel(String endDateLabel) {
		this.endDateLabel = endDateLabel;
	}
	
	
	

	
	
	
	
	


}
