package org.tip.puck.spacetime;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

import org.tip.puck.PuckException;
import org.tip.puck.geo.GeoLevel;
import org.tip.puck.geo.Geography;
import org.tip.puck.geo.Place;
import org.tip.puck.graphs.Graph;
import org.tip.puck.graphs.GraphProfile;
import org.tip.puck.graphs.Link;
import org.tip.puck.graphs.Node;
import org.tip.puck.graphs.workers.GraphUtils;
import org.tip.puck.net.Individual;
import org.tip.puck.net.Individuals;
import org.tip.puck.net.KinType;
import org.tip.puck.net.relations.Actor;
import org.tip.puck.net.relations.Relation;
import org.tip.puck.net.relations.workers.RelationWorker;
import org.tip.puck.net.workers.IndividualValuator;
import org.tip.puck.net.workers.NetUtils;
import org.tip.puck.partitions.Cluster;
import org.tip.puck.partitions.Partition;
import org.tip.puck.spacetime.SpaceTimeCriteria.RelationClassificationType;
import org.tip.puck.util.MathUtils;
import org.tip.puck.util.PuckUtils;
import org.tip.puck.util.ToolBox;
import org.tip.puck.util.Trafo;
import org.tip.puck.util.Value;

import com.mysql.jdbc.StringUtils;

import fr.devinsy.util.StringList;

/**
 * 
 * @author Klaus Hamberger
 *
 */
public class SequenceCensus {
	
	private Individual ego;
	private List<Relation> events;
	
	private String egoRoleName;
	private int[] maxDepths;
	private GeoLevel level;
	private List<String> placeNames;
	private List<String> relationModelNames; 
	private List<String> roleNames; 
	private List<RelationClassificationType> types; 
	private String alterRoleName; // for ego networks (temp)
	private List<RelationClassificationType>  mainEventTypes; // for parcours morphology and parcours networks
	private String relationModelName;
	private Geography geography;
	private List<Place> placeList;
	private String chainClassification;
	private String alterAttributeLabel; // for parcours networks
	private String alterAttributeValue; // for parcours networks
	
	private int length;
	
	private Map<String,Set<Individual>> altersByRoles;
	private Map<Individual,List<String>> rolesByAlter;
	private Map<Individual,List<String>> relationsByAlter;
	private Map<String,List<String>> relationsByRoles;
	private Map<Individual,List<Relation>> eventsByAlter;
	private Map<Integer,List<Individual>> coverage;
	private Map<RelationClassificationType,Map<Relation,String>> eventEvaluations;
	private Map<String,Double> specificDensities;
	
	private Map<String, GraphProfile<Individual>> networkProfiles;
	private Map<String, GraphProfile<Relation>> eventNetworkProfiles;
	
	private Map<String,List<String>> partitionLabels;

	private Map<RelationClassificationType,Graph<Set<Individual>>> parcoursSimilarityTrees;

	private Map<RelationClassificationType,SequenceProfile> sequenceProfiles;
	private String pattern;
	
	
	public SequenceCensus (Sequence source, SpaceTimeCriteria criteria){
		
//		this.source = source;

		this.ego = source.getEgo();
		this.events = new ArrayList<Relation>();
		this.length = 0;
		for (Relation event : source.getEvents().values()){
			events.add(event);
			if (!isBirth(event) && !isDeath(event)){
				length++;
			}
		}
		
		this.egoRoleName = criteria.getEgoRoleName();
		this.pattern = criteria.getPattern();
		this.maxDepths = ToolBox.stringsToInts(criteria.getPattern());
		this.level = criteria.getLevel();
		this.partitionLabels = criteria.getPartitionLabels();
		this.relationModelName = criteria.getRelationModelName();
		this.relationModelNames = criteria.getRelationModelNames(); 
		this.alterRoleName = criteria.getAlterFilterRoleName();
		this.geography = criteria.getGeography();
		this.roleNames = criteria.getRoleNames();
		this.types = criteria.getTypes();
		this.chainClassification = criteria.getChainClassification();
		this.alterAttributeLabel = criteria.getAlterFilterAttributeLabel();
		this.alterAttributeValue = criteria.getAlterFilterAttributeValue();
		this.mainEventTypes = criteria.getMainRelationClassificationTypes();
		this.placeNames = criteria.getMinimalPlaceNames();
		
		if (mainEventTypes.contains(RelationClassificationType.TREES)){
			for (Relation event : events){
				event.updateReferents(criteria.getDefaultReferentRoleName());
			}
		}
		
		if ((types.contains(RelationClassificationType.MIG) || types.contains(RelationClassificationType.HOSTMIG) || types.contains(RelationClassificationType.MIGRATIONTYPE)|| types.contains(RelationClassificationType.CHILDMIGRATIONTYPE)) && (!roleNames.contains("MIG"))){
			roleNames.add("MIG");
		}
		if ((types.contains(RelationClassificationType.HOST) || types.contains(RelationClassificationType.HOSTMIG) || types.contains(RelationClassificationType.MIGRATIONTYPE) || types.contains(RelationClassificationType.CHILDMIGRATIONTYPE)) && (!roleNames.contains("HOST"))){
			roleNames.add("HOST");
		}
				
		this.sequenceProfiles = new HashMap<RelationClassificationType,SequenceProfile>();
		this.networkProfiles = new HashMap<String,GraphProfile<Individual>>();
		this.eventNetworkProfiles = new HashMap<String,GraphProfile<Relation>>();
		
		setAlters();
		setAlterRelations("ALL");
		evaluateEvents();
		
		for (String networkTitle : criteria.getNetworkTitles()){
			if (networkTitle.contains("Ego Network")){
				setEgoNetworks();
				break;
			} else if (networkTitle.contains("Parcours Network")){
				setParcoursNetworks();
				break;
			}
		}
		
		
	}
	
	private void setEgoNetworks(){

		networkProfiles.put("Ego Network", new GraphProfile<Individual>(getEgoNetwork(),ego));
		networkProfiles.put("Nonmediated Ego Network", new GraphProfile<Individual>(getNonMediatedEgoNetwork(),ego));

		setSpecificDensities("Ego Network");
		setComponentTypes("Nonmediated Ego Network");

	}
	
	private List<String> getNetworkRelations(Graph<Individual> graph){
		List<String> result;
		
		result = new ArrayList<String>();
		for (Node<Individual> node : graph.getNodes().toListSortedById()){
			Individual referent = node.getReferent();
			for (String relation : relationsByAlter.get(referent)){
				if (!result.contains(relation)){
					result.add(relation);
				}
			}
		}
		Collections.sort(result);
		
		//
		return result;

	}
	
	private List<String[]> getConnectedRelations(Partition<Node<Individual>> partition){
		List<String[]> result;
		
		result = new ArrayList<String[]>();
		
		for (Cluster<Node<Individual>> cluster : partition.getClusters()){
			List<String> relations = new ArrayList<String>();
			for (Node<Individual> node : cluster.getItems()){
				Individual referent = node.getReferent();
				for (String relation : relationsByAlter.get(referent)){
					if (!relations.contains(relation)){
						relations.add(relation);
					}
				}
			}
			Collections.sort(relations);
			for (String firstRelation : relations){
				for (String secondRelation : relations){
					if (!firstRelation.equals(secondRelation)){
						String[] pair = new String[]{firstRelation,secondRelation};
						if (!result.contains(pair)){
							result.add(pair);
						}
					}
				}
			}
		}
		
		//
		return result;
	}
	
	private String getComponentType(Cluster<Node<Individual>> component){
		String result;
		
		result = null;
		
		List<String> relations = new ArrayList<String>();
		for (Node<Individual> node : component.getItems()){
			Individual referent = node.getReferent();
			relations.addAll(relationsByAlter.get(referent));
		}
		if (relations.contains("FATHER") && (relations.contains("MOTHER"))){
			result = "PARENT";
		} else if (relations.contains("FATHER")){
			result = "FATHER";
		} else if (relations.contains("MOTHER")){
			result = "MOTHER";
		} else if (relations.contains("SPOUSE")){
			result = "SPOUSE";
		} else {
			for (String relation : relations){
				if (relation.contains("RELATIVE")){
					result = "RELATIVE";
					break;
				}
			}
		}
		if (result == null) {
			if (relations.size()>0){
				result = "OTHER";
			} else {
				result = "UNKNOWN";
			}
		}
		//
		return result;
	}
	
	private void setParcoursNetworks(){
		
		parcoursSimilarityTrees = new HashMap<RelationClassificationType,Graph<Set<Individual>>>();
		
		for (RelationClassificationType relationClassificationType : mainEventTypes){

			networkProfiles.put("Parcours Network_"+relationClassificationType, new GraphProfile<Individual>(getParcoursNetwork(relationClassificationType),ego));
			
			if (partitionLabels.get("Parcours Network_"+relationClassificationType).contains("COMPONENT")){
				setComponentTypes("Parcours Network_"+relationClassificationType);
			}
		}
		
//		if (length>1){
			eventNetworkProfiles.put("Parcours", new GraphProfile<Relation>(getParcoursGraph()));
			eventNetworkProfiles.put("Extended Parcours", new GraphProfile<Relation>(getExtendedParcoursGraph()));
//		}
		eventNetworkProfiles.put("Multiple Parcours", new GraphProfile<Relation>(getMultipleParcoursGraph()));
		
	}
	
/*	private void makeAlterRelationPartition (Graph<Individual> network){
		
		for (Node<Individual> node : network.getNodes()){
			List<String> relations = relationsByAlter.get(node.getReferent());
			if (node.getReferent()==ego){
				node.setAttribute("EGO-RELATION", "EGO");
			} else if (relations != null){
				node.setAttribute("EGO-RELATION", relations.toString());
			}
		}
	}*/

	private <E> void makeAlterRelationPartition (Graph<E> network){
		
		for (Node<E> node : network.getNodes()){
			E referent = node.getReferent();
			Individual alter = null;
			if (referent instanceof Individual){
				alter = (Individual)referent;
			} else if (referent instanceof Set<?>){
				if (((Set) referent).size()==1){
					for (Individual element : (Set<Individual>)referent){
						alter = element;
					}
				}
			} 
			
			if (alter!=null){
				List<String> relations = relationsByAlter.get(alter);
				if (alter==ego){
					node.setAttribute("EGO-RELATION", "EGO");
				} else if (relations != null){
					node.setAttribute("EGO-RELATION", relations.toString());
				}
			}
		}
	}
	
	void writePajekNetwork (StringList pajekBuffer, String title) throws PuckException{

		if (title.contains("Parcours Similarity Tree")){
			RelationClassificationType relationClassificationType = RelationClassificationType.valueOf(title.substring(title.lastIndexOf("_")+1));
			Graph<Set<Individual>> network = parcoursSimilarityTrees.get(relationClassificationType);
			makeAlterRelationPartition(network);
			GraphUtils.getDepthPartition(network);
			GraphUtils.addNodeLabelsFromPartition(network, "EGO-RELATION");
			pajekBuffer.addAll(PuckUtils.writePajekNetwork(network,partitionLabels.get(title)));
		} else if (title.contains("Network")){
			Graph<Individual> network = networkProfiles.get(title).getGraph();
			makeAlterRelationPartition(network);
			GraphUtils.addNodeLabelsFromPartition(network, "EGO-RELATION");
			pajekBuffer.addAll(PuckUtils.writePajekNetwork(network,partitionLabels.get(title)));
		} else if (title.contains("Parcours")){
			Graph<Relation> network = eventNetworkProfiles.get(title).getGraph();
			if (network.nodeCount()>0){
				pajekBuffer.addAll(PuckUtils.writePajekNetwork(network,partitionLabels.get(title)));
			}
		}
	}
	
	private void setAlters() {
		
		altersByRoles = new HashMap<String,Set<Individual>>();
		altersByRoles.put("ALL", new HashSet<Individual>());
		for (String alterRoleName : roleNames){
			altersByRoles.put(alterRoleName, new HashSet<Individual>());
		}
		rolesByAlter = new HashMap<Individual, List<String>>();
		eventsByAlter = new HashMap<Individual, List<Relation>>();
		coverage = new TreeMap<Integer,List<Individual>>();
		
		//
		for (Relation event : events){
			if (event.hasRole(ego, egoRoleName) && !isBirth(event) && !isDeath(event)){
				for (Actor actor : event.actors()){
					Individual alter = actor.getIndividual();
					String roleName = actor.getRole().getName();
					if (altersByRoles.containsKey(roleName) && (!roleName.equals(egoRoleName) || alter!=ego)){
						altersByRoles.get("ALL").add(alter);
						altersByRoles.get(actor.getRole().getName()).add(alter);
						List<String> roleNameList = rolesByAlter.get(alter);
						if (roleNameList == null){
							roleNameList = new ArrayList<String>();
							rolesByAlter.put(alter, roleNameList);
						}
						roleNameList.add(roleName);
						List<Relation> eventList = eventsByAlter.get(alter);
						if (eventList == null){
							eventList = new ArrayList<Relation>();
							eventsByAlter.put(alter, eventList);
						}
						if (!eventList.contains(event)){
							eventList.add(event);
						}
					}
				}
			}
		}
		for (Individual alter : eventsByAlter.keySet()){
			Integer cov = eventsByAlter.get(alter).size();
			List<Individual> list = coverage.get(cov);
			if (list == null){
				list = new ArrayList<Individual>();
				coverage.put(cov, list);
			}
			if (!list.contains(alter)){
				list.add(alter);
			}
		}
		
	}
	
	private void setAlterRelations (String alterRoleName){
		
		//
		relationsByAlter = NetUtils.getAlterRelations1(ego,altersByRoles.get(alterRoleName),maxDepths,relationModelNames, chainClassification,null);
		relationsByRoles = new HashMap<String,List<String>>();
		for (String roleName : altersByRoles.keySet()){
			List<String> relations = new ArrayList<String>();
			relationsByRoles.put(roleName, relations);
			for (Individual alter : altersByRoles.get(roleName)){
				if (relationsByAlter.get(alter).size()==0){
					relations.add("UNKNOWN");
				} else {
					for (String relation : relationsByAlter.get(alter)){
						if (!relations.contains(relation)){
							relations.add(relation);
						}
					}
				}
			}
		}
		
		// Add other kin types (for parcours networks)
		for (KinType kinType : KinType.values()){
			for (Individual alter : ego.getKin(kinType)){
				if (relationsByAlter.get(alter)==null){
					List<String> list = new ArrayList<String>();
					list.add(kinType.toString());
					relationsByAlter.put(alter, list);
				}
			}
		}
	}
	
	private void setPlaceList(){
		
		//
		placeList = new ArrayList<Place>();
		
		for (Relation relation : events){
			Place place = geography.getPlace(relation.getAttributeValue("END_PLACE"),level);
			if (place==null){
				place = new Place(level,"UNKNOWN");
			}
			placeList.add(place);
		}
	}
	
	private Relation getPreviousEvent(Relation event){
		Relation result;
		
		int index = events.indexOf(event)-1;
		
		if (index>-1){
			result = events.get(index);
		} else {
			result = null;
		}
		//
		return result;
	}
	
	private Double getTurnover (Relation event){
		Double result;
		
		result = 0.;
		
		Relation previousEvent = getPreviousEvent(event);
		
		if (previousEvent!=null){
			
			Individuals previousAlters = previousEvent.getIndividuals(roleNames);
			Individuals currentAlters = event.getIndividuals(roleNames);

			Double n = new Double(previousAlters.size());
			
			for (Individual individual: previousAlters){
				if (currentAlters.contains(individual)){
					result++;
				}
			}
			
			result = result/n;
			
		}
		
		//
		return result;
		
	}
	
	private GeoLevel getMaxDistance(){
		GeoLevel result;
		
		result = null;
		
		for (Relation event : events){
			if (!isBirth(event) && !isDeath(event)){
				String distance = eventEvaluations.get(RelationClassificationType.DISTANCE).get(event);
				if (!StringUtils.isNullOrEmpty(distance) && !distance.equals("UNDEFINED")){
					if (result == null || GeoLevel.valueOf(distance).compareTo(result)<0){
						result = GeoLevel.valueOf(distance);
					}
				}
			}
		}
		
		//
		return result;
	}
	
	private List<String> getRelationTypes (Individuals alters){
		List<String> result;
		
		result = new ArrayList<String>();
		for (Individual alter : alters){
			if (relationsByAlter.get(alter)!=null){
				for (String relation : relationsByAlter.get(alter)){
					if (!result.contains(relation)){
						result.add(relation);
					}
				}
				if (result.size()==0){
					result.add("OTHER");
				}
			}
		}
		//
		Collections.sort(result);
		
		//
		return result;
	}
	
	private List<String> getRelationTypes (Relation event, String roleName){
		List<String> result;
		
		Individuals alters = event.getIndividuals(roleName);
		if (roleName.equals("MIG")){
			alters.removeById(ego.getId());
		}
		
		result = getRelationTypes(alters);
		
		//
		return result;
	}
	
	private String getChildMigrationType (Relation event) {
		String result;
		
		List<String> hosts = getRelationTypes(event,"HOST");
		List<String> migs = getRelationTypes(event,"MIG");
		
		result = null;
		
		if (isBirth(event)){
			result = "BIRTH";
		} else if (getAge(event)<16){
			if (!hosts.contains("FATHER") && !hosts.contains("MOTHER") && !migs.contains("FATHER") && !migs.contains("MOTHER")){
				result = "NOPARENTS";
			}
		}
		//
		return result;
		
		
	}
	
	private String getMigrationType (Relation event){
		String result;
		
		List<String> hosts = getRelationTypes(event,"HOST");
		List<String> migs = getRelationTypes(event,"MIG");
		
		if (isBirth(event)){
			result = "BIRTH";
		} else if (isBirth(event)){
				result = "DEATH";
		} else {
			String hostType = "OTHER";
			if (hosts.size()==0){
				hostType = "NOBODY";
			} else {
				String[] types = new String[]{"EGO","FATHER","MOTHER","SPOUSE","CHILD","SIBLING"};
				boolean end = false;
				for (String type : types){
					if (hosts.contains(type)){
						hostType = type;
						end = true;
						break;
					}
				}
				if (!end && (hosts.contains("RELATIVE_AGNATIC") || hosts.contains("RELATIVE_UTERINE") || hosts.contains("RELATIVE_COGNATIC")|| hosts.contains("RELATIVE"))){
					hostType = "KIN";
				} else if (!end && hosts.contains("AFFINE")){
					hostType = "AFFINE";
				}
			}
			
			String migType = "OTHER";
			if (migs.size()==0){
				migType = "NOBODY";
			} else {
				String[] types = new String[]{"EGO","FATHER","MOTHER","SPOUSE","CHILD","SIBLING"};
				boolean end = false;
				for (String type : types){
					if (migs.contains(type)){
						migType = type;
						end = true;
						break;
					}
				}
				if (!end && (migs.contains("RELATIVE_AGNATIC") || migs.contains("RELATIVE_UTERINE") || migs.contains("RELATIVE_COGNATIC") || migs.contains("RELATIVE"))){
					migType = "KIN";
				} else if (!end && migs.contains("AFFINE")){
					migType = "AFFINE";
				}

			}
			result = "WITH_"+migType+"_TO_"+hostType;
			result = result.replaceAll("WITH_NOBODY", "ALONE");
			
		}
		
		//
		return result;
	}
	
/*	public void putEvents (Partition<String> eventPartition, EventType eventType){
		
		for (Relation event : events){
			String type = getEventType(eventType,event);
			eventPartition.put(ego.getId()+" "+event.getTypedId(), Value.valueOf(type));
		}
	}
	
	public void putEventPairs (Partition<String> eventPairPartition, EventType eventType){
		
		String previousType = null;
		for (Relation event : events){
			if (previousType == null){
				previousType = getEventType(eventType,event);
			} else {
				String nextType = getEventType(eventType,event);
				eventPairPartition.put(ego.getId()+" "+event.getTypedId(), new Value(new String[]{previousType,nextType}));
				previousType = nextType;
			}
		}
	}

	
	public void putSubSequences (Partition<String> sequencePartition, EventType eventType){
		
		String type = null;
		for (Relation event :events){
			if (type == null){
				type = getEventType(eventType,event);
			} else {
				type += "-"+getEventType(eventType,event);
			}
			if (type!=null){
				sequencePartition.put(ego.getId()+" "+event.getTypedId(), new Value(type));
			}
		}
	}*/
	
/*	public Relation getEvent(int i){
		Relation result;
		
		if (events!=null && events.size()>i){
			result = events.get(i);
		} else {
			result = null;
		}
		//
		return result;
	}*/
	
	public static boolean isBirth (Relation event) {
		return event.getAttributeValue("END_PLACE")!=null && event.getAttributeValue("START_PLACE")==null;
	}
	
	public static boolean isDeath (Relation event) {
		return event.getAttributeValue("END_PLACE")==null && event.getAttributeValue("START_PLACE")!=null;
	}
	
	private Relation getFirstMigration (){
		Relation result;

		result = null;
		for (Relation event : events){
			if (!isBirth(event)){
				if (!isDeath(event)){
					result = event;
				}
				break;
			}
		}
		
		//
		return result;
		
	}
	
	private Relation getFirstEvent(String label, String value){
		Relation result;
		
		Map<Relation, String> eventType = eventEvaluations.get(RelationClassificationType.valueOf(label));
		
		result = null;
		if (eventType != null){
			for (Relation event : events){
				if (value.equals(eventType.get(event))){
					result = event;
					break;
				}
			}
		}
		//
		return result;
	}
	
/*	public String getEventType (EventType type, Relation event){
		String result;
		
		if (type==null || event == null){
			result = null;
		} else {
			result = eventTypes.get(type).get(event);
		}
		return result;
	}
	
	public Set<String> getEventTypes (EventType type){
		Set<String> result;
		
		result = new HashSet<String>();
		for (Relation event : events){
			result.add(getEventType(type,event));
		}
		
		//
		return result;
	}*/
	
	private String getRelationTypesAsShortCutString (Relation event, String roleName){
		String result;
		
		result = Trafo.asShortCutString(getRelationTypes(event, roleName), 2);
		
		if (result == null && roleName.equals("MIG")){
			result = "SG";
		}
		
		//
		return result;
	}

	/**
	 * Generalize within RelationValuator
	 */
	private void evaluateEvents (){
		
		eventEvaluations = new HashMap<RelationClassificationType,Map<Relation,String>>();
		
		for (RelationClassificationType type : types){
			
			eventEvaluations.put(type, new HashMap<Relation,String>());
			
			if (altersByRoles==null && (type==RelationClassificationType.HOST || type==RelationClassificationType.MIG || type==RelationClassificationType.HOSTMIG || type==RelationClassificationType.MIGRATIONTYPE)){
//				setAlters();
//				setAlterRelations("ALL");
			} else	if (type==RelationClassificationType.MOVEMENT){
				setPlaceList();
			} 
			
			int i = 0;
			for (Relation event : events){
				String eventType = null;
					switch (type){
					case TYPEDID:
						eventType = event.getTypedId()+"";
						break;
					case HOST: 
						eventType = getRelationTypesAsShortCutString(event,"HOST");
						break;
					case MIG:					
						eventType = getRelationTypesAsShortCutString(event,"MIG");
						break;
					case HOSTMIG:
						eventType = getRelationTypesAsShortCutString(event,"HOST")+":"+getRelationTypesAsShortCutString(event,"MIG");
						break;
					case MIGRATIONTYPE:
						eventType = getMigrationType(event);
						break;
					case CHILDMIGRATIONTYPE:
						eventType = getChildMigrationType(event);
						break;
					case TREES:
						eventType = RelationWorker.getLinkTrees(event, true, "GENDER", pattern);
						break;
					case DISTANCE: 
						
						if (isBirth(event)){
							eventType = "BIRTH";
						} else if (isDeath(event)){
							eventType = "DEATH";
						} else {
							GeoLevel distance = getDistance(event);
							if (distance != null){
								eventType = distance.dynamic().toString();
							}
						}
						break;
						
					case PLACE: 
						
						if (isDeath(event)){
							eventType = "DEATH";
						} else {
							Place place = geography.get(event.getAttributeValue("END_PLACE"),level);
							if (place!=null){
								eventType = place.getName();
							}
						}

						break;
						
					case REGION: 
						Place region = geography.getSup(event.getAttributeValue("END_PLACE"),placeNames);
						if (region!=null){
							eventType = region.getName();
						}
						break;
						
					case MOVEMENT:
						eventType = SequenceWorker.getMovement(placeList, i);	
						break;
						
					case TURNOVER:
						eventType = MathUtils.round(getTurnover(event), 2)+"";
						break;
						
					default:
						eventType = null;
					}
					
				if (eventType==null){
					System.err.println("Event type "+type+" is null for event "+event);
					eventType = "UNDEFINED";
				}
					
				eventEvaluations.get(type).put(event, eventType);
				i++;
			}
		}
	}
	
	private GeoLevel getDistance (Relation event){
		GeoLevel result;
		
		result = null;
		
		if (!isBirth(event) && !isDeath(event)){
			Place start = geography.getByHomonym(event.getAttributeValue("START_PLACE"));
			Place end = geography.getByHomonym(event.getAttributeValue("END_PLACE"));
			if (start != null && end != null){
				Place commonAncestor = geography.getCommonAncestor(start, end);
				result = commonAncestor.getLevel();
			}
		}
		
		//
		return result;
	}

	
	public Graph getExtendedParcoursGraph () {
		Graph result;
		
		result = new Graph<Relation>();
		result.setLabel("Extended Parcours "+ego);
		
		Relation lastEvent = null;
		for (Relation event : events){
			if (lastEvent != null){
				Link<Relation> eventLink = result.addArc(lastEvent, event,2);
				eventLink.setTag(ego.getId()+"");
				for (String label : roleNames){
					for (Individual alter : event.getIndividuals(label)){
						if (label!=egoRoleName || alter!=ego){
							Link alterLink = result.addArc(event, alter, 1);
							alterLink.setTag(label);
						}
					}
				}
			}
			lastEvent = event;
		}
		
		if (events.size()==1){
			result.addNode(lastEvent);
		}
		
		for (String label : roleNames){
			Set<Individual> cluster = altersByRoles.get(label);
			if (cluster!=null){
				for (Individual alter : cluster){
					for (Relation event : alter.relations().getByModel(lastEvent.getModel())){
						if (!events.contains(event)){
							if(!isBirth(event) && !isDeath(event) && Sequence.getYear(event)!=null && Sequence.getYear(lastEvent)!=null && Sequence.getYear(event)<=Sequence.getYear(lastEvent)){
								for (Individual tertius : event.getIndividuals()){
									if (tertius.getId()> alter.getId() && result.getNode(tertius)!=null){
										Link link1 = result.addArc(event, tertius, 1);
										link1.setTag(event.getRoleNamesAsString(tertius));
										Link link2 = result.addArc(event, alter, 1);
										link2.setTag(event.getRoleNamesAsString(alter));
									}
								}
							}
						}
					}
				}
			}
		}
		
		int order = 1;
		for (Object obj : result.getNodes()){
			Node node = (Node)obj;
			
			if (node.getReferent() instanceof Individual){
				
				node.setAttribute("TYPE", "INDIVIDUAL");

				for (RelationClassificationType type : types){
					
					List<String> list = null;
					if (type==RelationClassificationType.HOST || type==RelationClassificationType.MIG || type==RelationClassificationType.HOSTMIG){
						list = relationsByAlter.get((Individual)node.getReferent());
					} else {
						list = rolesByAlter.get((Individual)node.getReferent());
					}
					if (list!=null){
						node.setAttribute(type.toString(), list.toString());
					}
				}
				
				node.setAttribute("ORDER", "0");
				
			} else if (node.getReferent() instanceof Relation){
				
				node.setAttribute("TYPE", "EVENT");
				
				for (RelationClassificationType type : types){
					
					String value = eventEvaluations.get(type).get((Relation)node.getReferent());
					if (value!=null){
						node.setAttribute(type.toString(), value);
					}
					
				}
				
				node.setAttribute("ORDER", order+"");
				order++;
			}
		}
		
		//
		return result;
	}
	
	public Value getValue (String label){
		Value result;
				
		result = null;
		
		String parameter = null;
		if (label.contains("_")){
			parameter = label.substring(label.lastIndexOf("_")+1);
		}

		GraphProfile<Individual> networkProfile = null;

		if (networkProfiles!=null){
			networkProfile = networkProfiles.get("Nonmediated Ego Network");
			if (networkProfile == null){
				networkProfile = networkProfiles.get("Parcours Network_"+parameter);
			}
		}

		if (length == 0){
			
			result = null;
			
		} else if (label.contains("PROFILE")){
			
			SequenceProfile profile = getProfile(RelationClassificationType.valueOf(parameter));
			if (profile == null){
				result = null;
			} else {
				result = Value.valueOf(profile.getProfile());
			}

		} else if (label.contains("SUPPORT")){
			
			SequenceProfile profile = getProfile(RelationClassificationType.valueOf(parameter));
			if (profile == null){
				result = null;
			} else {
				result = Value.valueOf(profile.getSupportAsList());
			}

		} else if (label.contains("CENTERS")){
			
			SequenceProfile profile = getProfile(RelationClassificationType.valueOf(parameter));
			if (profile == null){
				result = null;
			} else {
				result = Value.valueOf(profile.getCentersAsList());
			}

		} else if (label.contains("NRINTERNALMOVES")){
			
			SequenceProfile profile = getProfile(RelationClassificationType.valueOf(parameter));
			if (profile == null){
				result = null;
			} else {
				result = Value.valueOf(profile.getNrLoops());
			}

		} else if (label.contains("NREXTERNALMOVES")){
			
			result = new Value(length - getValue("NRINTERNALMOVES_PLACE").intValue());

		} else if (label.contains("NRDIRECTRETURNS_NORM")){
			
			result = Value.valueOf(MathUtils.percent(getValue("NRDIRECTRETURNS_"+parameter).intValue(), length));

		} else if (label.contains("NRDIRECTRETURNS")){
			
			SequenceProfile profile = getProfile(RelationClassificationType.valueOf(parameter));
			if (profile == null){
				result = null;
			} else {
				result = Value.valueOf(profile.getNrDirectReturns());
			}

		} else if (label.contains("NRCYCLES_NORM")){
			
			result = Value.valueOf(MathUtils.percent(getValue("NRCYCLES_"+parameter).intValue(), length));

		} else if (label.contains("NRCYCLES")){
			
			SequenceProfile profile = getProfile(RelationClassificationType.valueOf(parameter));
			if (profile == null){
				result = null;
			} else {
				result = Value.valueOf(profile.getNrCycles());
			}

		} else if (label.contains("NRALTERSPEREVENT")){
			
			result = new Value(getMeanNr(parameter));
			
		} else if (label.contains("AGEFIRST")){
			
			String[] subLabels = label.split("_");
			Integer age = null;
			if (subLabels.length==1){
				age = getAge(getFirstMigration());
			} else if (subLabels.length==3){
				age = getAge(getFirstEvent(subLabels[1],subLabels[2]));
			}
			if (age!=null){
				result = new Value(age);
			}
			
		} else if (label.equals("MAX_DISTANCE")){
			
			GeoLevel maxDistance = getMaxDistance();
			if (maxDistance != null){
				result = new Value(maxDistance);
			}
			
		} else if (label.equals("MEAN_COVERAGE")){
			
			if (length==0){
				result = null;
			} else {
				result = new Value(getMeanCoverage());
			}
			
		} else if (label.equals("MAX_COVERAGE")){
			
			if (length==0){
				result = null;
			} else {
				result = new Value(getMaxCoverage());
			}
			
		} else if (label.contains("MEAN_BETWEENNESS")){
			
			result = new Value(networkProfile.getMeanBetweenness());
			
		} else if (label.contains("MAX_BETWEENNESS")){

			result = new Value(networkProfile.getMaxNonEgoBetweenness());
			
		} else if (label.equals("MAIN_ALTERS")){
			
			result = new Value(getMaxCoverageAlters());
			
		} else if (label.contains("CENTRAL_ALTERS")){
			
			if (networkProfile.nodeCount()<3){
				result = null;
			} else {
				result = new Value(networkProfile.getCentralAlters());
			}

		} else if (label.equals("MAIN_RELATIONS")){
			
			List<String> relations = new ArrayList<String>();
			for (Individual alter : getMaxCoverageAlters()){
				for (String relation : relationsByAlter.get(alter)){
					if (!relations.contains(relation)){
						relations.add(relation);
					}
				}
			}
			result = new Value(relations);
			
		} else if (label.contains("EVENTS_")){

			String typeName = label.substring(label.indexOf("_")+1);

			List<String> types = new ArrayList<String>();
			for (Relation event : events){
				String type = eventEvaluations.get(RelationClassificationType.valueOf(typeName)).get(event);
				if (!types.contains(type)){
					types.add(type);
				}
			}
			result = new Value(types);
			
		} else if (label.contains("CENTRAL_RELATIONS")){
			
			if (networkProfile.nodeCount()<3){
				result = null;
			} else {
				List<String> relations = new ArrayList<String>();
				for (Individual alter : networkProfile.getCentralAlters()){
					for (String relation : relationsByAlter.get(alter)){
						if (!relations.contains(relation)){
							relations.add(relation);
						}
					}
				}
				result = new Value(relations);
			}
			
		} else if (label.contains("CONNECTED_NETWORK_RELATIONS")){
			
			result = new Value(getConnectedRelations(networkProfile.getNonEgoComponents()));

		} else if (label.contains("NETWORK_RELATIONS")){
			
			result = new Value(getNetworkRelations(networkProfile.getGraphWithoutEgo()));

		} else if (label.contains("RELATIONS")){
			
			result = new Value(relationsByRoles.get(parameter));
			
		} else if (label.equals("NREVENTS")){
			
			result = new Value(length);
			
		} else if (label.contains("NRALTERS")){
			
			result = new Value(altersByRoles.get(parameter).size());
			
		} else if (label.contains("SAMESEXALTERS")){

			int sameSexAlters = 0;
			for (Individual alter : altersByRoles.get(parameter)){
				if (alter.getGender()==ego.getGender()){
					sameSexAlters++;
				}
			}
			
			if (altersByRoles.get(parameter).size()==0){
				result = null;
			} else {
				result = new Value(MathUtils.percent(sameSexAlters, altersByRoles.get(parameter).size()));
			}
			
		} else if (label.contains("SAMEPLACEALTERS")){

			int samePlaceAlters = 0;
			Place egoBirthPlace = geography.getPlace(ego.getAttributeValue("BIRT_PLAC"),level);
			
			for (Individual alter : altersByRoles.get(parameter)){
				Place alterBirthPlace = geography.getPlace(alter.getAttributeValue("BIRT_PLAC"),level);
				if (egoBirthPlace != null && egoBirthPlace.equals(alterBirthPlace)){
					samePlaceAlters++;
				}
			}
			
			if (altersByRoles.get(parameter).size()==0){
				result = null;
			} else {
				result = new Value(MathUtils.percent(samePlaceAlters, altersByRoles.get(parameter).size()));
			}
			
		} else if (label.contains("EGO-BETWEENNESS")){
			
			result = new Value(networkProfile.getEgoBetweenness());
			
		} else if (label.contains("SIZE")){
			
			result = new Value(networkProfile.nodeCount());
			
		} else if (label.contains("TIES")){

			result = new Value(networkProfile.nonNullLineCount());
			
		} else if (label.contains("DENSITY")){

			result = new Value(networkProfile.density());
			
		} else if (label.contains("DENSITY_NOLOOPS")){

			result = new Value(networkProfile.densityWithoutLoops());
			
		} else if (label.contains("SDENSITY")){
			
			if (specificDensities.get(parameter)==0.){
				result = null;
			} else {
				result = new Value(specificDensities.get(parameter));
			}
			
		} else if (label.contains("MEANDEGREE")){

			result = new Value(networkProfile.meanDegree());
						
		} else if (label.contains("MEANDEGREE_NOLOOPS")){

			result = new Value(networkProfile.meanDegreeWithoutLoops());
			
		} else if (label.contains("MEANDEGREE_NORM")){

			result = new Value(networkProfile.meanDegreeNormalized());
			
		} else if (label.contains("MEANDEGREE_NOLOOPS_NORM")){

			result = new Value(networkProfile.meanDegreeWithoutLoopsNormalized());
						
		} else if (label.contains("NRCOMPONENTS")){

			result = new Value(networkProfile.getNonEgoComponents().size());
						
		} else if (label.contains("NRISOLATES")){

			result = new Value(networkProfile.getNonEgoComponents().nrSingletons());
			
		} else if (label.contains("MAXCOMPONENT")){

			result = new Value(networkProfile.getNonEgoComponents().maxClusterSize());
						
		} else if (label.contains("NRCOMPONENTS_NORM")){

			if (networkProfile.getNonEgoComponents().size()==0){
				result = null;
			} else {
				result = new Value(networkProfile.getNonEgoComponents().meanShare());
			}
			
		} else if (label.contains("MAXCOMPONENT_NORM")){

			if (networkProfile.getNonEgoComponents().size()==0){
				result = null;
			} else {
				result = new Value(networkProfile.getNonEgoComponents().maxShare());
			}
						
		} else if (label.contains("NRISOLATES_NORM")){

			if (networkProfile.getNonEgoComponents().size()==0){
				result = null;
			} else {
				result = new Value(networkProfile.getNonEgoComponents().singletonShare());
			}
			
		} else if (label.contains("BROKERAGE")){

			result = new Value(networkProfile.brokerage());
			
		} else if (label.contains("EFFICIENT_SIZE")){

			result = Value.valueOf(networkProfile.efficientSize());
			
		} else if (label.contains("EFFICIENCY")){

			result = Value.valueOf(networkProfile.efficiency());
			
		} else if (label.contains("SIMILARITY")){
			
			result = new Value(getParcoursSimilarities(RelationClassificationType.valueOf(parameter)));
			
		} else if (label.contains("ECCENTRICITY")){

			result = new Value(networkProfile.getEccentricity());

		} else if (label.contains("CONCENTRATION")) {

			result = new Value(MathUtils.herfindahl(networkProfile.getNonEgoComponents().clusterSizes()));
		}
	//
		return result;
	}
	
	private Integer getAge (Relation event){
		Integer result;
		
		result = null;

		if (event!=null){
			int year = Sequence.getYear(event);
			result = IndividualValuator.ageAtYear(ego, year);
		}
		//
		return result;
	}
	
	private double getMaxCoverage (){
		double result;
		
		if (coverage.size()>0){
			result = MathUtils.percent(((TreeMap<Integer,List<Individual>>)coverage).lastKey(),length);
		} else {
			result = 0;
		}
		//
		return result;
	}
	
	private double getMeanCoverage (){
		double result;
		
		result = 0;
		
		if (coverage.size()>0){
			double sum = 0.;
			
			for (int i : coverage.keySet()){
				result += i;
				sum++;
			}
			
			result = MathUtils.percent(result, sum*length);
		}
		//
		return result;
	}
	
	private void setSpecificDensities (String networkTitle){
		
		Graph<Individual> network = networkProfiles.get(networkTitle).getGraph();
		
		specificDensities = new HashMap<String,Double>();
		specificDensities.put("PARENT-CHILD", 0.);
		specificDensities.put("SPOUSE", 0.);
		specificDensities.put("SIBLING", 0.);
		specificDensities.put("RELATIVE", 0.);
		specificDensities.put("AFFINE", 0.);
		specificDensities.put("EMPLOYMENT", 0.);
		specificDensities.put("RENT", 0.);
		
		for (Link<Individual> link : network.getLinks()){
			String tag = link.getTag();
			tag = tag.substring(tag.indexOf("'")+1,tag.lastIndexOf("'"));
			String key = null;
			if (tag.equals("FATHER") || tag.equals("MOTHER") || tag.equals("CHILD")){
				key = "PARENT-CHILD";
			} else if (tag.equals("SPOUSE") || tag.equals("SIBLING") || tag.equals("AFFINE")){
				key = tag;
			} else if (tag.contains("RELATIVE")){
				key = "RELATIVE";
			} else if (tag.contains("EMPLOY")){
				key = "EMPLOYMENT";
			} else if (tag.equals("LANDLORD") || tag.equals("LODGER")){
				key = "RENT";
			}
			
			if (key!=null){
				specificDensities.put(key, specificDensities.get(key)+1.);
			}
		}
		
		for (String key : specificDensities.keySet()){
			specificDensities.put(key, MathUtils.percent(specificDensities.get(key),network.lineCount()));
		}
		
	}
	

	
	private List<Individual> getMaxCoverageAlters (){
		List<Individual> result;
		
		if (coverage.size()>1){
			result = coverage.get(((TreeMap<Integer,List<Individual>>)coverage).lastKey());
		} else {
			result = new ArrayList<Individual>();
		}
		//
		return result; 
	}
	
	private void setComponentTypes(String networkTitle){
		
		Partition<Node<Individual>> nonEgoComponents = networkProfiles.get(networkTitle).getNonEgoComponents();
		
		if (nonEgoComponents.size()>0){
			for (Cluster<Node<Individual>> component : nonEgoComponents.getClusters().toList()){
				String type = getComponentType(component);
				Value value = new Value(type);
				if (nonEgoComponents.getValues().contains(value)){
					int i = 1;
					while (nonEgoComponents.getValues().contains(value)){
						i++;
						value = new Value(type+" "+i);
					}
				}
				nonEgoComponents.changeClusterValue(component, value);
			}
		}
	}
	
	
	private double getMeanNr(String roleName){
		double result;
		
		result = MathUtils.percent(altersByRoles.get(roleName).size(), 100*length);
		
		//
		return result;
		
	}
	
	private Graph<Individual> getEgoNetwork (){
		Graph<Individual> result;

		//
		result = new Graph<Individual>("Ego network " + ego+" "+ relationModelName+" "+egoRoleName+"-"+alterRoleName);

		Map<String,Integer> tagMap = new HashMap<String,Integer>();

		result.addNode(ego);
		//
		for (Individual alter : altersByRoles.get(alterRoleName)) {
			result.addNode(alter);
		}
		
		List<Individual> referents = result.getReferents();
		
		//
		for (int i=0;i<referents.size();i++){
			Individual ego = referents.get(i);
			for (int j=0;j<i;j++){
				Individual alter = referents.get(j);
				
				for (String tag: NetUtils.getAlterRoles(ego, alter, maxDepths, relationModelNames, chainClassification)){
					Link<Individual> link = result.addArc(ego, alter,1);
					Integer tagNumber = tagMap.get(tag);
					if (tagNumber == null){
						tagNumber = tagMap.size()+1;
						tagMap.put(tag, tagNumber);
					}
					link.setTag(":"+tagNumber+" '"+tag+"'");
				}
			}
		}
		
		//
		NetUtils.setGenderShapes(result);
		
		//
		return result;
	}
	
	private Graph<Relation> getParcoursGraph (){
		Graph<Relation> result;
		
		result = new Graph<Relation>();
		result.setLabel("Parcours "+ego);
		
		Relation previousEvent = null;
		for (Relation event : events){
			if (previousEvent != null){
				Link<Relation> link = result.addArc(previousEvent, event,1);
				link.setTag(ego.getId()+"");
			}
			previousEvent = event;
		}
		
		if (events.size()==1){
			result.addNode(previousEvent);
		}
		
		//
		return result;
	}
	
	private Graph<Relation> getMultipleParcoursGraph(){
		Graph<Relation> result;
		
		List<Graph<Relation>> graphs = new ArrayList<Graph<Relation>>();
		
		SpaceTimeCriteria criteria = new SpaceTimeCriteria();
		criteria.setMaxAge(1000);
		criteria.setRelationModelNames(relationModelNames);
		criteria.setRelationModelName(relationModelName);
		
		for (Individual alter : altersByRoles.get(alterRoleName)) {
			Sequence fullBiography = SequenceMaker.createPersonalSequence(alter, criteria);
			Sequence sortedBiography = SequenceWorker.split(fullBiography).getById(1);
			
			SequenceCensus census = new SequenceCensus(sortedBiography,criteria);
			if (sortedBiography.getNrEvents()>0){
				Graph<Relation> alterGraph = census.getParcoursGraph();
				graphs.add(alterGraph);
			}
		}
		result = GraphUtils.fuse(graphs);
		result.setLabel("Multiple parcours "+ego);
		//
		return result;
	}
	
	private Graph<Individual> getParcoursNetwork (RelationClassificationType relationClassificationType){
		Graph<Individual> result;
		
		//
		result = new Graph<Individual>("Parcours Network_"+relationClassificationType +" "+ ego+" "+ relationModelName+" "+egoRoleName+"-"+alterRoleName);
		
		result.addNode(ego);
		//
		for (Individual alter : altersByRoles.get(alterRoleName)) {
			String value = alter.getAttributeValue(alterAttributeLabel);
			if (value!=null && value.equals(alterAttributeValue)){
				result.addNode(alter);
			}
		}
		for (Individual alter : ego.getKin()){
			String value = alter.getAttributeValue(alterAttributeLabel);
			if (value!=null && value.equals(alterAttributeValue)){
				result.addNode(alter);
			}
		}
		
		List<Individual> referents = result.getReferents();
		
		Map<Individual,Set<String>> typeMap = new HashMap<Individual,Set<String>>();
		
		SpaceTimeCriteria criteria = new SpaceTimeCriteria();
		criteria.setMaxAge(1000);
		criteria.setRelationModelNames(relationModelNames);
		criteria.setRelationModelName(relationModelName);
		criteria.setChainClassification(chainClassification);
		criteria.setPattern(pattern);
		criteria.getTypes().addAll(types);
		
		for (Individual alter : referents){

			Sequence fullBiography = SequenceMaker.createPersonalSequence(alter, criteria);
			Sequence sortedBiography = SequenceWorker.split(fullBiography).getById(1);
			
			SequenceCensus census = new SequenceCensus(sortedBiography,criteria);
			
			typeMap.put(alter, census.getProfile(relationClassificationType).getSupport());
		}
		
		double[][] distanceMatrix = new double[referents.size()][referents.size()]; 
		
		//
		for (int i=0;i<referents.size();i++){
			Individual first = referents.get(i);
			for (int j=0;j<i;j++){
				Individual second = referents.get(j);
				
				double[] shares = MathUtils.intersectionRates(typeMap.get(first), typeMap.get(second));
				
//				if (shares[2]>0.){
					result.addArc(first, second,shares[0]);
					result.addArc(second, first,shares[1]);
//				}
				
				distanceMatrix[i][j] = 100. - shares[2];
				distanceMatrix[j][i] = distanceMatrix[i][j];
				
			}
		}
		
		parcoursSimilarityTrees.put(relationClassificationType, GraphUtils.createPhylogeneticTree(ego, "Parcours similarity tree_"+relationClassificationType+"_"+ego,referents, distanceMatrix));
		
		//
		NetUtils.setGenderShapes(result);
		
		//
		return result;
	}
	
	public Partition<Link<Individual>> getParcoursLinkPartition(RelationClassificationType relationClassificationType){
		Partition<Link<Individual>> result;
		
		GraphProfile<Individual> parcoursNetworkProfile = networkProfiles.get("Parcours Network_"+relationClassificationType);
		result = parcoursNetworkProfile.getLinkPartition();
		if (result==null){
			result = GraphUtils.getLinkPartitionByKinship(parcoursNetworkProfile.getGraph());
			parcoursNetworkProfile.setLinkPartition(result);
		}
		//
		return result;
	}
	
	public Map<Value, Double[]> getParcoursSimilarities(RelationClassificationType relationClassificationType) {
		Map<Value, Double[]> result;
		
		GraphProfile<Individual> parcoursNetworkProfile = networkProfiles.get("Parcours Network_"+relationClassificationType);
		if (parcoursNetworkProfile.getLinkPartition()==null){
			parcoursNetworkProfile.setLinkPartition(GraphUtils.getLinkPartitionByKinship(parcoursNetworkProfile.getGraph()));
		}
		result = parcoursNetworkProfile.aggregateWeights();
		
		//
		return result;
	}
	

	private Graph<Individual> getNonMediatedEgoNetwork (){
		Graph<Individual> result;

		//
		result = new Graph<Individual>("Nonmediated Ego Network " + ego+" "+ relationModelName+" "+egoRoleName+"-"+alterRoleName);

		Map<String,Integer> tagMap = new HashMap<String,Integer>();

		result.addNode(ego);
		//
		for (Individual alter : altersByRoles.get(alterRoleName)) {
			result.addNode(alter);
		}
		
		List<Individual> referents = result.getReferents();
		
		//
		for (int i=0;i<referents.size();i++){
			Individual ego = referents.get(i);
			for (int j=0;j<i;j++){
				Individual alter = referents.get(j);
				
				for (String tag: NetUtils.getAlterRoles(ego, alter, maxDepths, relationModelNames, chainClassification, referents)){
					Link<Individual> link = result.addArc(ego, alter,1);
					Integer tagNumber = tagMap.get(tag);
					if (tagNumber == null){
						tagNumber = tagMap.size()+1;
						tagMap.put(tag, tagNumber);
					}
					link.setTag(":"+tagNumber+" '"+tag+"'");
				}
			}
		}
		
		//
		NetUtils.setGenderShapes(result);
		
		//
		return result;
	}

	Partition<Node<Individual>> getComponents(String networkTitle) {
		return networkProfiles.get(networkTitle).getNonEgoComponents();
	}

	private SequenceProfile getProfile(RelationClassificationType relationClassificationType){
		SequenceProfile result;
		
		result = sequenceProfiles.get(relationClassificationType);
		if (result==null){
			result = new SequenceProfile(events, eventEvaluations, relationClassificationType);
			sequenceProfiles.put(relationClassificationType, result);
		}
		
		//
		return result;
	}
	
	public static String getEgoRolePartners(Relation event, SpaceTimeCriteria criteria){
		String result;
		
		result = "";
		if (event.getModel().getName().equals(criteria.getRelationModelName())){
			result = event.getIndividuals(criteria.getEgoRoleName()).toStringAsNameList();
		}
		
		//
		return result;
	}
	
	
	

}
