package org.tip.puck.net.workers;

import java.util.HashMap;
import java.util.Map;
import java.util.Stack;
import java.util.TreeMap;

import oldcore.calc.network.Edge;
import oldcore.calc.network.OldIndividual;
import oldcore.calc.network.OldNet;

import org.tip.puck.PuckException;
import org.tip.puck.net.Individual;
import org.tip.puck.net.Individuals;
import org.tip.puck.net.KinType;
import org.tip.puck.net.Net;

public class BicomponentWorker {
	
	/**
	 * the collection of individuals to be transformed
	 */
	private Individuals individuals;
	
    /**
     * the number of steps 
     * <p> used for dsf search procedures
     */
    private int step;

	/**
	 * the stack of edges of the maximal bicomponent 
	 */
	private Stack<Edge> bicomax;
	/**
	 * the map of bicomponent sizes
	 */
    private Map<Integer,Integer> bicomponents;
	
	/**
	 * the sum of keys
	 */
	private int volume;	

	/**
	 * the number of keys
	 */
	private int number;
	
	
	public BicomponentWorker(Net net){
		   step = 0;
		   bicomax = new Stack<Edge>();
		   bicomponents = new TreeMap<Integer,Integer>();
		   individuals = net.individuals();

	}
	
	/**
	 * puts a number with its value or overwrites the value if it is lower  
	 * @param i the number to be put
	 * @param threshold the (maximum) threshold value
	 */
	static void add(Map<Integer,Integer> bicomponents, int i, int threshold){
		if (bicomponents.get(i)==null || bicomponents.get(i)> threshold) bicomponents.put(i,threshold);
	}
	
	/**
	 * copies all mappings from a second map
	 * @param map the map from which the mappings are copied
	 */
	static void addAll(Map<Integer,Integer> bicomponents, Map<Integer,Integer> map){
		for (int i : map.keySet()){
			add(bicomponents,i,map.get(i));
		}
	}
	
	
	
	   public static Net getCore(Net net){
		   Net result;
		   
		   BicomponentWorker worker = new BicomponentWorker(net);
		   
		   result = worker.getMaximalBicomponent(true,true); 
		   
		   //
		   return result;
	   }

	   
	   //Check Parameter "all"
	   /**
	    * gets the maximal (standard or matrimonial) bicomponent of the Net
	    * @param matrimonial true if matrimonial bicomponents are computed, false if standard bicomponents are computed
	    * @param all true if all bicomponents are to be kept, false if only the maximal bicomponent shall be guarded
	    * @see OldNet.CopyOfNet#setColors(int)
	    */
	   private Net getMaximalBicomponent (boolean matrimonial, boolean all){
		   Net result;
		   
		   Stack<Edge> stack = new Stack<Edge>();
		   Map<Individual,Integer> colors = new HashMap<Individual,Integer>();
		   Map<Individual,Integer> discovery = new HashMap<Individual,Integer>();
		   
		   for (Individual individual: individuals){
			   if (colors.get(individual)==null) {
				   bicomp (individual,null,stack,0,KinType.CHILD,colors, discovery, matrimonial,all); 
			   }
		   }
		   getBicomponent(stack,null,all);
		   report(bicomponents,volume);
		   
		   result = fromEdges(bicomax);
		   return result;
	   }

		
		/**
		 * <p>Algorithm from {@link http://www.ecst.csuchico.edu/~juliano/csci356/JAWAA/Bicomponents.html}
		 * @see OldNet#getMaximalBicomponent(boolean, boolean) 
		 * @see elements.nodes.NewIndividual#bicomp(Individual, Stack, int, int, OldNet, boolean, boolean)
		 */
		public int bicomp (Individual ego, Individual p, Stack<Edge> stack, int back, KinType egoKinType, Map<Individual, Integer> colors, Map<Individual,Integer>discovery, boolean matrimonial,boolean all){
		
			step++;
			colors.put(ego, 1);
			discovery.put(ego, step);
			back = discovery.get(ego);
			   
			for (KinType alterKinType : KinType.values()){
				if (matrimonial && alterKinType == KinType.PARENT && egoKinType==KinType.CHILD) continue;
				if (ego.getKin(alterKinType)==null) continue;
				for (Individual alter : ego.getKin(alterKinType)){
					if (alter!=null && (p==null || !alter.equals(p)) && (colors.get(alter)==null || colors.get(alter)<2)) {
						back = bicomp1 (ego,alter, stack, back, alterKinType, colors, discovery, matrimonial,all);
					}
				}
			}
			if (matrimonial && back<discovery.get(ego) && egoKinType==KinType.CHILD) {
				for (Individual alter : ego.getParents()){
					if (alter!=null && (p==null || !alter.equals(p)) && (colors.get(alter)==null || colors.get(alter)<2)) {
						back = bicomp1 (ego,alter, stack, back, KinType.PARENT, colors, discovery,matrimonial,all);
					}
				}
			}
			colors.put(ego,2); 
			return back;
		}	   
		
		public int bicomp1 (Individual ego, Individual a, Stack<Edge> stack, int back, KinType type, Map<Individual, Integer> colors, Map<Individual,Integer>discovery, boolean matrimonial, boolean all){
			Edge e = new Edge(ego,a,type,true);
			stack.push(e);
			if (colors.get(a)==null) { //0
				int cback = bicomp (a,ego,stack,back,type,colors, discovery,matrimonial,all); 
				if (cback>discovery.get(ego)) {
//					System.out.println("cut tail until "+e.getEgo());
					clear(stack,e);
				} else if (cback==discovery.get(ego)) {
					if (matrimonial && e.inverse(ego).getType()==KinType.PARENT && stack.peek().inverse(ego).getType()==KinType.PARENT) {
//						System.out.println("xcut tail until "+e.getEgo());
						clear(stack,e);
					} else {
//						System.out.println("get Bicomponent until "+e.getEgo());
						getBicomponent(stack,e,all);
					}
				}
				back = Math.min(back, cback);
			} else {
				back = Math.min(back, discovery.get(a));
			}
			return back;
		}	

	   
		
		/**
		 * augments the value of the ith item by one or adds the item to the list if it is not yet in it
		 * @param i the item
		 */
		void count (Map<Integer,Integer> bicomponents, int i){

			try {
				bicomponents.put(i,bicomponents.get(i)+1);
			} catch (NullPointerException npe){
				bicomponents.put (i,1);
			}
			number++;
			volume=volume+i;
		}

		//change to public report with ArrayList<String> return
		/**
		 * reports the list of items with their frequencies
		 */
		static void report (Map<Integer,Integer> bicomponents, int volume){
			
			for (int i : bicomponents.keySet()){
				System.out.println(i+"\t"+bicomponents.get(i));
			}
			System.out.println(bicomponents.size()+" "+volume);
		}
		
		


		   public static Net getMaximalBicomponent(Net net){
			   Net result;
			   
			   BicomponentWorker worker = new BicomponentWorker(net);
			   
			   result = worker.getMaximalBicomponent(false,false); 
			   
			   //
			   return result;
		   }
		   

		   public static Net getKernel(Net net){
			   Net result;
			   
			   BicomponentWorker worker = new BicomponentWorker(net);
			   
			   result = worker.getMaximalBicomponent(true,false); 
			   
			   //
			   return result;
		   }
		   
		   
		   
		   public Net fromEdges (Stack<Edge> edges) {
			   Net result;
			   
			   result = new Net();
			   while (!edges.isEmpty()){
				   Edge edge = edges.pop();
				   Individual ego = result.getCloneWithAttributes(edge.getEgo());
				   Individual alter = result.getCloneWithAttributes(edge.getAlter());
				   try {
					NetUtils.setKin(result, ego, alter, edge.getType().inverse());
				} catch (PuckException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
			   }
			   
			   return result;
		   }
		   

			
			
		   /**
		    * Removes all the edges of a bicomponent from an edge stack and adds their number to the bicomponent census
		    * @param stack the edge stack
		    * @param edge the cut edge (down to which the edge stack is emptied)
		    * @param all true if all bicomponents are to be kept, false if only the maximal bicomponent shall be guarded
		    * @see OldNet.CopyOfNet#getMaximalBicomponent(boolean, boolean)
		    * @see elements.nodes.NewIndividual#bicomp(Individual, Stack, int, int, OldNet, boolean, boolean)
		    */
		   public void getBicomponent (Stack<Edge> stack, Edge e, boolean all){
			   Stack<Edge> bico = new Stack<Edge>();
			   if (e==null) bico = stack;
			   else {
				   bico.push(stack.pop());
				   while (bico.peek()!=e){
					   bico.push(stack.pop());
				   }
			   }
			   if (!bico.isEmpty()) count(bicomponents,bico.size());
			   if (!all && bico.size()<=bicomax.size()) return;
			   if (!all) bicomax.clear();
			   while (!bico.isEmpty()) bicomax.push(bico.pop());
		   }
		   
		    /**
		     * empties a stack of elements from top to bottom until a limiting element is reached
		     * @param <E> the element type
		     * @param stack the stack to be emptied
		     * @param e the limiting element
		     * @see OldIndividual#bicomp(OldIndividual, Stack, int, int, Net, boolean, boolean)
		     */
		   private static <E>void clear (Stack<E> stack, E e){
			   while (stack.pop()!=e){};
		   }	


}
