package org.tip.puck;

import java.io.File;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tip.puck.io.ged.GEDFile;
import org.tip.puck.io.iur.IURODSFile;
import org.tip.puck.io.iur.IURTXTFile;
import org.tip.puck.io.iur.IURXLSFile;
import org.tip.puck.io.ods.ODSFile;
import org.tip.puck.io.paj.PAJFile;
import org.tip.puck.io.permutation.PermutationFile;
import org.tip.puck.io.permutation.PermutationTable;
import org.tip.puck.io.pl.PLFile;
import org.tip.puck.io.puc.PUCFile;
import org.tip.puck.io.tip.TIPFile;
import org.tip.puck.io.txt.TXTFile;
import org.tip.puck.io.xls.XLSFile;
import org.tip.puck.net.Net;
import org.tip.puck.net.relations.RelationModel;
import org.tip.puck.net.relations.workers.RelationModelTXTFile;
import org.tip.puck.net.workers.UpdateWorker;
import org.tip.puck.net.workers.UpdateWorker.UpdateMode;
import org.tip.puck.report.Report;
import org.tip.puck.report.ReportTXTFile;
import org.tip.puck.report.ReportXLSFile;

import fr.devinsy.util.FileTools;

/**
 * 
 * @author TIP
 * 
 */
public class PuckManager {

	private static final Logger logger = LoggerFactory.getLogger(PuckManager.class);

	public static String DEFAULT_CHARSET_NAME = "UTF-8";

	/**
	 * 
	 * @param target
	 * @param sourceFile
	 * @param concordanceFile
	 * @return
	 * @throws PuckException
	 */
	public static Net fuseNet(final Net target, final File sourceFile, final File concordanceFile) throws PuckException {
		Net result;

		if ((target == null) || (sourceFile == null)) {
			//
			throw PuckExceptions.INVALID_PARAMETER.create("Null parameter.");

		} else {
			//
			Net source = PuckManager.loadNet(sourceFile);

			//
			PermutationTable permutations = PermutationFile.load(concordanceFile);

			//
			result = UpdateWorker.fuse(target, source, permutations);
		}

		//
		return result;
	}

	/**
	 * 
	 * @param pathname
	 * @return
	 * @throws PuckException
	 * @throws Exception
	 */
	public static Net loadNet(final File file) throws PuckException {
		Net result;

		result = loadNet(file, DEFAULT_CHARSET_NAME);

		//
		return result;
	}

	/**
	 * 
	 * @param pathname
	 * @return
	 * @throws PuckException
	 * @throws Exception
	 */
	public static Net loadNet(final File file, final String charsetName) throws PuckException {
		Net result;

		if (file == null) {
			//
			throw PuckExceptions.INVALID_PARAMETER.create("File=[" + file + "].");

		} else if (!file.exists()) {
			//
			throw PuckExceptions.FILE_NOT_FOUND.create("File=[" + file + "].");

		} else if (!file.isFile()) {
			//
			throw PuckExceptions.NOT_A_FILE.create("File=[" + file + "].");

		} else {
			String extension = StringUtils.lowerCase(FileTools.getExtension(file.getName()));

			if (extension == null) {
				//
				throw PuckExceptions.UNSUPPORTED_FILE_FORMAT.create("File=[" + file + "].");

			} else if (extension.endsWith("ged")) {
				//
				result = GEDFile.load(file, charsetName);

			} else if (extension.endsWith("ods")) {
				//
				result = ODSFile.load(file);

			} else if (extension.endsWith("paj")) {
				//
				result = PAJFile.load(file, charsetName);

			} else if (extension.endsWith("pl")) {
				//
				result = PLFile.load(file, charsetName);

			} else if (extension.endsWith("puc")) {
				//
				result = PUCFile.load(file);

			} else if (extension.endsWith("tip")) {
				//
				result = TIPFile.load(file, charsetName);

			} else if (extension.endsWith("txt")) {
				//
				result = TXTFile.load(file);

			} else if (extension.endsWith("txt3")) {
				// In past, IUR format was named TXT version 3, TXT3.
				result = IURTXTFile.load(file);

			} else if (extension.endsWith("xls")) {
				//
				result = XLSFile.load(file);

			} else {
				//
				throw PuckExceptions.UNSUPPORTED_FILE_FORMAT.create("File=[" + file + "].");
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @return
	 */
	public static Net newEmptyNet() {
		Net result;

		result = null;

		//
		return result;
	}

	/**
	 * 
	 * @return
	 */
	public static Net newRandomNet() {
		Net result;

		result = null;

		//
		return result;
	}

	/**
	 * 
	 * @param pathname
	 * @param net
	 * @throws PuckException
	 */
	public static void saveNet(final File file, final Net net) throws PuckException {

		try {
			//
			if (file == null) {
				//
				throw PuckExceptions.INVALID_PARAMETER.create("File=[" + file + "].");

			} else if (file.isDirectory()) {
				//
				throw PuckExceptions.NOT_A_FILE.create("File=[" + file + "].");

			} else {
				// Define temporary target.
				File temporaryFile = new File(file.getAbsolutePath() + "-tmp");
				temporaryFile.delete();

				// Save.
				String extension = StringUtils.lowerCase(FileTools.getExtension(file.getName()));
				if (extension.endsWith("ged")) {
					//
					GEDFile.save(temporaryFile, net);

				} else if (extension.endsWith("ods")) {
					//
					IURODSFile.save(temporaryFile, net);
					// ODSWriter adds ".ods" extension systematically.
					temporaryFile = new File(temporaryFile.getAbsolutePath() + ".ods");

				} else if (extension.endsWith("pl")) {
					//
					PLFile.save(temporaryFile, net);

				} else if (extension.endsWith("puc")) {
					//
					PUCFile.save(temporaryFile, net, "PUCK");

				} else if (extension.endsWith("tip")) {
					//
					TIPFile.save(temporaryFile, net);

				} else if (extension.endsWith("txt")) {
					//
					IURTXTFile.save(temporaryFile, net);

				} else if (extension.endsWith("xls")) {
					//
					IURXLSFile.save(temporaryFile, net);

				} else {
					//
					throw PuckExceptions.UNSUPPORTED_FILE_FORMAT.create("File=[" + file + "].");
				}

				// Backup existing file.
				if (file.exists()) {
					File backupFile = new File(file.getAbsolutePath() + ".bak");
					backupFile.delete();
					boolean succeed = file.renameTo(backupFile);
					if (!succeed) {
						logger.warn("Backup failed for [" + file + "][" + backupFile + "]");
					}
				}

				// Rename temporary file.
				boolean succeed = temporaryFile.renameTo(file);
				if (!succeed) {
					logger.error("Error renaming temporary file for [" + temporaryFile + "]: ");
					throw PuckExceptions.IO_ERROR.create("Can't rename temporary file=[" + temporaryFile + "].");
				}

			}
		} catch (final SecurityException exception) {
			//
			exception.printStackTrace();
			logger.error("Permissions denied: " + exception.getMessage());
			throw PuckExceptions.IO_ERROR.create("Permission denied: " + exception.getMessage(), exception);
		}
	}

	/**
	 * 
	 * @param pathname
	 * @param net
	 * @throws PuckException
	 */
	public static void saveReport(final File file, final Report net) throws PuckException {

		if (file == null) {
			//
			throw PuckExceptions.INVALID_PARAMETER.create("File=[" + file + "].");

		} else if (file.isDirectory()) {
			//
			throw PuckExceptions.NOT_A_FILE.create("File=[" + file + "].");

		} else if (file.getName().toLowerCase().endsWith(".txt")) {
			//
			ReportTXTFile.save(file, net);

		} else if (file.getName().toLowerCase().endsWith(".xls")) {
			//
			ReportXLSFile.save(file, net);

		} else {
			//
			throw PuckExceptions.UNSUPPORTED_FILE_FORMAT.create("File=[" + file + "].");
		}
	}

	/**
	 * 
	 * @param pathname
	 * @param net
	 * @throws PuckException
	 */
	public static void saveRelationModel(final File file, final RelationModel model) throws PuckException {

		if (file == null) {
			//
			throw PuckExceptions.INVALID_PARAMETER.create("File=[" + file + "].");

		} else if (file.isDirectory()) {
			//
			throw PuckExceptions.NOT_A_FILE.create("File=[" + file + "].");

		} else if (file.getName().toLowerCase().endsWith(".txt")) {
			//
			RelationModelTXTFile.save(file, model);

		} else if (file.getName().toLowerCase().endsWith(".xls")) {
			//
//			ReportXLSFile.save(file, net);

		} else {
			//
			throw PuckExceptions.UNSUPPORTED_FILE_FORMAT.create("File=[" + file + "].");
		}
	}

	/**
	 * 
	 * @param net
	 * @param file
	 * @throws PuckException
	 */
	public static Net updateNetAppending(final Net target, final File file) throws PuckException {
		Net result;

		if ((target == null) || (file == null)) {
			//
			throw PuckExceptions.INVALID_PARAMETER.create("Null parameter.");

		} else {
			//
			Net source = PuckManager.loadNet(file);

			//
			result = UpdateWorker.update(target, source, UpdateMode.APPEND);
		}

		//
		return result;
	}

	/**
	 * 
	 * @param net
	 * @param file
	 * @throws PuckException
	 */
	public static Net updateNetOverwriting(final Net target, final File file) throws PuckException {
		Net result;

		if ((target == null) || (file == null)) {
			//
			throw PuckExceptions.INVALID_PARAMETER.create("Null parameter.");

		} else {
			//
			Net source = PuckManager.loadNet(file);

			//
			result = UpdateWorker.update(target, source, UpdateMode.OVERWRITE);
		}

		//
		return result;
	}
}
