package org.tip.puck.sequences;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.Set;
import java.util.Stack;
import java.util.TreeMap;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.tip.puck.PuckException;
import org.tip.puck.geo.GeoLevel;
import org.tip.puck.geo.Geography;
import org.tip.puck.geo.Place;
import org.tip.puck.graphs.Graph;
import org.tip.puck.graphs.GraphProfile;
import org.tip.puck.graphs.Node;
import org.tip.puck.io.paj.PAJFile;
import org.tip.puck.matrix.MatrixStatistics.Indicator;
import org.tip.puck.matrix.MatrixStatistics.Mode;
import org.tip.puck.net.Attribute;
import org.tip.puck.net.FiliationType;
import org.tip.puck.net.Gender;
import org.tip.puck.net.Individual;
import org.tip.puck.net.IndividualComparator.Sorting;
import org.tip.puck.net.Individuals;
import org.tip.puck.net.Net;
import org.tip.puck.net.relations.Relation;
import org.tip.puck.net.relations.Relations;
import org.tip.puck.net.workers.ExpansionMode;
import org.tip.puck.net.workers.IndividualValuator;
import org.tip.puck.net.workers.NetUtils;
import org.tip.puck.partitions.Cluster;
import org.tip.puck.partitions.Partition;
import org.tip.puck.partitions.PartitionCriteria;
import org.tip.puck.partitions.PartitionCriteria.PartitionType;
import org.tip.puck.partitions.PartitionCriteria.SizeFilter;
import org.tip.puck.partitions.PartitionCriteria.ValueFilter;
import org.tip.puck.partitions.PartitionMaker;
import org.tip.puck.report.Report;
import org.tip.puck.report.ReportChart;
import org.tip.puck.report.ReportChart.GraphType;
import org.tip.puck.report.ReportRawData;
import org.tip.puck.report.ReportTable;
import org.tip.puck.segmentation.Segmentation;
import org.tip.puck.sequences.SequenceCensusCriteria.CensusType;
import org.tip.puck.sequences.SequenceCensusCriteria.RelationAttributeType;
import org.tip.puck.statistics.StatisticsReporter;
import org.tip.puck.util.Chronometer;
import org.tip.puck.util.MathUtils;
import org.tip.puck.util.NumberedValues;
import org.tip.puck.util.PuckUtils;
import org.tip.puck.util.ToolBox;
import org.tip.puck.util.Value;

import fr.devinsy.util.StringList;

/**
 * 
 * @author Klaus Hamberger
 *
 */
public class SequenceReporter {
	
	public static Report reportSteps (Segmentation segmentation, String seedLabel, Value seedValue, String reachLabel, ExpansionMode expansionMode, int maxStep, int year) throws PuckException{
		Report result;
		
		//
		Chronometer chrono = new Chronometer();
		result = new Report("Steps");
		result.setOrigin("Sequence reporter");
		
		Report report1 = new Report("Steps");
		
		Individuals seeds = IndividualValuator.extract(segmentation.getCurrentIndividuals(), seedLabel, null, seedValue);
		Individuals neighbors = NetUtils.expand(seeds, expansionMode, FiliationType.COGNATIC, seedLabel, maxStep);
		Individuals livingNeighbors = new Individuals();
		Individuals reachedNeighbors = new Individuals();
		Individuals reachedAdultNeighbors = new Individuals();
		for (Individual neighbor : neighbors){
			if (!IndividualValuator.lifeStatusAtYear(neighbor, year).equals("DEAD")){
				livingNeighbors.put(neighbor);
			}
			if (neighbor.getAttributeValue(reachLabel)!=null){
				reachedNeighbors.put(neighbor);
				if (neighbor.getAttributeValue("CHILD")==null) {
					reachedAdultNeighbors.put(neighbor);
				}
			}
		}

		String label = "STEP_" + seedLabel.replaceAll(" ", "_") + "_" + expansionMode;

		Partition<Individual> neighborPartition = PartitionMaker.createRaw("Environment", neighbors, label);
		Partition<Individual> livingNeighborPartition = PartitionMaker.createRaw("Environment (Living)", livingNeighbors, label);
		Partition<Individual> reachedNeighborPartition = PartitionMaker.createRaw("Environment (Living)", reachedNeighbors, label);
		Partition<Individual> reachedAdultNeighborPartition = PartitionMaker.createRaw("Environment (Living)", reachedAdultNeighbors, label);
		
		PartitionCriteria stepCriteria = new PartitionCriteria(seedLabel);
		stepCriteria.setValueFilter(ValueFilter.NULL);
		PartitionCriteria splitCriteria = new PartitionCriteria(reachLabel);
		splitCriteria.setValueFilter(ValueFilter.NULL);
		
		List<ReportChart> charts = new ArrayList<ReportChart>();
		List<ReportTable> tables = new ArrayList<ReportTable>();
		
		ReportChart chart1 = StatisticsReporter.createPartitionChart(neighborPartition, stepCriteria, splitCriteria);
		chart1.setTitle("Total");
		charts.add(chart1);
		tables.add(ReportTable.transpose(chart1.createReportTableWithSum()));
	
		ReportChart chart2 = StatisticsReporter.createPartitionChart(livingNeighborPartition, stepCriteria, splitCriteria);
		chart2.setTitle("Alive");
		charts.add(chart2);
		tables.add(ReportTable.transpose(chart2.createReportTableWithSum()));
		
		splitCriteria.setLabel("GENDER");
	
		ReportChart chart3 = StatisticsReporter.createPartitionChart(reachedNeighborPartition, stepCriteria, splitCriteria);
		chart3.setTitle("Reached");
		charts.add(chart3);
		tables.add(ReportTable.transpose(chart3.createReportTableWithSum()));
	
		ReportChart chart4 = StatisticsReporter.createPartitionChart(reachedAdultNeighborPartition, stepCriteria, splitCriteria);
		chart4.setTitle("Interviewed");
		charts.add(chart4);
		tables.add(ReportTable.transpose(chart4.createReportTableWithSum()));
	
		// Manage the number of chart by line.
		for (int chartIndex = 0; chartIndex < charts.size(); chartIndex++) {
			report1.outputs().append(charts.get(chartIndex));
			if (chartIndex % 4 == 3) {
				report1.outputs().appendln();
			}
		}

		// Add chart tables.
		for (ReportTable table : tables) {
			report1.outputs().appendln(table.getTitle());
			report1.outputs().appendln(table);
		}
		
		result.outputs().append(report1);
		
		Report report2 = new Report("List");
		
		for (Cluster<Individual> cluster : neighborPartition.getClusters().toListSortedByValue()){
			report2.outputs().appendln(neighborPartition.getLabel()+" "+cluster.getLabel());
			report2.outputs().appendln();
			List<Individual> items = cluster.getItems();
			Collections.sort(items);
			for (Individual neighbor : items){
				report2.outputs().appendln(neighbor+"\t"+IndividualValuator.lifeStatusAtYear(neighbor, 2009)+"\t"+neighbor.getAttributeValue("INTERV"));
			}
			report2.outputs().appendln();
		}
		
		result.outputs().append(report2);
		
		
		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;
		
	}
	
	private static StringList getStories (Relation event){
		StringList result;
		
		result = new StringList();
		
		for (Attribute attribute : event.attributes()){
			String story = "";
			if (attribute.getLabel().contains("NOTE")){
				String[] label = attribute.getLabel().split("_");
				if (label.length>1 && StringUtils.isNumeric(label[1])){
					int id = Integer.parseInt(label[1]);
					Individual indi = event.getIndividuals().getById(id);
					story += indi.signature()+": ";
				}
				story += attribute.getValue();
				result.appendln(story);
			}
		}
		
		//
		return result;
	}
	
	public static Report reportSequenceCensus (Segmentation segmentation, CensusType censusType) throws PuckException{
		Report result;
		
		//
		Chronometer chrono = new Chronometer();
		
		result = new Report(censusType.toString());
		result.setOrigin("Sequence reporter");
		
		// Create Reports
		Report overallReport = new Report("Survey");
		Report diagramReport = new Report("Diagrams");
		Report detailReport = new Report("Details");
		Report componentReport = new Report("Components");
		Report treeReport = new Report("Trees");
				
		// Create Partition charts and tables
		List<ReportChart> charts = new ArrayList<ReportChart>();
		List<ReportTable> tables = new ArrayList<ReportTable>();
		
		// Set census criteria
		SequenceCensusCriteria censusCriteria = new SequenceCensusCriteria(censusType);
		
		// Get (coherent) itineraries
		Sequences sequences = new Sequences();
		for (Individual ego : segmentation.getCurrentIndividuals().toSortedList()){
			sequences.addRenumbered(SequenceWorker.getCoherentItinerarySegment(ego, censusCriteria));
		}

		// Proceed census
		SequencesCensus census = new SequencesCensus(sequences,censusCriteria);
		
		// Make overall report and diagrams
		overallReport.outputs().appendln("Measure\tAverage (Male)\tAverage Pos. (Male)\tMedian (Male)\tMaximum (Male)\tSum (Male)\tAverage (Female)\tAverage Pos. (Female)\tMedian (Female)\tMaximum (Female)\tSum (Female)\tAverage (All)\tAverage Pos. (All)\tMedian (All)\tMaximum (All)\tSum (All)");

		// Set partition criteria 
		for (String label : censusCriteria.getLabels()){
			
			PartitionCriteria partitionCriteria = new PartitionCriteria(label);
//			partitionCriteria.setValueFilter(ValueFilter.NULL);
			
/*			if (label.contains("PROFILE")){
				partitionCriteria.setType(PartitionType.PARTIALIZATION);
			} */
			
			if (label.equals("NREVENTS")){
				partitionCriteria.setType(PartitionType.RAW);
//				partitionCriteria.setCumulationType(CumulationType.DESCENDANT);
			} else if (label.contains("AGEFIRST")){
				partitionCriteria.setType(PartitionType.SIZED_GROUPING);
				partitionCriteria.setStart(0.);
				partitionCriteria.setSize(5.);
			} else if (label.equals("ECCENTRICITY")){
				partitionCriteria.setType(PartitionType.SIZED_GROUPING);
				partitionCriteria.setStart(-100.);
				partitionCriteria.setSize(20.);
			} else if (label.contains("COVERAGE") || label.contains("SAME")|| label.contains("NORM")|| label.contains("DENSITY")|| label.contains("BETWEENNESS") || label.contains("EFFICIENCY")|| label.contains("CONCENTRATION")){
				partitionCriteria.setType(PartitionType.SIZED_GROUPING);
				partitionCriteria.setStart(0.);
				partitionCriteria.setSize(20.);
			} else if (label.contains("MEAN") || label.contains("COVERAGE") || label.contains("PEREVENT") || label.contains("BETWEENNESS")|| label.contains("BROKERAGE")|| label.contains("EFFICIENT_SIZE")){
				partitionCriteria.setType(PartitionType.SIZED_GROUPING);
				partitionCriteria.setStart(0.);
				partitionCriteria.setSize(1.);
			} else {
				partitionCriteria.setType(PartitionType.RAW);
			}
			
			ReportChart chart = null;
			
			if (label.contains("SIMILARITY")){
				RelationAttributeType relationAttributeType = RelationAttributeType.valueOf(label.substring(label.lastIndexOf("_")+1));
				Map<Value,Double[]> similaritiesMap = census.getSimilaritiesMap(relationAttributeType);
				chart = StatisticsReporter.createMapChart(similaritiesMap, label, new String[]{"HH", "FH", "HF", "FF","All"},GraphType.LINES);
				
				for (Value key : similaritiesMap.keySet()){
					overallReport.outputs().appendln(label+"_"+key+"\t"+MathUtils.percent(similaritiesMap.get(key)[4],100));
				}
				
				
			} else if (!label.contains("ALTERS") && !label.contains("PROFILE")){
				NumberedValues values = census.getValues(label);
				
				Partition<Individual> partition = PartitionMaker.create(label, segmentation.getCurrentIndividuals(), values, partitionCriteria);
			
				PartitionCriteria splitCriteria = new PartitionCriteria(censusCriteria.getPartitionLabel());
				chart = StatisticsReporter.createPartitionChart(partition, partitionCriteria, splitCriteria);

				if (label.substring(0, 3).equals("AGE")){
					
					partitionCriteria.setType(PartitionType.RAW);
					partitionCriteria.setSizeFilter(SizeFilter.HOLES);
					partitionCriteria.setValueFilter(ValueFilter.NULL);
				
					partition = PartitionMaker.create(label, segmentation.getCurrentIndividuals(), values, partitionCriteria);
					
					if (partition.maxValue()!=null){
						ReportChart survivalChart = StatisticsReporter.createSurvivalChart(partition, splitCriteria);
						charts.add(survivalChart);
					} else {
						System.err.println(label+" no max value");
					}
				}
				
				NumberedValues[] genderedValues = PuckUtils.getGenderedNumberedValues(values, segmentation.getCurrentIndividuals());
				
				overallReport.outputs().append(label+"\t");
				for (int gender=0;gender<3;gender++){
					String sum = "";
					if (label.startsWith("NR")){
						sum = new Double(genderedValues[gender].sum()).intValue()+"";
					}
					overallReport.outputs().append(MathUtils.round(genderedValues[gender].average(),2)+"\t"+MathUtils.round(genderedValues[gender].averagePositives(),2)+"\t"+values.median()+"\t"+genderedValues[gender].max()+"\t"+sum+"\t");
				}
				overallReport.outputs().appendln();
				
			}
		
			if (chart != null) {
				charts.add(chart);
				if (label.equals("SIMILARITY")){
					tables.add(ReportTable.transpose(chart.createReportTable()));
				} else {
					ReportTable table = ReportTable.transpose(chart.createReportTableWithSum());
					tables.add(table);
					if (!label.contains("EVENTS_") && !label.contains("RELATIONS")){
						tables.add(ReportTable.normalize(table));
					}
				}

			}
		
		}
		overallReport.outputs().appendln();
		
		// Create detailed report
		detailReport.outputs().append("Nr\tEgo\tGender");
		for (String partitionLabel : censusCriteria.getLabels()){
			if (partitionLabel.contains("SIMILARITY")){
				RelationAttributeType relationAttributeType = RelationAttributeType.valueOf(partitionLabel.substring(partitionLabel.lastIndexOf("_")+1));
				detailReport.outputs().append("\tSIMILARITY_PARENT_"+relationAttributeType+"\tSIMILARITY_CHILD_"+relationAttributeType+"\tSIMILARITY_SIBLING_"+relationAttributeType+"\tSIMILARITY_SPOUSE_"+relationAttributeType);
			} else {
				detailReport.outputs().append("\t"+partitionLabel);
			}
		}
		
		Map<String,Map<String,Double>> componentChartMap = new TreeMap<String,Map<String,Double>>();
/*		String[] types = new String[]{"FATHER","MOTHER","PARENT","SPOUSE","RELATIVE","OTHER","UNKNOWN"};
		for (String type : types){
			Map<String,Double> map = new TreeMap<String,Double>();
			componentChartMap.put(type,map);
			for (Gender gender : Gender.values()){
				map.put(gender.toString(), 0.);
			}
		}*/
		
		detailReport.outputs().appendln();
		for (Individual ego : segmentation.getCurrentIndividuals().toSortedList()){
			
			if ((((censusType==CensusType.GENERAL) || (censusType==CensusType.PARCOURS) || (censusType==CensusType.PARCOURSNETWORKS) ) && (census.getValues("NREVENTS").get(ego.getId())!=null)) || ((censusType == CensusType.EGONETWORKS) && (census.getValues("SIZE").get(ego.getId())!=null))) {
				detailReport.outputs().append(ego.getId()+"\t"+ego+"\t"+ego.getGender());
				for (String label : censusCriteria.getLabels()){
					if (label.contains("SIMILARITY")){
						Value value = census.getValues(label).get(ego.getId());
						Map<Value,Double[]> indiSimilaritiesMap = (Map<Value,Double[]>)value.mapValue();
						String[] keys = new String[]{"PARENT","CHILD","SIBLING","SPOUSE"};
						for (String key : keys){
							Double[] sim = indiSimilaritiesMap.get(new Value(key));
							if (sim!=null){
								detailReport.outputs().append("\t"+MathUtils.round(sim[4], 2));
							}
						}
					} else {
						detailReport.outputs().append("\t"+census.getValues(label).get(ego.getId()));
					}
				}
				detailReport.outputs().appendln();
			}
			
			if ((censusType==CensusType.EGONETWORKS || censusType ==CensusType.PARCOURSNETWORKS)){
				
				for (String networkTitle : censusCriteria.getNetworkTitles()){
					Map<Integer,Partition<Node<Individual>>> componentsMap = census.getComponents(networkTitle);
					if (componentsMap!=null){
						Partition<Node<Individual>> components = componentsMap.get(ego.getId());
						
						componentReport.outputs().appendln("Components "+networkTitle);
						componentReport.outputs().appendln(ego+"\t"+components.size());
						int i=1;
						for (Cluster<Node<Individual>> cluster : components.getClusters().toListSortedByValue()){
							componentReport.outputs().appendln("\t"+i+"\t"+cluster.getValue()+"\t("+cluster.size()+")\t"+cluster.getItemsAsString());
							i++;
						}
						componentReport.outputs().appendln();
						
						for (Value value : components.getValues()){
							String label = value.toString();
							Map<String,Double> map = componentChartMap.get(label);
							if (map==null){
								map = new TreeMap<String,Double>();
								for (Gender gender : Gender.values()){
									map.put(gender.toString(), 0.);
								}
								componentChartMap.put(label, map);
							}
							map.put(ego.getGender().toString(), map.get(ego.getGender().toString())+1);
						}
					}
				}
			}
		}
		
		if ((censusType==CensusType.EGONETWORKS || censusType ==CensusType.PARCOURSNETWORKS)){
			ReportChart componentChart = StatisticsReporter.createChart("COMPONENTS", componentChartMap);
			charts.add(componentChart);
			tables.add(ReportTable.transpose(componentChart.createReportTableWithSum()));
		
			if (census.getRelationConnectionMatrix()!=null){
				for (ReportChart chart : census.getRelationConnectionMatrix().getCharts()){
					charts.add(chart);
				}
				tables.add(census.getRelationConnectionMatrix().getTable("Component Connections"));
			}
		
		}

		
		// SequenceAnalysis
		
		if (censusType == CensusType.PARCOURS){
			
			for (RelationAttributeType relationAttributeType : censusCriteria.getMainRelationAttributeTypes()){
				
				if (censusCriteria.getNetworkTitles().contains("Event Type Network")){

					CorrelationMatrix eventSequenceMatrix = census.getEventSequenceMatrix(relationAttributeType.toString());
					
					if (eventSequenceMatrix!=null){
						for (ReportChart chart : eventSequenceMatrix.getCharts()){
							charts.add(chart);
						}
						tables.add(eventSequenceMatrix.getTable("Event Type Sequences"));
					}
					
					overallReport.outputs().appendln();
					overallReport.outputs().appendln("Sequence Network Statistics "+relationAttributeType);
					overallReport.outputs().appendln("\tDensity\tInertia\t(Divergence)\tConcentration\t(Divergence)\tSymmetry\t(Divergence)\tCentral nodes");
					
					for (Gender gender : Gender.values()){
						GraphProfile<Cluster<String>> profile = eventSequenceMatrix.getProfile(gender);

						String centralReferents = "";
						for (Cluster<String> centralReferent : profile.getCentralReferents()){
							centralReferents+=centralReferent.getValue()+" ";
						}
						double maxBetweenness = profile.getMaxBetweenness();
						double density = profile.density();
						double endo = MathUtils.round(profile.getStatistics(Indicator.LOOPS, Mode.NORMALIZED),2);
						double endoExp = MathUtils.round(profile.getStatistics(Indicator.LOOPS, Mode.DIVERGENCE_NORMALIZED),2);
						double conc = MathUtils.round(profile.getStatistics(Indicator.CONCENTRATION, Mode.SIMPLE),2);
						double concExp = MathUtils.round(profile.getStatistics(Indicator.CONCENTRATION, Mode.DIVERGENCE),2);
						double sym = MathUtils.round(profile.getStatistics(Indicator.SYMMETRY, Mode.SIMPLE),2);
						double symExp = MathUtils.round(profile.getStatistics(Indicator.SYMMETRY, Mode.DIVERGENCE),2);

						overallReport.outputs().appendln(gender+"\t"+density+"\t"+endo+"\t"+endoExp+"\t"+conc+"\t"+concExp+"\t"+sym+"\t"+symExp+"\t"+centralReferents +"("+maxBetweenness+") betweenness centrality");
					}
					overallReport.outputs().appendln();

				}
				
				if (censusCriteria.getNetworkTitles().contains("Sequence Type Network")){

					CorrelationMatrix subSequenceMatrix = census.getSubSequenceMatrix(relationAttributeType.toString());
					
					if (subSequenceMatrix!=null){
						charts.add(subSequenceMatrix.getRamificationChart());
					}
				}
				
				
//				reportSequencePartition(overallReport,"",census.getEventPartition(eventTypeName),census.getEventPairPartition(eventTypeName),census.nrSequences(),census.nrEvents());
//				reportSequenceTree(treeReport,"",census.getSequenceTypeNetwork(eventTypeName));
				
/*				int maxPositions = census.getNrValues(eventType,eventType.toString());
						
				ReportChart diversityChart = StatisticsReporter.createDiversityChart(census.getDepthPartition(eventType.toString()), maxPositions);
				charts.add(diversityChart);*/
			}
		}
		
		// Manage the number of chart by line.
		for (int chartIndex = 0; chartIndex < charts.size(); chartIndex++) {
			diagramReport.outputs().append(charts.get(chartIndex));
			if (chartIndex % 4 == 3) {
				diagramReport.outputs().appendln();
			}
		}
		
		// Add chart tables.
		for (ReportTable table : tables) {
			diagramReport.outputs().appendln(table.getTitle());
			diagramReport.outputs().appendln(table);
		}
		
		// Finalize reports
		result.outputs().append(overallReport);
		result.outputs().append(diagramReport);
		result.outputs().append(detailReport);
		
		if (censusType == CensusType.EGONETWORKS || censusType ==CensusType.PARCOURSNETWORKS){
			result.outputs().append(componentReport);
		}
		if (censusType == CensusType.PARCOURS) {
			result.outputs().append(treeReport);
		}
		
		//addPajekData
		
		Map<String,StringList> pajekBuffers = census.getPajekBuffers();
		
		for (String title : pajekBuffers.keySet()){
			
			StringList pajekBuffer = pajekBuffers.get(title);
			if (pajekBuffer.length() != 0) {
				File targetFile = ToolBox.setExtension(ToolBox.addToName(new File(segmentation.getLabel()), "-"+title), ".paj");
				ReportRawData rawData = new ReportRawData("Export "+title+"s to Pajek", "Pajek", "paj", targetFile);
				rawData.setData(PAJFile.convertToMicrosoftEndOfLine(pajekBuffer.toString()));

				result.outputs().appendln();
				result.outputs().append(rawData);
			}
		}

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;

	}
	
/*	public static Report reportEventSpace (Segmentation segmentation) throws PuckException{
		Report result;
		
		//
		Chronometer chrono = new Chronometer();
		
		result = new Report("Parcours");
		result.setOrigin("Sequence reporter");
		
		Report overallReport = new Report("Survey");
		Report diagramReport = new Report("Diagrams");
		Report detailReport = new Report("Details");
		
		

		
		// === Build Pajek data.
		StringList pajekBufferParcours = new StringList();
		StringList pajekBufferExtendedParcours = new StringList();
		StringList pajekBufferMultipleParcours = new StringList();
		
		List<String> labels = new ArrayList<String>();
		labels.add("DATE");
		
		List<String> labels2 = new ArrayList<String>();
//		labels2.add("DATE");
//		labels2.add("TYPE");
//		labels2.add("RELATIONS");
		labels2.add("MOVEMENT");
		labels2.add("ORDER");
		
		SequenceCensusCriteria censusCriteria = new SequenceCensusCriteria();
		
		censusCriteria.getRoleNames().add("HOST");
		censusCriteria.getRoleNames().add("MIG");
		
		censusCriteria.getTypes().add(EventType.HOSTMIG);
		censusCriteria.getTypes().add(EventType.MIGRATIONTYPE);
		censusCriteria.getTypes().add(EventType.PLACE);
		censusCriteria.getTypes().add(EventType.DISTANCE);
		censusCriteria.getTypes().add(EventType.MOVEMENT);
		
		censusCriteria.getRelationModelNames().add("Apprenticeship");
		censusCriteria.getRelationModelNames().add("Housing");
		censusCriteria.getRelationModelNames().add("Employment");
		censusCriteria.getRelationModelNames().add("Friend");
		

		
		Map<String, NumberedValues> valuesMap = new HashMap<String, NumberedValues>();
		
		List<String> censusLabels = new ArrayList<String>();
		censusLabels.add("NREVENTS");
		censusLabels.add("AGEFIRST");
		censusLabels.add("AGEFIRST_MIGRATIONTYPE_NOPARENTS");
		censusLabels.add("AGEFIRST_DISTANCE_CONTINENT");
		censusLabels.add("MAX_DISTANCE");
		censusLabels.add("NRALTERS");
		for (String roleName : censusCriteria.getRoleNames()){
			censusLabels.add("MEANNR_"+roleName);
		}
		censusLabels.add("MEAN_COVERAGE");
		censusLabels.add("MAX_COVERAGE");
		for (String roleName : censusCriteria.getRoleNames()){
			censusLabels.add("RELATIONS_"+roleName);
		}
		censusLabels.add("MAX_ALTERS_COVERAGE");
		censusLabels.add("MAX_ALTERS_BETWEENNESS");
		censusLabels.add("MAX_RELATIONS_COVERAGE");
		censusLabels.add("MAX_RELATIONS_BETWEENNESS");
		for (EventType type : censusCriteria.getTypes()){
			censusLabels.add("PROFILE_"+type);
		}
		
		censusLabels.add("BETWEENNESS");
		censusLabels.add("EGO-BETWEENNESS");
		censusLabels.add("SIZE");
		censusLabels.add("TIES");
		censusLabels.add("DENSITY");
		censusLabels.add("DENSITY_NOLOOPS");
		censusLabels.add("MEANDEGREE");
		censusLabels.add("MEANDEGREE_NOLOOPS");
		censusLabels.add("NRCOMPONENTS");
		censusLabels.add("MAXCOMPONENT");
		censusLabels.add("NRCOMPONENTS_NORM");
		censusLabels.add("MAXCOMPONENT_NORM");
		censusLabels.add("BROKERAGE");
		censusLabels.add("EFFICIENT_SIZE");
		censusLabels.add("EFFICIENCY");
		censusLabels.add("JACQUARD");

		
		for (String label : censusLabels){
			valuesMap.put(label, new NumberedValues());
		}
		
		
		for (Individual ego : segmentation.getCurrentIndividuals().toSortedList()){
			
			Sequence fullBiography = SequenceWorker.getItinerary(ego, censusCriteria.getEgoRoleName());
			Sequence sortedBiography = SequenceWorker.sortAndSplit(fullBiography).getById(1);
			Sequence biography = SequenceWorker.truncate(sortedBiography, censusCriteria.getMinAge(), censusCriteria.getMaxAge());
		
			SequenceCensus census = new SequenceCensus(biography, censusCriteria);
			
//			biography.setAlterRelations(censusCriteria.getMaxDepth(), censusCriteria.getMaxOrder(), censusCriteria.getRelationModelNames());
//			SequenceWorker.setEventTypes(biography, censusCriteria);
			
			List<Graph<Relation>> graphs = new ArrayList<Graph<Relation>>();
			
			if (biography.getNrEvents()>1){
				
				Graph<Relation> egoGraph = biography.getParcoursGraph();
				pajekBufferParcours.addAll(PuckUtils.writePajekNetwork(egoGraph,labels));
				pajekBufferParcours.appendln();
				graphs.add(egoGraph);
				
				Graph<Relation> extendedGraph = census.getExtendedParcoursGraph();
				pajekBufferExtendedParcours.addAll(PuckUtils.writePajekNetwork(extendedGraph,labels2,null));
				pajekBufferExtendedParcours.appendln();
				
				for (String label : censusLabels){
					valuesMap.get(label).addValue(ego.getId(), census.getValue(label));
				}
				
				for (Individual alter : biography.alters()){
					Sequence alterBiography = SequenceWorker.getItinerary(alter, censusCriteria.getEgoRoleName());
					if (alterBiography.getNrEvents()>0){
						Graph<Relation> alterGraph = alterBiography.getParcoursGraph();
						graphs.add(alterGraph);
					}
				}
				
				Graph<Relation> fusedGraph = GraphUtils.fuse(graphs);
				fusedGraph.setLabel("Event space "+ego);
				pajekBufferMultipleParcours.addAll(PuckUtils.writePajekNetwork(fusedGraph,labels,null));
				pajekBufferMultipleParcours.appendln();
			}
		}
		
		if (pajekBufferParcours.length() != 0) {
			File targetFile = ToolBox.setExtension(ToolBox.addToName(new File(segmentation.getLabel()), "-Parcours"), ".paj");
			ReportRawData rawData = new ReportRawData("Export Parcours Networks to Pajek", "Pajek", "paj", targetFile);
			rawData.setData(PAJFile.convertToMicrosoftEndOfLine(pajekBufferParcours.toString()));

			result.outputs().appendln();
			result.outputs().append(rawData);
		}
		
		if (pajekBufferMultipleParcours.length() != 0) {
			File targetFile = ToolBox.setExtension(ToolBox.addToName(new File(segmentation.getLabel()), "-Multiparcours"), ".paj");
			ReportRawData rawData = new ReportRawData("Export Multiple Parcours Networks to Pajek", "Pajek", "paj", targetFile);
			rawData.setData(PAJFile.convertToMicrosoftEndOfLine(pajekBufferMultipleParcours.toString()));

			result.outputs().appendln();
			result.outputs().append(rawData);
		}
		
		if (pajekBufferExtendedParcours.length() != 0) {
			File targetFile = ToolBox.setExtension(ToolBox.addToName(new File(segmentation.getLabel()), "-Event Spaces"), ".paj");
			ReportRawData rawData = new ReportRawData("Export Ego Space Networks to Pajek", "Pajek", "paj", targetFile);
			rawData.setData(PAJFile.convertToMicrosoftEndOfLine(pajekBufferExtendedParcours.toString()));

			result.outputs().appendln();
			result.outputs().append(rawData);
		}
		
		// Create Partition charts and tables
		List<ReportChart> charts = new ArrayList<ReportChart>();
		List<ReportTable> tables = new ArrayList<ReportTable>();
		
		overallReport.outputs().appendln("Measure\tAverage\tMaximum");

		for (String label : censusLabels){
			
			PartitionCriteria partitionCriteria = new PartitionCriteria(label);
			
			if (label.contains("PROFILE")){
				partitionCriteria.setType(PartitionType.PARTIALIZATION);
			} else if (label.contains("AGEFIRST")){
				partitionCriteria.setType(PartitionType.SIZED_GROUPING);
				partitionCriteria.setStart(0.);
				partitionCriteria.setSize(5.);
			} else {
				partitionCriteria.setType(PartitionType.RAW);
			}
			
			NumberedValues values = valuesMap.get(label);
			
			Partition<Individual> partition = PartitionMaker.create(label, segmentation.getCurrentIndividuals(), values, partitionCriteria);
		
			PartitionCriteria splitCriteria = new PartitionCriteria("GENDER");
			ReportChart chart = StatisticsReporter.createPartitionChart(partition, partitionCriteria, splitCriteria);
		
			if (chart != null) {
				charts.add(chart);
				tables.add(ReportTable.transpose(chart.createReportTableWithSum()));
			}
			
			overallReport.outputs().appendln(label+"\t"+MathUtils.round(values.average(),2)+"\t"+values.max());
		}
				
		// Manage the number of chart by line.
		for (int chartIndex = 0; chartIndex < charts.size(); chartIndex++) {
			diagramReport.outputs().append(charts.get(chartIndex));
			if (chartIndex % 4 == 3) {
				diagramReport.outputs().appendln();
			}
		}
		
		// Add chart tables.
		for (ReportTable table : tables) {
			diagramReport.outputs().appendln(table.getTitle());
			diagramReport.outputs().appendln(table);
		}
		
		// Report individual values
		detailReport.outputs().append("Ego");
		for (String partitionLabel : censusLabels){
			detailReport.outputs().append("\t"+partitionLabel);
		}
		detailReport.outputs().appendln();
		for (Individual ego : segmentation.getCurrentIndividuals().toSortedList()){
			if (valuesMap.get("NREVENTS").get(ego.getId())!=null){
				detailReport.outputs().append(ego);
				for (String label : censusLabels){
					detailReport.outputs().append("\t"+valuesMap.get(label).get(ego.getId()));
				}
				detailReport.outputs().appendln();
			}
		}
		
		result.outputs().append(overallReport);
		result.outputs().append(diagramReport);
		result.outputs().append(detailReport);

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;

		
	}*/

	
	public static Report reportSequences (Net net, Segmentation segmentation, SequenceCensusCriteria censusCriteria, Geography geography) throws PuckException{
		Report result;
		
		String spouseFilterLabel = "INTERV";
		
		//
		Chronometer chrono = new Chronometer();
		
		result = new Report("Itineraries");
		result.setOrigin("Sequence reporter");
		
		//
		Report surveyReport = new Report("Survey");
		Report detailedReport = new Report("Details");

		Sequences sequences = new Sequences();
		
		// === Build Pajek data.
		StringList pajekBuffer = new StringList();
		
		Graph<Individual> graph = NetUtils.createRelationGraph(segmentation,censusCriteria.getRelationModelName());
		List<String> partitionLabels = new ArrayList<String>();
		pajekBuffer.addAll(PuckUtils.writePajekNetwork(graph,partitionLabels));
		pajekBuffer.appendln();

		// Create sequences
		
		for (Individual ego : segmentation.getCurrentIndividuals().toSortedList()){

			surveyReport.outputs().appendln(ego.signature());
			surveyReport.outputs().appendln();

			detailedReport.outputs().appendln(ego.signature());
			detailedReport.outputs().appendln();
			
			Sequence sequence = SequenceMaker.createPersonalSequence(ego, censusCriteria);
			sequences.add(sequence);

			// Move to appropriate census!
			for (Individual spouse : ego.spouses()){
				if (spouse.getAttributeValue(spouseFilterLabel)!=null){
//					surveyReport.outputs().appendln("Common events with spouse ("+spouse+"): "+SequenceWorker.getCommonEvents(ego,spouse,censusCriteria.getRelationModelName()).size());
				}
			}
			
			// Create coherent subsequences
			
			Sequences subSequences = SequenceWorker.sortAndSplit(sequence);
			
			for (Sequence subSequence : subSequences){
				
				if (subSequences.size()>1) {
					surveyReport.outputs().appendln(subSequence.getId());
					detailedReport.outputs().appendln(subSequence.getId());
				}
				
				
				for (Ordinal key : subSequence.getEvents().keySet()){
					
					Relation event = subSequence.getEvent(key);

					Place start = geography.getByHomonym(event.getAttributeValue("START_PLACE"));
					Place end = geography.getByHomonym(event.getAttributeValue("END_PLACE"));
					Place ancestor = geography.getCommonAncestor(start, end);
					GeoLevel commonLevel = null;
					String commonPlaceName = null;
					if (ancestor !=null){
						commonLevel = ancestor.getLevel();
						commonPlaceName = ancestor.getName();
					}
					
					String order = SequenceWorker.order(event,ego);
					
					surveyReport.outputs().appendln(key+"\t"+order+"\t("+subSequence.getAge(key.getYear())+")\t"+event.getTypedId()+"\t"+event.getAttributeValue("START_PLACE")+"\t"+event.getAttributeValue("END_PLACE")+"\t"+commonLevel+ "\t("+commonPlaceName+")\t");

					detailedReport.outputs().appendln(key+"\t"+order+"\t("+subSequence.getAge(key.getYear())+")\t"+event.getTypedId()+"\t"+event.getAttributeValue("START_PLACE")+"\t"+event.getAttributeValue("END_PLACE")+"\t"+commonLevel+ "\t("+commonPlaceName+")\t");
					detailedReport.outputs().appendln();
					detailedReport.outputs().appendln(getStories(event));
					
				}
				
				surveyReport.outputs().appendln();
				detailedReport.outputs().appendln();

			}
		}
	
		// Create sequences including life events
		
		Report biographyReport = new Report("Biographies");
		
		for (Sequence biography : SequenceMaker.createBiographies(net, segmentation).toSortedList()){
			
			Individual ego = segmentation.getCurrentIndividuals().getById(biography.getId());
		
			biographyReport.outputs().appendln(ego.signature());
			for (Ordinal key : biography.getEvents().keySet()){
				Relation event = biography.getEvent(key);
				biographyReport.outputs().appendln(key+"\t"+event.getRoles(ego).toString()+" ("+biography.getAge(key.getYear())+")\t"+event.getName()+" "+SequenceCensus.getEgoRolePartners(event, censusCriteria));
			}
			biographyReport.outputs().appendln();
		}
		
		// Create extended biographies
		
		Report extendedBiographyReport = new Report("Extended biographies");
		for (Sequence extendedBiography : SequenceMaker.createExtendedBiographies(net, segmentation, censusCriteria).toSortedList()){
			
			Individual ego = segmentation.getCurrentIndividuals().getById(extendedBiography.getId());
			
			extendedBiographyReport.outputs().appendln(ego.signature());
			for (Ordinal key : extendedBiography.getEvents().keySet()){
				Relation event = extendedBiography.getEvent(key);
				extendedBiographyReport.outputs().appendln(key+"\t"+event.getRoles(ego).toString()+" ("+extendedBiography.getAge(key.getYear())+")\t"+event.getName()+" "+SequenceCensus.getEgoRolePartners(event, censusCriteria));
			}
			extendedBiographyReport.outputs().appendln();
		}
		
		// Create Pajek File
		
		if (pajekBuffer.length() != 0) {
			File targetFile = ToolBox.setExtension(ToolBox.addToName(new File(segmentation.getLabel()), "-Relation Network"), ".paj");
			ReportRawData rawData = new ReportRawData("Export Relation Network to Pajek", "Pajek", "paj", targetFile);
			rawData.setData(PAJFile.convertToMicrosoftEndOfLine(pajekBuffer.toString()));

			result.outputs().appendln();
			result.outputs().append(rawData);
		}
		
		result.outputs().append(surveyReport);
		result.outputs().append(detailedReport);
		result.outputs().append(biographyReport);
		result.outputs().append(extendedBiographyReport);

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;
	}
	
/*	public static Report reportEventNetworks (Segmentation segmentation, String eventType) throws PuckException{
		Report result;
		
		//
		Chronometer chrono = new Chronometer();
		
		//
		result = new Report("Events "+eventType);
		result.setOrigin("Sequence reporter");

		Report overallReport = new Report("Survey");
		Report diagramReport = new Report("Diagrams");
		Report treeReport = new Report("Trees");
		Report detailReport = new Report ("Details");
		StringList morph = new StringList();
		
//		Partition<Individual> patterns = new Partition<Individual>();
		
		SequenceCensusCriteria censusCriteria = new SequenceCensusCriteria();
		censusCriteria.setType(EventType.valueOf(eventType));

		PartitionCriteria criteria = new PartitionCriteria(censusCriteria.getPartitionLabel());
		
		censusCriteria.getRelationModelNames().add("Apprenticeship");
		censusCriteria.getRelationModelNames().add("Housing");
		censusCriteria.getRelationModelNames().add("Employment");

		// === Build Pajek data.
		StringList pajekBufferEventTypeNetwork = new StringList();
		
		// Create Event Graph
		
		Partition<Individual> individualPartition = PartitionMaker.create("Sequences", segmentation.getCurrentIndividuals(), criteria);

		Map<Value,Sequences> sortedBiographiesByValue = new HashMap<Value,Sequences>();
		Map<Value,Partition<String>> eventPartitionByValue = new HashMap<Value,Partition<String>>();
		Map<Value,Partition<String>> subsequencePartitionByValue = new HashMap<Value,Partition<String>>();
		Map<Value,Partition<String>> sequencePartitionByValue = new HashMap<Value,Partition<String>>();
		Map<Value,Graph<Cluster<String>>> eventTypeNetworksByValue = new HashMap<Value,Graph<Cluster<String>>>();
		Map<Value,Graph<Cluster<String>>> sequenceTypeNetworksByValue = new HashMap<Value,Graph<Cluster<String>>>();
		
		Sequences sortedBiographies = new Sequences();
		for (Value value : individualPartition.getValues()){
			sortedBiographiesByValue.put(value, new Sequences());
		}
		
		// Get event networks
		for (Individual ego : segmentation.getCurrentIndividuals().toSortedList()){
	
			Sequence biography = SequenceWorker.getItinerary(ego, censusCriteria.getEgoRoleName());
			SequenceWorker.setEventTypes(biography, censusCriteria);
			
//			biography.setAlters(egoRoleName);
//			biography.setAlterRelations(maxDepth, maxOrder, relationModelNames);

			Sequences subBiographies = SequenceWorker.sortAndSplit(biography);
			Sequences sortedBiographiesByCluster = sortedBiographiesByValue.get(individualPartition.getValue(ego));
			
			for (Sequence subBiography : subBiographies){

				sortedBiographies.addRenumbered(subBiography);
				sortedBiographiesByCluster.addRenumbered(subBiography);
				
				morph.appendln(ego.signature()+"\t"+subBiography.getNrEvents()+" events"+"\t"+subBiography.getNrEventTypes()+" eventTypes");
				morph.appendln("\tItinerary: "+subBiography.getEventTypePattern());
				morph.appendln("\tStations: "+subBiography.getEventTypesAsString());
				morph.appendln();
				
				SequenceCensus census = new SequenceCensus(subBiography,level);
				patterns.put(ego, new Value(census.pattern()));
				
				morph.appendln(ego.signature()+"\t"+census.getLength()+"\t"+census.getPlaces().size());
				morph.appendln("Pattern: "+census.pattern());
				morph.appendln("Itinerary: "+census.getStations().toString());
				morph.appendln("Places: "+census.getPlaces().toString());
				morph.appendln();

			}
		}
		
		PartitionCriteria morphCriteria = new PartitionCriteria("_");
		morphCriteria.setType(PartitionType.RAW);
		ReportChart morphChart = StatisticsReporter.createPartitionChart(patterns, morphCriteria, new PartitionCriteria("GENDER"));
		morphologyReport.outputs().appendln(morphChart);
		morphologyReport.outputs().appendln(ReportTable.transpose(morphChart.createReportTableWithSum()));
		detailReport.outputs().appendln(morph);
		detailReport.outputs().appendln();

		Partition<String> eventPartition = SequenceWorker.getEventPartition(sortedBiographies);
		Partition<String> subsequencePartition = SequenceWorker.getSubsequencePartition(sortedBiographies);
		Graph<Cluster<String>> eventTypeNetwork = SequenceWorker.getSequenceNetwork(sortedBiographies, eventPartition);
		Graph<Cluster<String>> sequenceTypeNetwork = SequenceWorker.getSequenceNetwork(sortedBiographies, subsequencePartition);
		Partition<String> sequencePartition = SequenceWorker.getSequencePartition(sortedBiographies);

		pajekBufferEventTypeNetwork.addAll(PuckUtils.writePajekNetwork(sequenceTypeNetwork, ClusterNetworkReporter.ALLIANCE_NETWORK_PARTITION_LABELS));
		pajekBufferEventTypeNetwork.addAll(PuckUtils.writePajekNetwork(eventTypeNetwork, ClusterNetworkReporter.ALLIANCE_NETWORK_PARTITION_LABELS));

		for (Value value : individualPartition.getValues()){
			eventPartitionByValue.put(value, SequenceWorker.getEventPartition(sortedBiographiesByValue.get(value)));
			subsequencePartitionByValue.put(value, SequenceWorker.getSubsequencePartition(sortedBiographiesByValue.get(value)));
			sequencePartitionByValue.put(value, SequenceWorker.getSequencePartition(sortedBiographiesByValue.get(value)));
		
			eventTypeNetworksByValue.put(value, SequenceWorker.getSequenceNetwork(sortedBiographiesByValue.get(value),eventPartitionByValue.get(value)));
			sequenceTypeNetworksByValue.put(value, SequenceWorker.getSequenceNetwork(sortedBiographiesByValue.get(value),subsequencePartitionByValue.get(value)));
			
			pajekBufferEventTypeNetwork.addAll(PuckUtils.writePajekNetwork(eventTypeNetworksByValue.get(value), ClusterNetworkReporter.ALLIANCE_NETWORK_PARTITION_LABELS,null));
			pajekBufferEventTypeNetwork.addAll(PuckUtils.writePajekNetwork(sequenceTypeNetworksByValue.get(value), ClusterNetworkReporter.ALLIANCE_NETWORK_PARTITION_LABELS,null));

		}

		// Create diagrams
		
		List<ReportChart> charts = new ArrayList<ReportChart>();
//		List<ReportTable> tables = new ArrayList<ReportTable>();

		// Sequence analysis
		
		reportSequencePartition(overallReport,"",eventPartition,sequencePartition,sortedBiographies.size(),sortedBiographies.getNrEvents());
		reportSequenceTree(treeReport,"",sequenceTypeNetwork);
		
		charts.add(StatisticsReporter.createPartitionChartBySize(eventPartition,censusCriteria.getThreshold()));
		charts.add(StatisticsReporter.createPartitionChartBySize(sequencePartition,censusCriteria.getThreshold()));
		
		for (Value value : individualPartition.getValues()){
			
			reportSequencePartition(overallReport,value.toString(),eventPartitionByValue.get(value),sequencePartitionByValue.get(value),sortedBiographiesByValue.get(value).size(),sortedBiographiesByValue.get(value).getNrEvents());
			reportSequenceTree(treeReport,value.toString(),sequenceTypeNetworksByValue.get(value));
			
			charts.add(StatisticsReporter.createPartitionChartBySize(eventPartitionByValue.get(value),censusCriteria.getThreshold()));
			charts.add(StatisticsReporter.createPartitionChartBySize(sequencePartitionByValue.get(value),censusCriteria.getThreshold()));
		}
		
		
		Partition<Relation> types = new Partition<Relation>("TYPES");
		PartitionCriteria typeCriteria = new PartitionCriteria("TYPE");
		typeCriteria.setValueFilter(ValueFilter.NULL);
		PartitionCriteria splitCriteria1 = new PartitionCriteria("DISTANCE");
		splitCriteria1.setValueFilter(ValueFilter.NULL);

		ReportChart typeChart = StatisticsReporter.createPartitionChart(types, typeCriteria, splitCriteria1);
		charts.add(typeChart);
		tables.add(ReportTable.transpose(typeChart.createReportTableWithSum()));
		
		// Manage the number of chart by line.
		for (int chartIndex = 0; chartIndex < charts.size(); chartIndex++) {
			diagramReport.outputs().append(charts.get(chartIndex));
			if (chartIndex % 4 == 3) {
				diagramReport.outputs().appendln();
			}
		}


		// Create Pajek Event Network
		if (pajekBufferEventTypeNetwork.length() != 0) {
			File targetFile = ToolBox.setExtension(ToolBox.addToName(new File(segmentation.getLabel()), "-Ego Networks"), ".paj");
			ReportRawData rawData = new ReportRawData("Export Event Networks to Pajek", "Pajek", "paj", targetFile);
			rawData.setData(PAJFile.convertToMicrosoftEndOfLine(pajekBufferEventTypeNetwork.toString()));

			result.outputs().appendln();
			result.outputs().append(rawData);
		}
		
		result.outputs().append(overallReport);
		result.outputs().append(diagramReport);
		result.outputs().append(treeReport);
		result.outputs().append(detailReport);

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;
	}*/
	
	
/*	private static void reportSequencePartition(Report report, String value, Partition<String> eventPartition, Partition<String> eventPairPartition, int nrTotalSequences, int nrTotalEvents){
		
		report.outputs().appendln("Events "+value);
		report.outputs().appendln("Type\tnrEvents\t%\tnrItineraries\t%\tEvents/Itinerary");
		for (Cluster<String> cluster : eventPartition.getClusters().toListSortedByDescendingSize()){
			int nrEvents = cluster.size();
			int nrSequences = sequenceNumber(cluster);
			report.outputs().appendln(cluster.getValue()+"\t"+nrEvents+"\t"+MathUtils.percent(nrEvents, nrTotalEvents)+"\t"+nrSequences+"\t"+MathUtils.percent(nrSequences, nrTotalSequences)+"\t"+MathUtils.percent(nrEvents, 100*nrSequences));
		}
		report.outputs().appendln();

		report.outputs().appendln("Sequences "+value);
		report.outputs().appendln("Type\tnrEvents\t%\tnrItineraries\t%\tEvents/Itinerary\tConfidence\tLift");
		for (Cluster<String> cluster : eventPairPartition.getClusters().toListSortedByDescendingSize()){
			int nrEvents = cluster.size();
			int nrSequences = sequenceNumber(cluster);
			String[] values = cluster.getValue().stringArrayValue();
			Cluster<String> previous = eventPartition.getCluster(Value.valueOf(values[0]));
			Cluster<String> next = eventPartition.getCluster(Value.valueOf(values[1]));
			double confidence = MathUtils.percent(nrEvents, previous.size());
			double lift = MathUtils.round(confidence/MathUtils.percent(next.size(), nrTotalEvents),2);
					
			report.outputs().appendln(values[0]+" - "+values[1]+"\t"+nrEvents+"\t"+MathUtils.percent(nrEvents, nrTotalEvents)+"\t"+nrSequences+"\t"+MathUtils.percent(nrSequences, nrTotalSequences)+"\t"+MathUtils.percent(nrEvents, 100*nrSequences)+"\t"+confidence+"\t"+lift);
		}
		report.outputs().appendln();
	}*/
	
	private static void reportSequenceTree (Report report, String value, Graph<Cluster<String>> sequenceTypeNetwork){
		
		report.outputs().appendln("Sequence Tree "+value);
		Node<Cluster<String>> start = sequenceTypeNetwork.getNode(1);
		Stack<Node<Cluster<String>>> stack = new Stack<Node<Cluster<String>>>();
		report.outputs().appendln(start.getReferent()+"\t"+start.getReferent().size());
		stack.push(start);
		
		while (!stack.isEmpty()){
			Node<Cluster<String>> node = stack.pop();
			for (Node<Cluster<String>> next : node.getOutNodes().toListSortedByLabel()){
				report.outputs().appendln(next.getReferent()+"\t"+next.getReferent().size());
				stack.push(next);
			}
		}
		report.outputs().appendln();

	}
	
	public static Report reportResidenceSequence(Segmentation segmentation){
		Report result;
		
		Integer[] dates = new Integer[]{2004,2009};
				
		result = new Report("Residence Sequence");
		
		Relations residences = segmentation.getAllRelations().getByModelName("Residence");
		
		Map<Integer,Map<Individual,String>> residencesByIndividuals = new TreeMap<Integer,Map<Individual,String>>();
		Map<Integer,Individuals> residencesByHouses = new TreeMap<Integer,Individuals>();
		
		for (Relation residence : residences){
			String dateAsString = residence.getAttributeValue("MOMENT_DATE");
			
			if (dateAsString!=null){
				Integer date = Integer.parseInt(dateAsString);
				Individual individual = residence.getFirstIndividual();
				if (residencesByIndividuals.get(date)==null){
					residencesByIndividuals.put(date, new TreeMap<Individual,String>());
				}
				
				String value = "";
				if (residence.getAttributeValue("HOUSE")!=null){
					value += residence.getAttributeValue("HOUSE");
				} else if (residence.getAttributeValue("PLACE")!=null){
					value += residence.getAttributeValue("PLACE");
				}
				if (residence.getAttributeValue("MODE")!=null){
					value += " "+residence.getAttributeValue("MODE");
				}
				value+="\t";

				if (residence.getAttributeValue("START_DATE")!=null){
					value += residence.getAttributeValue("START_DATE");
				}
				if (residence.getAttributeValue("END_DATE")!=null){
					value += "-"+residence.getAttributeValue("END_DATE");
				}
				if (residence.getAttributeValue("NOTE")!=null){
					value += " ["+residence.getAttributeValue("NOTE")+"]";
				}
				residencesByIndividuals.get(date).put(individual, value);
				
				String houseIdAsString = residence.getAttributeValue("HOUSE");
				if (houseIdAsString!=null && Arrays.asList(dates).contains(date)){
					Integer houseId = Integer.parseInt(houseIdAsString);
					Individuals individuals = residencesByHouses.get(houseId);
					if (individuals == null){
						individuals = new Individuals();
						residencesByHouses.put(houseId, individuals);
					}
					individuals.put(individual);
				}
			}
		}
		
		StringList list = new StringList();
		
		for (Integer houseId : residencesByHouses.keySet()){
			list.appendln(houseId);
			list.appendln();
			
			List<Individual> residents = residencesByHouses.get(houseId).toSortedList(Sorting.BIRT_YEAR);
			for (Individual individual : residents){
				list.append(individual.signature()+" ("+IndividualValuator.lifeStatusAtYear(individual, 2015)+")\t");
				for (Integer date : dates){
					String value = residencesByIndividuals.get(date).get(individual);
					if (value!=null){
						list.append(value+"\t");
					} else {
						Integer deathYear = IndividualValuator.getDeathYear(individual);
						Integer birthYear = IndividualValuator.getBirthYear(individual);
						
						if (deathYear!=null && deathYear<=date){
							list.append("+"+deathYear+"\t\t");
						} else if (birthYear!=null && birthYear>=date){
							list.append("*"+birthYear+"\t\t");
						} else {
							list.append("?\t\t");
						}
					}
				}
				list.appendln();
				
			}
			list.appendln();
		}
		result.outputs().append(list);
		
		//
		return result;
		
	}
	
	private static void reportTriangles(Report report, Segmentation segmentation) throws PuckException{
		
		Map<Integer, Partition<EventTriangle>> trianglesMap = new HashMap<Integer,Partition<EventTriangle>>(); 
		Partition<EventTriangle> allTriangles = new Partition<EventTriangle>();
		Partition<Individual> triangleTypes = new Partition<Individual>();
		
		// Get triangles
		
		for (Individual ego : segmentation.getCurrentIndividuals()){
			Individuals individuals = ego.getRelated("Migevent");
			individuals.add(ego);
			Partition<EventTriangle> triangles = SequenceWorker.getTriangles(individuals, "Migevent");
			allTriangles.add(triangles);
			trianglesMap.put(ego.getId(), triangles);
			for (Cluster<EventTriangle> cluster : triangles.getClusters()){
				triangleTypes.put(ego, cluster.getValue());
			}
		}
		
		// Report triangles
		Report trianglesReport = new Report("Triangles");
		trianglesReport.outputs().appendln("Type\tnrTriangles\tnrEgoNetworks");
		
		for (Cluster<EventTriangle> cluster: allTriangles.getClusters().toListSortedByValue()){
			trianglesReport.outputs().appendln(cluster.getValue()+"\t"+cluster.size()+"\t"+triangleTypes.getCluster(cluster.getValue()).size());
		}
		trianglesReport.outputs().appendln();

		PartitionCriteria partitionCriteria = new PartitionCriteria("Triangles");
		ReportChart chart5 = StatisticsReporter.createPartitionChart(allTriangles, partitionCriteria, null);
		trianglesReport.outputs().appendln(chart5);

		for (Individual ego : segmentation.getCurrentIndividuals().toSortedList()){
			Partition<EventTriangle> triangles = trianglesMap.get(ego.getId());
			trianglesReport.outputs().appendln(ego+"\t"+triangles.size()+" types");
			for (Cluster<EventTriangle> cluster : triangles.getClusters().toListSortedByValue()){
				trianglesReport.outputs().appendln(cluster.getValue()+"\t"+cluster.size());
				for (EventTriangle triangle : cluster.getItems()){
					trianglesReport.outputs().appendln(triangle.getEventPattern());
				}
				trianglesReport.outputs().appendln();
			}
			trianglesReport.outputs().appendln();
		}
	}
	
	private static int sequenceNumber (Cluster<String> cluster){
		int result;
		
		Set<String> set = new HashSet<String>();
		
		for (String string : cluster.getItems()){
			set.add(string.split("\\s")[0]);
		}
		
		result = set.size();
		
		//
		return result;
	}

	
/*	public static Report reportEgoNetworks (Segmentation segmentation, String alterRoleName) throws PuckException{
		Report result;
		
		//
		Chronometer chrono = new Chronometer();
		
		//
		result = new Report("Ego Networks "+alterRoleName);
		result.setOrigin("Sequence reporter");

		Report overallReport = new Report("Survey");
		Report diagramReport = new Report("Diagrams");
		Report detailReport = new Report("Details");
		Report componentReport = new Report("Components");
		
//		Sequences biographies = new Sequences();
		
		// === Build Pajek data.
		StringList pajekBuffer = new StringList();
		
		// Create total relation graph
		Graph<Individual> graph = NetUtils.createRelationGraph(segmentation,"Migevent");
		Map<Integer,Integer> nodeIds = GraphUtils.getIds(graph);

		double[] graphBetweenness = GraphUtils.betweenness(graph);
		
		Map<String, NumberedValues> valuesMap = new HashMap<String,NumberedValues>(); 
		Map<Integer, Partition<EventTriangle>> trianglesMap = new HashMap<Integer,Partition<EventTriangle>>(); 
		Partition<EventTriangle> allTriangles = new Partition<EventTriangle>();
		Partition<Individual> triangleTypes = new Partition<Individual>();
		
		List<String> pajekPartitionLabels = new ArrayList<String>();
		pajekPartitionLabels.add("BETWEENNESS");

		List<String> partitionLabels = new ArrayList<String>();
		partitionLabels.add("BETWEENNESS");
		partitionLabels.add("EGO-BETWEENNESS");
		partitionLabels.add("SIZE");
		partitionLabels.add("TIES");
		partitionLabels.add("DENSITY");
		partitionLabels.add("DENSITY_NOLOOPS");
		partitionLabels.add("MEANDEGREE");
		partitionLabels.add("MEANDEGREE_NOLOOPS");
		partitionLabels.add("NRCOMPONENTS");
		partitionLabels.add("MAXCOMPONENT");
		partitionLabels.add("NRCOMPONENTS_NORM");
		partitionLabels.add("MAXCOMPONENT_NORM");
		partitionLabels.add("BROKERAGE");
		partitionLabels.add("EFFICIENT_SIZE");
		partitionLabels.add("EFFICIENCY");
		partitionLabels.add("JACQUARD");
		
		List<String> roleNames = new ArrayList<String>();
		for (Role role : segmentation.getCurrentRelations().getRelationModels().getByName("Migevent").roles()){
			roleNames.add(role.getName());
			partitionLabels.add("NR"+role.getName());
			partitionLabels.add("NR"+role.getName()+"_NORM");
		}
		
		for (String partitionLabel : partitionLabels){
			valuesMap.put(partitionLabel, new NumberedValues());
		}
		
		SequenceCensusCriteria criteria = new SequenceCensusCriteria();
		criteria.setAlterRoleName(alterRoleName);
		
		// Get ego networks
		for (Individual ego : segmentation.getCurrentIndividuals().toSortedList()){
	
			Sequence biography = SequenceWorker.getItinerary(ego, criteria.getEgoRoleName());
//			biographies.add(biography);
	
			Graph<Individual> egoNetwork = NetUtils.createRelationGraph(ego, "Migevent", criteria.getEgoRoleName(), criteria.getAlterRoleName());
			egoNetwork.setLabel("ego Network "+ego);
			pajekBuffer.addAll(PuckUtils.writePajekNetwork(egoNetwork,pajekPartitionLabels));
			pajekBuffer.appendln();
			
			Graph<Individual> egoNetworkWithoutEgo = GraphUtils.cloneWithoutEgo(egoNetwork, egoNetwork.getNode(ego));
			Partition<Node<Individual>> components = GraphUtils.components(egoNetworkWithoutEgo);
			
			componentReport.outputs().appendln(ego+"\t"+components.size());
			int i=1;
			for (Cluster<Node<Individual>> cluster : components.getClusters().toListSortedByDescendingSize()){
				componentReport.outputs().appendln("\t"+i+"("+cluster.size()+"): "+cluster.getItemsAsString());
				i++;
			}
			componentReport.outputs().appendln();
			
			
			valuesMap.get("BETWEENNESS").put(ego.getId(),new Value(100.*graphBetweenness[nodeIds.get(ego.getId())-1]));
			valuesMap.get("EGO-BETWEENNESS").put(ego.getId(),new Value(100.*GraphUtils.betweenness(egoNetwork)[0]));
			valuesMap.get("SIZE").put(ego.getId(),new Value(egoNetworkWithoutEgo.nodeCount()));
			valuesMap.get("TIES").put(ego.getId(),new Value(egoNetworkWithoutEgo.lineCount()));
			valuesMap.get("DENSITY").put(ego.getId(),new Value(GraphUtils.density(egoNetworkWithoutEgo)));
			valuesMap.get("DENSITY_NOLOOPS").put(ego.getId(),new Value(GraphUtils.densityWithoutLoops(egoNetworkWithoutEgo)));
			valuesMap.get("MEANDEGREE").put(ego.getId(),new Value(GraphUtils.meanDegree(egoNetworkWithoutEgo)));
			valuesMap.get("MEANDEGREE_NOLOOPS").put(ego.getId(),new Value(GraphUtils.meanDegreeWithoutLoops(egoNetworkWithoutEgo)));
			valuesMap.get("NRCOMPONENTS").put(ego.getId(),new Value(components.size()));
			valuesMap.get("MAXCOMPONENT").put(ego.getId(),new Value(components.maxClusterSize()));
			valuesMap.get("NRCOMPONENTS_NORM").put(ego.getId(),new Value(100.*components.meanShare()));
			valuesMap.get("MAXCOMPONENT_NORM").put(ego.getId(),new Value(100.*components.maxShare()));
			valuesMap.get("BROKERAGE").put(ego.getId(),new Value(GraphUtils.unconnectedPairsNormalized(egoNetworkWithoutEgo)));
			valuesMap.get("EFFICIENT_SIZE").put(ego.getId(),new Value(GraphUtils.efficientSize(egoNetworkWithoutEgo)));
			valuesMap.get("EFFICIENCY").put(ego.getId(),new Value(GraphUtils.efficiency(egoNetworkWithoutEgo)));
			
			
			double jacquard = 0.;
			double nrAlters = 0.;
			for (Individual alter : egoNetwork.getReferents()){
				if (alter!=ego){
					nrAlters++;
					jacquard+= SequenceWorker.jacquardDistance(ego,alter,"Migevent",criteria.getEgoRoleName(),criteria.getAlterRoleName());
				}
			}
			valuesMap.get("JACQUARD").put(ego.getId(),new Value(MathUtils.percent(jacquard,100*nrAlters)));

			for (String roleName : roleNames){
				nrAlters = 0;
				for (Individual alter : biography.alters(roleName)){
					if (egoNetwork.getReferents().contains(alter)){
						nrAlters++;
						
					}
				}
				valuesMap.get("NR"+roleName).put(ego.getId(), new Value(nrAlters));
				valuesMap.get("NR"+roleName+"_NORM").put(ego.getId(), new Value(MathUtils.percent(nrAlters,egoNetwork.nodeCount()-1)));
			}
			
			
			
			// Get triangles
			
			Individuals individuals = ego.getRelated("Migevent");
			individuals.add(ego);
			Partition<EventTriangle> triangles = SequenceWorker.getTriangles(individuals, "Migevent");
			allTriangles.add(triangles);
			trianglesMap.put(ego.getId(), triangles);
			for (Cluster<EventTriangle> cluster : triangles.getClusters()){
				triangleTypes.put(ego, cluster.getValue());
			}
		}
		
		
		// Create Pajek Egonetworks
		if (pajekBuffer.length() != 0) {
			File targetFile = ToolBox.setExtension(ToolBox.addToName(new File(segmentation.getLabel()), "-Ego Networks"), ".paj");
			ReportRawData rawData = new ReportRawData("Export Ego Networks to Pajek", "Pajek", "paj", targetFile);
			rawData.setData(PAJFile.convertToMicrosoftEndOfLine(pajekBuffer.toString()));

			result.outputs().appendln();
			result.outputs().append(rawData);
		}
		
		// Create Partition charts and tables
		List<ReportChart> charts = new ArrayList<ReportChart>();
		List<ReportTable> tables = new ArrayList<ReportTable>();
		
		overallReport.outputs().appendln("Measure\tAverage\tMaximum");

		for (String partitionLabel : partitionLabels){
			
			PartitionCriteria partitionCriteria = new PartitionCriteria(partitionLabel);
			
			if (partitionLabel.equals("SIZE") || partitionLabel.equals("TIES") || partitionLabel.equals("NRCOMPONENTS") || partitionLabel.equals("MAXCOMPONENT")){
				partitionCriteria.setType(PartitionType.RAW);
			} else {
				partitionCriteria.setType(PartitionType.SIZED_GROUPING);
				partitionCriteria.setStart(0.);
				partitionCriteria.setEnd(100.);
				partitionCriteria.setSize(10.);
			}
			
			NumberedValues values = valuesMap.get(partitionLabel);
			
			Partition<Individual> partition = PartitionMaker.create(partitionLabel, segmentation.getCurrentIndividuals(), values, partitionCriteria);
		
			PartitionCriteria splitCriteria = new PartitionCriteria("GENDER");
			ReportChart chart = StatisticsReporter.createPartitionChart(partition, partitionCriteria, splitCriteria);
		
			if (chart != null) {
				charts.add(chart);
				tables.add(ReportTable.transpose(chart.createReportTableWithSum()));
			}
			
			overallReport.outputs().appendln(partitionLabel+"\t"+MathUtils.round(values.average(),2)+"\t"+values.median()+"\t"+values.max());
		}
				
		// Manage the number of chart by line.
		for (int chartIndex = 0; chartIndex < charts.size(); chartIndex++) {
			diagramReport.outputs().append(charts.get(chartIndex));
			if (chartIndex % 4 == 3) {
				diagramReport.outputs().appendln();
			}
		}
		
		// Add chart tables.
		for (ReportTable table : tables) {
			diagramReport.outputs().appendln(table.getTitle());
			diagramReport.outputs().appendln(table);
		}
		
		// Report individual values
		detailReport.outputs().append("Ego");
		for (String partitionLabel : partitionLabels){
			detailReport.outputs().append("\t"+partitionLabel);
		}
		detailReport.outputs().appendln();
		for (Individual ego : segmentation.getCurrentIndividuals().toSortedList()){
			detailReport.outputs().append(ego);
			for (String partitionLabel : partitionLabels){
				detailReport.outputs().append("\t"+valuesMap.get(partitionLabel).get(ego.getId()));
			}
			detailReport.outputs().appendln();
		}
		
		
		// Report triangles
		Report trianglesReport = new Report("Triangles");
		trianglesReport.outputs().appendln("Type\tnrTriangles\tnrEgoNetworks");
		for (Cluster<EventTriangle> cluster: allTriangles.getClusters().toListSortedByValue()){
			trianglesReport.outputs().appendln(cluster.getValue()+"\t"+cluster.size()+"\t"+triangleTypes.getCluster(cluster.getValue()).size());
		}
		trianglesReport.outputs().appendln();
		
		PartitionCriteria partitionCriteria = new PartitionCriteria("Triangles");
		ReportChart chart5 = StatisticsReporter.createPartitionChart(allTriangles, partitionCriteria, null);
		trianglesReport.outputs().appendln(chart5);
		
		for (Individual ego : segmentation.getCurrentIndividuals().toSortedList()){
			Partition<EventTriangle> triangles = trianglesMap.get(ego.getId());
			trianglesReport.outputs().appendln(ego+"\t"+triangles.size()+" types");
			for (Cluster<EventTriangle> cluster : triangles.getClusters().toListSortedByValue()){
				trianglesReport.outputs().appendln(cluster.getValue()+"\t"+cluster.size());
				for (EventTriangle triangle : cluster.getItems()){
					trianglesReport.outputs().appendln(triangle.getEventPattern());
				}
				trianglesReport.outputs().appendln();
			}
			trianglesReport.outputs().appendln();
		}
		
		result.outputs().append(overallReport);
		result.outputs().append(diagramReport);
		result.outputs().append(detailReport);
		result.outputs().append(componentReport);
		result.outputs().append(trianglesReport);

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;
	}*/
	
	public static Report reportSequences (Sequences sequences){
		Report result;
		
		//
		Chronometer chrono = new Chronometer();

		//
		result = new Report();
		result.setTitle("Sequence report");
		result.setOrigin("Sequence reporter");

		Report report1 = new Report("Sequence list");
		
		for (Sequence sequence : sequences){
			report1.outputs().appendln(sequence.getId());
			for (Ordinal order : sequence.getEvents().keySet()){
				report1.outputs().appendln(order+"\t"+sequence.getEvents().get(order).getName());
			}
			report1.outputs().appendln();
		}
		
		result.outputs().append(report1);
		
		Report report2 = new Report("Actor-Event tables");
		
		for (Sequence sequence : sequences){
			report2.outputs().append(sequence.roleTable());
			report2.outputs().appendln();
		}
		
		result.outputs().append(report2);

		Report report3 = new Report("Actor-Event tables");
		
		for (Sequence sequence : sequences){
			report3.outputs().append(sequence.interactionTable());
			report3.outputs().appendln();
		}
		
		result.outputs().append(report3);

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;
	}

	public static Report reportUnknownPlaces(Segmentation segmentation, final ResourceBundle bundle) {
		Report result;
		
		//
		Chronometer chrono = new Chronometer();

		Geography geography = Geography.getInstance();
		
		Partition<String> unknownPlaces = new Partition<String>();
		
		int errorCount = 0;
		StringList errors = new StringList();

		for (Individual individual : segmentation.getCurrentIndividuals()){
			for (Attribute attribute : individual.attributes()){
				if (attribute.getLabel().contains("PLAC")){
					if (geography.getByHomonym(attribute.getValue())==null){
						unknownPlaces.put(attribute.getValue(), new Value(attribute.getLabel()));
						errorCount++;
					}
				}
			}
		}
		for (Relation relation : segmentation.getCurrentRelations()){
			for (Attribute attribute : relation.attributes()){
				if (attribute.getLabel().contains("PLAC")){
					if (geography.getByHomonym(attribute.getValue())==null){
						unknownPlaces.put(attribute.getValue(), new Value(attribute.getLabel()));
						errorCount++;
					}
				}
			}
		}
		
		for (Cluster<String> cluster : unknownPlaces.getClusters()){
			errors.appendln(cluster.getValue().toString());
			for (String placeName : cluster.getItems()){
				errors.appendln("\t"+placeName);
			}
			errors.appendln();
		}
		
		//
		result = new Report();
		result.setTitle("Unknown Places");
		result.setOrigin("Control reporter");
		result.setTarget(segmentation.getLabel());

		//
		errors.add(0, errorCount + " " + Report.translate(bundle, "Unknown Places") + "\n");
		result.outputs().append(errors.toString());

		//
		result.setStatus(errorCount);

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;
	}

	public static Report reportDiscontinuousItineraries(Segmentation segmentation, SequenceCensusCriteria criteria, ResourceBundle bundle) {
		Report result;
		
		//
		Chronometer chrono = new Chronometer();
		
		int errorCount = 0;
		StringList errors = new StringList();

		//
		result = new Report();
		result.setTitle("Discontinuous Itineraries");
		result.setOrigin("Control reporter");
		result.setTarget(segmentation.getLabel());
		
		for (Sequence itinerary : SequenceMaker.createPersonalSequences(segmentation, criteria).toSortedList()){
			Sequences partials = SequenceWorker.sortAndSplit(itinerary);
			if (partials.size()>1){
				errorCount++;
				errors.appendln(itinerary.getEgo().signature());
				int j=0;
				for (Sequence partial : partials){
					if (partial.getEvents().size()!=0 && j>0){
						errors.appendln(partial.getFirst()+"\t"+partial.getEvents().get(partial.getFirst()));
					}
					j++;
				}
				errors.appendln();
			}
		}

		//
		errors.add(0, errorCount + " " + Report.translate(bundle, "Discontinuous Itineraries") + "\n");
		result.outputs().append(errors.toString());

		//
		result.setStatus(errorCount);

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;
	}

	public static Report reportMissingTestimonies(Segmentation segmentation, ResourceBundle bundle) {
		Report result;
		
		//
		Chronometer chrono = new Chronometer();
		
		int errorCount = 0;
		StringList errors = new StringList();

		//
		result = new Report();
		result.setTitle("Missing Testimonies");
		result.setOrigin("Control reporter");
		result.setTarget(segmentation.getLabel());
		
		for (Relation event : segmentation.getCurrentRelations()){
			for (Individual witness : event.getIndividuals("MIG")){
				if (segmentation.getCurrentIndividuals().contains(witness)){
					boolean missing = true;
					for (Attribute attribute : event.attributes()){
						if (attribute.getLabel().contains("NOTE") && attribute.getLabel().indexOf("_")>-1){
							String id = attribute.getLabel().substring(attribute.getLabel().indexOf("_")+1);
							if (NumberUtils.isNumber(id) && Integer.parseInt(id)==witness.getId()){
								missing = false;
								break;
							}
						}
					}
					if (missing) {
						errorCount++;
						errors.appendln(witness.signature()+"\t"+event.getTypedId()+"\t"+event.getName());
					}
				}
			}
		}

		//
		errors.add(0, errorCount + " " + Report.translate(bundle, "Missing Testimonies") + "\n");
		result.outputs().append(errors.toString());

		//
		result.setStatus(errorCount);

		//
		result.setTimeSpent(chrono.stop().interval());

		//
		return result;		
	}

}
