package org.tip.puck.net.workers;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.Stack;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tip.puck.PuckException;
import org.tip.puck.PuckExceptions;
import org.tip.puck.census.chains.Chain;
import org.tip.puck.census.chains.ChainFinder;
import org.tip.puck.census.chains.Couple;
import org.tip.puck.census.chains.Notation;
import org.tip.puck.census.workers.ChainValuator;
import org.tip.puck.census.workers.CircuitFinder;
import org.tip.puck.graphs.Graph;
import org.tip.puck.graphs.Link;
import org.tip.puck.graphs.Node;
import org.tip.puck.net.Attribute;
import org.tip.puck.net.Families;
import org.tip.puck.net.Family;
import org.tip.puck.net.FiliationType;
import org.tip.puck.net.Gender;
import org.tip.puck.net.Individual;
import org.tip.puck.net.Individuals;
import org.tip.puck.net.KinType;
import org.tip.puck.net.Net;
import org.tip.puck.net.relations.Actor;
import org.tip.puck.net.relations.Actors;
import org.tip.puck.net.relations.Relation;
import org.tip.puck.net.relations.RelationModel;
import org.tip.puck.net.relations.Relations;
import org.tip.puck.net.relations.Role;
import org.tip.puck.net.workers.UpdateWorker.UpdateMode;
import org.tip.puck.partitions.Cluster;
import org.tip.puck.partitions.Partition;
import org.tip.puck.partitions.PartitionCriteria;
import org.tip.puck.partitions.PartitionMaker;
import org.tip.puck.segmentation.Segment;
import org.tip.puck.segmentation.Segmentation;
import org.tip.puck.util.Value;

import fr.devinsy.util.StringList;

/**
 * 
 * Methods added in this classes has to be static and autonomous.
 * 
 * @author TIP
 */
public class NetUtils {

	private static final Logger logger = LoggerFactory.getLogger(NetUtils.class);

	/**
	 * This method anonymizes a net leaving only the first name.
	 * 
	 * @param net
	 *            Source to anonymize.
	 * 
	 * @return the number of field anonymized
	 */
	public static long anonymizeByFirstName(final Individuals source) {
		return AttributeWorker.anonymizeByFirstName(source);
	}

	/**
	 * This method anonymizes a net leaving only the first name.
	 * 
	 * @param net
	 *            Source to anonymize.
	 * 
	 * @return the number of field anonymized
	 */
	public static long anonymizeByFirstName(final Net source) {
		return AttributeWorker.anonymizeByFirstName(source);
	}

	/**
	 * Anonymizes a net replacing name by gender and id.
	 * 
	 * @param net
	 *            Source to anonymize.
	 * 
	 * @return the number of field anonymized
	 */
	public static long anonymizeByGenderAndId(final Individuals source) {
		return AttributeWorker.anonymizeByGenderAndId(source);
	}

	/**
	 * Anonymizes a net replacing name by gender and id.
	 * 
	 * @param net
	 *            Source to anonymize.
	 * 
	 * @return the number of field anonymized
	 */
	public static long anonymizeByGenderAndId(final Net source) {
		return AttributeWorker.anonymizeByGenderAndId(source);
	}

	/**
	 * Anonymizes a net leaving only the first name.
	 * 
	 * @param net
	 *            Source to anonymize.
	 * 
	 * @return the number of field anonymized
	 */
	public static long anonymizeByLastName(final Individuals source) {
		return AttributeWorker.anonymizeByLastName(source);
	}

	/**
	 * This method anonymizes a net leaving only the first name.
	 * 
	 * @param net
	 *            Source to anonymize.
	 * 
	 * @return the number of field anonymized
	 */
	public static long anonymizeByLastName(final Net source) {
		return AttributeWorker.anonymizeByLastName(source);
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Net buildCleanedNet(final Net source) throws PuckException {
		Net result;

		//
		result = new Net();

		//
		result.setLabel(source.getLabel());

		//
		for (Attribute attribute : source.attributes().toSortedList()) {
			result.attributes().put(attribute.getLabel(), attribute.getValue());
		}

		// Create individuals (with bad data).
		for (Individual sourceIndividual : source.individuals()) {

			/**
			 * Set individual data. Warning:
			 * <ul>
			 * <li>origin family is set with source data and it will be fixed
			 * later
			 * <li>personal families is not set and it will be fixed later
			 * <li>relations is not set and it will be done later.
			 * </ul>
			 */
			Individual targetIndividual = new Individual(sourceIndividual.getId());
			targetIndividual.setGender(sourceIndividual.getGender());
			targetIndividual.setName(sourceIndividual.getName());
			targetIndividual.setOriginFamily(sourceIndividual.getOriginFamily());
			targetIndividual.attributes().addAll(sourceIndividual.attributes());

			//
			result.individuals().add(targetIndividual);
		}

		// Create family and fix data.
		for (Family sourceFamily : source.families()) {
			//
			int fatherId;
			if (sourceFamily.getHusband() == null) {
				fatherId = 0;
			} else {
				fatherId = sourceFamily.getHusband().getId();
			}

			//
			int motherId;
			if (sourceFamily.getWife() == null) {
				motherId = 0;
			} else {
				motherId = sourceFamily.getWife().getId();
			}

			// Get parent family or create it.
			// Note: If one parent is unknown, we can't associate
			// the individual to an existent family, we have to
			// avoid german relation. So, if one parent is
			// unknown, we have to create a new family.
			Family targetFamily;
			if ((fatherId == 0) || (motherId == 0)) {
				targetFamily = null;
			} else {
				targetFamily = result.families().getBySpouses(fatherId, motherId);
			}

			if (targetFamily == null) {
				targetFamily = new Family(result.families().size() + 1);
				result.families().add(targetFamily);

				//
				if (sourceFamily.getHusband() != null) {
					targetFamily.setHusband(result.individuals().getById(sourceFamily.getHusband().getId()));
				}
				if (sourceFamily.getWife() != null) {
					targetFamily.setWife(result.individuals().getById(sourceFamily.getWife().getId()));
				}
				targetFamily.setUnionStatus(sourceFamily.getUnionStatus());

				for (Individual sourceChild : sourceFamily.getChildren()) {
					Individual targetChild = result.individuals().getById(sourceChild.getId());
					targetFamily.getChildren().add(targetChild);
					targetChild.setOriginFamily(targetFamily);
				}

				targetFamily.attributes().addAll(sourceFamily.attributes());

			} else {
				for (Individual sourceChild : sourceFamily.getChildren()) {
					if (targetFamily.getChildren().getById(sourceChild.getId()) == null) {
						Individual targetChild = result.individuals().getById(sourceChild.getId());
						targetFamily.getChildren().add(targetChild);
						targetChild.setOriginFamily(targetFamily);
					}
				}

				if (sourceFamily.isMarried()) {
					targetFamily.setMarried(true);
				}

				targetFamily.attributes().addAll(sourceFamily.attributes());
			}

			//
			if (sourceFamily.getHusband() != null) {
				Individual targetParent = result.individuals().getById(sourceFamily.getHusband().getId());
				targetParent.getPersonalFamilies().add(targetFamily);
			}

			//
			if (sourceFamily.getWife() != null) {
				Individual targetParent = result.individuals().getById(sourceFamily.getWife().getId());
				targetParent.getPersonalFamilies().add(targetFamily);
			}
		}

		//
		for (RelationModel model : source.relationModels()) {
			result.relationModels().add(new RelationModel(model));
		}

		//
		for (Relation sourceRelation : source.relations()) {
			//
			Relation targetRelation = new Relation(sourceRelation.getId(), sourceRelation.getTypedId(), result.relationModels().getByName(
					sourceRelation.getModel().getName()), sourceRelation.getName());
			result.relations().add(targetRelation);

			//
			for (Actor sourceActor : sourceRelation.actors()) {
				result.createRelationActor(targetRelation, sourceActor.getId(), sourceActor.getRole().getName());
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 */
	public static Net copy(final Net source) {
		Net result;

		result = new Net(source);

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 */
	public static Net copyWithMarriedCoparents(final Net source) {
		Net result;

		result = new Net(source);
		marryCoparents(result);

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Net copyWithoutMarkedDoubles(final Net source) throws PuckException {
		Net result;

		result = new Net(source);
		eliminateMarkedDoubles(result);

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 */
	public static Net copyWithoutSingles(final Net source) {
		Net result;

		result = new Net(source);
		eliminateSingles(result);

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 */
	public static Net copyWithoutStructuralChildren(final Net source) {
		Net result;

		result = new Net(source);
		eliminateStructuralChildren(result);

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 */
	public static Net copyWithoutVirtuals(final Net source) {
		Net result;

		result = new Net(source);
		eliminateVirtuals(result);

		//
		return result;
	}

	public static void createFamiliesFromRelations(final Net net, final String relationLabel, final Map<String, String> labels) throws PuckException {

		String husbandLabel = labels.get("husband");
		String wifeLabel = labels.get("wife");
		String sonLabel = labels.get("son");
		String daughterLabel = labels.get("daughter");
		String otherWifeLabel = labels.get("otherWife");
		String otherSonLabel = labels.get("otherSon");
		String otherDaughterLabel = labels.get("otherDaughter");
		String naturalSonLabel = labels.get("naturalSon");
		String naturalDaughterLabel = labels.get("naturalDaughter");
		String husbandMotherLabel = labels.get("husbandMother");
		String husbandFatherLabel = labels.get("husbandFather");
		String brotherLabel = labels.get("brother");
		String sisterLabel = labels.get("sister");

		for (Relation relation : net.relations().getByModelName(relationLabel)) {

			Individual husband = null;
			Individual wife = null;
			Individuals children = new Individuals();

			Actors husbandActors = relation.actors().getByRole(husbandLabel);
			if (husbandActors != null) {
				if (husbandActors.size() == 1) {
					husband = husbandActors.get(0).getIndividual();
				} else {
					System.err.println("Multiple fathers for role " + husbandLabel);
				}
			}
			Actors wifeActors = relation.actors().getByRole(wifeLabel);
			if (wifeActors != null) {
				if (wifeActors.size() == 1) {
					wife = wifeActors.get(0).getIndividual();
				} else {
					System.err.println("Multiple mothers for role " + wifeLabel);
				}
			}
			Actors sonActors = relation.actors().getByRole(sonLabel);
			if (sonActors != null) {
				for (Actor childActor : sonActors) {
					children.add(childActor.getIndividual());
				}
			}

			Actors daughterActors = relation.actors().getByRole(daughterLabel);
			if (daughterActors != null) {
				for (Actor childActor : daughterActors) {
					children.add(childActor.getIndividual());
				}
			}

			if (husband != null || wife != null || children.size() > 0) {
				net.createFamily(husband, wife, children).setMarried();;
			}

			Actors otherWifeActors = relation.actors().getByRole(otherWifeLabel);
			if (otherWifeActors != null) {
				for (Actor otherWifeActor : otherWifeActors) {
					net.createFamily(husband, otherWifeActor.getIndividual()).setMarried();;
				}
			}

			Actors otherSonActors = relation.actors().getByRole(otherSonLabel);
			if (otherSonActors != null) {
				for (Actor childActor : otherSonActors) {
					Individuals otherChildren = new Individuals();
					otherChildren.add(childActor.getIndividual());
					net.createFamily(husband, null, otherChildren).setMarried();
				}
			}

			Actors otherDaughterActors = relation.actors().getByRole(otherDaughterLabel);
			if (otherDaughterActors != null) {
				for (Actor childActor : otherDaughterActors) {
					Individuals otherChildren = new Individuals();
					otherChildren.add(childActor.getIndividual());
					net.createFamily(husband, null, otherChildren).setMarried();
				}
			}

			Actors naturalSonActors = relation.actors().getByRole(naturalSonLabel);
			if (naturalSonActors != null) {
				for (Actor childActor : naturalSonActors) {
					Individuals naturalChildren = new Individuals();
					naturalChildren.add(childActor.getIndividual());
					net.createFamily(husband, null, naturalChildren);
				}
			}

			Actors naturalDaughterActors = relation.actors().getByRole(naturalDaughterLabel);
			if (naturalDaughterActors != null) {
				for (Actor childActor : naturalDaughterActors) {
					Individuals naturalChildren = new Individuals();
					naturalChildren.add(childActor.getIndividual());
					net.createFamily(husband, null, naturalChildren);
				}
			}

		}

		for (Individual husband : net.individuals()) {
			for (Relation relation : husband.relations().getByModelName(relationLabel)) {
				if (relation.getIndividuals(husbandLabel).contains(husband)) {
					if (relation.hasActors(brotherLabel, sisterLabel, husbandMotherLabel)) {
						Individual husbandFather = null;
						Individual husbandMother = null;
						Individuals husbandSiblings = new Individuals();
						husbandSiblings.add(husband);

						Actors husbandFatherActors = relation.actors().getByRole(husbandFatherLabel);
						if (husbandFatherActors != null) {
							if (husbandFatherActors.size() == 1) {
								husbandFather = husbandFatherActors.get(0).getIndividual();
							} else {
								System.err.println("Multiple husbandFathers for role " + husbandFatherLabel);
							}
						}
						Actors husbandMotherActors = relation.actors().getByRole(husbandMotherLabel);
						if (husbandMotherActors != null) {
							if (husbandMotherActors.size() == 1) {
								husbandMother = husbandMotherActors.get(0).getIndividual();
							} else {
								System.err.println("Multiple husbandMothers for role " + husbandMotherLabel);
							}
						}
						Actors brotherActors = relation.actors().getByRole(brotherLabel);
						if (brotherActors != null) {
							for (Actor childActor : brotherActors) {
								husbandSiblings.add(childActor.getIndividual());
							}
						}

						Actors sisterActors = relation.actors().getByRole(sisterLabel);
						if (sisterActors != null) {
							for (Actor childActor : sisterActors) {
								husbandSiblings.add(childActor.getIndividual());
							}
						}

						Family husbandOriginFamily = husband.getOriginFamily();
						if (husbandOriginFamily == null) {
							husbandOriginFamily = net.createFamily(husbandFather, husbandMother, husbandSiblings);
						}
						if (husbandOriginFamily.getFather() == null) {
							husbandOriginFamily.setFather(husbandFather);
						} else if (husbandOriginFamily.getFather().equals(husbandFather)) {
							System.err.println("Multiple husbandFathers for role " + husbandFatherLabel);
						}
						if (husbandOriginFamily.getMother() == null) {
							husbandOriginFamily.setMother(husbandMother);
						} else if (husbandOriginFamily.getMother().equals(husbandMother)) {
							System.err.println("Multiple husbandMothers for role " + husbandMotherLabel);
						}
						for (Individual husbandSibling : husbandSiblings) {
							husbandOriginFamily.getChildren().add(husbandSibling);
							husbandSibling.setOriginFamily(husbandOriginFamily);
						}
					}

				}
			}

		}

	}

	public static void createIndividualsFromFamilyAttributes(final Net net, final Family family, final String husbandFirstNameLabel,
			final String husbandLastNameLabel, final String wifeFirstNameLabel, final String wifeLastNameLabel, final Map<String, String> husbandLabels,
			final Map<String, String> wifeLabels) {

		String husbandFirstName = family.getAttributeValue(husbandFirstNameLabel);
		if (husbandFirstName == null) {
			husbandFirstName = "";
		}
		String husbandLastName = family.getAttributeValue(husbandLastNameLabel);
		if (husbandLastName == null) {
			husbandLastName = "";
		}
		String wifeFirstName = family.getAttributeValue(wifeFirstNameLabel);
		if (wifeFirstName == null) {
			wifeFirstName = "";
		}
		String wifeLastName = family.getAttributeValue(wifeLastNameLabel);
		if (wifeLastName == null) {
			wifeLastName = "";
		}

		Individual husband = null;
		Individual wife = null;

		if (husbandFirstName != null || husbandLastName != null) {
			husband = new Individual(net.individuals().nextFreeId(net.getDefaultIdStrategy()), husbandFirstName + " / " + husbandLastName, Gender.MALE);
			for (String famLabel : husbandLabels.keySet()) {
				String indLabel = husbandLabels.get(famLabel);
				String value = family.getAttributeValue(famLabel);
				if (value != null) {
					husband.setAttribute(indLabel, value);
				}
			}
			net.individuals().put(husband);
			family.setHusband(husband);
			husband.addPersonalFamily(family);
		}
		if (wifeFirstName != null || wifeLastName != null) {
			wife = new Individual(net.individuals().nextFreeId(net.getDefaultIdStrategy()), wifeFirstName + " / " + wifeLastName, Gender.FEMALE);
			for (String famLabel : wifeLabels.keySet()) {
				String indLabel = wifeLabels.get(famLabel);
				String value = family.getAttributeValue(famLabel);
				if (value != null) {
					wife.setAttribute(indLabel, value);
				}
			}
			net.individuals().put(wife);
			family.setWife(wife);
			wife.addPersonalFamily(family);
		}
	}

	/**
	 * 
	 * 
	 * @param net
	 * @throws PuckException
	 */
	public static RelationModel createLifeEvents(final Net net) throws PuckException {
		RelationModel result;

		//
		RelationModel model = net.createRelationModel("BIO");

		//
		model.roles().add(new Role("EGO"));
		model.roles().add(new Role("SPOUSE"));
		model.roles().add(new Role("PARENT"));
		model.roles().add(new Role("CHILD"));

		//
		for (Individual individual : net.individuals()) {

			for (Attribute attribute : individual.attributes()) {
				if (attribute.getLabel().contains("DATE") && attribute.getLabel().indexOf("_") > -1 && !attribute.getValue().equals("0")) {
					String label = attribute.getLabel().substring(0, attribute.getLabel().indexOf("_"));
					Relation relation = net.createRelation(net.relations().getFirstFreeId(), label + " " + individual.toString(), model);
					//
					relation.setAttribute("DATE", attribute.getValue());
					String place = individual.getAttributeValue(label + "_PLAC");
					if (place != null) {
						relation.setAttribute("PLAC", place);
					}

					if (label.equals("BIRT")) {
						net.createRelationActor(relation, individual.getId(), "CHILD");
						if (individual.getFather() != null) {
							net.createRelationActor(relation, individual.getFather().getId(), "PARENT");
						}
						if (individual.getMother() != null) {
							net.createRelationActor(relation, individual.getMother().getId(), "PARENT");
						}
					} else {
						net.createRelationActor(relation, individual.getId(), "EGO");
					}
				}
			}
		}

		//
		for (Family family : net.families()) {

			for (Attribute attribute : family.attributes()) {
				if (attribute.getLabel().contains("DATE") && attribute.getLabel().indexOf("_") > -1 && !attribute.getValue().equals("0")) {
					String label = attribute.getLabel().substring(0, attribute.getLabel().indexOf("_"));
					Relation relation = net.createRelation(net.relations().getFirstFreeId(), label + " " + family.toString(), model);

					relation.setAttribute("DATE", attribute.getValue());
					String place = family.getAttributeValue(label + "_PLAC");
					if (place != null) {
						relation.setAttribute("PLAC", place);
					}

					//
					if (family.getHusband() != null) {
						net.createRelationActor(relation, family.getHusband().getId(), "SPOUSE");
					}
					if (family.getWife() != null) {
						net.createRelationActor(relation, family.getWife().getId(), "SPOUSE");
					}
				}
			}
		}

		//
		result = model;

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Graph<Individual> createOreGraph(final Segmentation source) throws PuckException {
		Graph<Individual> result;

		//
		result = new Graph<Individual>("Ore graph " + source.getLabel());

		//
		for (Individual individual : source.getCurrentIndividuals().toSortedList()) {
			result.addNode(individual);
		}

		//
		for (Family family : source.getCurrentFamilies()) {
			Individual father = family.getHusband();
			Individual mother = family.getWife();
			if (father != null && mother != null && family.isMarried()) {
				result.addEdge(father, mother, 1);
			}
			if (!family.isSterile()) {
				for (Individual child : family.getChildren()) {
					if (father != null) {
						result.addArc(father, child, 1);
					}
					if (mother != null) {
						result.addArc(mother, child, 1);
					}
				}
			}
		}

		//
		NetUtils.setGenderShapes(result);

		//
		return result;
	}

	public static void createOriginFamilyFromIndividualAttributes(final Net net, final Individual ego, final String fatherFirstNameLabel,
			final String fatherLastNameLabel, final String motherFirstNameLabel, final String motherLastNameLabel) {

		String fatherFirstName = ego.getAttributeValue(fatherFirstNameLabel);
		if (fatherFirstName == null) {
			fatherFirstName = "";
		}
		String fatherLastName = ego.getAttributeValue(fatherLastNameLabel);
		if (fatherLastName == null) {
			fatherLastName = "";
		}
		String motherFirstName = ego.getAttributeValue(motherFirstNameLabel);
		if (motherFirstName == null) {
			motherFirstName = "";
		}
		String motherLastName = ego.getAttributeValue(motherLastNameLabel);
		if (motherLastName == null) {
			motherLastName = "";
		}

		Individual father = null;
		Individual mother = null;

		if (!StringUtils.isBlank(fatherFirstName + fatherLastName)) {
			father = new Individual(net.individuals().nextFreeId(net.getDefaultIdStrategy()), fatherFirstName + " / " + fatherLastName, Gender.MALE);
			net.individuals().put(father);
		}
		if (!StringUtils.isBlank(motherFirstName + motherLastName)) {
			mother = new Individual(net.individuals().nextFreeId(net.getDefaultIdStrategy()), motherFirstName + " / " + motherLastName, Gender.FEMALE);
			net.individuals().put(mother);
		}

		if (father != null || mother != null) {
			net.createFamily(father, mother, ego);
		}
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Graph<Family> createPGraph(final Segmentation source) throws PuckException {
		Graph<Family> result;
		//
		result = new Graph<Family>("Pgraph " + source.getLabel());

		// getAllFamilies would be better
		int familyCount = source.getCurrentFamilies().getLastId();

		for (Individual individual : source.getCurrentIndividuals()) {
			Family originFamily = individual.getOriginFamily();
			if (originFamily == null) {
				familyCount++;
				originFamily = new Family(familyCount);
				originFamily.getChildren().add(individual);
			}
			Families personalFamilies = individual.getPersonalFamilies();
			if (personalFamilies.size() == 0) {
				familyCount++;
				Family personalFamily = new Family(familyCount);
				if (individual.isMale()) {
					personalFamily.setHusband(individual);
				} else if (individual.isFemale()) {
					personalFamily.setWife(individual);
				}
			}
			for (Family personalFamily : personalFamilies) {
				int weight = 1 - 2 * (individual.getGender().toInt() % 2);

				result.addArc(originFamily, personalFamily, weight);
			}
		}

		//
		return result;

	}

	public static Graph<Individual> createRelationGraph(final Individual source, final String relationModelName, final String egoRoleName,
			final String alterRoleName) {
		Graph<Individual> result;

		//
		result = new Graph<Individual>("Ego network " + source + " " + relationModelName + " " + egoRoleName + "-" + alterRoleName);

		Map<String, Integer> tagMap = new HashMap<String, Integer>();

		result.addNode(source);
		//
		for (Relation relation : source.relations()) {
			if (relation.getRoleNames(source).contains(egoRoleName)) {
				Individuals alters;
				if (alterRoleName.equals("ALL")) {
					alters = relation.getIndividuals();
				} else {
					alters = relation.getIndividuals(alterRoleName);
				}
				for (Individual alter : alters) {
					result.addNode(alter);
				}
			}
		}

		//
		for (Individual ego : result.getReferents()) {
			for (Relation relation : ego.relations().getByModelName(relationModelName)) {
				for (Individual alter : relation.getIndividuals()) {
					if (result.getReferents().contains(alter)) {
						for (String egoRole : relation.getRoleNames(ego)) {
							for (String alterRole : relation.getRoleNames(alter)) {
								Link<Individual> link = null;
								if (egoRole.equals(alterRole)) {
									if (ego.getId() < alter.getId()) {
										link = result.addEdge(ego, alter, 1);
									}
								} else if (egoRole.compareTo(alterRole) < 0) {
									link = result.addArc(ego, alter, 1);
								}
								if (link != null) {
									String tag = egoRole + "-" + alterRole;
									Integer tagNumber = tagMap.get(tag);
									if (tagNumber == null) {
										tagNumber = tagMap.size() + 1;
										tagMap.put(tag, tagNumber);
									}
									link.setTag(":" + tagNumber + " '" + tag + "'");
								}
							}
						}
					}
				}
			}
		}

		//
		NetUtils.setGenderShapes(result);

		//
		return result;
	}

	public static Graph<Individual> createRelationGraph(final Segmentation source, final String relationName) {
		Graph<Individual> result;

		//
		result = new Graph<Individual>("Relation graph " + relationName + " " + source.getLabel());

		//
		for (Individual individual : source.getCurrentIndividuals().toSortedList()) {
			result.addNode(individual);
		}

		Map<String, Integer> tagMap = new HashMap<String, Integer>();

		//
		for (Relation relation : source.getCurrentRelations().getByModelName(relationName)) {
			for (Actor egoActor : relation.actors()) {
				String egoRole = egoActor.getRole().getName();
				Individual ego = egoActor.getIndividual();
				for (Actor alterActor : relation.actors()) {
					String alterRole = alterActor.getRole().getName();
					Individual alter = alterActor.getIndividual();
					Link<Individual> link = null;
					if (egoRole.equals(alterRole)) {
						if (ego.getId() < alter.getId()) {
							link = result.addEdge(ego, alter, 1);
						}
					} else if (egoRole.compareTo(alterRole) < 0) {
						link = result.addArc(ego, alter, 1);
					}
					if (link != null) {
						String tag = egoRole + "-" + alterRole;
						Integer tagNumber = tagMap.get(tag);
						if (tagNumber == null) {
							tagNumber = tagMap.size() + 1;
							tagMap.put(tag, tagNumber);
						}
						link.setTag(":" + tagNumber + " '" + tag + "'");
					}
				}
			}
		}

		//
		NetUtils.setGenderShapes(result);

		//
		return result;
	}

	public static RelationModel createRelationsFromAttributes(final Net net, final String relationLabel, final String roleLabel) throws PuckException {
		RelationModel result;

		//
		result = net.createRelationModel(relationLabel);

		//
		for (Individual individual : net.individuals()) {

			String relationValue = null;
			String roleValue = null;

			for (Attribute attribute : individual.attributes()) {
				if (attribute.getLabel().equals(relationLabel)) {
					relationValue = attribute.getValue();
				}
				if (attribute.getLabel().equals(roleLabel)) {
					roleValue = attribute.getValue();
				}
			}

			if (relationValue != null && roleValue == null) {
				roleValue = "OTHER";
			}

			int typedId = Integer.parseInt(relationValue);

			Relation relation = net.relations().getByTypedId(typedId, result);
			if (relation == null) {
				relation = net.createRelation(typedId, relationLabel + " " + relationValue, result);
			}

			net.createRelationRole(result, roleValue);
			net.createRelationActor(relation, individual.getId(), roleValue);

		}

		//
		return result;

	}

	/**
	 * 
	 * 
	 * @param net
	 * @throws PuckException
	 */
	public static RelationModel createRelationsFromCircuits(final Net net, final CircuitFinder finder, final RelationModel model) throws PuckException {
		RelationModel result;

		//
		model.roles().add(new Role("PIVOT"));
		model.roles().add(new Role("INTERMEDIARY"));

		//
		int i = 1;
		for (Cluster<Chain> cluster : finder.getCircuits().getClusters().toListSortedByValue()) {
			for (Chain r : cluster.getItems()) {
				//
				Relation relation = net.createRelation(i, r.signature(Notation.NUMBERS), model);
				for (int j = 0; j < 2 * r.dim(); j++) {
					Individual pivot = r.getPivot(j);
					if (!relation.hasActor(pivot)) {
						net.createRelationActor(relation, pivot.getId(), "PIVOT");
					}
				}
				relation.attributes().add(new Attribute("TYPE", cluster.getValue().toString()));
				relation.attributes().add(new Attribute("CLASSIC", r.signature(Notation.CLASSIC_GENDERED)));

				for (Individual indi : r) {
					if (!relation.hasActor(indi)) {
						if (indi instanceof Couple) {
							if (((Couple) indi).getFirstId() > 0) {
								net.createRelationActor(relation, ((Couple) indi).getFirstId(), "INTERMEDIARY");
							}
							if (((Couple) indi).getSecondId() > 0) {
								net.createRelationActor(relation, ((Couple) indi).getSecondId(), "INTERMEDIARY");
							}
						} else {
							net.createRelationActor(relation, indi.getId(), "INTERMEDIARY");
						}
					}
				}
				i++;
			}
		}

		//
		result = model;

		//
		return result;
	}

	/**
	 * 
	 * 
	 * @param net
	 * @throws PuckException
	 */
	public static RelationModel createRelationsFromFamilies(final Net net) throws PuckException {
		RelationModel result;

		//
		RelationModel model = net.createRelationModel("FamiliesPlus");

		//
		model.roles().add(new Role("HUSBAND"));
		model.roles().add(new Role("WIFE"));
		model.roles().add(new Role("CHILD"));

		//
		for (Family family : net.families()) {

			//
			Relation relation = net.createRelation(family.getId(), family.hashKey(), model);

			//
			if (family.getHusband() != null) {
				Actor husband = net.createRelationActor(relation, family.getHusband().getId(), "HUSBAND");
				husband.setRelationOrder(family.getHusbandOrder());
			}

			if (family.getWife() != null) {
				Actor wife = net.createRelationActor(relation, family.getWife().getId(), "WIFE");
				wife.setRelationOrder(family.getWifeOrder());
			}

			for (Individual child : family.getChildren()) {
				Actor actor = net.createRelationActor(relation, child.getId(), "CHILD");
				actor.setRelationOrder(child.getBirthOrder());
			}

			//
			relation.attributes().addAll(family.attributes());
		}

		//
		result = model;

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 * @return
	 * @throws PuckException
	 */
	public static Graph<Individual> createTipGraph(final Segmentation source) throws PuckException {

		Graph<Individual> result;
		//
		result = new Graph<Individual>("Tip graph " + source.getLabel());

		for (Individual individual : source.getCurrentIndividuals().toSortedList()) {
			result.addNode(individual);
		}

		//
		for (Family family : source.getCurrentFamilies()) {
			//
			Individual father = family.getHusband();
			Individual mother = family.getWife();

			//
			if (father != null && mother != null && family.isMarried()) {
				Link<Individual> link = result.addArc(mother, father, 1);
				link.setTag("1 'H.F'");
			}

			//
			if (!family.isSterile()) {
				for (Individual child : family.getChildren()) {
					if (father != null) {
						//
						int relationCode;
						String relationPattern;
						switch (child.getGender()) {
							case FEMALE:
								relationCode = 4;
								relationPattern = "'F(H)'";
							break;
							case MALE:
								relationCode = 5;
								relationPattern = "'H(H)'";
							break;
							default:
								relationCode = 7;
								relationPattern = "'X(H)'";
						}

						//
						Link<Individual> link = result.addArc(father, child, relationCode);
						link.setTag(relationCode + " " + relationPattern);
					}

					if (mother != null) {
						//
						int relationCode;
						String relationPattern;
						switch (child.getGender()) {
							case FEMALE:
								relationCode = 2;
								relationPattern = "'F(F)'";
							break;
							case MALE:
								relationCode = 3;
								relationPattern = "'M(F)'";
							break;
							default:
								relationCode = 6;
								relationPattern = "'X(F)'";
						}

						//
						Link<Individual> link = result.addArc(mother, child, relationCode);
						link.setTag(relationCode + " " + relationPattern);
					}
				}
			}
		}

		//
		NetUtils.setGenderShapes(result);

		//
		return result;
	}

	/**
	 * Eliminates the individuals marked as double (i.e., individuals having a
	 * number instead of a name).
	 * <p>
	 * The "name" of a double is the ID number of the original.
	 * 
	 * @param net
	 *            Source.
	 * 
	 * @return The number of single individuals eliminated.
	 * @throws PuckException
	 */
	public static int eliminateDoubleIndividuals(final Net source, final Individuals sample) throws PuckException {
		int result;

		List<Individual> individuals = sample.toList();

		Map<Integer, Integer> originalIds = new HashMap<Integer, Integer>();
		List<Integer> inexistentOriginals = new ArrayList<Integer>();
		List<Integer> selfReferentialOriginals = new ArrayList<Integer>();
		List<Integer> unreplacedIds = new ArrayList<Integer>();

		result = 0;
		for (Individual sourceIndividual : individuals) {
			if (NumberUtils.isNumber(sourceIndividual.getFirstName())) {
				//
				int originalId = Integer.parseInt(sourceIndividual.getFirstName());

				if (originalId == sourceIndividual.getId()) {
					selfReferentialOriginals.add(originalId);
					continue;
				}

				Individual targetIndividual = source.individuals().getById(originalId);

				while (targetIndividual == null) {
					if (originalIds.get(originalId) == null) {
						inexistentOriginals.add(originalId);
						break;
					} else {
						originalId = originalIds.get(originalId);
						targetIndividual = source.individuals().getById(originalId);
					}
				}
				if (targetIndividual == null) {
					unreplacedIds.add(sourceIndividual.getId());
					continue;
				}

				originalIds.put(sourceIndividual.getId(), originalId);

				String track = targetIndividual.getAttributeValue("DOUBLE");
				if (track == null) {
					track = String.valueOf(sourceIndividual.getId());
				} else {
					track += ";" + String.valueOf(sourceIndividual.getId());
				}

				targetIndividual.setAttribute("DOUBLE", track);

				// Append attributes.
				UpdateWorker.update(targetIndividual.attributes(), sourceIndividual.attributes(), UpdateMode.APPEND);

				// Append parents.
				if ((targetIndividual.getFather() == null) && (sourceIndividual.getFather() != null)) {
					NetUtils.setFatherRelation(source, sourceIndividual.getFather().getId(), targetIndividual.getId());
				}
				if ((targetIndividual.getMother() == null) && (sourceIndividual.getMother() != null)) {
					NetUtils.setMotherRelation(source, sourceIndividual.getMother().getId(), targetIndividual.getId());
				}

				// Append or fuse personalFamilies
				for (Family sourceFamily : sourceIndividual.getPersonalFamilies()) {
					Family targetFamily;
					if (sourceIndividual == sourceFamily.getHusband()) {
						targetFamily = source.families().getBySpouses(targetIndividual, sourceFamily.getWife());
						if (targetFamily == null) {
							sourceFamily.setHusband(targetIndividual);
							targetIndividual.addPersonalFamily(sourceFamily);
						} else {
							targetFamily.getChildren().add(sourceFamily.getChildren());
							for (Individual child : sourceFamily.getChildren()) {
								child.setOriginFamily(targetFamily);
							}
							if (sourceFamily.getWife() != null) {
								sourceFamily.getWife().getPersonalFamilies().removeById(sourceFamily.getId());
							}
							source.families().removeById(sourceFamily.getId());
						}
					} else if (sourceIndividual == sourceFamily.getWife()) {
						targetFamily = source.families().getBySpouses(sourceFamily.getHusband(), targetIndividual);
						if (targetFamily == null) {
							sourceFamily.setWife(targetIndividual);
							targetIndividual.addPersonalFamily(sourceFamily);
						} else {
							targetFamily.getChildren().add(sourceFamily.getChildren());
							for (Individual child : sourceFamily.getChildren()) {
								child.setOriginFamily(targetFamily);
							}
							if (sourceFamily.getHusband() != null) {
								sourceFamily.getHusband().getPersonalFamilies().removeById(sourceFamily.getId());
							}
							source.families().removeById(sourceFamily.getId());
						}
					}
				}

				source.remove(sourceIndividual);

				result += 1;
			}
		}

		// Make error report
		StringList reportList = new StringList();

		reportList.append("Inexistent originals: ");
		for (int id : inexistentOriginals) {
			reportList.append(id + " ");
		}
		reportList.appendln();

		reportList.append("Selfreferential originals: ");
		for (int id : selfReferentialOriginals) {
			reportList.append(id + " ");
		}
		reportList.appendln();

		reportList.append("Unreplaced ids: ");
		for (int id : unreplacedIds) {
			reportList.append(id + " ");
		}
		reportList.appendln();

		System.out.println(reportList);

		//
		return result;
	}

	/**
	 * Eliminates the relations marked as double (i.e., relations having a
	 * number instead of a name).
	 * <p>
	 * The "name" of a double is the ID number of the original.
	 * 
	 * @param net
	 *            Source.
	 * 
	 * @return The number of single relations eliminated.
	 * @throws PuckException
	 */
	public static int eliminateDoubleRelations(final Net source, final Relations sample) throws PuckException {
		int result;

		List<Relation> relations = sample.toSortedList();

		Map<Integer, Integer> originalIds = new HashMap<Integer, Integer>();
		List<Integer> inexistentOriginals = new ArrayList<Integer>();
		List<Integer> selfReferentialOriginals = new ArrayList<Integer>();
		List<Integer> unreplacedIds = new ArrayList<Integer>();

		result = 0;
		for (Relation sourceRelation : relations) {

			if (NumberUtils.isNumber(sourceRelation.getName())) {
				//
				int originalTypedId = Integer.parseInt(sourceRelation.getName());
				RelationModel model = sourceRelation.getModel();

				if (originalTypedId == sourceRelation.getTypedId()) {
					selfReferentialOriginals.add(originalTypedId);
					continue;
				}

				Relation targetRelation = source.relations().getByTypedId(originalTypedId, model);

				while (targetRelation == null) {
					if (originalIds.get(originalTypedId) == null) {
						inexistentOriginals.add(originalTypedId);
						break;
					} else {
						originalTypedId = originalIds.get(originalTypedId);
						targetRelation = source.relations().getByTypedId(originalTypedId, model);
					}
				}
				
				if (targetRelation == null) {
					unreplacedIds.add(sourceRelation.getTypedId());
					continue;
				}

				originalIds.put(sourceRelation.getTypedId(), originalTypedId);

				String track = targetRelation.getAttributeValue("DOUBLE");
				if (track == null) {
					track = String.valueOf(sourceRelation.getTypedId());
				} else {
					track += ";" + String.valueOf(sourceRelation.getTypedId());
				}

				targetRelation.setAttribute("DOUBLE", track);

				// Append attributes.
				UpdateWorker.update(targetRelation.attributes(), sourceRelation.attributes(), UpdateMode.APPEND);

				// Append or fuse actors
				for (Actor sourceActor : sourceRelation.actors().toList()) {
					Individual individual = sourceActor.getIndividual();
					if (!targetRelation.actors().contains(sourceActor)) {
						targetRelation.actors().add(sourceActor);
						individual.relations().add(targetRelation);
					} else {
						Actor targetActor = targetRelation.getActor(sourceActor);
						UpdateWorker.update(targetActor.attributes(), sourceActor.attributes(), UpdateMode.APPEND);
					}
					sourceRelation.removeActor(sourceActor);
					individual.relations().remove(sourceRelation);
				}

				source.remove(sourceRelation);
				result += 1;
			}
		}
		
		// Make error report
		StringList reportList = new StringList();

		reportList.append("Inexistent original relations: ");
		for (int id : inexistentOriginals) {
			reportList.append(id + " ");
		}
		reportList.appendln();

		reportList.append("Selfreferential original relations: ");
		for (int id : selfReferentialOriginals) {
			reportList.append(id + " ");
		}
		reportList.appendln();

		reportList.append("Unreplaced relation ids: ");
		for (int id : unreplacedIds) {
			reportList.append(id + " ");
		}
		reportList.appendln();

		System.out.println(reportList);


		//
		return result;
	}

	/**
	 * Eliminates the individuals marked as double (i.e., individuals having a
	 * number instead of a name).
	 * <p>
	 * The "name" of a double is the ID number of the original.
	 * 
	 * @param net
	 *            Source.
	 * 
	 * @return The number of single individuals eliminated.
	 * @throws PuckException
	 */
	public static int eliminateMarkedDoubles(final Net source) throws PuckException {
		int result;

		result = eliminateDoubleIndividuals(source, source.individuals());

		//
		return result;
	}

	/**
	 * Eliminates single individual.
	 * 
	 * @param net
	 *            Source.
	 * 
	 * @return The number of single individuals eliminated.
	 */
	public static int eliminateSingles(final Net source) {
		int result;

		result = eliminateSingles(source, source.individuals());

		//
		return result;
	}

	/**
	 * Eliminates single individual.
	 * 
	 * @param net
	 *            Source.
	 * 
	 * @return The number of single individuals eliminated.
	 */
	public static int eliminateSingles(final Net source, final Individuals sample) {
		int result;

		List<Individual> individuals = sample.toList();
		result = 0;
		for (Individual individual : individuals) {
			if (individual.isSingle()) {
				//
				source.remove(individual);

				//
				result += 1;
			}
		}

		//
		return result;
	}

	/**
	 * @param net
	 *            Source.
	 * 
	 * @return The number of single individuals eliminated.
	 */
	public static int eliminateStructuralChildren(final Net source) {
		int result;

		result = eliminateStructuralChildren(source, source.individuals());

		//
		return result;
	}

	/**
	 * @param net
	 *            Source.
	 * 
	 * @return The number of single individuals eliminated.
	 */
	public static int eliminateStructuralChildren(final Net source, final Individuals sample) {
		int result;

		List<Individual> individuals = sample.toList();
		result = 0;
		for (Individual individual : individuals) {
			if ((individual.isSingle()) && (individual.isSterile())) {
				source.remove(individual);
				result += 1;
			}
		}

		//
		return result;
	}

	/**
	 * @param net
	 *            Source.
	 * 
	 * @return The number of single individuals eliminated.
	 */
	public static int eliminateVirtuals(final Net source) {
		int result;

		result = eliminateVirtuals(source, source.individuals());

		//
		return result;
	}

	/**
	 * @param net
	 *            Source.
	 * 
	 * @return The number of single individuals eliminated.
	 */
	public static int eliminateVirtuals(final Net source, final Individuals sample) {
		int result;

		List<Individual> individuals = sample.toList();
		result = 0;
		for (Individual individual : individuals) {
			if ((StringUtils.isBlank(individual.getName())) || individual.getName().equals("?") || (individual.getName().charAt(0) == '#')) {
				source.remove(individual);
				result += 1;
			}
		}

		//
		return result;
	}

	public static void enlargeNetFromAttributes(final Net net, final String fatherFirstNameLabel, final String fatherLastNameLabel,
			final String motherFirstNameLabel, final String motherLastNameLabel, final String husbandFirstNameLabel, final String husbandLastNameLabel,
			final String wifeFirstNameLabel, final String wifeLastNameLabel, final Map<String, String> husbandLabels, final Map<String, String> wifeLabels) {

		for (Family family : net.families().toSortedList()) {
			createIndividualsFromFamilyAttributes(net, family, husbandFirstNameLabel, husbandLastNameLabel, wifeFirstNameLabel, wifeLastNameLabel,
					husbandLabels, wifeLabels);
		}
		for (Individual ego : net.individuals().toSortedList()) {
			createOriginFamilyFromIndividualAttributes(net, ego, fatherFirstNameLabel, fatherLastNameLabel, motherFirstNameLabel, motherLastNameLabel);
		}
	}
	
	public static Individuals neighbors(final Individual ego, final ExpansionMode expansionMode, final FiliationType filiationType){
		Individuals result;
		
		result = new Individuals();

		switch (expansionMode) {
			case ALL:
				result = ego.getRelated();
				result.add(ego.getKin());
			break;

			case CHILD:
				result = ego.getKin(KinType.CHILD, filiationType);
			break;

			case KIN:
				result = ego.getKin();
			break;

			case PARENT:
				result = ego.getKin(KinType.PARENT, filiationType);

			break;

			case RELATED:
				result = ego.getRelated();
			break;

			case SPOUSE:
				result = ego.getKin(KinType.SPOUSE, filiationType);
			break;

			default:
				KinType kinType = KinType.valueOf(expansionMode.toString());
				result = ego.getKin(kinType, filiationType);
		}
		//
		return result;

	}

	/**
	 * 
	 * @param seeds
	 * @param expansionMode
	 * @param filiationType
	 * @param seedLabel
	 * @param maxStep
	 * @return
	 */
	public static Individuals expand(final Individuals seeds, final ExpansionMode expansionMode, final FiliationType filiationType, final String seedLabel,
			final int maxStep) {
		Individuals result;

		result = new Individuals();

		String label = "STEP_" + seedLabel.replaceAll(" ", "_") + "_" + expansionMode;
		if (filiationType != FiliationType.COGNATIC) {
			label += "_" + filiationType;
		}

		Queue<Individual> queue = new LinkedList<Individual>();
		for (Individual seed : seeds) {
			queue.add(seed);
			result.add(seed);
			seed.setAttribute(label, 0 + "");
		}

		while (!queue.isEmpty()) {

			Individual ego = queue.remove();
			int step = Integer.parseInt(ego.getAttributeValue(label)) + 1;

			if (maxStep <= 0 || step <= maxStep) {

				Individuals neighbors = neighbors(ego, expansionMode, filiationType);

				for (Individual neighbor : neighbors) {
					//
					if (!result.contains(neighbor)) {
						//
						queue.add(neighbor);
						result.add(neighbor);
						neighbor.setAttribute(label, step + "");
					}
				}
			}
		}

		//
		return result;
	}

	/**
	 * This method expands a net.
	 * 
	 * @param source
	 *            The net to expand.
	 * @param seeds
	 * @param seedLabel
	 * @param expansionMode
	 *            How expand the net (ALL, RELATED, KIN, PARENT, CHILD, SPOUSE).
	 * @param filiationType
	 *            Expansion parameter (COGNATIC, AGNATIC, UTERINE).
	 * @param maxStep
	 *            The limit of step expansion.
	 * @return A new net.
	 */
	public static Net expand(final Net source, final Individuals seeds, final String seedLabel, final ExpansionMode expansionMode,
			final FiliationType filiationType, final int maxStep) {
		Net result;

		Individuals individuals = expand(seeds, expansionMode, filiationType, seedLabel, maxStep);

		result = extract(source, individuals);

		String title = result.getLabel() + "_" + seedLabel + "_" + expansionMode;
		result.setLabel(title);

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 *            Net to update.
	 * 
	 * @param sourceFamilies
	 *            Families of the current individuals segment.
	 * 
	 * @return A new Net based on the current families segment.
	 */
	public static Net extract(final Net source, final Families sourceFamilies) {
		Net result;

		//
		result = new Net();
		result.setLabel(source.getLabel());
		result.attributes().addAll(source.attributes());

		// Extract families.
		for (Family sourceFamily : sourceFamilies) {
			//
			Family targetFamily = new Family(sourceFamily.getId());
			targetFamily.setMarried(sourceFamily.isMarried());
			targetFamily.setHusbandOrder(sourceFamily.getHusbandOrder());
			targetFamily.setWifeOrder(sourceFamily.getWifeOrder());
			targetFamily.attributes().addAll(sourceFamily.attributes());

			// targetFamily.setFather(sourceFamily.getFather());
			// targetFamily.setMother(sourceFamily.getMother());
			// targetFamily.getChildren().add(sourceFamily.getChildren());

			//
			result.families().add(targetFamily);
		}

		// Extract individuals.
		for (Individual sourceIndividual : source.individuals()) {
			//
			Individual targetIndividual = new Individual(sourceIndividual.getId());
			targetIndividual.setName(sourceIndividual.getName());
			targetIndividual.setGender(sourceIndividual.getGender());
			targetIndividual.setBirthOrder(sourceIndividual.getBirthOrder());
			targetIndividual.attributes().addAll(sourceIndividual.attributes());

			//
			if (sourceIndividual.getOriginFamily() != null) {
				//
				Family targetOriginFamily = result.families().getById(sourceIndividual.getOriginFamily().getId());

				//
				if (targetOriginFamily != null) {
					//
					targetIndividual.setOriginFamily(targetOriginFamily);

					targetOriginFamily.getChildren().add(targetIndividual);
				}
			}

			//
			for (Family sourcePersonalFamily : sourceIndividual.getPersonalFamilies()) {
				//
				Family targetPersonalFamily = result.families().getById(sourcePersonalFamily.getId());

				//
				if (targetPersonalFamily != null) {
					//
					targetIndividual.getPersonalFamilies().add(targetPersonalFamily);

					//
					if (sourcePersonalFamily.isFather(sourceIndividual)) {
						//
						targetPersonalFamily.setFather(targetIndividual);
					}

					// Do not add else there.

					//
					if (sourcePersonalFamily.isMother(sourceIndividual)) {
						//
						targetPersonalFamily.setMother(targetIndividual);
					}
				}
			}

			//
			if ((targetIndividual.getOriginFamily() != null) || (!targetIndividual.getPersonalFamilies().isEmpty())) {
				//
				result.individuals().add(targetIndividual);
			}
		}

		// Extract relation models.
		result.relationModels().addAll(source.relationModels());

		// Extract relations.
		for (Relation sourceRelation : source.relations()) {
			//
			RelationModel targetModel = result.relationModels().getByName(sourceRelation.getModel().getName());

			//
			Relation targetRelation = new Relation(sourceRelation.getId(), sourceRelation.getTypedId(), targetModel, sourceRelation.getName(), new Actor[0]);
			targetRelation.attributes().addAll(sourceRelation.attributes());

			//
			for (Actor sourceActor : sourceRelation.actors()) {
				//
				Individual targetIndividual = result.individuals().getById(sourceActor.getId());

				if (targetIndividual != null) {
					//
					Actor targetActor = new Actor(targetIndividual, targetModel.roles().getByName(sourceActor.getRole().getName()));
					targetActor.setRelationOrder(sourceActor.getRelationOrder());

					targetRelation.actors().add(targetActor);

					targetIndividual.relations().add(targetRelation);
				}
			}

			//
			if (!targetRelation.actors().isEmpty()) {
				//
				result.relations().add(targetRelation);
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 *            Net to update.
	 * 
	 * @param sourceIndividuals
	 *            Individuals of the current individuals segment.
	 * 
	 * @return A new Net based on the current individuals segment.
	 */
	public static Net extract(final Net source, final Individuals sourceIndividuals) {
		Net result;

		//
		result = new Net();
		result.setLabel(source.getLabel());
		result.attributes().addAll(source.attributes());

		// Extract individuals.
		for (Individual sourceIndividual : sourceIndividuals) {
			//
			Individual targetIndividual = new Individual(sourceIndividual.getId());
			targetIndividual.setName(sourceIndividual.getName());
			targetIndividual.setGender(sourceIndividual.getGender());
			targetIndividual.setBirthOrder(sourceIndividual.getBirthOrder());
			targetIndividual.attributes().addAll(sourceIndividual.attributes());

			//
			result.individuals().add(targetIndividual);
		}

		// Extract families.
		for (Family sourceFamily : source.families()) {
			//
			Family targetFamily = new Family(sourceFamily.getId());
			targetFamily.setMarried(sourceFamily.isMarried());
			targetFamily.setHusbandOrder(sourceFamily.getHusbandOrder());
			targetFamily.setWifeOrder(sourceFamily.getWifeOrder());
			targetFamily.attributes().addAll(sourceFamily.attributes());

			//
			Individual sourceFather = sourceFamily.getFather();
			if (sourceFather != null) {
				//
				Individual targetFather = result.individuals().getById(sourceFather.getId());

				if (targetFather != null) {
					//
					targetFamily.setFather(targetFather);
					targetFather.getPersonalFamilies().add(targetFamily);
				}
			}

			//
			Individual sourceMother = sourceFamily.getMother();
			if (sourceMother != null) {
				//
				Individual targetMother = result.individuals().getById(sourceMother.getId());

				if (targetMother != null) {
					//
					targetFamily.setMother(targetMother);
					targetMother.getPersonalFamilies().add(targetFamily);
				}
			}

			//
			for (Individual sourceChild : sourceFamily.getChildren()) {
				//
				Individual targetChild = result.individuals().getById(sourceChild.getId());
				if (targetChild != null) {
					//
					targetFamily.getChildren().add(targetChild);

					targetChild.setOriginFamily(targetFamily);
				}
			}

			//
			if ((targetFamily.getFather() != null) || (targetFamily.getMother() != null) || (!targetFamily.getChildren().isEmpty())) {
				//
				result.families().add(targetFamily);
			}
		}

		// Extract relation models.
		result.relationModels().addAll(source.relationModels());

		// Extract relations.
		for (Relation sourceRelation : source.relations()) {
			//
			RelationModel targetModel = result.relationModels().getByName(sourceRelation.getModel().getName());

			//
			Relation targetRelation = new Relation(sourceRelation.getId(), sourceRelation.getTypedId(), targetModel, sourceRelation.getName(), new Actor[0]);

			//
			targetRelation.attributes().addAll(sourceRelation.attributes());

			//
			for (Actor sourceActor : sourceRelation.actors()) {
				//
				Individual targetIndividual = result.individuals().getById(sourceActor.getId());

				if (targetIndividual != null) {
					//
					Actor targetActor = new Actor(targetIndividual, targetModel.roles().getByName(sourceActor.getRole().getName()));
					targetActor.setRelationOrder(sourceActor.getRelationOrder());

					targetRelation.actors().add(targetActor);

					targetIndividual.relations().add(targetRelation);
				}
			}

			//
			if (!targetRelation.actors().isEmpty()) {
				//
				result.relations().add(targetRelation);
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 *            Net to update.
	 * 
	 * @param sourceRelations
	 *            Relations of the current individuals segment.
	 * 
	 * @return A new Net based on the current relations segment.
	 */
	public static Net extract(final Net source, final Relations sourceRelations) {
		Net result;

		//
		result = new Net();
		result.setLabel(source.getLabel());
		result.attributes().addAll(source.attributes());

		// Extract relation models.
		result.relationModels().addAll(source.relationModels());

		// Extract Relations.
		for (Relation sourceRelation : sourceRelations) {
			//
			RelationModel targetModel = result.relationModels().getByName(sourceRelation.getModel().getName());

			//
			Relation targetRelation = new Relation(sourceRelation.getId(), sourceRelation.getTypedId(), targetModel, sourceRelation.getName(), new Actor[0]);
			targetRelation.attributes().addAll(sourceRelation.attributes());

			//
			result.relations().add(targetRelation);
		}

		// Extract individuals.
		for (Individual sourceIndividual : source.individuals()) {
			//
			Individual targetIndividual = new Individual(sourceIndividual.getId());
			targetIndividual.setName(sourceIndividual.getName());
			targetIndividual.setGender(sourceIndividual.getGender());
			targetIndividual.setBirthOrder(sourceIndividual.getBirthOrder());
			targetIndividual.attributes().addAll(sourceIndividual.attributes());

			for (Relation sourceRelation : sourceIndividual.relations()) {
				//
				RelationModel targetModel = result.relationModels().getByName(sourceRelation.getModel().getName());

				Relation targetRelation = result.relations().getById(sourceRelation.getId());

				if (targetRelation != null) {
					//
					targetIndividual.relations().add(targetRelation);

					for (Actor sourceActor : sourceRelation.actors()) {
						//
						if (sourceActor.getIndividual() == sourceIndividual) {
							//
							Actor targetActor = new Actor(targetIndividual, targetModel.roles().getByName(sourceActor.getRole().getName()));
							targetActor.setRelationOrder(sourceActor.getRelationOrder());

							targetRelation.actors().add(targetActor);
						}
					}
				}
			}

			//
			if (targetIndividual.relations().isEmpty()) {
				//
				result.individuals().add(targetIndividual);
			}
		}

		// Extract families.
		for (Family sourceFamily : source.families()) {
			//
			Family targetFamily = new Family(sourceFamily.getId());
			targetFamily.setMarried(sourceFamily.isMarried());
			targetFamily.setHusbandOrder(sourceFamily.getHusbandOrder());
			targetFamily.setWifeOrder(sourceFamily.getWifeOrder());
			targetFamily.attributes().addAll(sourceFamily.attributes());

			//
			Individual sourceFather = sourceFamily.getFather();
			if (sourceFather != null) {
				//
				Individual targetFather = result.individuals().getById(sourceFather.getId());

				if (targetFather != null) {
					//
					targetFamily.setFather(targetFather);
					targetFather.getPersonalFamilies().add(targetFamily);
				}
			}

			//
			Individual sourceMother = sourceFamily.getMother();
			if (sourceMother != null) {
				//
				Individual targetMother = result.individuals().getById(sourceMother.getId());

				if (targetMother != null) {
					//
					targetFamily.setMother(targetMother);
					targetMother.getPersonalFamilies().add(targetFamily);
				}
			}

			//
			for (Individual sourceChild : sourceFamily.getChildren()) {
				//
				Individual targetChild = result.individuals().getById(sourceChild.getId());
				if (targetChild != null) {
					//
					targetFamily.getChildren().add(targetChild);

					targetChild.setOriginFamily(targetFamily);
				}
			}

			//
			if ((targetFamily.getFather() != null) || (targetFamily.getMother() != null) || (!targetFamily.getChildren().isEmpty())) {
				//
				result.families().add(targetFamily);
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 *            Network from which to extract.
	 * @param segment
	 * @param segmentLabel
	 * @return
	 * @throws PuckException
	 */
	public static Net extract(final Net source, final Segment segment, final String segmentLabel) throws PuckException {
		Net result;

		if ((source == null) || (segment == null)) {
			//
			result = null;

		} else {
			//
			PartitionCriteria criteria = segment.getCriteria();

			//
			if (criteria.getRelationModelName().equals(PartitionCriteria.RelationModelCanonicalNames.INDIVIDUAL.toString())) {
				//
				Partition<Individual> partition = (Partition<Individual>) segment.getPartition();
				//
				Individuals individuals = new Individuals();
				for (Cluster<Individual> cluster : partition.getClusters()) {
					//
					if (!cluster.isNull()) {
						//
						individuals.add(cluster.getItems());
					}
				}

				//
				result = extract(source, individuals);

			} else if (criteria.getRelationModelName().equals(PartitionCriteria.RelationModelCanonicalNames.FAMILY.toString())) {
				//
				Partition<Family> partition = (Partition<Family>) segment.getPartition();
				//
				Families families = new Families();
				for (Cluster<Family> cluster : partition.getClusters()) {
					//
					if (!cluster.isNull()) {
						//
						families.add(cluster.getItems());
					}
				}

				//
				result = extract(source, families);

			} else {
				//
				Partition<Relation> partition = (Partition<Relation>) segment.getPartition();

				//
				Relations relations = new Relations();
				for (Cluster<Relation> cluster : partition.getClusters()) {
					//
					if (!cluster.isNull()) {
						//
						relations.add(cluster.getItems());
					}
				}

				//
				result = extract(source, relations);
			}

			//
			result.setLabel(segmentLabel);
		}

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 *            Network from which to extract.
	 * @param segment
	 * @param segmentLabel
	 * @param minimalValue
	 * @return
	 * @throws PuckException
	 */
	public static Net extractByClusterSize(final Net source, final Segment segment, final String segmentLabel, final int minimalNumberOfMembers)
			throws PuckException {
		Net result;

		if ((source == null) || (segment == null)) {
			//
			result = null;

		} else {
			//
			PartitionCriteria criteria = segment.getCriteria();

			//
			if (criteria.getRelationModelName().equals(PartitionCriteria.RelationModelCanonicalNames.INDIVIDUAL.toString())) {
				//
				Partition<Individual> partition = (Partition<Individual>) segment.getPartition();
				//
				Individuals individuals = new Individuals();
				for (Cluster<Individual> cluster : partition.getClusters()) {
					//
					if ((!cluster.isNull()) && (cluster.count() >= minimalNumberOfMembers)) {
						//
						individuals.add(cluster.getItems());
					}
				}

				//
				result = extract(source, individuals);

			} else if (criteria.getRelationModelName().equals(PartitionCriteria.RelationModelCanonicalNames.FAMILY.toString())) {
				//
				Partition<Family> partition = (Partition<Family>) segment.getPartition();
				//
				Families families = new Families();
				for (Cluster<Family> cluster : partition.getClusters()) {
					//
					if ((!cluster.isNull()) && (cluster.count() >= minimalNumberOfMembers)) {
						//
						families.add(cluster.getItems());
					}
				}

				//
				result = extract(source, families);

			} else {
				//
				Partition<Relation> partition = (Partition<Relation>) segment.getPartition();

				//
				Relations relations = new Relations();
				for (Cluster<Relation> cluster : partition.getClusters()) {
					//
					if ((!cluster.isNull()) && (cluster.count() >= minimalNumberOfMembers)) {
						//
						relations.add(cluster.getItems());
					}
				}

				//
				result = extract(source, relations);
			}

			//
			result.setLabel(segmentLabel);
		}

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 *            Network from which to extract.
	 * @param label
	 * @param labelParameter
	 *            Parameter of the label (example: 3 for PEDG 3).
	 * @param minimalNumberOfMembers
	 * @return
	 * @throws PuckException
	 */
	public static Net extractByClusterSize(final Net source, final String label, final String labelParameter, final int minimalNumberOfMembers)
			throws PuckException {
		Net result;

		//
		Partition<Individual> partition = PartitionMaker.createRaw(source, label, labelParameter);

		//
		Individuals individuals = new Individuals();
		for (Cluster<Individual> cluster : partition.getClusters()) {
			if (!cluster.isNull() && cluster.count() >= minimalNumberOfMembers) {
				individuals.add(cluster.getItems());
			}
		}

		//
		result = extract(source, individuals);

		//
		String netLabel;
		if (StringUtils.isBlank(labelParameter)) {
			//
			netLabel = result.getLabel() + "_" + label + "_min_" + minimalNumberOfMembers;

		} else {
			//
			netLabel = result.getLabel() + "_" + label + " " + labelParameter + "_min_" + minimalNumberOfMembers;
		}
		result.setLabel(netLabel);

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 *            Network from which to extract.
	 * @param segment
	 * @param segmentLabel
	 * @param minimalValue
	 * @return
	 * @throws PuckException
	 */
	public static Net extractByClusterValue(final Net source, final Segment segment, final String segmentLabel, final int minimalValue) throws PuckException {
		Net result;

		if ((source == null) || (segment == null)) {
			//
			result = null;

		} else {
			//
			PartitionCriteria criteria = segment.getCriteria();

			//
			if (criteria.getRelationModelName().equals(PartitionCriteria.RelationModelCanonicalNames.INDIVIDUAL.toString())) {
				//
				Partition<Individual> partition = (Partition<Individual>) segment.getPartition();
				//
				Individuals individuals = new Individuals();
				for (Cluster<Individual> cluster : partition.getClusters()) {
					//
					if ((!cluster.isNull()) && (cluster.getValue().doubleValue() >= minimalValue)) {
						//
						individuals.add(cluster.getItems());
					}
				}

				//
				result = extract(source, individuals);

			} else if (criteria.getRelationModelName().equals(PartitionCriteria.RelationModelCanonicalNames.FAMILY.toString())) {
				//
				Partition<Family> partition = (Partition<Family>) segment.getPartition();
				//
				Families families = new Families();
				for (Cluster<Family> cluster : partition.getClusters()) {
					//
					if ((!cluster.isNull()) && (cluster.getValue().doubleValue() >= minimalValue)) {
						//
						families.add(cluster.getItems());
					}
				}

				//
				result = null; // extract(source, families);

			} else {
				//
				Partition<Relation> partition = (Partition<Relation>) segment.getPartition();

				//
				Relations relations = new Relations();
				for (Cluster<Relation> cluster : partition.getClusters()) {
					//
					if ((!cluster.isNull()) && (cluster.getValue().doubleValue() >= minimalValue)) {
						//
						relations.add(cluster.getItems());
					}
				}

				//
				result = null; // extract(source, relations);
			}

			//
			result.setLabel(segmentLabel);
		}

		//
		return result;
	}

	/**
	 * 
	 * @param source
	 *            Network from which to extract.
	 * @param label
	 * @param labelParameter
	 *            Parameter of the label (example: 3 for PEDG 3).
	 * @param minimalValue
	 * @return
	 * @throws PuckException
	 */
	public static Net extractByClusterValue(final Net source, final String label, final String labelParameter, final int minimalValue) throws PuckException {
		Net result;

		//
		Partition<Individual> partition = PartitionMaker.createRaw(source, label, labelParameter);

		//
		Individuals individuals = new Individuals();
		for (Cluster<Individual> cluster : partition.getClusters()) {
			if (!cluster.isNull() && cluster.getValue().doubleValue() >= minimalValue) {
				individuals.add(cluster.getItems());
			}
		}

		//
		result = extract(source, individuals);

		//
		String netLabel;
		if (StringUtils.isBlank(labelParameter)) {
			//
			netLabel = result.getLabel() + "_" + label + "_min_" + minimalValue;

		} else {
			//
			netLabel = result.getLabel() + "_" + label + " " + labelParameter + "_min_" + minimalValue;
		}
		result.setLabel(netLabel);

		//
		return result;
	}

	/*	private static void expand(final Individuals individuals, final Individual ego) {
			if (!individuals.contains(ego)) {
				individuals.add(ego);
				ego.setAttribute("EXPANDED", "true");

				Individuals related = ego.getRelated();
				related.add(ego.getKin());

				for (Individual alter : related) {
					expand(individuals, alter);
				}
			}
		}

		
		public static Net expand(final Net source, final Segment segment) {
			Net result;

			Individuals individuals = new Individuals();
			for (Individual individual : segment.getCurrentIndividuals()) {
				expand(individuals, individual);
			}

			result = extract(source, individuals);

			String label = result.getLabel() + "_" + segment.getLabel() + "_expanded_ALL";
			result.setLabel(label);

			//
			return result;
		}


		private static void expandByKin(final Individuals individuals, final Individual ego, final KinType kinType) {
			if (!individuals.contains(ego)) {
				individuals.add(ego);
				ego.setAttribute("EXPANDED_" + kinType, "true");

				Individuals kin;
				if (kinType == null) {
					kin = ego.getKin();
				} else {
					kin = ego.getKin(kinType);
				}

				for (Individual alter : kin) {
					expandByKin(individuals, alter, kinType);
				}
			}
		}

		public static Net expandByKin(final Net source, final Segment segment, final KinType direction) {
			Net result;

			Individuals individuals = new Individuals();
			for (Individual individual : segment.getCurrentIndividuals()) {
				expandByKin(individuals, individual, direction);
			}

			result = extract(source, individuals);

			String label = result.getLabel() + "_" + segment.getLabel() + "_expanded";
			if (direction != null) {
				label = label + "_" + direction;
			} else {
				label = label + "_ALL_KIN";
			}
			result.setLabel(label);

			//
			return result;
		}

		private static void expandByRelations(final Individuals individuals, final Individual ego) {
			if (!individuals.contains(ego)) {
				individuals.add(ego);
				ego.setAttribute("EXPANDED_RELATED", "true");

				Individuals related = ego.getRelated();

				for (Individual alter : related) {
					expandByRelations(individuals, alter);
				}
			}
		}

		public static Net expandByRelations(final Net source, final Segment segment) {
			Net result;

			Individuals individuals = new Individuals();
			for (Individual individual : segment.getCurrentIndividuals()) {
				expandByRelations(individuals, individual);
			}

			result = extract(source, individuals);

			String label = result.getLabel() + "_" + segment.getLabel() + "_expanded_RELATIONS";
			result.setLabel(label);

			//
			return result;
		}*/

	/**
	 * 
	 * @param source
	 *            Network from which to extract.
	 * @param segmentation
	 * @return
	 * @throws PuckException
	 */
	public static Net extractCurrentCluster(final Net source, final Segmentation segmentation) throws PuckException {
		Net result;

		if ((source == null) || (segmentation == null)) {
			//
			result = null;

		} else if (segmentation.isAtTheTop()) {
			//
			result = new Net(source);

		} else {
			//
			PartitionCriteria criteria = segmentation.getCurrentSegment().getCriteria();

			//
			if (criteria.getRelationModelName().equals(PartitionCriteria.RelationModelCanonicalNames.INDIVIDUAL.toString())) {
				//
				result = extract(source, segmentation.getCurrentIndividuals());

			} else if (criteria.getRelationModelName().equals(PartitionCriteria.RelationModelCanonicalNames.FAMILY.toString())) {
				//
				result = extract(source, segmentation.getCurrentFamilies());

			} else {
				//
				result = extract(source, segmentation.getCurrentRelations());
			}

			//
			result.setLabel("Extact current cluster");
		}

		//
		return result;
	}

	/**
	 * 
	 * @param parent1
	 * @param parent2
	 * @return
	 */
	public static Individual fixFatherByGender(final Individual ego, final Individual alter) {
		Individual result;

		if ((ego == null) && (alter == null)) {
			result = null;
		} else if (ego == null) {
			if (alter.isFemale()) {
				result = ego;
			} else {
				result = alter;
			}
		} else if (alter == null) {
			if (ego.isMale()) {
				result = ego;
			} else {
				result = alter;
			}
		} else {
			if (ego.isMale() || (alter.isFemale() && ego.isUnknown())) {
				result = ego;
			} else {
				result = alter;
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @param parent1
	 * @param parent2
	 * @return
	 */
	public static Individual fixMotherByGender(final Individual ego, final Individual alter) {
		Individual result;

		Individual newHusband = fixFatherByGender(ego, alter);

		if (newHusband == ego) {
			result = alter;
		} else {
			result = ego;
		}

		//
		return result;
	}

	/**
	 * 
	 * @param family
	 */
	public static void fixSpouseRolesByGender(final Family family) {

		if (family != null) {
			//
			Individual newHusband = fixFatherByGender(family.getHusband(), family.getWife());

			if (newHusband != family.getHusband()) {
				swapParents(family);
			}
		}
	}

	/**
	 * 
	 * @param family
	 */
	public static void fixSpouseRolesByGender2(final Family family) {

		if (family != null) {
			Individual ego = family.getHusband();
			Individual alter = family.getWife();

			Individual husband;
			Individual wife;
			if (ego.isMale() || (alter.isFemale() && ego.isUnknown())) {
				husband = ego;
				wife = alter;
			} else {
				husband = alter;
				wife = ego;
			}

			family.setHusband(husband);
			family.setWife(wife);
		}
	}

	public static void fuseIndividuals(final Net source, final Individual sourceIndividual, final Individual targetIndividual) throws PuckException {

		String track = targetIndividual.getAttributeValue("DOUBLE");
		if (track == null) {
			track = String.valueOf(sourceIndividual.getId());
		} else {
			track += ";" + String.valueOf(sourceIndividual.getId());
		}

		targetIndividual.setAttribute("DOUBLE", track);

		// Append attributes.
		UpdateWorker.update(targetIndividual.attributes(), sourceIndividual.attributes(), UpdateMode.APPEND);

		// Append parents.
		if ((targetIndividual.getFather() == null) && (sourceIndividual.getFather() != null)) {
			NetUtils.setFatherRelation(source, sourceIndividual.getFather().getId(), targetIndividual.getId());
		}
		if ((targetIndividual.getMother() == null) && (sourceIndividual.getMother() != null)) {
			NetUtils.setMotherRelation(source, sourceIndividual.getMother().getId(), targetIndividual.getId());
		}

		// Append or fuse personalFamilies
		for (Family sourceFamily : sourceIndividual.getPersonalFamilies()) {
			Family targetFamily;
			if (sourceIndividual == sourceFamily.getHusband()) {
				targetFamily = source.families().getBySpouses(targetIndividual, sourceFamily.getWife());
				if (targetFamily == null) {
					sourceFamily.setHusband(targetIndividual);
					targetIndividual.addPersonalFamily(sourceFamily);
				} else {
					targetFamily.getChildren().add(sourceFamily.getChildren());
					for (Individual child : sourceFamily.getChildren()) {
						child.setOriginFamily(targetFamily);
					}
					if (sourceFamily.getWife() != null) {
						sourceFamily.getWife().getPersonalFamilies().removeById(sourceFamily.getId());
					}
					source.families().removeById(sourceFamily.getId());
				}
			} else if (sourceIndividual == sourceFamily.getWife()) {
				targetFamily = source.families().getBySpouses(sourceFamily.getHusband(), targetIndividual);
				if (targetFamily == null) {
					sourceFamily.setWife(targetIndividual);
					targetIndividual.addPersonalFamily(sourceFamily);
				} else {
					targetFamily.getChildren().add(sourceFamily.getChildren());
					for (Individual child : sourceFamily.getChildren()) {
						child.setOriginFamily(targetFamily);
					}
					if (sourceFamily.getHusband() != null) {
						sourceFamily.getHusband().getPersonalFamilies().removeById(sourceFamily.getId());
					}
					source.families().removeById(sourceFamily.getId());
				}
			}
		}

		source.remove(sourceIndividual);
	}

	public static Map<Individual, String> getAlterRelations(final Individual ego, final List<Individual> alters, final int[] maxDegrees,
			final List<String> relationModelNames) {
		Map<Individual, String> result;

		result = new HashMap<Individual, String>();

		for (Individual alter : alters) {
			result.put(alter, getAlterRole(ego, alter, maxDegrees, relationModelNames));
		}
		//
		return result;
	}

	public static Map<Individual, List<String>> getAlterRelations1(final Individual ego, final Set<Individual> alters, final int[] maxDegrees,
			final List<String> relationModelNames, final String chainClassification, final List<Individual> excludedIntermediaries) {
		Map<Individual, List<String>> result;

		result = new HashMap<Individual, List<String>>();

		for (Individual alter : alters) {
			result.put(alter, getAlterRoles(ego, alter, maxDegrees, relationModelNames, chainClassification, excludedIntermediaries));
		}
		//
		return result;
	}

	public static String getAlterRole(final Individual ego, final Individual alter, final int[] maxDegrees, final List<String> relationModelNames) {
		String result;

		result = "UNKNOWN";

		// Check for relatives

		if (ego == alter) {
			result = "IDENTITY";
		} else if (ego.getFather()!=null && ego.getFather().equals(alter)) {
			result = "FATHER";
		} else if (ego.getMother()!=null && ego.getMother().equals(alter)) {
			result = "MOTHER";
		} else if (ego.spouses().contains(alter)) {
			result = "SPOUSE";
		} else if (ego.children().contains(alter)) {
			result = "CHILD";
		} else {
			Chain chain = ChainFinder.findShortestChain(ego, alter, maxDegrees);
			if (chain != null) {
				if (chain.dim() > 1) {
					result = "AFFINE";
				} else {
					String chainType = "";
					chain.getSubchains();
					Value chainValue = ChainValuator.get(chain, "LINE");
					if (chainValue != null) {
						chainType = "_" + chainValue.toString();
					}
					result = "RELATIVE" + chainType;
				}
			} else if (relationModelNames!=null) {
				for (Relation relation : ego.relations()) {
					if (relationModelNames.contains(relation.getModel().getName())) {
						if (relation.getIndividuals().contains(alter)) {
							result = relation.getRoleNames(alter).get(0).toString();
							break;
						}
					}
				}

			}
		}

		//
		return result;
	}

	public static List<String> getAlterRoles(final Individual ego, final Individual alter, final int[] maxDegrees, final List<String> relationModelNames,
			final String chainClassification) {
		return getAlterRoles(ego, alter, maxDegrees, relationModelNames, chainClassification, null);
	}

	public static List<String> getAlterRoles(final Individual ego, final Individual alter, final int[] maxDegrees, final List<String> relationModelNames,
			final String chainClassification, final List<Individual> excludedIntermediaries) {
		List<String> result;

		result = new ArrayList<String>();

		// Check for relatives

		if (ego == alter) {
			result.add("EGO");
		} else if (ego.getFather() != null && ego.getFather().equals(alter)) {
			result.add("FATHER");
		} else if (ego.getMother() != null && ego.getMother().equals(alter)) {
			result.add("MOTHER");
		} else if (ego.spouses().contains(alter)) {
			result.add("SPOUSE");
		} else if (ego.children().contains(alter)) {
			result.add("CHILD");
		} else {
			// Check for siblings
			if (ego.siblings().contains(alter)) {
				if (excludedIntermediaries == null) {
					result.add("SIBLING");
				} else {
					boolean noExcludedIntermediaries = true;
					for (Individual parent : ego.getParents()) {
						if (alter.getParents().contains(parent) && excludedIntermediaries.contains(parent)) {
							noExcludedIntermediaries = false;
							break;
						}
					}
					if (noExcludedIntermediaries) {
						result.add("SIBLING");
					}
				}
			}
			// Check for other relatives
			if (!result.contains("SIBLING")) {

				Chain chain = ChainFinder.findShortestChain(ego, alter, maxDegrees);

				if (chain != null) {

					boolean noExcludedIntermediaries = true;
					if (excludedIntermediaries != null && chain.size() > 2) {
						for (Individual intermediary : excludedIntermediaries) {
							for (int i = 1; i < chain.size() - 1; i++) {
								if (chain.get(i) instanceof Couple) {
									for (Individual apex : ((Couple) (chain.get(i))).individuals()) {
										if (apex.equals(intermediary)) {
											noExcludedIntermediaries = false;
											break;
										}
									}
								}
								if (chain.get(i).equals(intermediary)) {
									noExcludedIntermediaries = false;
									break;
								}
							}
							if (!noExcludedIntermediaries) {
								break;
							}
						}
					}
					if (noExcludedIntermediaries) {
						if (chain.dim() > 1) {
							result.add("AFFINE");
						} else {
							String chainType = "";
							if (chainClassification != null) {
								chain.getSubchains();
								Value chainValue = ChainValuator.get(chain, chainClassification);
								if (chainValue != null) {
									chainType = "_" + chainValue.toString();
								}
							}
							result.add("RELATIVE" + chainType);
						}
					}
				}
			}
		}
		if (ego != alter) {
			for (Relation relation : ego.relations()) {
				if (relationModelNames.contains(relation.getModel().getName())) {
					if (relation.getIndividuals().contains(alter)) {
						for (String roleName : relation.getRoleNames(alter)) {
							if (!result.contains(roleName)) {
								result.add(roleName);
							}
						}
					}
				}
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @param father
	 * @param mother
	 * @return
	 */
	public static boolean isBirthOrderUsed(final Individuals source) {
		boolean result;

		if (source == null) {
			//
			result = false;
		} else {
			result = true;
			boolean ended = false;
			Iterator<Individual> iterator = source.iterator();
			while (!ended) {
				if (iterator.hasNext()) {
					//
					Individual current = iterator.next();

					//
					if (current.getBirthOrder() != null) {
						ended = true;
						result = true;
					}
				} else {
					//
					ended = true;
					result = false;
				}
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @param father
	 * @param mother
	 * @return
	 */
	public static boolean isFemaleFatherOrMaleMother(final Individual father, final Individual mother) {
		boolean result;

		if ((father == null) || (mother == null)) {
			result = false;
		} else if ((father.isFemale()) || (mother.isMale())) {
			result = true;
		} else {
			result = false;
		}

		//
		return result;
	}

	/**
	 * 
	 * @param father
	 * @param mother
	 * @return
	 */
	public static boolean isParentChildMarriage(final Family family, final Individual children) {
		boolean result;

		if ((family == null) || (children == null)) {
			//
			result = false;
		} else {
			//
			Individuals partners = children.spouses();

			if ((partners.contains(family.getFather())) || (partners.contains(family.getMother()))) {
				//
				result = true;
			} else {
				//
				result = false;
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @param father
	 * @param mother
	 * @return
	 */
	public static boolean isParentChildMarriage(final Individual parent, final Individual children) {
		boolean result;

		if ((parent == null) || (children == null)) {
			result = false;
		} else if (children.spouses().contains(parent)) {
			result = true;
		} else {
			result = false;
		}

		//
		return result;
	}

	/**
	 * 
	 * @param father
	 * @param mother
	 * @return
	 */
	public static boolean isParentChildMarriage(final Individual parent, final Individual... children) {
		boolean result;

		if (parent == null) {
			result = false;
		} else {

			boolean ended = false;
			result = false;
			int childIndex = 0;
			while (!ended) {
				if (childIndex < children.length) {
					if (isParentChildMarriage(parent, children[childIndex])) {
						ended = true;
						result = true;
					} else {
						childIndex += 1;
					}
				} else {
					ended = true;
					result = false;
				}
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @param father
	 * @param mother
	 * @return
	 */
	public static boolean isParentChildMarriage(final Individual father, final Individual mother, final Individual... children) {
		boolean result;

		if ((isParentChildMarriage(father, children)) || (isParentChildMarriage(mother, children))) {
			result = true;
		} else {
			result = false;
		}

		//
		return result;
	}

	/**
	 * 
	 * @param parent1
	 * @param parent2
	 * @return
	 */
	public static boolean isRolesFixedByGender(final Individual ego, final Individual alter) {
		boolean result;

		Individual newHusband = fixFatherByGender(ego, alter);

		if (newHusband == ego) {
			result = true;
		} else {
			result = false;
		}

		//
		return result;
	}

	/**
	 * 
	 * @return
	 */
	public static boolean isSame(final Individual ego, final Individual alter) {
		boolean result;

		if ((ego == null) || (alter == null)) {
			result = false;
		} else if (ego == alter) {
			result = true;
		} else {
			result = false;
		}

		//
		return result;
	}

	/**
	 * 
	 * @return
	 */
	public static boolean isSameSex(final Individual ego, final Individual alter) {
		boolean result;

		if ((ego == null) || (alter == null) || (ego.isUnknown()) || (alter.isUnknown())) {
			result = false;
		} else if (ego.getGender() == alter.getGender()) {
			result = true;
		} else {
			result = false;
		}

		//
		return result;
	}

	public static int marryCoparents(final Net source) {
		int result;

		//
		result = 0;

		for (Family family : source.families()) {
			if (!family.isSterile() && !family.isSingleParent() && !family.isMarried()) {
				family.setMarried();
				result++;
			}
		}

		//
		return result;

	}

	/**
	 * Adds the identity number between parentheses to the original name (e.g.
	 * "Pierre Dupont (5)").
	 * 
	 * @param net
	 *            Source to modify.
	 */
	public static void numberNames(final Individuals source) {
		if (source != null) {
			for (Individual individual : source) {
				individual.setName(individual.getTrimmedName() + " (" + individual.getId() + ")");
			}
		}
	}

	/**
	 * Adds the identity number between parentheses to the original name (e.g.
	 * "Pierre Dupont (5)").
	 * 
	 * @param net
	 *            Source to modify.
	 */
	public static void numberNames(final Net source) {
		if (source != null) {
			numberNames(source.individuals());
		}
	}

	/**
	 * 
	 * @param family
	 */
	public static void removeFather(final Family family) {
		if (family != null) {
			family.getFather().getPersonalFamilies().removeById(family.getId());
			family.setFather(null);
		}
	}

	/**
	 * 
	 * @param family
	 */
	public static void removeMother(final Family family) {
		if ((family != null) && (family.getMother() != null)) {
			family.getMother().getPersonalFamilies().removeById(family.getId());
			family.setMother(null);
		}
	}

	/**
	 * 
	 * @param family
	 */
	public static void removeSpouse(final Family family, final Individual spouse) {
		if ((family != null) && (spouse != null)) {
			if (family.getFather() == spouse) {
				removeFather(family);
			} else if (family.getMother() == spouse) {
				removeMother(family);
			}
		}
	}

	/**
	 * This method renumerates individuals of a net.
	 * 
	 * @param net
	 *            Source to renumberate.
	 */
	public static void renumber(final Net source) {
		//
		if (source != null) {
			//
			List<Individual> individuals = source.individuals().toSortedList();

			//
			source.individuals().clear();

			//
			int index = 0;
			for (Individual individual : individuals) {
				//
				individual.setId(index);
				index += 1;
				source.individuals().add(individual);
			}
		}
	}

	/**
	 * Re-numberate a net.
	 * 
	 * @param net
	 *            Source to renumberate.
	 */
	public static int renumberFromAttribute(final Net source, final String attributeLabel) {
		int result;

		//
		result = 0;

		//
		if ((source != null) && (attributeLabel != null)) {
			//
			Individuals candidates = source.individuals().searchByAttribute(attributeLabel, "^I?\\d+$");
			Individuals unchanged = source.individuals().copy().remove(candidates);

			logger.debug("candidates: " + candidates.size());
			logger.debug("unchanged:  " + unchanged.size());

			// Check new ids.
			// This step is required because id change touch individual. So, if
			// an eror comes after an individual touch, then the net is
			// partially touched.
			Set<Integer> newIds = new HashSet<Integer>(source.size());
			for (Individual individual : unchanged) {
				//
				newIds.add(individual.getId());
			}

			for (Individual individual : candidates) {
				//
				String value = individual.getAttributeValue(attributeLabel);
				int id;
				if (value.startsWith("I")) {
					//
					id = Integer.parseInt(value.substring(1));

				} else {
					//
					id = Integer.parseInt(value);
				}

				if (newIds.contains(id)) {
					//
					throw new IllegalArgumentException(String.format("value (%d) is already set.", id));

				} else {
					//
					newIds.add(id);
				}
			}

			//
			Individuals targets = new Individuals(source.individuals().size());
			targets.add(unchanged);

			//
			for (Individual individual : candidates) {
				//
				String value = individual.getAttributeValue(attributeLabel);
				int id;
				if (value.startsWith("I")) {
					//
					id = Integer.parseInt(value.substring(1));

				} else {
					//
					id = Integer.parseInt(value);
				}

				//
				individual.setAttribute(attributeLabel, String.valueOf(individual.getId()));
				individual.setId(id);
				targets.add(individual);

				result += 1;
			}

			//
			source.individuals().clear();
			source.individuals().add(targets);
		}

		//
		return result;
	}

	/**
	 * 
	 * @param parentId
	 * @param childId
	 * @throws PuckException
	 */
	public static void setAttribute(final Net net, final int individualId, final String label, final String value) throws PuckException {
		if ((net != null) && (individualId != 0) && (StringUtils.isNotBlank(label)) && (StringUtils.isNotBlank(value))) {
			//
			Individual individual = net.individuals().getById(individualId);
			if (individual == null) {
				individual = new Individual(individualId);
				net.individuals().add(individual);
			}

			//
			individual.attributes().put(label, value);
		}
	}

	/**
	 * WARNING: the use of this method can created duplicated families. WARNING:
	 * use only if source does not contain family ids.
	 * 
	 * @param parentId
	 * @param childId
	 * @throws PuckException
	 */
	public static void setFatherRelation(final Net net, final int fatherId, final int childId) throws PuckException {
		if ((net != null) && (fatherId != 0) && (childId != 0)) {
			//
			Individual parent = net.individuals().getById(fatherId);
			if (parent == null) {
				parent = new Individual(fatherId);
				net.individuals().add(parent);
			}

			//
			Individual child = net.individuals().getById(childId);
			if (child == null) {
				child = new Individual(childId);
				net.individuals().add(child);
			}

			//
			Family childSourceFamily;
			if (child.getOriginFamily() == null) {
				childSourceFamily = new Family(net.families().size() + 1);
				net.families().add(childSourceFamily);
				child.setOriginFamily(childSourceFamily);
				childSourceFamily.getChildren().add(child);
			} else {
				childSourceFamily = child.getOriginFamily();
				if (childSourceFamily.getFather() != null) {
					childSourceFamily.getFather().getPersonalFamilies().removeById(childSourceFamily.getId());
				}
			}
			childSourceFamily.setFather(parent);

			//
			Family parentPersonalFamily = parent.getPersonalFamilies().getById(childSourceFamily.getId());
			if (parentPersonalFamily == null) {
				parent.getPersonalFamilies().add(childSourceFamily);
			}
		}
	}

	/**
	 * 
	 * @param graph
	 */
	public static void setGenderShapes(final Graph<Individual> graph) {
		for (Node<Individual> node : graph.getNodes()) {
			node.setTag(node.getReferent().getGender().toShapeString());
		}
	}

	/**
	 * 
	 * This method needs not null parameters.
	 * 
	 * The case "ego equals alter" is managed normally to be detectable in
	 * controls.
	 * 
	 * In case of PARENT kintype, the gender will fix the role (father, mother).
	 * 
	 * @param ego
	 *            spouse or parent
	 * @param alter
	 *            spouse or child
	 * @param type
	 * 
	 * @throws PuckException
	 */
	public static void setKin(final Net net, final Individual ego, final Individual alter, final KinType type) throws PuckException {

		if ((ego == null) || (alter == null) || (type == null)) {
			throw PuckExceptions.INVALID_PARAMETER.create("Null parameter.");
		} else {
			switch (type) {
				case PARENT:
					setKinParent(net, ego, alter);
				break;

				case CHILD:
					setKinParent(net, alter, ego);
				break;

				case SPOUSE:
					setKinSpouse(net, ego, alter);
				break;
			}
		}
	}

	/**
	 * 
	 * @param net
	 * @param family
	 * @param newFather
	 */
	public static void setKinFather(final Family family, final Individual newFather) {
		if (family != null) {
			//
			if (family.getFather() != null) {
				removeFather(family);
			}

			//
			if (newFather != null) {
				if ((family.getMother() == null) || (newFather.getId() != family.getMother().getId())) {
					family.setFather(newFather);
					newFather.getPersonalFamilies().add(family);
				}
			}
		}
	}

	/**
	 * 
	 * @param net
	 * @param father
	 * @param child
	 */
	public static void setKinFather(final Net net, final Individual father, final Individual child) {

		setKinParent(net, father, KinType.FATHER, child);
	}

	/**
	 * 
	 * @param net
	 * @param family
	 * @param newFather
	 */
	public static void setKinMother(final Family family, final Individual newMother) {
		if (family != null) {
			//
			if (family.getMother() != null) {
				removeMother(family);
			}

			//
			if (newMother != null) {
				if ((family.getFather() == null) || (newMother.getId() != family.getFather().getId())) {
					family.setMother(newMother);
					newMother.getPersonalFamilies().add(family);
				}
			}
		}
	}

	/**
	 * 
	 * @param net
	 * @param mother
	 * @param child
	 */
	public static void setKinMother(final Net net, final Individual mother, final Individual child) {

		setKinParent(net, mother, KinType.MOTHER, child);
	}

	/**
	 * This method sets a parent kinship determining parent role by gender.
	 * 
	 * In case of another parent gender value than FATHER or MOTHER, this method
	 * do nothing.
	 * 
	 * 
	 * @param net
	 * @param ego
	 * @param alter
	 */
	public static void setKinParent(final Net net, final Individual parent, final Individual child) {

		//
		switch (parent.getGender()) {
			case FEMALE:
				setKinParent(net, parent, KinType.MOTHER, child);
			break;

			case MALE:
				setKinParent(net, parent, KinType.FATHER, child);
			break;
		}
	}

	/**
	 * In case of another parentRole value than FATHER or MOTHER, this method do
	 * nothing.
	 * 
	 * Action is done with the point of child view.
	 * 
	 * @param net
	 * @param newParent
	 * @param parentRole
	 * @param child
	 */
	public static void setKinParent(final Net net, final Individual newParent, final KinType parentRole, final Individual child) {
		// Notation:
		// F = Father
		// M = Mother
		// f = family
		// C = Child
		// C1 = Child number 1
		// (A) = kinship objects
		// + A = new kin parameter
		// ' = new object
		if ((child != null) && (parentRole != null) && ((parentRole == KinType.FATHER) || (parentRole == KinType.MOTHER))) {

			if (parentRole == KinType.FATHER) {
				// Remove father.
				if ((child.getFather() != null) && (child.getFather() != newParent)) {
					//
					Family previousFamily = child.getOriginFamily();
					previousFamily.getChildren().removeById(child.getId());
					child.setOriginFamily(null);

					//
					if (previousFamily.getMother() != null) {
						net.createFamily((Individual) null, previousFamily.getMother(), child);
					}

					// Clean
					if ((previousFamily.isSterile()) && (previousFamily.isSingleParent()) && (previousFamily.attributes().size() == 0)) {
						net.remove(previousFamily);
					}
				}

				// Add father.
				if (newParent != null) {
					if (child.getOriginFamily() == null) {
						net.createFamily(newParent, null, child);
					} else if ((child.getFather() == null) && (child.getMother() == null)) {
						child.getOriginFamily().setFather(newParent);
					} else if ((child.getFather() == null) && (child.getMother() != null)) {
						Family newOriginFamily = net.families().getBySpouses(newParent, child.getOriginFamily().getMother());
						if (newOriginFamily == null) {
							child.getOriginFamily().setFather(newParent);
							child.getFather().getPersonalFamilies().add(child.getOriginFamily());
						} else {
							//
							Family previousFamily = child.getOriginFamily();
							previousFamily.getChildren().removeById(child.getId());
							if (previousFamily.isEmpty()) {
								net.remove(previousFamily);
							}

							//
							child.setOriginFamily(newOriginFamily);
							newOriginFamily.getChildren().add(child);

							// Clean
							if ((previousFamily.isSterile()) && (previousFamily.isSingleParent()) && (previousFamily.attributes().size() == 0)) {
								net.remove(previousFamily);
							}
						}
					}
				}
			} else if (parentRole == KinType.MOTHER) {
				// Remove Mother.
				if ((child.getMother() != null) && (child.getMother() != newParent)) {
					//
					Family previousFamily = child.getOriginFamily();
					previousFamily.getChildren().removeById(child.getId());
					child.setOriginFamily(null);

					//
					if (previousFamily.getFather() != null) {
						net.createFamily(previousFamily.getFather(), (Individual) null, child);
					}

					// Clean
					if ((previousFamily.isSterile()) && (previousFamily.isSingleParent()) && (previousFamily.attributes().size() == 0)) {
						net.remove(previousFamily);
					}
				}

				// Add mother.
				if (newParent != null) {
					if (child.getOriginFamily() == null) {
						net.createFamily((Individual) null, newParent, child);
					} else if ((child.getFather() == null) && (child.getMother() == null)) {
						child.getOriginFamily().setMother(newParent);
					} else if ((child.getFather() != null) && (child.getMother() == null)) {
						Family newOriginFamily = net.families().getBySpouses(child.getOriginFamily().getFather(), newParent);
						if (newOriginFamily == null) {
							child.getOriginFamily().setMother(newParent);
							child.getMother().getPersonalFamilies().add(child.getOriginFamily());
						} else {
							//
							Family previousFamily = child.getOriginFamily();
							previousFamily.getChildren().removeById(child.getId());
							if (previousFamily.isEmpty()) {
								net.remove(previousFamily);
							}

							//
							child.setOriginFamily(newOriginFamily);
							newOriginFamily.getChildren().add(child);

							// Clean.
							if ((previousFamily.isSterile()) && (previousFamily.isSingleParent()) && (previousFamily.attributes().size() == 0)) {
								net.remove(previousFamily);
							}
						}
					}
				}
			}
		}
	}

	/**
	 * In case of another parentRole value than FATHER or MOTHER, this method do
	 * nothing.
	 * 
	 * Bad way because there is no matter about attributes.
	 * 
	 * @param net
	 * @param newParent
	 * @param parentRole
	 * @param child
	 */
	public static void setKinParent2(final Net net, final Individual newParent, final KinType parentRole, final Individual child) {
		// Notation:
		// F = Father
		// M = Mother
		// f = family
		// C = Child
		// C1 = Child number 1
		// (A) = kinship objects
		// + A = new kin parameter
		// ' = new object

		if ((parentRole != null) && ((parentRole == KinType.FATHER) || (parentRole == KinType.MOTHER))) {
			//
			if (child.isOrphan()) {
				// Rule: (null-C) + F-C -> F-f-C
				// Rule: (null-C) + M-C -> M-f-C

				if (child.getOriginFamily() == null) {
					if (parentRole == KinType.MOTHER) {
						child.getOriginFamily().setMother(newParent);
					} else {
						child.getOriginFamily().setFather(newParent);
					}
				} else {
					if (parentRole == KinType.MOTHER) {
						net.createFamily((Individual) null, newParent, child);
					} else {
						net.createFamily(newParent, null, child);
					}
				}

				//
				newParent.getPersonalFamilies().add(child.getOriginFamily());

			} else if ((child.isOrphanOfMother()) && (parentRole == KinType.FATHER)) {
				// Check if the newParent is already parent.
				if (child.getOriginFamily().getFather() != newParent) {

					//
					if (child.getOriginFamily().getChildren().size() == 1) {
						// Rule: (F-f-C) + F'-C -> (F,F'-f-C)

						//
						Individual previousFather = child.getOriginFamily().getFather();
						previousFather.getPersonalFamilies().removeById(child.getOriginFamily().getId());

						//
						child.getOriginFamily().setFather(newParent);
					} else {
						// Rule: (F-f-C1C2) + F'-C1 -> (F-f-C2,F'-f'-C1)

						//
						child.getOriginFamily().getChildren().removeById(child.getId());

						//
						net.createFamily(newParent, null, child);
					}
				}

			} else if ((child.isOrphanOfFather()) && (parentRole == KinType.MOTHER)) {
				// Check if the newParent is already parent.
				if (child.getOriginFamily().getMother() != newParent) {

					//
					if (child.getOriginFamily().getChildren().size() == 1) {
						// Rule: (M-f-C) + M'-C -> (M,M'-f-C)

						//
						child.getOriginFamily().getMother().getPersonalFamilies().removeById(child.getOriginFamily().getId());

						//
						child.getOriginFamily().setMother(newParent);
					} else {
						// Rule: (M-f-C1C2) + M'-C1 -> (M-f-C2,M'-f'-C1)

						//
						child.getOriginFamily().getChildren().removeById(child.getId());

						//
						net.createFamily(child.getFather(), newParent, child);
					}
				}

			} else if ((child.isOrphanOfMother()) && (parentRole == KinType.MOTHER)) {

				//
				Family newChildOriginFamily = net.families().getBySpouses(child.getOriginFamily().getFather(), newParent);

				if (newChildOriginFamily == null) {
					if (child.getOriginFamily().getChildren().size() == 1) {
						// Rule: (F-f-C) + M-C -> (FM-f-C)
						child.getOriginFamily().setMother(newParent);
						newParent.getPersonalFamilies().add(child.getOriginFamily());
					} else {
						// Rule: (F-f-C1C2) + M-C1 -> (F-f-C2, FM-f'-C)

						//
						child.getOriginFamily().getChildren().removeById(child.getId());

						//
						net.createFamily(child.getFather(), newParent, child);
					}
				} else {
					if (child.getOriginFamily().getChildren().size() == 1) {
						// Rule: (F-f1-C, FM-f2) + M-C -> (f1 removed, FM-f2-C)
						child.getFather().getPersonalFamilies().removeById(child.getOriginFamily().getId());
						child.getOriginFamily().getChildren().removeById(child.getId());
						net.families().removeById(child.getOriginFamily().getId());

						//
						child.setOriginFamily(newChildOriginFamily);
					} else {
						// Rule: (F-f1-C1C2, FM-f2) + M-C1 -> (F-f1-C2,
						// FM-f2-C1)
						child.getOriginFamily().getChildren().removeById(child.getId());
						child.setOriginFamily(newChildOriginFamily);
						child.getOriginFamily().getChildren().add(child);
					}
				}

			} else if ((child.isOrphanOfFather()) && (parentRole == KinType.FATHER)) {
				//
				Family newChildOriginFamily = net.families().getBySpouses(newParent, child.getOriginFamily().getMother());

				if (newChildOriginFamily == null) {
					if (child.getOriginFamily().getChildren().size() == 1) {
						// Rule: (M-f-C) + F-C -> (FM-f-C)
						child.getOriginFamily().setFather(newParent);
						newParent.getPersonalFamilies().add(child.getOriginFamily());
					} else {
						// Rule: (M-f-C1C2) + F-C1 -> (F-f-C2, FM-f'-C)

						//
						child.getOriginFamily().getChildren().removeById(child.getId());

						//
						net.createFamily(child.getFather(), newParent, child);
					}
				} else {
					if (child.getOriginFamily().getChildren().size() == 1) {
						// Rule: (M-f1-C, FM-f2) + F-C -> (f1 removed, FM-f2-C)
						child.getMother().getPersonalFamilies().removeById(child.getOriginFamily().getId());
						child.getOriginFamily().getChildren().removeById(child.getId());
						net.families().removeById(child.getOriginFamily().getId());

						//
						child.setOriginFamily(newChildOriginFamily);
					} else {
						// Rule: (M-f1-C1C2, FM-f2) + F-C1 -> (F-f1-C2,
						// FM-f2-C1)
						child.getOriginFamily().getChildren().removeById(child.getId());
						child.setOriginFamily(newChildOriginFamily);
						child.getOriginFamily().getChildren().add(child);
					}
				}

			} else {
				// Note: child is not orphan.

				if ((parentRole == KinType.FATHER) && (child.getFather() != newParent)) {
					//
					Family newChildOriginFamily = net.families().getBySpouses(newParent, child.getOriginFamily().getMother());

					if (newChildOriginFamily == null) {
						// Rule: (FM-f-C) + F'-C -> (FM-f, F'M-f'-C)

						//
						child.getOriginFamily().getChildren().removeById(child.getId());

						//
						net.createFamily(newParent, child.getMother(), child);
					} else {
						// Rule: (FM-f1-C, F'M-f2) + F'-C -> (FM-f1, F'M-f2-C)

						//
						child.getOriginFamily().getChildren().removeById(child.getId());
						child.setOriginFamily(newChildOriginFamily);
						child.getOriginFamily().getChildren().put(child);
					}
				} else if ((parentRole == KinType.MOTHER) && (child.getMother() != newParent)) {
					//
					Family newChildOriginFamily = net.families().getBySpouses(child.getOriginFamily().getFather(), newParent);

					if (newChildOriginFamily == null) {
						// Rule: (FM-f-C) + M'-C -> (FM-f, FM'-f'-C)

						//
						child.getOriginFamily().getChildren().removeById(child.getId());

						//
						net.createFamily(child.getFather(), newParent, child);
					} else {
						// Rule: (FM-f1-C, FM'-f2) + M'-C -> (FM-f1, FM'-f2-C)

						//
						child.getOriginFamily().getChildren().removeById(child.getId());
						child.setOriginFamily(newChildOriginFamily);
						child.getOriginFamily().getChildren().put(child);
					}
				}
			}
		}
	}

	/**
	 * WARNING: the use of this method can created duplicated families. WARNING:
	 * use only if source does not contain family ids.
	 * 
	 * @param net
	 * @param egoId
	 * @param alterId
	 * @param kinType
	 * @throws PuckException
	 */
	public static void setKinRelation(final Net net, final Individual ego, final Individual alter, final KinType kinType) throws PuckException {
		int egoId = ego.getId();
		int alterId = alter.getId();

		switch (kinType) {
			case PARENT:
				switch (alter.getGender()) {
					case MALE:
						setFatherRelation(net, alterId, egoId);
					break;
					case FEMALE:
						setMotherRelation(net, alterId, egoId);
					break;
				}
			break;
			case CHILD:
				switch (ego.getGender()) {
					case MALE:
						setFatherRelation(net, egoId, alterId);
					break;
					case FEMALE:
						setMotherRelation(net, egoId, alterId);
					break;
				}
			break;
			case SPOUSE:
				switch (ego.getGender()) {
					case MALE:
						setSpouseRelation(net, egoId, alterId);
					break;
					case FEMALE:
						setSpouseRelation(net, alterId, egoId);
					break;
				}
			break;
		}
	}

	/**
	 * This method sets a spouse kinship determining role by gender.
	 * 
	 * @param net
	 * @param ego
	 * @param alter
	 */
	public static void setKinSpouse(final Net net, final Individual ego, final Individual alter) {
		//
		Individual husband;
		Individual wife;
		if (ego.isMale() || (alter.isFemale() && ego.isUnknown())) {
			husband = ego;
			wife = alter;
		} else {
			wife = ego;
			husband = alter;
		}

		//
		setKinSpouseByRole(net, husband, wife);
	}

	/**
	 * This method sets a spouse kinship.
	 * 
	 * @param net
	 * @param ego
	 * @param alter
	 */
	public static void setKinSpouseByRole(final Net net, final Individual husband, final Individual wife) {
		//
		Family family = net.families().getBySpouses(husband, wife);

		//
		if (family == null) {
			family = new Family(net.families().getFirstFreeId());
			net.families().add(family);

			//
			husband.getPersonalFamilies().add(family);
			wife.getPersonalFamilies().add(family);
		}

		//
		family.setHusband(husband);
		family.setWife(wife);

		//
		family.setMarried(true);
	}

	/**
	 * WARNING: the use of this method can created duplicated families. WARNING:
	 * use only if source does not contain family ids.
	 * 
	 * @param parentId
	 * @param childId
	 * @throws PuckException
	 */
	public static void setMotherRelation(final Net net, final int motherId, final int childId) throws PuckException {
		if ((net != null) && (motherId != 0) && (childId != 0)) {

			//
			Individual parent = net.individuals().getById(motherId);
			if (parent == null) {
				parent = new Individual(motherId);
				net.individuals().add(parent);
			}

			//
			Individual child = net.individuals().getById(childId);
			if (child == null) {
				child = new Individual(childId);
				net.individuals().add(child);
			}

			//
			Family childSourceFamily;
			if (child.getOriginFamily() == null) {
				childSourceFamily = new Family(net.families().size() + 1);
				net.families().add(childSourceFamily);
				child.setOriginFamily(childSourceFamily);
				childSourceFamily.getChildren().add(child);
			} else {
				childSourceFamily = child.getOriginFamily();
				if (childSourceFamily.getMother() != null) {
					childSourceFamily.getMother().getPersonalFamilies().removeById(childSourceFamily.getId());
				}
			}
			childSourceFamily.setMother(parent);

			//
			Family parentPersonalFamily = parent.getPersonalFamilies().getById(childSourceFamily.getId());
			if (parentPersonalFamily == null) {
				parent.getPersonalFamilies().add(childSourceFamily);
			}
		}
	}

	/**
	 * WARNING: the parent role is determined by gender.
	 * 
	 * WARNING: the use of this method can created duplicated families.
	 * 
	 * WARNING: use only if source does not contain family ids.
	 * 
	 * @param parentId
	 * @param childId
	 * @throws PuckException
	 */
	public static void setParentRelation(final Net net, final int parentId, final int childId) throws PuckException {
		if ((net != null) && (parentId != 0) && (childId != 0)) {
			//
			Individual parent = net.individuals().getById(parentId);
			if (parent == null) {
				parent = new Individual(parentId);
				net.individuals().add(parent);
			}

			//
			Individual child = net.individuals().getById(childId);
			if (child == null) {
				child = new Individual(childId);
				net.individuals().add(child);
			}

			//
			Family childSourceFamily;
			if (child.getOriginFamily() == null) {
				childSourceFamily = new Family(net.families().size() + 1);
				net.families().add(childSourceFamily);
				child.setOriginFamily(childSourceFamily);
				childSourceFamily.getChildren().add(child);
			} else {
				childSourceFamily = child.getOriginFamily();
			}

			//
			switch (parent.getGender()) {
				case FEMALE:
					childSourceFamily.setMother(parent);
				break;

				case MALE:
				case UNKNOWN:
					childSourceFamily.setFather(parent);
			}

			//
			Family parentPersonalFamily = parent.getPersonalFamilies().getById(childSourceFamily.getId());
			if (parentPersonalFamily == null) {
				parent.getPersonalFamilies().add(childSourceFamily);
			}
		}
	}

	/**
	 * WARNING: the use of this method can created duplicated families.
	 * 
	 * @param parentId
	 * @param childId
	 * @throws PuckException
	 */
	public static Family setSpouseRelation(final Net net, final int husbandId, final int wifeId) throws PuckException {
		Family result;

		if ((net == null) || (husbandId == 0) || (wifeId == 0)) {
			result = null;
		} else {
			//
			boolean isNewFamily = false;

			//
			Individual husband = net.individuals().getById(husbandId);
			if (husband == null) {
				husband = new Individual(husbandId);
				net.individuals().add(husband);
				isNewFamily = true;
			}

			//
			Individual wife = net.individuals().getById(wifeId);
			if (wife == null) {
				wife = new Individual(wifeId);
				net.individuals().add(wife);
				isNewFamily = true;
			}

			//
			Family family;
			if (isNewFamily) {
				family = null;
			} else {
				family = net.families().getBySpouses(husband, wife);
			}

			//
			if (family == null) {
				family = new Family(net.families().size() + 1);
				net.families().add(family);
				family.setHusband(husband);
				family.setWife(wife);
				family.setMarried(true);
				husband.getPersonalFamilies().add(family);
				wife.getPersonalFamilies().add(family);
			}

			//
			result = family;
		}

		//
		return result;
	}

	/**
	 * WARNING: the use of this method can created duplicated families.
	 * 
	 * @param parentId
	 * @param childId
	 * @throws PuckException
	 */
	public static void setSpouseRelationAndFixRoles(final Net net, final int husbandId, final int wifeId) throws PuckException {
		//
		Family family = setSpouseRelation(net, husbandId, wifeId);
		if (family != null) {
			fixSpouseRolesByGender(family);
		}
	}

	/**
	 * 
	 * @param source
	 */
	public static void swapParents(final Family source) {
		if (source != null) {
			Individual pivot = source.getFather();
			source.setFather(source.getMother());
			source.setMother(pivot);
		}
	}

	/**
	 * 
	 * @param seeds
	 * @param label
	 * @param filiationType
	 * @param maxGenerations
	 */
	public static int transmitAttributeValue(final Individuals seeds, final String label, final FiliationType filiationType, final Integer maxGenerations) {
		int result;

		Individuals target = new Individuals();

		result = 0;

		Stack<Individual> stack = new Stack<Individual>();
		for (Individual seed : seeds) {
			Value seedValue = IndividualValuator.get(seed, label);
			if (seedValue != null) {
				String value = seedValue.stringValue();
				stack.push(seed);
				target.add(seed);
				while (!stack.isEmpty()) {
					Individual ego = stack.pop();
					for (Individual neighbor : ego.getKin(KinType.CHILD, filiationType)) {
						if (!target.contains(neighbor)) {
							stack.push(neighbor);
							target.add(neighbor);
							neighbor.setAttribute(label, value);
							result += 1;
						}
					}
					// stack.pop();
				}
			}
		}

		//
		return result;
	}

	/**
	 * 
	 * @param seeds
	 * @param criteria
	 * @return
	 */
	public static int transmitAttributeValue(final Individuals seeds, final TransmitAttributeValueCriteria criteria) {
		int result;

		if (TransmitAttributeValueCriteria.isNotValid(criteria)) {
			//
			throw new IllegalArgumentException("Invalid criteria.");

		} else {
			//
			result = transmitAttributeValue(seeds, criteria.getAttributeLabel(), criteria.getFiliationType(), criteria.getMaxGenerations());
		}

		//
		return result;
	}
}
