package org.tip.puck.net.relations;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Queue;

import org.apache.commons.lang3.StringUtils;
import org.tip.puck.graphs.Graph;
import org.tip.puck.graphs.Node;
import org.tip.puck.net.Gender;
import org.tip.puck.net.Individual;
import org.tip.puck.net.Net;
import org.tip.puck.net.relations.RoleDefinition.AlterAge;
import org.tip.puck.net.relations.RoleDefinition.Primary;

import fr.devinsy.util.StringList;

public class RelationModelMaker {

	public static RelationModel create (String name, StringList roleDefinitionsList){
		RelationModel result;
		
		result = new RelationModel(name);
		createRoleDefinitions (result,roleDefinitionsList);
		
		//
		return result;
	}
	
	public static RelationModel create (Net net){
		RelationModel result;
		
		result = new RelationModel(net.getLabel());
		createRoleDefinitions (result,net);
		
		//
		return result;
	}

	
	public static RoleDefinitions createRoleDefinitions (RelationModel model, StringList roleDefinitionsList){
		RoleDefinitions result;
		
		result = new RoleDefinitions();
		model.setRoleDefinitions(result);

		int id = 0;
		
		for (String roleDefinitionLine : roleDefinitionsList){
			
			String[] items = roleDefinitionLine.split("\t");
			
			id++;
			Role role = null;
			Primary primary = null;
			Role inversion = null;
			Roles composition = new Roles();
			AlterAge alterAge = null;
			Gender alterGender = null;
			Gender egoGender = null;
			
			for (int idx = 0;idx < items.length; idx++){
				
				String item = items[idx];
				if (StringUtils.isEmpty(item)){
					continue;
				}
				
				switch (idx){
				case 0: 
					role = model.role(item);
					break;
				case 1:
					primary = Primary.valueOf(item);
					break;
				case 2:
					inversion = model.role(item);
					break;
				case 3:
					composition.add(model.role(item));
					break;
				case 4:
					composition.add(model.role(item));
					break;
				case 5:
					alterGender = Gender.valueOf(item);
					break;
				case 6:
					alterAge = AlterAge.valueOf(item);
					break;
				case 7:
					egoGender = Gender.valueOf(item);
					break;
				}
			}
						
			result.add(new RoleDefinition(id, role, primary, inversion, composition, alterGender, alterAge, egoGender));
			
		}
		
		//
		return result;
	}
	
	public static RoleDefinitions createRoleDefinitions (RelationModel model, Net net){
		RoleDefinitions result;
		
		result = new RoleDefinitions();
		model.setRoleDefinitions(result);
		
		Individual maleEgo = null;
		Individual femaleEgo = null;
		
		// Determine egos
		for (Individual indi : net.individuals()){
			if (indi.getName().equals("Ego")){
				indi.setAttribute("EGOGENDER", indi.getGender().toString());
				indi.setAttribute("GENERATION","0");
				if (indi.isMale()){
					maleEgo = indi;
				} else if (indi.isFemale()){
					femaleEgo = indi;
				}
				if (maleEgo!=null && femaleEgo !=null){
					result.setEgoGenderDistinction(true);
					break;
				}
			}
		}
		
		List<String> egoNeutral = new ArrayList<String>();
		
		if (maleEgo!=null && maleEgo.isSterile() || femaleEgo!=null && femaleEgo.isSterile()){
			egoNeutral.add("CHILD");
		}
		if (maleEgo!=null && maleEgo.isOrphan() || femaleEgo!=null && femaleEgo.isOrphan()){
			egoNeutral.add("PARENT");
		}
		if (maleEgo!=null && maleEgo.isSingle() || femaleEgo!=null && femaleEgo.isSingle()){
			egoNeutral.add("SPOUSE");
		}
		if (maleEgo!=null && maleEgo.isUnique() || femaleEgo!=null && femaleEgo.isUnique()){
			egoNeutral.add("SIBLING");
		}
		
		Queue<Individual> queue = new LinkedList<Individual>();
		List<Individual> visited = new ArrayList<Individual>();
		
		//Temporary: preliminary separate parent composition for male and female ego to assure disponibility of child roles for all genders
		
		if (maleEgo!=null) {
			queue.add(maleEgo);
			visited.add(maleEgo);
			composeParents(model,maleEgo,Gender.MALE,queue,visited,egoNeutral);
		}
		if (femaleEgo!=null) {
			queue.add(femaleEgo);
			visited.add(femaleEgo);
			composeParents(model,femaleEgo,Gender.FEMALE,queue,visited,egoNeutral);
		}

		if (maleEgo!=null) {
			compose(model,maleEgo,Gender.MALE,queue,visited,egoNeutral);
		}

		if (femaleEgo!=null) {
			compose(model,femaleEgo,Gender.FEMALE,queue,visited,egoNeutral);
		}
		
		System.out.println("Ego environments complete: "+queue);
		
		while (!queue.isEmpty()){
			Individual indi = queue.remove();
			Gender egoGender = Gender.valueOf(indi.getAttributeValue("EGOGENDER"));
			compose(model,indi,egoGender,queue,visited,egoNeutral);
		}
				
		//
		return result;
	}
	
	private static AlterAge getAlterAge(Individual indi){
		AlterAge result;
		
		result = null;
		
		if (indi.getAttributeValue("GENERATION").equals("0") && indi.getBirthOrder()!=null){
			if (indi.getBirthOrder()==3){
				result = AlterAge.YOUNGER;
			} else if (indi.getBirthOrder()==1){
				result = AlterAge.ELDER;
			}
		}
		//
		return result;
	}
	
	private static void composeParents (RelationModel model, Individual indi, Gender egoGender, Queue<Individual> queue, List<Individual> visited, List<String> egoNeutral){
		
		RoleDefinitions roleDefinitions = model.roleDefinitions();
		
		if (indi.getFather()!=null && indi.getFather().getName()!=null && !visited.contains(indi.getFather())){
			indi.getFather().setAttribute("EGOGENDER", egoGender.toString());
			indi.getFather().setAttribute("GENERATION",(Integer.parseInt(indi.getAttributeValue("GENERATION"))+1)+"");
			Gender childGender = indi.getGender();
			if (egoNeutral.contains("PARENT")){
				childGender = null;
			}
			if (indi.getName().equals("Ego")){
				roleDefinitions.addNew(new RoleDefinition(roleDefinitions.size(), model.role(indi.getFather()), Primary.PARENT, null, null, Gender.MALE, null, childGender));
			} else {
				for (Role fatherRole : roleDefinitions.getRoles(Primary.PARENT,Gender.MALE,null,childGender)){
					Roles composition = new Roles();
					composition.add(roleDefinitions.getRoleByName(indi.getName()));
					composition.add(fatherRole);
					roleDefinitions.addNew(new RoleDefinition(roleDefinitions.size(), model.role(indi.getFather()), null, null, composition, null, null, egoGender));
				}
			}
			queue.add(indi.getFather());
			visited.add(indi.getFather());
		}
		
		if (indi.getMother()!=null && indi.getMother().getName()!=null && !visited.contains(indi.getMother())){
			indi.getMother().setAttribute("EGOGENDER", egoGender.toString());
			indi.getMother().setAttribute("GENERATION",(Integer.parseInt(indi.getAttributeValue("GENERATION"))+1)+"");
			Gender childGender = indi.getGender();
			if (egoNeutral.contains("PARENT")){
				childGender = null;
			}
			if (indi.getName().equals("Ego")){
				roleDefinitions.addNew(new RoleDefinition(roleDefinitions.size(), model.role(indi.getMother()), Primary.PARENT, null, null, Gender.FEMALE, null, childGender));
			} else {
				for (Role motherRole : roleDefinitions.getRoles(Primary.PARENT,Gender.FEMALE,null,childGender)){
					Roles composition = new Roles();
					composition.add(roleDefinitions.getRoleByName(indi.getName()));
					composition.add(motherRole);
					roleDefinitions.addNew(new RoleDefinition(roleDefinitions.size(), model.role(indi.getMother()), null, null, composition, null, null, egoGender));
				}
			}
			queue.add(indi.getMother());
			visited.add(indi.getMother());
		}

	}

	
	private static void compose (RelationModel model, Individual indi, Gender egoGender, Queue<Individual> queue, List<Individual> visited, List<String> egoNeutral){
		
		RoleDefinitions roleDefinitions = model.roleDefinitions();
		
		if (roleDefinitions.getRoleByName(indi.getName())==null) System.out.println("Ego role missing: "+indi.getName());

		if (indi.getFather()!=null && indi.getFather().getName()!=null && !visited.contains(indi.getFather())){
			indi.getFather().setAttribute("EGOGENDER", egoGender.toString());
			indi.getFather().setAttribute("GENERATION",(Integer.parseInt(indi.getAttributeValue("GENERATION"))+1)+"");
			Gender childGender = indi.getGender();
			if (indi.getName().equals("Ego")){
				if (egoNeutral.contains("PARENT")){
					childGender = null;
				}
				roleDefinitions.addNew(new RoleDefinition(roleDefinitions.size(), model.role(indi.getFather()), Primary.PARENT, null, null, Gender.MALE, null, childGender));
			} else {
				for (Role fatherRole : roleDefinitions.getRoles(Primary.PARENT,Gender.MALE,null,childGender)){
					Roles composition = new Roles();
					composition.add(roleDefinitions.getRoleByName(indi.getName()));
					composition.add(fatherRole);
					roleDefinitions.addNew(new RoleDefinition(roleDefinitions.size(), model.role(indi.getFather()), null, null, composition, Gender.MALE, null, egoGender));
				}
			}
			queue.add(indi.getFather());
			visited.add(indi.getFather());
		}
		
		if (indi.getMother()!=null && indi.getMother().getName()!=null && !visited.contains(indi.getMother())){
			indi.getMother().setAttribute("EGOGENDER", egoGender.toString());
			indi.getMother().setAttribute("GENERATION",(Integer.parseInt(indi.getAttributeValue("GENERATION"))+1)+"");
			Gender childGender = indi.getGender();
			if (indi.getName().equals("Ego")){
				if (egoNeutral.contains("PARENT")){
					childGender = null;
				}
				roleDefinitions.addNew(new RoleDefinition(roleDefinitions.size(), model.role(indi.getMother()), Primary.PARENT, null, null, Gender.FEMALE, null, childGender));
			} else {
				for (Role motherRole : roleDefinitions.getRoles(Primary.PARENT,Gender.FEMALE,null,childGender)){
					Roles composition = new Roles();
					composition.add(roleDefinitions.getRoleByName(indi.getName()));
					composition.add(motherRole);
					roleDefinitions.addNew(new RoleDefinition(roleDefinitions.size(), model.role(indi.getMother()), null, null, composition, Gender.FEMALE, null, egoGender));
				}
			}
			queue.add(indi.getMother());
			visited.add(indi.getMother());
		}
		
		for (Individual sibling : indi.siblings()){

			Gender siblingGender = indi.getGender();
			if (egoNeutral.contains("SIBLING")){
				siblingGender = null;
			}

			if (sibling.getName()!=null && !visited.contains(sibling)){
				sibling.setAttribute("EGOGENDER", egoGender.toString());
				sibling.setAttribute("GENERATION",indi.getAttributeValue("GENERATION"));
				if (indi.getName().equals("Ego")){
					roleDefinitions.addNew(new RoleDefinition(roleDefinitions.size(), model.role(sibling), Primary.SIBLING, null, null, sibling.getGender(), getAlterAge(sibling,indi), siblingGender));
				} else {
					for (Role siblingRole : roleDefinitions.getRoles(Primary.SIBLING,sibling.getGender(),getAlterAge(sibling,indi),siblingGender)){
						Roles composition = new Roles();
						composition.add(roleDefinitions.getRoleByName(indi.getName()));
						composition.add(siblingRole);
						roleDefinitions.addNew(new RoleDefinition(roleDefinitions.size(), model.role(sibling), null, null, composition, sibling.getGender(), null, egoGender));
					}
				}
				queue.add(sibling);
				visited.add(sibling);
			}
		}

		for (Individual spouse : indi.spouses()){

			Gender spouseGender = indi.getGender();

			if (spouse.getName()!=null && !visited.contains(spouse)){
				spouse.setAttribute("EGOGENDER", egoGender.toString());
				spouse.setAttribute("GENERATION",indi.getAttributeValue("GENERATION"));
				if (indi.getName().equals("Ego")){
					if (egoNeutral.contains("SPOUSE")){
						spouseGender = null;
					}
					roleDefinitions.addNew(new RoleDefinition(roleDefinitions.size(), model.role(spouse), Primary.SPOUSE, null, null, spouse.getGender(), null, spouseGender));
				} else {
					for (Role spouseRole : roleDefinitions.getRoles(Primary.SPOUSE,spouse.getGender(),null,spouseGender)){
						Roles composition = new Roles();
						composition.add(roleDefinitions.getRoleByName(indi.getName()));
						composition.add(spouseRole);
						roleDefinitions.addNew(new RoleDefinition(roleDefinitions.size(), model.role(spouse), null, null, composition, spouse.getGender(), null, egoGender));
					}
				}
				queue.add(spouse);
				visited.add(spouse);
			}
		}

		for (Individual child : indi.children()){
			
			Gender parentGender = indi.getGender();

			if (child.getName()!=null && !visited.contains(child)){
				child.setAttribute("EGOGENDER", egoGender.toString());
				child.setAttribute("GENERATION",(Integer.parseInt(indi.getAttributeValue("GENERATION"))-1)+"");

				if (indi.getName().equals("Ego")){
					if (egoNeutral.contains("CHILD")){
						parentGender = null;
					}
					Roles parentRoles = roleDefinitions.getRoles(Primary.PARENT,parentGender,null,child.getGender());
					for (Role role : parentRoles){
						roleDefinitions.addNew(new RoleDefinition(roleDefinitions.size(), model.role(child), null, role, null, child.getGender(), null, parentGender));
					}
				} else {
					Roles parentRoles = roleDefinitions.getRoles(Primary.PARENT,parentGender,null,child.getGender());
					for (Role parentRole : parentRoles){
						Role childRole = roleDefinitions.getInverseRole(parentRole, child.getGender(), parentGender);
						if (childRole!=null){
							Roles composition = new Roles();
							composition.add(roleDefinitions.getRoleByName(indi.getName()));
							composition.add(childRole);
							roleDefinitions.addNew(new RoleDefinition(roleDefinitions.size(), model.role(child), null, null, composition, child.getGender(), getAlterAge(child), egoGender));
						} else {
							System.out.println("Missing "+indi.getGender()+" ("+parentGender+") "+" parentRole for "+child.getGender()+" child :"+indi+" for "+child);
						}
					}
				}
				queue.add(child);
				visited.add(child);
			}
		}
		
	}
	
	private static AlterAge getAlterAge (Individual alter, Individual ego){
		AlterAge result;
		
		result = null;
		
		if (alter==null || ego == null){
			result = null;
		} else if (alter.isYoungerThan(ego)){
			result = AlterAge.YOUNGER;
		} else if (alter.isElderThan(ego)){
			result = AlterAge.ELDER;
		}
		//
		return result;
	}

	
	public static Graph<Role> relationModelGraph (RelationModel model){
		Graph<Role> result;
		
		result = new Graph<Role>();
		
		Map<Role,String> arcTypes = new HashMap<Role,String>();
		
		if (model.roleDefinitions().egoGenderDistinction){
			
		}
		
		Role ego = new Role("Ego");
		
		result.addNode(ego);
			
		for (Role role : model.roles()){
			result.addNode(role);
		}
		
		for (RoleDefinition definition : model.roleDefinitions().toSortedList()){
			if (definition.primary()!=null){
				switch (definition.primary()){
				case PARENT:
					if (definition.alterGender().isMale()){
						arcTypes.put(definition.role(), "FATHER");
						result.addArc(ego, definition.role(),1);
					} else if (definition.alterGender().isFemale()){
						arcTypes.put(definition.role(), "MOTHER");
						result.addArc(ego, definition.role(),-1);
					}
					break;
				case SIBLING:
					arcTypes.put(definition.role(), "SIBLING");
					result.addEdge(ego, definition.role(),-1);
					break;
				case SPOUSE:
					arcTypes.put(definition.role(), "SPOUSE");
					result.addEdge(ego, definition.role(),1);
					break;
				}
			}
		}

		
		for (RoleDefinition definition : model.roleDefinitions().toSortedList()){
			if (definition.inversion()!=null){
				Node<Role> alterNode = result.getNode(definition.role());
				Node<Role> egoNode = result.getNode(definition.inversion());
				String arcType = arcTypes.get(definition.inversion());

				if (arcType.equals("FATHER")){
					result.addArc(alterNode, egoNode, 1);
				} else if (arcType.equals("MOTHER")){
					result.addArc(alterNode, egoNode, -1);
				}
			}
			if (definition.composition()!=null){
				Node<Role> alterNode = result.getNode(definition.role());
				Node<Role> egoNode = result.getNode(definition.composition().get(0));
				String arcType = arcTypes.get(definition.composition().get(1));
				if (arcType!=null){
					if (arcType.equals("FATHER")){
						result.addArc(egoNode, alterNode, 1);
					} else if (arcType.equals("MOTHER")){
						result.addArc(egoNode, alterNode, -1);
					} else if (arcType.equals("SPOUSE")){
						result.addEdge(egoNode, alterNode, 1);
					} else if (arcType.equals("SIBLING")){
						result.addEdge(egoNode, alterNode, -1);
					}
				} else {
																	
					for (Gender egoGender : model.roleDefinitions().getAlterGenders(definition.composition().get(0))){
						for (Gender alterGender : model.roleDefinitions().getAlterGenders(definition.composition().get(1))){
							if (definition.alterGender()!=null && definition.alterGender().equals(alterGender)){
								Role inverseRole = model.roleDefinitions().getInverseRole(definition.composition().get(1),egoGender,alterGender);
								arcType = arcTypes.get(inverseRole);
								if (arcType.equals("FATHER")){
									result.addArc(alterNode, egoNode, 1);
								} else if (arcType.equals("MOTHER")){
									result.addArc(alterNode, egoNode, -1);
								}
							}
						}
					}
				}
					
//				System.out.println(definition+" "+alterNode+" "+egoNode+" "+arcType);

			}
		}

		
		//
		return result;
	}
	
}
