package org.tip.puck.sequences;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

import org.tip.puck.PuckException;
import org.tip.puck.census.chains.Chain;
import org.tip.puck.graphs.Graph;
import org.tip.puck.graphs.Link;
import org.tip.puck.graphs.Node;
import org.tip.puck.graphs.workers.GraphUtils;
import org.tip.puck.graphs.workers.NodeValuator;
import org.tip.puck.net.Gender;
import org.tip.puck.net.Individual;
import org.tip.puck.net.relations.Relation;
import org.tip.puck.partitions.Cluster;
import org.tip.puck.partitions.Partition;
import org.tip.puck.partitions.PartitionMaker;
import org.tip.puck.sequences.SequenceCensusCriteria.RelationAttributeType;
import org.tip.puck.util.MathUtils;
import org.tip.puck.util.NumberedValues;
import org.tip.puck.util.PuckUtils;
import org.tip.puck.util.Value;
import org.tip.puck.util.Values;

import fr.devinsy.util.StringList;

/**
 * 
 * @author Klaus Hamberger
 *
 */
public class SequencesCensus {
	
	private Sequences sequences;
	private Map<String, NumberedValues> valuesMap;
	private List<String> labels;
	private Map<String, StringList> pajekBuffers;
	
	private Map<String,CorrelationMatrix> eventSequenceMatrices;
	private Map<String,CorrelationMatrix> subSequenceMatrices;
		
	private Map<String,Map<Integer,Partition<Node<Individual>>>> componentsMap;
	private CorrelationMatrix relationConnectionMatrix;

	private Map<RelationAttributeType,Partition<Link<Individual>>> linkPartitions;
	private Map<RelationAttributeType,Map<Value,Double[]>> similaritiesMaps;
	
	Map<RelationAttributeType,Map<String,Integer>> nrValues;

	public SequencesCensus (Sequences sequences, SequenceCensusCriteria criteria) throws PuckException{
		
		this.sequences = sequences;
		this.labels = criteria.getLabels();
		valuesMap = new HashMap<String, NumberedValues>();

		for (String label : labels){
			if (label.contains("SIMILARITY")){
				if (similaritiesMaps == null){
					similaritiesMaps = new HashMap<RelationAttributeType,Map<Value,Double[]>>();
					linkPartitions = new HashMap<RelationAttributeType,Partition<Link<Individual>>>();
				}
				RelationAttributeType relationAttributeType = RelationAttributeType.valueOf(label.substring(label.lastIndexOf("_")+1));
				similaritiesMaps.put(relationAttributeType, new HashMap<Value,Double[]>());
				linkPartitions.put(relationAttributeType, new Partition<Link<Individual>>());
			} 
			valuesMap.put(label, new NumberedValues());
		}
		
		pajekBuffers = new HashMap<String, StringList>();
		for (String title : criteria.getNetworkTitles()){
			pajekBuffers.put(title, new StringList());
		}
		
		for (String networkTitle : criteria.getNetworkTitles()){
			if (networkTitle.contains("Ego Network") || networkTitle.contains("Parcours Network")){
				if (componentsMap==null){
					componentsMap = new HashMap<String,Map<Integer,Partition<Node<Individual>>>>();
				}
				componentsMap.put(networkTitle, new HashMap<Integer,Partition<Node<Individual>>>());
			}
		}

		eventSequenceMatrices = new HashMap<String,CorrelationMatrix>();
		subSequenceMatrices = new HashMap<String,CorrelationMatrix>();
		nrValues = new HashMap<RelationAttributeType,Map<String,Integer>>();

		for (RelationAttributeType relationAttributeType : criteria.getMainRelationAttributeTypes()){
			nrValues.put(relationAttributeType, new HashMap<String,Integer>());
		}
		
		for (Sequence sequence : sequences){
			
			SequenceCensus census = new SequenceCensus(sequence, criteria);
			
			for (String label : labels){
				if (label.contains("SIMILARITY")){
					
					RelationAttributeType relationAttributeType = RelationAttributeType.valueOf(label.substring(label.lastIndexOf("_")+1));
					linkPartitions.get(relationAttributeType).add(census.getParcoursLinkPartition(relationAttributeType));
					
				} 
				valuesMap.get(label).put(sequence.getEgo().getId(), census.getValue(label));				
			}
			
			if (componentsMap!=null){
				
				for (String networkTitle : criteria.getNetworkTitles()){
					Map<Integer,Partition<Node<Individual>>> components = componentsMap.get(networkTitle);
					if (components!=null){
						if (networkTitle.contains("Ego Network")){
							components.put(sequence.getEgo().getId(), census.getComponents("Nonmediated Ego Network"));
						} else if (networkTitle.contains("Parcours Network")){
							RelationAttributeType relationAttributeType = RelationAttributeType.valueOf(networkTitle.substring(networkTitle.lastIndexOf("_")+1));
							components.put(sequence.getEgo().getId(), census.getComponents("Parcours Network_"+relationAttributeType));
						}
					}
				}
			}
			

			
/*			for (EventType eventType : criteria.getMainEventTypes()){
				
				String eventTypeName = eventType.toString();

				if (criteria.getNetworkTitles().contains("Event Type Network")){
					
					Partition<String> eventPartition = eventPartitions.get(eventTypeName);
					Partition<String> eventPairPartition = eventPairPartitions.get(eventTypeName);
					
					census.putEvents(eventPartition,EventType.valueOf(eventTypeName));
					census.putEventPairs(eventPairPartition, EventType.valueOf(eventTypeName));

					
				} 
				
				if (criteria.getNetworkTitles().contains("Sequence Type Network")){

					Partition<String> subSequencePartition = subSequencePartitions.get(eventTypeName);
					
					census.putSubSequences(subSequencePartition,EventType.valueOf(eventTypeName));
				} 
			}*/
			
			
			for (String title : criteria.getNetworkTitles()){
				
				if (!title.equals("Event Type Network") && !title.equals("Sequence Type Network")){
					
					census.writePajekNetwork(pajekBuffers.get(title), title);

				}
				
			}
			
		}
		
		// Make Event Type Networks
		
		if (criteria.getNetworkTitles().contains("Event Type Network")){

			for (RelationAttributeType relationAttributeType : criteria.getMainRelationAttributeTypes()){
				
				String eventTypeName = relationAttributeType.toString();
				Map<Individual,List<String>> singles = new TreeMap<Individual,List<String>>();
				Map<Individual,List<String[]>> pairs  = new TreeMap<Individual,List<String[]>>();
				
				for (Sequence sequence : sequences){
					
					Individual ego = sequence.getEgo();
					
					Value singlesValue = valuesMap.get("PROFILE_"+eventTypeName).get(ego.getId());
					
					if (singlesValue != null){


						List<String> singlesList = (List<String>)singlesValue.listValue();
						List<String[]> pairsList = new ArrayList<String[]>();
						for (int i=1;i<singlesList.size();i++){
							pairsList.add(new String[]{singlesList.get(i-1),singlesList.get(i)});
						}
						
						//
						singles.put(ego, singlesList);
						pairs.put(ego, pairsList);
					}
				}
				
				eventSequenceMatrices.put(eventTypeName, new CorrelationMatrix("Event Type Network",eventTypeName,singles,pairs));
			}
		}
		
		if (criteria.getNetworkTitles().contains("Sequence Type Network")){

			for (RelationAttributeType relationAttributeType : criteria.getMainRelationAttributeTypes()){
				
				String eventTypeName = relationAttributeType.toString();
				Map<Individual,List<String>> singles = new TreeMap<Individual,List<String>>();
				Map<Individual,List<String[]>> pairs  = new TreeMap<Individual,List<String[]>>();
				
				for (Sequence sequence : sequences){
					
					Individual ego = sequence.getEgo();
					
					Value singlesValue = valuesMap.get("PROFILE_"+eventTypeName).get(ego.getId());
					
					if (singlesValue != null){

						List<String> singlesList = PuckUtils.cumulateList((List<String>)singlesValue.listValue());
						List<String[]> pairsList = new ArrayList<String[]>();
						for (int i=1;i<singlesList.size();i++){
							pairsList.add(new String[]{singlesList.get(i-1),singlesList.get(i)});
						}
						//
						singles.put(ego, singlesList);
						pairs.put(ego, pairsList);
					}
				}
				
				subSequenceMatrices.put(eventTypeName, new CorrelationMatrix("Sequence Type Network",eventTypeName,singles,pairs));
			}
		}
		
		
		if (labels.contains("CONNECTED_NETWORK_RELATIONS")){

			Map<Individual,List<String>> singles = new TreeMap<Individual,List<String>>();
			Map<Individual,List<String[]>> pairs  = new TreeMap<Individual,List<String[]>>();
			
			for (Sequence sequence : sequences){
				
				Individual ego = sequence.getEgo();
				Value singlesValue = valuesMap.get("NETWORK_RELATIONS").get(ego.getId());
				
				if (singlesValue != null){
					singles.put(ego, (List<String>)singlesValue.listValue());
				}
								
				Value pairsValue = valuesMap.get("CONNECTED_NETWORK_RELATIONS").get(ego.getId());
				
				if (pairsValue != null){
					pairs.put(ego, (List<String[]>)pairsValue.listValue());
				}
			}
			
			relationConnectionMatrix = new CorrelationMatrix("Component connections",null,singles,pairs);
			
		}

		for (String networkTitle : criteria.getNetworkTitles()){
			if (networkTitle.contains("Parcours Network")){
				
				for (RelationAttributeType relationAttributeType : linkPartitions.keySet()){
					Partition<Link<Individual>> linkPartition = linkPartitions.get(relationAttributeType);
					Map<Value,Double[]> similaritiesMap = similaritiesMaps.get(relationAttributeType);
					
					for (Value linkValue : linkPartition.getValues()){
						
						Double[] values = new Double[]{0.,0.,0.,0.,0.};
						Double[] sums = new Double[]{0.,0.,0.,0.,0.};

						for (Link<Individual> link : linkPartition.getCluster(linkValue).getItems()){
							Individual ego = (Individual)link.getSourceNode().getReferent();
							Individual alter = (Individual)link.getTargetNode().getReferent();
							
							int egoGender = ego.getGender().toInt();
							int alterGender = alter.getGender().toInt();
							int idx = egoGender + 2 * alterGender;
							
							values[idx] += link.getWeight();
							values[4] += link.getWeight();
							sums[idx]++;
							sums[4]++;
							
						}
						
						for (int idx=0;idx<sums.length;idx++){
							values[idx] = MathUtils.percent(values[idx], 100*sums[idx]);
						}
						
						similaritiesMap.put(linkValue, values);
					}
				}
			}
		}
		
		for (RelationAttributeType relationAttributeType : criteria.getMainRelationAttributeTypes()){

			if (criteria.getNetworkTitles().contains("Event Type Network")){

				CorrelationMatrix matrix = eventSequenceMatrices.get(relationAttributeType.toString());

				Graph<Cluster<String>>[] eventTypeNetworks = matrix.getSequenceNetworks();
				
				List<String> partitionLabels = new ArrayList<String>();
				partitionLabels.add(relationAttributeType.toString());
				Map<String,Map<Value,Integer>> partitionNumbersMaps = getPartitionNumbersMaps(partitionLabels, eventTypeNetworks[2]);
				for (String label : partitionLabels){
					if (partitionNumbersMaps.get(label)!=null){
						nrValues.get(relationAttributeType).put(label, partitionNumbersMaps.get(label).size());
					}
				}
				partitionLabels.add("SIZE");

				for (Gender gender : Gender.values()){
					
					pajekBuffers.get("Event Type Network").addAll(PuckUtils.writePajekNetwork(eventTypeNetworks[gender.toInt()],partitionLabels,partitionNumbersMaps));

				}
			}
				
			if (criteria.getNetworkTitles().contains("Sequence Type Network")){
				
				CorrelationMatrix matrix = subSequenceMatrices.get(relationAttributeType.toString());

				Graph<Cluster<String>>[] sequenceTypeNetworks = matrix.getSequenceNetworks();
				matrix.getDepthPartitions();
						
				List<String> partitionLabels = new ArrayList<String>();
				partitionLabels.add(relationAttributeType.toString());
				Map<String,Map<Value,Integer>> partitionNumbersMaps = getPartitionNumbersMaps(partitionLabels, sequenceTypeNetworks[2]);
				for (String label : partitionLabels){
					nrValues.get(relationAttributeType).put(label, partitionNumbersMaps.get(label).size());
				}
				partitionLabels.add("SIZE");
				partitionLabels.add("STEP");

				for (Gender gender : Gender.values()){
					pajekBuffers.get("Sequence Type Network").addAll(PuckUtils.writePajekNetwork(sequenceTypeNetworks[gender.toInt()],partitionLabels,partitionNumbersMaps));
				}
			}
		}
	}
	
	public static <E> Map<String,Map<Value,Integer>> getPartitionNumbersMaps(List<String> labels, Graph<E> model){
		Map<String,Map<Value,Integer>> result;
		
		result = new HashMap<String,Map<Value,Integer>>();
		
		for (String label : labels){
			
			Values values = NodeValuator.get(model, label);
			if (!values.isNumeric()) {
				Partition<Value> partition = PartitionMaker.create(label, values);
				result.put(label, PartitionMaker.getPartitionNumbersMap(partition));
			}

		}
		
		//
		return result;
		
	}
	

	public Map<String, StringList> getPajekBuffers() {
		return pajekBuffers;
	}



	public Map<Integer, Partition<Node<Individual>>> getComponents(String networkTitle) {
		return componentsMap.get(networkTitle);
	}


	public NumberedValues getValues (String label){
		
		return valuesMap.get(label);
	}
	
	public Graph<Cluster<String>> getSequenceNetwork (String title, RelationAttributeType relationAttributeType, Partition<String> partition){
		Graph<Cluster<String>> result;
		
		if (sequences == null) {
			throw new IllegalArgumentException("Null parameter detected.");
		} else {
			
			result = new Graph<Cluster<String>>(title+"_"+relationAttributeType);
			
			//
			for (Cluster<String> cluster : partition.getClusters().toListSortedByDescendingSize()) {
				if (!cluster.isNull()) {
					result.addNode(cluster);
				}
			}
			
			//
			for (Sequence biography : sequences){
				Cluster<String> previous = null;
				for (Relation event : biography.getEvents().values()){
					Cluster<String> next = partition.getCluster(biography.getEgo().getId()+" "+event.getTypedId());
					if (previous!=null){
						result.incArcWeight(previous, next);
					}
					previous = next;
				}
			}
			
			for (Node<Cluster<String>> node : result.getNodes()){
				Cluster<String> referent = node.getReferent();
				if (referent !=null){
					Value clusterValue = referent.getValue();
					if (clusterValue!=null){
						String value = clusterValue.toString();
						if (value.lastIndexOf("-")>-1){
							value = value.substring(value.lastIndexOf("-")+1);
						}
						node.setAttribute(relationAttributeType.toString(), value);
					}
				}
			}
		}
		
		//
		return result;
	}




/*	public Graph<Cluster<String>> getEventTypeNetwork(String eventTypeName) {
		return eventTypeNetworks.get(eventTypeName);
	}




	public Graph<Cluster<String>> getSequenceTypeNetwork(String eventTypeName) {
		return sequenceTypeNetworks.get(eventTypeName);
	}




	public Partition<String> getEventPartition(String eventTypeName) {
		return eventPartitions.get(eventTypeName);
	}*/
	
	public Integer getNrValues (RelationAttributeType relationAttributeType, String label){
		Integer result;
		
		if (relationAttributeType==null || label==null || nrValues.get(relationAttributeType)==null){
			result = null;
		} else {
			result = nrValues.get(relationAttributeType).get(label);
		}
		
		return result;
	}




/*	public Partition<String> getSubSequencePartition(String eventTypeName) {
		return subSequencePartitions.get(eventTypeName);
	}*/
	
	public Set<Relation> events(){
		Set<Relation> result;
		
		result = new HashSet<Relation>();
		
		for (Sequence sequence : sequences){
			for (Relation event : sequence.getEvents().values()){
				result.add(event);
			}
		}
		//
		return result;
		
	}
	
	public int nrEvents(){
		return events().size();
	}
	
	public int nrSequences(){
		return sequences.size();
	}


/*	public Partition<String> getEventPairPartition(String eventTypeName) {
		return eventPairPartitions.get(eventTypeName);
	}*/


	public Map<Value, Double[]> getSimilaritiesMap(RelationAttributeType relationAttributeType) {
		return similaritiesMaps.get(relationAttributeType);
	}

/*	public Partition<Node<Cluster<String>>>[] getDepthPartition(String eventTypeName) {
		return depthPartitions.get(eventTypeName);
	}*/

	public CorrelationMatrix getRelationConnectionMatrix() {
		return relationConnectionMatrix;
	}
	
	public CorrelationMatrix getEventSequenceMatrix(String eventTypeName) {
		CorrelationMatrix result;
		
		if (eventSequenceMatrices == null || eventTypeName == null){
			result = null;
		} else {
			result = eventSequenceMatrices.get(eventTypeName);
		}
		
		return result;
	}
	
	public CorrelationMatrix getSubSequenceMatrix(String eventTypeName) {
		CorrelationMatrix result;
		
		if (subSequenceMatrices == null || eventTypeName == null){
			result = null;
		} else {
			result = subSequenceMatrices.get(eventTypeName);
		}
		
		return result;
	}
	

	

}
