package org.tip.puck.sequences;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.tip.puck.geo.GeoLevel;
import org.tip.puck.geo.Geography;
import org.tip.puck.net.FiliationType;
import org.tip.puck.net.workers.ExpansionMode;

/**
 * 
 * @author Klaus Hamberger
 *
 */
public class SequenceCensusCriteria {
	
	public enum RelationAttributeType {
		TYPEDID,
		HOST,
		MIG,
		HOSTMIG,
		MIGRATIONTYPE,
		CHILDMIGRATIONTYPE,
		TREES,
		DISTANCE,
		PLACE,
		REGION,
		MOVEMENT,
		TURNOVER,
		COMPONENTS
	}
	
	public enum CensusType {
		GENERAL,
		EGONETWORKS,
		PARCOURSNETWORKS,
		PARCOURS;
		
		public String toString(){
			String result;
			
			switch (this){
			case GENERAL:
				result = "General";
				break;
			case EGONETWORKS:
				result = "Ego Networks";
				break;
			case PARCOURSNETWORKS:
				result = "Parcours Networks";
				break;
			case PARCOURS:
				result = "Parcours";
				break;
			default:
				result = null;
			}
			//
			return result;
			
		}
	}

	private CensusType censusType;
	private RelationAttributeType relationAttributeType;
	private List<RelationAttributeType> relationAttributeTypes;
	private String egoRoleName;
	private double threshold;
	private GeoLevel level;
	private String splitLabel;
	private Map<String,List<String>> partitionLabels;
	private String eventModelName;
	private String chainClassification;
	private String pattern;
	private String defaultReferentRoleName;
	
	// parameters for sequence extension
	private ExpansionMode expansionMode;
	private FiliationType filiationType;
	
	
	public String getChainClassification() {
		return chainClassification;
	}

	public void setChainClassification(String chainClassification) {
		this.chainClassification = chainClassification;
	}

	private List<String> placeNames;
	private List<String> relationModelNames; 
	private String alterRoleName; // for ego networks (temp)
	private List<RelationAttributeType> mainRelationAttributeTypes; // for parcours networks
	private List<String> eventRoleNames; 
	private List<String> labels;
	private int maxAge;
	private int minAge;
	private Geography geography;
	private boolean withEgoNetworks;
	private boolean withParcoursNetworks;
	private List<String> networkTitles;
	
	private String alterAttributeLabel; // for parcours networks
	private String alterAttributeValue; // for parcours networks
	
	
	
	/**
	 * Temporary Method to set default criteria 
	 * @param censusType
	 */
	public SequenceCensusCriteria (CensusType censusType){
		
		this.censusType = censusType;
		relationAttributeType = RelationAttributeType.HOSTMIG;
		defaultReferentRoleName = "HOST";

		// Set event model name [Choose among possible relation model names]
		eventModelName = "Migevent";

		// Set event role names [Choose among roles of relationModel that has been chosen as eventModel; in addition: "ALL"]
		eventRoleNames = new ArrayList<String>();
		eventRoleNames.add("ALL");
		eventRoleNames.add("HOST");
		eventRoleNames.add("MIG");
//		roleNames.add("FIN");
//		roleNames.add("OTHER");
		
		// Set ego role name [Choose among event role names]
		egoRoleName = "MIG";
		
		// Set alter role name [Choose among event role names]
		alterRoleName = "ALL";
		
		// Set alter attribute label+value [choose from individual attribute labels]
		if (censusType == CensusType.PARCOURSNETWORKS){
			alterAttributeLabel = "INTERV";
			alterAttributeValue = "Yes";
		}
		
		// Set relation model names [Choose among remaining relation model names]
		relationModelNames = new ArrayList<String>();
		relationModelNames.add("Apprenticeship");
		relationModelNames.add("Housing");
		relationModelNames.add("Employment");
		relationModelNames.add("Friendship");
		relationModelNames.add("Relatedness");

		// Set age thresholds [Spinner integers with interval 1; Default: min = 0, max = 1000]
		minAge = 0;
		maxAge = 1000;
//		minAge = 16;
//		maxAge = 16;
		
		// Set event types [choose among values of enum EventType]
		relationAttributeTypes = new ArrayList<RelationAttributeType>();
		switch (censusType){
		case GENERAL:
			relationAttributeTypes.add(RelationAttributeType.PLACE);
			relationAttributeTypes.add(RelationAttributeType.MIGRATIONTYPE);
			relationAttributeTypes.add(RelationAttributeType.TREES);
			relationAttributeTypes.add(RelationAttributeType.CHILDMIGRATIONTYPE);
			relationAttributeTypes.add(RelationAttributeType.DISTANCE);
			relationAttributeTypes.add(RelationAttributeType.REGION);
			break;
		case EGONETWORKS:
			relationAttributeTypes.add(RelationAttributeType.PLACE);
			break;
		case PARCOURSNETWORKS:
			relationAttributeTypes.add(RelationAttributeType.PLACE);
			break;
		case PARCOURS:
			relationAttributeTypes.add(RelationAttributeType.PLACE);
			break;
		}

		// Set main event types [choose among values of enum EventType] 
		mainRelationAttributeTypes = new ArrayList<RelationAttributeType>();
		switch (censusType){
		case GENERAL:
			break;
		case EGONETWORKS:
			break;
		case PARCOURSNETWORKS:
			mainRelationAttributeTypes.add(RelationAttributeType.MIGRATIONTYPE);
			mainRelationAttributeTypes.add(RelationAttributeType.TYPEDID);
			break;
		case PARCOURS:
			mainRelationAttributeTypes.add(RelationAttributeType.PLACE);
			mainRelationAttributeTypes.add(RelationAttributeType.DISTANCE);
			mainRelationAttributeTypes.add(RelationAttributeType.TREES);
			mainRelationAttributeTypes.add(RelationAttributeType.MIGRATIONTYPE);
//			mainRelationAttributeTypes.add(EventType.COMPONENTS);
			break;
		}
		// Consolidate main event types with event types
		for (RelationAttributeType mainEventType : mainRelationAttributeTypes){
			if (!relationAttributeTypes.contains(mainEventType)){
				relationAttributeTypes.add(mainEventType);
			}
		}
		
		// Set geography criteria [consolidate with new module!]
		geography = Geography.getInstance();
		level = GeoLevel.TOWN;
		// Set place names [write as string separated by comma?]
		if (censusType == CensusType.GENERAL){
			placeNames = Arrays.asList(new String[]{"Afagnan","Lom","Bas-Mono","Togo","Bnin","Ghana","Nigria","Cte d'Ivoire","Afrique"});
		}
		
		// Set chain census criteria
		pattern = "3 1";
		chainClassification = "LINE";
		threshold = 0.01;
		
		// Set statistics criteria
		splitLabel = "GENDER";
		
		// Set network titles [choose from list]
		networkTitles = new ArrayList<String>();
		partitionLabels = new HashMap<String,List<String>>();
		switch (censusType){
		case GENERAL:
			break;
		case EGONETWORKS:
			addNetworkTitle("Ego Network");
			addNetworkTitle("Nonmediated Ego Network");
			break;
		case PARCOURSNETWORKS:
			addNetworkTitle("Parcours");
			addNetworkTitle("Extended Parcours");
			addNetworkTitle("Multiple Parcours");
			for (RelationAttributeType mainEventType : mainRelationAttributeTypes){
				addNetworkTitle("Parcours Network_"+mainEventType);
				addNetworkTitle("Parcours Similarity Tree_"+mainEventType);
			}
			break;
		case PARCOURS:
			addNetworkTitle("Event Type Network");
			addNetworkTitle("Sequence Type Network");
			break;
		}
		
		// Set partition labels
		for (String networkTitle : networkTitles){
			partitionLabels.put(networkTitle,new ArrayList<String>());
		}
		switch (censusType){
		case GENERAL:
			break;
		case EGONETWORKS:
			partitionLabels.get("Ego Network").add("EGO-RELATION");
			partitionLabels.get("Ego Network").add("DEGREE");
			partitionLabels.get("Nonmediated Ego Network").add("BETWEENNESS");
			partitionLabels.get("Nonmediated Ego Network").add("EGO-RELATION");
			partitionLabels.get("Nonmediated Ego Network").add("DEGREE");
			partitionLabels.get("Nonmediated Ego Network").add("COMPONENT");
			break;
		case PARCOURSNETWORKS:
			partitionLabels.get("Parcours").add("DATE");
			partitionLabels.get("Parcours").add("DISTANCE");
			partitionLabels.get("Multiple Parcours").add("DATE");
			partitionLabels.get("Multiple Parcours").add("DISTANCE");
			partitionLabels.get("Extended Parcours").add("ORDER");
			partitionLabels.get("Extended Parcours").add("TYPE");
			break;
		case PARCOURS:
			break;
		}
		
		// Set ego network and tree network partition labels
		if (censusType == CensusType.PARCOURSNETWORKS){
			for (RelationAttributeType mainEventType : mainRelationAttributeTypes){
				List<String> egoNetworkPartitionLabels = partitionLabels.get("Parcours Network_"+mainEventType);
				egoNetworkPartitionLabels.add("BETWEENNESS");
				egoNetworkPartitionLabels.add("EGO-RELATION");
				egoNetworkPartitionLabels.add("DEGREE");
				egoNetworkPartitionLabels.add("COMPONENT");
				
				List<String> treePartitionLabels = partitionLabels.get("Parcours Similarity Tree_"+mainEventType);
				treePartitionLabels.add("SIZE");
				treePartitionLabels.add("STEP");
				treePartitionLabels.add("EGO-RELATION");
			}
		}
		
		// Set labels of census operations
		labels = new ArrayList<String>();
		switch (censusType){
		case GENERAL:
			
			labels.add("NREVENTS");
			labels.add("NREXTERNALMOVES");
			
			labels.add("AGEFIRST");
			labels.add("AGEFIRST_CHILDMIGRATIONTYPE_NOPARENTS");
			labels.add("AGEFIRST_DISTANCE_TRANSNATIONAL");
			labels.add("MAX_DISTANCE");
			labels.add("SAMESEXALTERS_ALL");
			labels.add("SAMEPLACEALTERS_ALL");
			
			for (String roleName : eventRoleNames){
				labels.add("NRALTERS_"+roleName);
				labels.add("NRALTERSPEREVENT_"+roleName);
			}
			
			for (RelationAttributeType type : relationAttributeTypes){
				labels.add("EVENTS_"+type);
			}
			
			labels.add("MEAN_COVERAGE");
			labels.add("MAX_COVERAGE");
			
			for (String roleName : eventRoleNames){
				labels.add("RELATIONS_"+roleName);
			}
			
			labels.add("MAIN_ALTERS");
			labels.add("MAIN_RELATIONS");
			
			break;
			
		case EGONETWORKS:

			labels.add("NREVENTS");
			labels.add("NREXTERNALMOVES");

			labels.add("SIZE");
			labels.add("TIES");
			labels.add("EGO-BETWEENNESS");
			labels.add("MEAN_BETWEENNESS");
			labels.add("MAX_BETWEENNESS");
			labels.add("ECCENTRICITY");
			labels.add("DENSITY");
			labels.add("DENSITY_NOLOOPS");
			labels.add("MEANDEGREE");
			labels.add("MEANDEGREE_NOLOOPS");
			labels.add("MEANDEGREE_NOLOOPS_NORM");
			labels.add("NRCOMPONENTS");
			labels.add("NRISOLATES");
			labels.add("MAXCOMPONENT");
			labels.add("NRCOMPONENTS_NORM");
			labels.add("NRISOLATES_NORM");
			labels.add("MAXCOMPONENT_NORM");
			labels.add("CONCENTRATION");
			labels.add("BROKERAGE");
			labels.add("EFFICIENT_SIZE");
			labels.add("EFFICIENCY");
			labels.add("NETWORK_RELATIONS");
			labels.add("CONNECTED_NETWORK_RELATIONS");
			
			labels.add("CENTRAL_ALTERS");
			labels.add("CENTRAL_RELATIONS");

			labels.add("SDENSITY_PARENT-CHILD");
			labels.add("SDENSITY_SPOUSE");
			labels.add("SDENSITY_SIBLING");
			labels.add("SDENSITY_RELATIVE");
			labels.add("SDENSITY_AFFINE");
			labels.add("SDENSITY_EMPLOYMENT");
			labels.add("SDENSITY_RENT");

			break;
			
		case PARCOURSNETWORKS:
			
			labels.add("NREVENTS");
			labels.add("NREXTERNALMOVES");
			
			for (RelationAttributeType mainEventType : mainRelationAttributeTypes){
				labels.add("SIZE_"+mainEventType);
				labels.add("TIES_"+mainEventType);
				labels.add("EGO-BETWEENNESS_"+mainEventType);
				labels.add("MEAN_BETWEENNESS_"+mainEventType);
				labels.add("MAX_BETWEENNESS_"+mainEventType);
				labels.add("ECCENTRICITY_"+mainEventType);
				labels.add("DENSITY_"+mainEventType);
				labels.add("DENSITY_NOLOOPS_"+mainEventType);
				labels.add("MEANDEGREE_"+mainEventType);
				labels.add("MEANDEGREE_NOLOOPS_"+mainEventType);
				labels.add("MEANDEGREE_NOLOOPS_NORM_"+mainEventType);
				labels.add("NRCOMPONENTS_"+mainEventType);
				labels.add("NRISOLATES_"+mainEventType);
				labels.add("MAXCOMPONENT_"+mainEventType);
				labels.add("NRCOMPONENTS_NORM_"+mainEventType);
				labels.add("NRISOLATES_NORM_"+mainEventType);
				labels.add("MAXCOMPONENT_NORM_"+mainEventType);
				labels.add("CONCENTRATION_"+mainEventType);
				labels.add("BROKERAGE_"+mainEventType);
				labels.add("EFFICIENT_SIZE_"+mainEventType);
				labels.add("EFFICIENCY_"+mainEventType);
				labels.add("NETWORK_RELATIONS_"+mainEventType);
				labels.add("CONNECTED_NETWORK_RELATIONS_"+mainEventType);
				labels.add("SIMILARITY_"+mainEventType);
			}

			break;
			
		case PARCOURS:
			labels.add("NREVENTS");
			labels.add("NREXTERNALMOVES");
			
			for (RelationAttributeType relationAttributeType : mainRelationAttributeTypes){
				labels.add("PROFILE_"+relationAttributeType);
				labels.add("SUPPORT_"+relationAttributeType);
				if (relationAttributeType == RelationAttributeType.PLACE){
					labels.add("CENTERS_"+relationAttributeType);
					labels.add("NRINTERNALMOVES_"+relationAttributeType);
					labels.add("NRDIRECTRETURNS_"+relationAttributeType);
					labels.add("NRCYCLES_"+relationAttributeType);
					labels.add("NRDIRECTRETURNS_NORM_"+relationAttributeType);
					labels.add("NRCYCLES_NORM_"+relationAttributeType);
				}
			}

			break;
		}

	}

	
	public SequenceCensusCriteria (){
		
		egoRoleName = "MIG";
		alterRoleName = "ALL";
		pattern = "3, 1";
		minAge = 0;
		maxAge = 1000;
		threshold = 0.01;
		level = GeoLevel.TOWN;
		splitLabel = "GENDER";
		eventModelName = "Migevent";
		relationModelNames = new ArrayList<String>();
		eventRoleNames = new ArrayList<String>();
		labels = new ArrayList<String>();
		networkTitles = new ArrayList<String>();
		eventRoleNames.add("ALL");
		relationAttributeTypes = new ArrayList<RelationAttributeType>();
		mainRelationAttributeTypes = new ArrayList<RelationAttributeType>();
		relationAttributeType = RelationAttributeType.HOSTMIG;
		geography = Geography.getInstance();
		partitionLabels = new HashMap<String,List<String>>();
		for (String networkTitle : networkTitles){
			partitionLabels.put(networkTitle,new ArrayList<String>());
		}

	}
	
	public void addNetworkTitle (String networkTitle){
		
		networkTitles.add(networkTitle);
		partitionLabels.put(networkTitle,new ArrayList<String>());
		
	}

	public void setType(RelationAttributeType type) {
		this.relationAttributeType = type;
	}

	public String getEgoRoleName() {
		return egoRoleName;
	}

	public void setEgoRoleName(String egoRoleName) {
		this.egoRoleName = egoRoleName;
	}

	public int getMaxAge() {
		return maxAge;
	}

	public void setMaxAge(int maxAge) {
		this.maxAge = maxAge;
	}

	public int getMinAge() {
		return minAge;
	}

	public void setMinAge(int minAge) {
		this.minAge = minAge;
	}

	public double getThreshold() {
		return threshold;
	}
	public void setThreshold(double threshold) {
		this.threshold = threshold;
	}
	public String getRelationModelName() {
		return eventModelName;
	}

	public void setRelationModelName(String relationModelName) {
		this.eventModelName = relationModelName;
	}

	public GeoLevel getLevel() {
		return level;
	}
	public List<String> getLabels() {
		return labels;
	}

	public void setLabels(List<String> labels) {
		this.labels = labels;
	}

	public void setLevel(GeoLevel level) {
		this.level = level;
	}
	public String getPartitionLabel() {
		return splitLabel;
	}
	public void setPartitionLabel(String partitionLabel) {
		this.splitLabel = partitionLabel;
	}

	public List<String> getRelationModelNames() {
		return relationModelNames;
	}

	public String getAlterRoleName() {
		return alterRoleName;
	}

	public void setAlterRoleName(String alterRoleName) {
		this.alterRoleName = alterRoleName;
	}

	public Geography getGeography() {
		return geography;
	}

	public List<RelationAttributeType> getMainRelationAttributeTypes() {
		return mainRelationAttributeTypes;
	}

	public void setMainEventType(List<RelationAttributeType> mainEventTypes) {
		this.mainRelationAttributeTypes = mainEventTypes;
	}

	public void setGeography(Geography geography) {
		this.geography = geography;
	}

	public void setRelationModelNames(List<String> relationModelNames) {
		this.relationModelNames = relationModelNames;
	}

	public List<String> getRoleNames() {
		return eventRoleNames;
	}

	public void setRoleNames(List<String> alterRoleNames) {
		this.eventRoleNames = alterRoleNames;
	}

	public List<RelationAttributeType> getTypes() {
		return relationAttributeTypes;
	}

	public void setTypes(List<RelationAttributeType> types) {
		this.relationAttributeTypes = types;
	}

	public boolean isWithEgoNetworks() {
		return withEgoNetworks;
	}

	public void setWithEgoNetworks(boolean withEgoNetworks) {
		this.withEgoNetworks = withEgoNetworks;
	}

	public boolean isWithParcoursNetworks() {
		return withParcoursNetworks;
	}

	public void setWithParcoursNetworks(boolean withParcoursNetworks) {
		this.withParcoursNetworks = withParcoursNetworks;
	}

	public List<String> getNetworkTitles() {
		return networkTitles;
	}

	public String getAlterAttributeLabel() {
		return alterAttributeLabel;
	}

	public void setAlterAttributeLabel(String alterAttributeLabel) {
		this.alterAttributeLabel = alterAttributeLabel;
	}

	public String getAlterAttributeValue() {
		return alterAttributeValue;
	}

	public void setAlterAttributeValue(String alterAttributeValue) {
		this.alterAttributeValue = alterAttributeValue;
	}

	public Map<String, List<String>> getPartitionLabels() {
		return partitionLabels;
	}

	public List<String> getPlaceNames() {
		return placeNames;
	}
	
	public void setPlaceNames(List<String> placeNames) {
		this.placeNames = placeNames;
	}

	public String getPattern() {
		return pattern;
	}

	public void setPattern(String pattern) {
		this.pattern = pattern;
	}

	public String getDefaultReferentRoleName() {
		return defaultReferentRoleName;
	}

	public void setDefaultReferentRoleName(String defaultReferentRoleName) {
		this.defaultReferentRoleName = defaultReferentRoleName;
	}

	public void setCensusType(CensusType censusType) {
		this.censusType = censusType;
	}

	public CensusType getCensusType() {
		return censusType;
	}

	public List<RelationAttributeType> getEventTypes() {
		return relationAttributeTypes;
	}

	public void setEventTypes(List<RelationAttributeType> relationAttributeTypes) {
		this.relationAttributeTypes = relationAttributeTypes;
	}

	public ExpansionMode getExpansionMode() {
		return expansionMode;
	}

	public void setExpansionMode(ExpansionMode expansionMode) {
		this.expansionMode = expansionMode;
	}

	public FiliationType getFiliationType() {
		return filiationType;
	}

	public void setFiliationType(FiliationType filiationType) {
		this.filiationType = filiationType;
	}
	
	
	
	
	


}
